/* der_lib.c */
/* Copyright (C) 1995 Eric Young (eay@mincom.oz.au).
 * All rights reserved.
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.
 * See the COPYRIGHT file in the SSLeay distribution for more details.
 */

#include <stdio.h>
#include "crypto.h"
#include "der.h"
#include "x509.h"

char *DER_version="\0DER part of SSLeay v 0.4.4 17/07/95";

#ifdef PROTO
static unsigned long der_get_length(unsigned char **pp,int *inf);
static void der_put_length(unsigned char **pp, int length);
static void der_print_info(FILE *file, int tag, int class);
static void der_parse2(FILE *fp, unsigned char **pp, unsigned int length,
	int offset, int depth);
#else
static unsigned long der_get_length();
static void der_put_length();
static void der_print_info();
static void der_parse2();
#endif

int DER_get_object(pp, plength, ptag, pclass)
unsigned char **pp;
unsigned long *plength;
int *ptag;
int *pclass;
	{
	int i,ret;
	unsigned long l;
	unsigned char *p= *pp;
	int tag,class,inf;

	ret=(*p&DER_CONSTRUCTED);
	class=(*p&DER_PRIVATE);
	i= *p&DER_PRIMATIVE_TAG;
	if (i == DER_PRIMATIVE_TAG)
		{		/* high-tag */
		p++;
		l=0;
		while (*p&0x80)
			{
			l<<=7L;
			l|= *(p++)&0x7f;
			}
		l<<=7L;
		l|= *(p++)&0x7f;
		tag=l;
		}
	else
		{ /* constructed */
		tag=i;
		p++;
		}
	*pp=p;
	*ptag=tag;
	*pclass=class;
	*plength=der_get_length(pp,&inf);
	return(ret+inf);
	}


int DER_object_size(constructed, length, tag)
int constructed;
int length;
int tag;
	{
	int ret;

	ret=length;
	ret++;
	if (tag >= 31)
		{
		while (tag > 0)
			{
			tag>>=7;
			ret++;
			}
		}
	if ((length == 0) && constructed)
		ret+=2;
	ret++;
	if (length > 127)
		{
		while (length > 0)
			{
			length>>=8;
			ret++;
			}
		}
	return(ret);
	}

/* class 0 is constructed */
void DER_put_object(pp, constructed, length, tag, class)
unsigned char **pp;
int constructed;
int length;
int tag;
int class;
	{
	unsigned char *p= *pp;
	int i;

	i=(constructed)?DER_CONSTRUCTED:0;
	i|=(class&DER_PRIVATE);
	if (tag < 31)
		*(p++)=i|(tag&DER_PRIMATIVE_TAG);
	else
		{
		*(p++)=i|DER_PRIMATIVE_TAG;
		while (tag > 0x3f)
			{
			*(p++)=(tag&0x3f)|0x80;
			tag>>=7;
			}
		*(p++)=(tag&0x3f);
		}
	if (constructed && (length == 0))
		*(p++)=0x80; /* der_put_length would output 0 instead */
	else
		der_put_length(&p,length);
	*pp=p;
	}

static void der_put_length(pp, length)
unsigned char **pp;
int length;
	{
	unsigned char *p= *pp;
	int i,l;
	if (length < 0x7f)
		*(p++)=(unsigned char)length;
	else
		{
		l=length;
		for (i=0; l > 0; i++)
			l>>=8;
		*(p++)=i|0x80;
		l=i;
		while (i-- > 0)
			{
			p[i]=length&0xff;
			length>>=8;
			}
		p+=l;
		}
	*pp=p;
	}

static unsigned long der_get_length(pp,inf)
unsigned char **pp;
int *inf;
	{
	unsigned char *p= *pp;
	unsigned long ret=0;
	int i;

	if (*p == 0x80)
		{
		ret=0;
		*inf=1;
		p++;
		}
	else
		{
		*inf=0;
		i= *p&0x7f;
		if (*(p++) & 0x80)
			{
			while (i-- > 0)
				{
				ret<<=8L;
				ret|= *(p++);
				}
			}
		else
			ret=i;
		}
	*pp=p;
	return(ret);
	}

int i2D_DER_BIT_STRING(a, pp)
DER_BIT_STRING *a;
unsigned char **pp;
	{
	int ret,i,j,r,bits;
	unsigned char *p;

	/* our bit strings are always a multiple of 8 :-) */
	bits=0;
	ret=1+a->length;
	r=DER_object_size(0,ret,DER_BIT_STRING_IDENTIFIER);
	if (pp == NULL) return(r);
	p= *pp;

	DER_put_object(&p,0,ret,DER_BIT_STRING_IDENTIFIER,DER_UNIVERSAL);
	if (bits == 0)
		j=0;
	else	j=8-bits;
	*(p++)=(unsigned char)j;
	for (i=0; i<a->length; i++)
		*(p++)=a->data[i];
	if (a->length > 0) p[-1]&=(0xff<<j);
	*pp= p;
	return(r);
	}

int i2D_OCTET_STRING(a, pp)
DER_BIT_STRING *a;
unsigned char **pp;
	{
	int ret,i,r;
	unsigned char *p;

	ret=a->length;
	r=DER_object_size(0,ret,DER_OCTET_STRING);
	if (pp == NULL) return(r);
	p= *pp;

	DER_put_object(&p,0,ret,DER_OCTET_STRING,DER_UNIVERSAL);
	for (i=0; i<a->length; i++)
		*(p++)=a->data[i];
	*pp= p;
	return(r);
	}

int i2D_TAG(a,c,pp)
int a,c;
unsigned char **pp;
	{
	unsigned char *p;

	if (pp == NULL) return(2);
	p= *pp;

	DER_put_object(&p,1,0,a,c);
	*pp=p;
	return(2);
	}

int i2D_UTCTime(a, pp)
char *a;
unsigned char **pp;
	{
	int ret,r;
	unsigned char *p;

	ret=strlen((char *)a);
	r=DER_object_size(0,ret,DER_UTCTIME);
	if (pp == NULL) return(r);
	p= *pp;

	DER_put_object(&p,0,ret,DER_UTCTIME,DER_UNIVERSAL);
	memcpy(p,a,(unsigned int)ret);
	p+=ret;
	*pp=p;
	return(r);
	}

int i2D_DER_OBJECT(a, pp)
DER_OBJECT *a;
unsigned char **pp;
	{
	int i,j,r,n,ret,f;
	unsigned long l;
	unsigned char *p;

	ret=0;
	if (a->num >= 2) ret=1;
	for (i=2; i<a->num; i++)
		{
		l=a->values[i];
		do	{
			ret++;
			l>>=7L;
			} while (l);
		}
	r=DER_object_size(0,ret,DER_OBJECT_IDENTIFIER);
	if (pp == NULL) return(r);

	p= *pp;
	DER_put_object(&p,0,ret,DER_OBJECT_IDENTIFIER,DER_UNIVERSAL);

	if (a->num == 0)
		*(p++)=(unsigned char)(a->values[0]*40);
	else if (a->num >= 2)
		{
		*(p++)=(unsigned char)(a->values[0]*40+((a->values[1])%40));
		for (i=2; i<a->num; i++)
			{
			l=a->values[i];
			j=0;
			do	{
				j++;
				l>>=7L;
				} while (l);
			l=a->values[i];
			n=j; /* n is number of bytes */
			f=1;
			for (j--; j >= 0; j--)
				{
				p[j]=(unsigned char)(l&0x7f);
				l>>=7L;
				if (f)
					f=0;
				else
					p[j]|=0x80;
				}
			p+=n;
			}
		}
	*pp=p;
	return(r);
	}

int i2D_INTEGER(a, pp)
DER_BIT_STRING *a;
unsigned char **pp;
	{
	int pad=0,ret,r;
	unsigned char *p;

	ret=a->length;
	if ((ret > 0) && (a->data[0]&0x80))
		pad=1;
	/* add one if top bit is set - we are only saving +ve numbers */
	ret+=pad;
	r=DER_object_size(0,ret,DER_INTEGER);
	if (pp == NULL) return(r);
	p= *pp;

	DER_put_object(&p,0,ret,DER_INTEGER,DER_UNIVERSAL);
	if (pad) *(p++)=0;
	memcpy(p,a->data,(unsigned int)a->length);
	p+=a->length;
	*pp=p;
	return(r);
	}

void i2f_DER_BIT_STRING(fp, a)
FILE *fp;
DER_BIT_STRING *a;
	{
	int i;
	static char *h="0123456789ABCDEF";

	if (a->length == 0)
		Fprintf(fp,"= NULL");
	else
		{
		Fprintf(fp,"%c ",(a->length > 35)?'=':'-');
		for (i=0; i<a->length; i++)
			{
			if ((i != 0) && (i%35 == 0))
				{
				if (i+35 >= a->length)
					Fprintf(fp,"\n- ");
				else
					Fprintf(fp,"\n= ");
				}
			Fputc((h[((unsigned char)a->data[i]>>4)&0x0f]),fp);
			Fputc((h[((unsigned char)a->data[i]   )&0x0f]),fp);
			}
		}
	Fprintf(fp,"\n");
	}

void i2f_DER_OBJECT(fp, a)
FILE *fp;
DER_OBJECT *a;
	{
	int i;
	char *s;

	Fprintf(fp,"= ");

	s=X509_nid2ln(X509_obj2nid(a));
	if (s != NULL)
		Fprintf(fp,"%s\n",s);
	else
		{
		for (i=0; i<a->num; i++)
			Fprintf(fp,"%ld%s",a->values[i],
				((i+1) == a->num)?"":" ");
		Fprintf(fp,"\n");
		}
	}

int D2i_INTEGER(bs, pp)
DER_BIT_STRING *bs;
unsigned char **pp;
	{
	unsigned char *p,*s;
	unsigned long len;
	int tag,class;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_INTEGER)
		{
		DERerr(DER_F_D2I_INTEGER,DER_R_EXPECTING_AN_INTEGER);
		return(0);
		}
	if (len != 0)
		{
		s=(unsigned char *)malloc(len);
		if (s == NULL)
			{
			DERerr(DER_F_D2I_INTEGER,ERR_R_MALLOC_FAILURE);
			return(0);
			}
		memcpy(s,p,len);
		p+=len;
		}
	else
		s=NULL;

	bs->length=len;
	if (bs->data != NULL) free(bs->data);
	bs->data=s;
	*pp=p;
	return(1);
	}

int D2i_DER_OBJECT(o, pp)
DER_OBJECT *o;
unsigned char **pp;
	{
	unsigned char *p;
	unsigned long len,ret;
	int tag,class;
	int i,n;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_OBJECT_IDENTIFIER)
		{
		DERerr(DER_F_D2I_DER_OBJECT,DER_R_EXPECTING_AN_OBJECT);
		return(0);
		}
	n=0;
	if (len >= 1) n=2;
	for (i=1; i<len; i++)
		if (!(p[i]&0x80)) n++;
	o->num=n;
	if (o->values != NULL) free(o->values);
	o->values=(unsigned long *)malloc(sizeof(unsigned long)*n);
	if (o->values == NULL)
		{
		DERerr(DER_F_D2I_DER_OBJECT,ERR_R_MALLOC_FAILURE);
		return(0);
		}
	if (len >= 1)
		{
		o->values[0]=(unsigned long)p[0]/40;
		o->values[1]=(unsigned long)p[0]%40;
		}
	p++;
	
	for (i=2; i<n; i++)
		{
		ret=0;
		do	{
			ret=(ret<<7L)|(*p&0x7f);
			} while (*(p++) & 0x80);
		o->values[i]=ret;
		}
	*pp=p;
	return(1);
	}

int D2i_PRINTABLESTRING(sp, pp)
unsigned char **sp;
unsigned char **pp;
	{
	unsigned char *p,*s;
	unsigned long len;
	int tag,class;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_PRINTABLESTRING)
		{
		DERerr(DER_F_D2I_PRINTABLESTRING,
			DER_R_EXPECTING_A_PRINTABLESTRING);
		return(0);
		}
	if (*sp != NULL) free(*sp);
	s=(unsigned char *)malloc(len+1);
	if (s == NULL)
		{
		DERerr(DER_F_D2I_PRINTABLESTRING,ERR_R_MALLOC_FAILURE);
		return(0);
		}
	memcpy(s,p,len);
	p+=len;

	s[len]='\0';
	*sp=s;
	*pp=p;
	return(1);
	}

int D2i_DER_BIT_STRING(bp, pp)
DER_BIT_STRING *bp;
unsigned char **pp;
	{
	unsigned char *p,*s;
	unsigned long len;
	int tag,class;
	int i;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_BIT_STRING_IDENTIFIER)
		{
		DERerr(DER_F_D2I_DER_BIT_STRING,DER_R_EXPECTING_A_BIT_STRING);
		return(0);
		}
	i= *(p++);
	if (len-- > 1) /* using one because of the bits left byte */
		{
		s=(unsigned char *)malloc(len);
		if (s == NULL)
			{
			DERerr(DER_F_D2I_DER_BIT_STRING,ERR_R_MALLOC_FAILURE);
			return(0);
			}
		memcpy(s,p,len);
		s[len-1]&=(0xff<<i);
		p+=len;
		}
	else
		s=NULL;

	bp->length=len;
	if (bp->data != NULL) free(bp->data);
	bp->data=s;
	*pp=p;
	return(1);
	}

int D2i_OCTET_STRING(bp, pp)
DER_BIT_STRING *bp;
unsigned char **pp;
	{
	unsigned char *p,*s;
	unsigned long len;
	int tag,class;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_OCTET_STRING)
		{
		DERerr(DER_F_D2I_DER_OCTET_STRING,
			DER_R_EXPECTING_AN_OCTET_STRING);
		return(0);
		}
	if (len != 0)
		{
		s=(unsigned char *)malloc(len);
		if (s == NULL)
			{
			DERerr(DER_F_D2I_DER_OCTET_STRING,ERR_R_MALLOC_FAILURE);
			return(0);
			}
		memcpy(s,p,len);
		p+=len;
		}
	else
		s=NULL;

	bp->length=len;
	if (bp->data != NULL) free(bp->data);
	bp->data=s;
	*pp=p;
	return(1);
	}

int D2i_UTCTime(sp, pp)
unsigned char **sp;
unsigned char **pp;
	{
	unsigned char *p,*s;
	unsigned long len;
	int tag,class;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_UTCTIME)
		{
		DERerr(DER_F_D2I_UTCTIME,DER_R_EXPECTING_AN_UTCTIME);
		return(0);
		}
	if (*sp != NULL) free(*sp);
	s=(unsigned char *)malloc(len+1);
	if (s == NULL)
		{
		DERerr(DER_F_D2I_UTCTIME,ERR_R_MALLOC_FAILURE);
		return(0);
		}
	memcpy(s,p,len);
	p+=len;

	s[len]='\0';
	*sp=s;
	*pp=p;
	return(1);
	}

static void der_print_info(file, tag, class)
FILE *file;
int tag;
int class;
	{
	static char *fmt="%-18s";

	if (class & DER_CONSTRUCTED)
		Fprintf(file,"cons: ");
	else
		Fprintf(file,"prim: ");
	if ((class & DER_PRIVATE) == DER_PRIVATE)
		{
		Fprintf(file,"priv: %02X",tag);
		return;
		}
	else if ((class & DER_CONTEXT_SPECIFIC) == DER_CONTEXT_SPECIFIC)
		{
		Fprintf(file,"cont: %02X",tag);
		return;
		}
	else if ((class & DER_APPLICATION) == DER_APPLICATION)
		{
		Fprintf(file,"appl: %02X",tag);
		return;
		}
	else if ((class & DER_UNIVERSAL) /* lint complained */
		/* == DER_UNIVERSAL*/)
		Fprintf(file,"univ: ");

	if ((tag == DER_EOC) && (class == 0))
		Fprintf(file,fmt,"EOC");
	else if (tag == DER_INTEGER)
		Fprintf(file,fmt,"INTEGER");
	else if (tag == DER_BIT_STRING_IDENTIFIER)
		Fprintf(file,fmt,"DER_BIT_STRING");
	else if (tag == DER_OCTET_STRING)
		Fprintf(file,fmt,"OCTET_STRING");
	else if (tag == DER_NULL)
		Fprintf(file,fmt,"NULL");
	else if (tag == DER_OBJECT_IDENTIFIER)
		Fprintf(file,fmt,"OBJECT_IDENTIFIER");
	else if (tag == DER_SEQUENCE)
		Fprintf(file,fmt,"SEQUENCE");
	else if (tag == DER_SET)
		Fprintf(file,fmt,"SET");
	else if (tag == DER_PRINTABLESTRING)
		Fprintf(file,fmt,"PRINTABLESTRING");
	else if (tag == DER_T61STRING)
		Fprintf(file,fmt,"T61STRING");
	else if (tag == DER_IA5STRING)
		Fprintf(file,fmt,"IA5STRING");
	else if (tag == DER_UTCTIME)
		Fprintf(file,fmt,"UTCTIME");
	else	Fprintf(file,fmt,"unknown");
	}

void DER_parse(fp, pp, len)
FILE *fp;
unsigned char *pp;
unsigned int len;
	{
	der_parse2(fp,&pp,len,0,0);
	}

static void der_parse2(fp, pp, length, offset, depth)
FILE *fp;
unsigned char **pp;
unsigned int length;
int offset;
int depth;
	{
	unsigned char *p,*ep,*tot,*op,*opp;
	unsigned long len;
	int tag,class;
	int nl,i,j;

	p= *pp;
	tot=p+length;
	op=p-1;
	while ((p < tot) && (op < p))
		{
		op=p;
		j=DER_get_object(&p,&len,&tag,&class);
#ifdef LINT
		j=j;
#endif
		/* if j == 21 it is a constructed indefinite length object */
		Fprintf(fp,"%5ld:",(long)offset+(long)(op- *pp));
		Fprintf(fp,"d=%d hl=%ld l=%3ld ",depth,(long)(p-op),len);
		der_print_info(fp,tag,class);
		if (class & DER_CONSTRUCTED)
			{
			ep=p+len;
			Fputc('\n',fp);
			if (len > length) break; /* error */
			while (p < ep)
				der_parse2(fp,&p,(unsigned int)len,
					offset+(p - *pp),depth+1);
			}
		else
			{
			nl=0;
			if (	(tag == DER_PRINTABLESTRING) ||
				(tag == DER_T61STRING) ||
				(tag == DER_IA5STRING) ||
				(tag == DER_UTCTIME))
				{
				Fputc(':',fp);
				for (i=0; i<len; i++)
					Fputc(p[i],fp);
				}
			if (tag == DER_OBJECT_IDENTIFIER)
				{
				DER_OBJECT *o;
				char *s;

				opp=op;
				o=DER_OBJECT_new();
				i=D2i_DER_OBJECT(o,&opp);
				s=(char *)X509_nid2ln(X509_obj2nid(o));
				if (s == NULL)
					{
					i2f_DER_OBJECT(fp,o);
					DER_OBJECT_free(o);
					nl=1;
					}
				else
					Fprintf(fp,":%s",
						X509_nid2ln(X509_obj2nid(o)));
				DER_OBJECT_free(o);
				}
			if (!nl) Fputc('\n',fp);
			p+=len;
			}
		}
	*pp=p;
	}
