/* crl.c */
/* Copyright (C) 1995 Eric Young (eay@mincom.oz.au).
 * All rights reserved.
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.
 * See the COPYRIGHT file in the SSLeay distribution for more details.
 */

#include <stdio.h>
#include "crypto.h"
#include "der.h"
#include "bn.h"
#include "x509.h"
#include "pem.h"

#define	POSTFIX	".revoke"

#define FORMAT_UNDEF	0
#define FORMAT_DER	1
#define FORMAT_TEXT	2
#define FORMAT_PEM	3

char *usage[]={
"usage: crl args\n",
"\n",
" -inform arg     - input format - default PEM (one of DER, TXT or PEM)\n",
" -outform arg    - output format - default PEM\n",
" -in arg         - input file - default stdin\n",
" -out arg        - output file - default stdout\n",
" -hash           - print hash value\n",
" -issuer         - print issuer DN\n",
" -lastupdate     - lastUpdate field\n",
" -nextupdate     - nextUpdate field\n",
" -noout          - CRL output\n",
NULL
};

#ifdef PROTO
static X509_CRL *load_crl(char *file, int format);
static int str2fmt (char *s);
#else
static X509_CRL *load_crl();
static int str2fmt ();
#endif

int main(argc, argv)
int argc;
char **argv;
	{
	X509_CRL *x;
	int i,num,badops=0;
	FILE *out=NULL;
	int informat,outformat;
	char *infile=NULL,*outfile=NULL;
	char *str=NULL;
	int hash=0,issuer=0,lastupdate=0,nextupdate=0,noout=0;
	char **pp;

	informat=FORMAT_PEM;
	outformat=FORMAT_PEM;

	argc--;
	argv++;
	num=0;
	while (argc >= 1)
		{
		if 	(strcmp(*argv,"-inform") == 0)
			{
			if (argc-- < 1) goto bad;
			informat=str2fmt(*(++argv));
			}
		else if (strcmp(*argv,"-outform") == 0)
			{
			if (argc-- < 1) goto bad;
			outformat=str2fmt(*(++argv));
			}
		else if (strcmp(*argv,"-in") == 0)
			{
			if (argc-- < 1) goto bad;
			infile= *(++argv);
			}
		else if (strcmp(*argv,"-out") == 0)
			{
			if (argc-- < 1) goto bad;
			outfile= *(++argv);
			}
		else if (strcmp(*argv,"-hash") == 0)
			hash= ++num;
		else if (strcmp(*argv,"-issuer") == 0)
			issuer= ++num;
		else if (strcmp(*argv,"-lastupdate") == 0)
			lastupdate= ++num;
		else if (strcmp(*argv,"-nextupdate") == 0)
			nextupdate= ++num;
		else if (strcmp(*argv,"-noout") == 0)
			noout= ++num;
		else
			{
			fprintf(stderr,"unknown option %s\n",*argv);
			badops=1;
			break;
			}
		argc--;
		argv++;
		}

	if (badops)
		{
bad:
		for (pp=usage; (*pp != NULL); pp++)
			fprintf(stderr,*pp);
		exit(1);
		}

	x=load_crl(infile,informat);

	if (num)
		{
		for (i=1; i<=num; i++)
			{
			if (issuer == i)
				{
				str=X509_oneline_X509_NAME(x->crl->issuer);
				if (str == NULL)
					{
					fprintf(stderr,"unable to get issuer Name from CRL\n");
					ERR_print_error_stack(stderr);
					exit(1);
					}
				fprintf(stdout,"issuer= %s\n",str);
				free(str);
				}

			if (hash == i)
				{
				fprintf(stdout,"%08lx\n",
					X509_name_hash(x->crl->issuer));
				}
			if (lastupdate == i)
				{
				fprintf(stdout,"lastUpdate=%s\n",
					x->crl->lastUpdate);
				}
			if (nextupdate == i)
				{
				fprintf(stdout,"nextUpdate=%s\n",
					x->crl->nextUpdate);
				}
			}
		}

	if (noout) goto end;

	if (outfile == NULL)
		out=stdout;
	else
		{
		out=fopen(outfile,"w");
		if (out == NULL) { perror(outfile); exit(1); }
		}

	if 	(outformat == FORMAT_DER)
		i=i2D_X509_CRL_fp(out,x);
	else if (outformat == FORMAT_TEXT)
		{ i2f_X509_CRL(out,x); i=1; }
	else if (outformat == FORMAT_PEM)
		i=PEM_write_X509_CRL(out,x);
	else	{
		fprintf(stderr,"bad output format specified for outfile\n");
		exit(1);
		}
	if (!i) { fprintf(stderr,"unable to write CRL\n"); exit(1); }
	fclose(out);
end:
	exit(0);
	return(0);
	}

static int str2fmt(s)
char *s;
	{
	if 	((*s == 'D') || (*s == 'd'))
		return(FORMAT_DER);
	else if ((*s == 'T') || (*s == 't'))
		return(FORMAT_TEXT);
	else if ((*s == 'P') || (*s == 'p'))
		return(FORMAT_PEM);
	else
		return(FORMAT_UNDEF);
	}

static X509_CRL *load_crl(file, format)
char *file;
int format;
	{
	X509_CRL *x;
	FILE *crl;
	int i;

	if (file == NULL)
		crl=stdin;
	else
		{
		crl=fopen(file,"r");
		if (crl == NULL) { perror(file); exit(1); }
		}
	x=X509_CRL_new();
	if (x == NULL)
		{
		ERR_print_error_stack(stderr);
		exit(1);
		}
	if 	(format == FORMAT_DER)
		i=D2i_X509_CRL_fp(crl,x);
	else if (format == FORMAT_TEXT)
		i=f2i_X509_CRL(crl,x);
	else if (format == FORMAT_PEM)
		i=PEM_read_X509_CRL(crl,x);
	else	{
		fprintf(stderr,"bad input format specified for input crl\n");
		exit(1);
		}
	if (!i)
		{
		fprintf(stderr,"unable to load CRL\n");
		ERR_print_error_stack(stderr);
		exit(1);
		}
	fclose(crl);
	return(x);
	}

