/* bn.c */
/* Copyright (C) 1995 Eric Young (eay@mincom.oz.au).
 * All rights reserved.
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.
 * See the COPYRIGHT file in the SSLeay distribution for more details.
 */

#include <stdio.h>
#include "crypto.h"
#include "bn.h"
#include "X509.h"

#ifdef PROTO
static void bn_SUB(BIGNUM *r, BIGNUM *a, BIGNUM *b);
static void bn_fix_top(BIGNUM *a);
#else
static void bn_SUB();
static void bn_fix_top();
#endif

int bn_num_bits(a)
BIGNUM *a;
	{
	int i;
	BN_ULONG l;
	static char bits[256]={
		0,1,2,2,3,3,3,3,4,4,4,4,4,4,4,4,
		5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
		6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
		6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
		7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
		7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
		7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
		7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
		8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
		8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
		8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
		8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
		8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
		8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
		8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
		8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
		};

	l=a->d[a->top-1];
	i=(a->top-1)*BITS2;
	if (l == 0)
		{
		fprintf(stderr,"BAD TOP VALUE\n");
		abort();
		}

#ifdef SIXTY_FOUR_BIT
	if (l & 0xffffffff00000000)
		{
		if (l & 0xffff000000000000)
			{
			if (l & 0xff00000000000000)
				{
				return(i+bits[l>>56]+56);
				}
			else	return(i+bits[l>>48]+48);
			}
		else
			{
			if (l & 0x0000ff0000000000)
				{
				return(i+bits[l>>40]+40);
				}
			else	return(i+bits[l>>32]+32);
			}
		}
	else
#endif
		{
#if defined(THIRTY_TWO_BIT) || defined(SIXTY_FOUR_BIT)
		if (l & 0xffff0000)
			{
			if (l & 0xff000000)
				return(i+bits[l>>24]+24);
			else	return(i+bits[l>>16]+16);
			}
		else
#endif
			{
			if (l & 0xff00)
				return(i+bits[l>>8]+8);
			else	return(i+bits[l   ]  );
			}
		}
	}

void bn_free(a)
BIGNUM *a;
	{
	if (a == NULL) return;
	if (a->d != NULL) free(a->d);
	free(a);
	}

static void bn_fix_top(a)
BIGNUM *a;
	{
	int i;
	BN_ULONG *l;

	l=a->d;
	for (i=a->top-1; i>0; i--)
		if (l[i]) break;
	a->top=i+1;
	}

BIGNUM *bn_new()
	{
	BIGNUM *ret;
	register int i;
	BN_ULONG *p;

	ret=(BIGNUM *)malloc(sizeof(BIGNUM));
	if (ret == NULL)
		{ RSA_errno=RSA_ERR_OUT_OF_MEM; return(NULL); }
	ret->top=0;
	ret->neg=0;
	ret->max=(DEFAULT_BITS/BITS2);
	p=(BN_ULONG *)malloc(sizeof(BN_ULONG)*(ret->max+1));
	if (p == NULL)
		{ RSA_errno=RSA_ERR_OUT_OF_MEM; return(NULL); }
	ret->d=p;

	for (i=0; i<ret->max+1; i++)
		p[i]=0;
	return(ret);
	}

BIGNUM *bn_expand(b, bits)
BIGNUM *b;
int bits;
	{
	BN_ULONG *p;
	register int i,n;

	if (b == NULL)
		{ RSA_errno=RSA_ERR_NULL_ARG_TO_BN_EXPAND; return(NULL); }
	while (bits > b->max*BITS2)
		{
		n=((bits+BITS2-1)/BITS2)*2;
		p=b->d=(BN_ULONG *)realloc(b->d,sizeof(BN_ULONG)*n+1);
		if (p == NULL)
			{ RSA_errno=RSA_ERR_OUT_OF_MEM; return(NULL); }
		for (i=b->max; i<(n+1); i++)
			p[i]=0;
		b->max=n;
		}
	return(b);
	}

BIGNUM *bn_dup(a)
BIGNUM *a;
	{
	BIGNUM *r;

	r=bn_new();
	if (r == NULL) return(NULL);
	return((BIGNUM *)bn_copy(r,a));
	}

BIGNUM *bn_copy(a, b)
BIGNUM *a;
BIGNUM *b;
	{
	int i;

	if (bn_expand(a,b->top*BITS2) == NULL) return(NULL);
	for (i=0; i<b->top; i++)
		a->d[i]=b->d[i];
	for (;i<a->max; i++)
		a->d[i]=0;
	a->top=b->top;
	a->neg=b->neg;
	return(a);
	}

void bn_zero(a)
BIGNUM *a;
	{
	int i;

	for (i=0; i<a->max; i++)
		a->d[i]=0;
	a->top=0;
	a->neg=0;
	}

int bn_one(a)
BIGNUM *a;
	{
	int i;

	if (bn_expand(a,1) == NULL) return(0);
	for (i=0; i<a->max; i++)
		a->d[i]=0;
	a->d[0]=1;
	a->top=1;
	a->neg=0;
	return(1);
	}

/* ignore negative */
BIGNUM *bn_bin2bn(len, s, ret)
int len;
unsigned char *s;
BIGNUM *ret;
	{
	unsigned int i,m;
	int n;
	BN_ULONG l;

	if (ret == NULL) ret=bn_new();
	if (ret == NULL) return(NULL);
	l=0;
	n=len;
	if (n == 0)
		{
		ret->top=0;
		return(ret);
		}
	if (bn_expand(ret,n*8) == NULL)
		return(NULL);
	i=((n-1)/BYTES)+1;
	m=((n-1)%(BYTES));
	ret->top=i;
	while (n-- > 0)
		{
		l=(l<<8)| *(s++);
		if (m-- == 0)
			{
			ret->d[--i]=l;
			l=0;
			m=BYTES-1;
			}
		}
	/* need to call this due to clear byte at top if avoiding
	 * having the top bit set (-ve number) */
	bn_fix_top(ret);
	return(ret);
	}

void bn_print(fp, a)
FILE *fp;
BIGNUM *a;
	{
	int i,j;
	static char *hex="0123456789ABCDEF";

	if (a->neg) fputc('-',fp);
	if (a->top == 0) fputc('0',fp);
	for (i=a->top-1; i >=0; i--)
		{
		for (j=BITS2-4; j >= 0; j-=4)
			{
			fputc(hex[(a->d[i]>>j)&0x0f],fp);
			}
		}
	}

/* ignore negative */
int bn_bn2bin(a, to)
BIGNUM *a;
unsigned char *to;
	{
	int n,i;
	BN_ULONG l;

	n=i=bn_num_bytes(a);
	while (i-- > 0)
		{
		l=a->d[i/BYTES];
		*(to++)=(unsigned char)(l>>(8*(i%BYTES)))&0xff;
		}
	return(n);
	}

/* r can == a or b */
int bn_add(r, a, b)
BIGNUM *r;
BIGNUM *a;
BIGNUM *b;
	{
	register int i;
	int max,min;
	BN_ULONG *ap,*bp,*rp,carry,t1,t2;
	BIGNUM *tmp;

	/*  a +  b	a+b
	 *  a + -b	a-b
	 * -a +  b	b-a
	 * -a + -b	-(a+b)
	 */
	if (a->neg ^ b->neg)
		{
		if (a->neg)
			{ a->neg=0; i=bn_sub(r,b,a); if (a != r) a->neg=1; }
		else
			{ b->neg=0; i=bn_sub(r,a,b); if (b != r) b->neg=1; }
		return(i);
		}
	if (a->neg) /* both are neg */
		{
		a->neg=0; b->neg=0; i=bn_add(r,a,b);
		if (a != r) a->neg=1;
		if (b != r) b->neg=1;
		return(i);
		}
	if (a->top < b->top)
		{ tmp=a; a=b; b=tmp; }
		
	max=a->top;
	min=b->top;
	if (bn_expand(r,(max+1)*BITS2) == NULL) return(0);
	r->top=max;
	r->neg=0;

	ap=a->d;
	bp=b->d;
	rp=r->d;
	carry=0;
	for (i=0; i<min; i++)
		{
		t1= *(ap++);
		t2= *(bp++);
		if (carry)
			{
			carry=(t2 >= ((~t1)&MASK2));
			t2=(t1+t2+1)&MASK2;
			}
		else
			{
			t2=(t1+t2)&MASK2;
			carry=(t2 < t1);
			}
		*(rp++)=t2;
		}
	if (carry)
		{
		while (i < max)
			{
			t1= *(ap++);
			t2=(t1+1)&MASK2;
			*(rp++)=t2;
			carry=(t2 < t1);
			i++;
			if (!carry) break;
			}
		if ((i >= max) && carry)
			{
			*(rp++)=1;
			r->top++;
			}
		}
	for (; i<max; i++)
		*(rp++)= *(ap++);
	return(1);
	}

int bn_Ucmp(a, b)
BIGNUM *a;
BIGNUM *b;
	{
	int i;
	BN_ULONG t1,t2,*ap,*bp;

	i=a->top-b->top;
	if (i != 0) return(i);
	ap=a->d;
	bp=b->d;
	for (i=a->top-1; i>=0; i--)
		{
		t1=ap[i];
		t2=bp[i];
		if (t1 > t2) return(1);
		if (t1 < t2) return(-1);
		}
	return(0);
	}

int bn_cmp(a, b)
BIGNUM *a;
BIGNUM *b;
	{
	int i;
	int gt,lt;
	BN_ULONG t1,t2;

	if (a->neg != b->neg)
		{
		if (a->neg)
			return(-1);
		else	return(1);
		}
	if (a->neg == 0)
		{ gt=1; lt=-1; }
	else	{ gt=-1; lt=1; }

	if (a->top > b->top) return(gt);
	if (a->top < b->top) return(lt);
	for (i=a->top-1; i>=0; i--)
		{
		t1=a->d[i];
		t2=b->d[i];
		if (t1 > t2) return(gt);
		if (t1 < t2) return(lt);
		}
	return(0);
	}

static void bn_SUB(r, a, b)
BIGNUM *r;
BIGNUM *a;
BIGNUM *b;
	{
	int max,min;
	register BN_ULONG t1,t2,*ap,*bp,*rp;
	int i,carry;
#ifdef IRIX_CC_BUG
	int dummy;
#endif

	max=a->top;
	min=b->top;
	ap=a->d;
	bp=b->d;
	rp=r->d;

	carry=0;
	for (i=0; i<min; i++)
		{
		t1= *(ap++);
		t2= *(bp++);
		if (carry)
			{
			carry=(t1 <= t2);
			t1=(t1-t2-1);
			}
		else
			{
			carry=(t1 < t2);
			t1=(t1-t2);
			}
#ifdef IRIX_CC_BUG
		dummy=t1;
#endif
		*(rp++)=t1&MASK2;
		}
	if (carry) /* subtracted */
		{
		while (i < max)
			{
			i++;
			t1= *(ap++);
			t2=(t1-1)&MASK2;
			*(rp++)=t2;
			if (t1 > t2) break;
			}
		}
	for (; i<max; i++)
		*(rp++)=*(ap++);

	r->top=max;
	bn_fix_top(r);
	}

int bn_sub(r, a, b)
BIGNUM *r;
BIGNUM *a;
BIGNUM *b;
	{
	int max,i;

	/*  a -  b	a-b
	 *  a - -b	a+b
	 * -a -  b	-(a+b)
	 * -a - -b	b-a
	 */
	if (a->neg)
		{
		if (b->neg)
			{
			a->neg=b->neg=0;
			i=bn_sub(r,b,a);
			if (a != r) a->neg=1;
			if (b != r) b->neg=1;
			}
		else
			{
			a->neg=0;
			i=bn_add(r,a,b);
			r->neg=a->neg=1;
			}
		return(i);
		}
	else
		{
		if (b->neg)
			{
			b->neg=0;
			i=bn_add(r,a,b);
			if (r != b) b->neg=1;
			return(i);
			}
		}

	max=(a->top > b->top)?a->top:b->top;
	if (bn_cmp(a,b) < 0)
		{
		if (bn_expand(r,max*BITS2) == NULL) return(0);
		bn_SUB(r,b,a);
		r->neg=1;
		}
	else
		{
		if (bn_expand(r,max*BITS2) == NULL) return(0);
		bn_SUB(r,a,b);
		r->neg=0;
		}
	return(1);
	}

#ifdef undef
void bn_mul3(r,a,b)
BIGNUM *r,*a,*b;
	{
	register int i,j;
	int max,tos;
	BN_ULONG *ap,*bp,*t1p,*t2p;
	BN_ULONG m;
	BN_ULLONG t;
	BIGNUM *t1,*t2,*ret;

	tos=bn_get_tos();
	t1=bn_get_reg();
	t2=bn_get_reg();
	ret=bn_get_reg();

	max=(a->top+b->top+1);
	bn_expand(r,(max+1)*BITS2);
	bn_expand(t1,(max+1)*BITS2);
	bn_expand(t2,(max+1)*BITS2);
	bn_zero(r);
	bn_zero(t1);
	bn_zero(t2);
	ap=a->d;
	bp=b->d;
	t1p=t1->d;
	t2p=t2->d;
	r->neg=a->neg^b->neg;

	for (i=0; i<b->top; i++)
		{
		m=bp[i];
		for (j=0; j<a->top; j++)
			{
			t=(BN_ULLONG)m*(BN_ULLONG)ap[j]; /**/
			t1p[i+j]=t&MASK2;
			t2p[i+j+1]=(t>>BITS2)&MASK2;
			}
		t1->top=i+j+1;
		t2->top=i+j+2;
		bn_add(ret,t1,t2);
		bn_add(r,r,ret);
		t1p[i]=t2p[i+1]=0;
		}
	r->top=max+1;
	bn_fix_top(r);
	bn_set_tos(tos);
	}

void bn_mul2(r,a,b)
BIGNUM *r,*a,*b;
	{
	register int i,j;
	int max,tos;
	BN_ULONG *ap,*bp,*t1p;
	BN_ULONG carry,m;
	BN_ULLONG t;
	BIGNUM *t1;

	tos=bn_get_tos();
	t1=bn_get_reg();

	max=(a->top+b->top+1);
	bn_expand(r,(max+1)*BITS2);
	bn_expand(t1,(max+1)*BITS2);
	bn_zero(r);
	bn_zero(t1);
	ap=a->d;
	bp=b->d;
	t1p=t1->d;
	r->neg=a->neg^b->neg;

	for (i=0; i<b->top; i++)
		{
		m=bp[i];
		carry=0;
		for (j=0; j<a->top; j++)
			{
			t=(BN_ULLONG)m*ap[j]+carry;
			t1p[i+j]=(BN_ULONG)t&MASK2;
			carry=(t>>BITS2)&MASK2;
			}
		t1->top=j+i+1;
		if (carry)
			{
			for (;;)
				{
				m=t1p[i+j]+carry;

				t1p[i+j]=m;
				t1->top++;
				if (m < carry)
					{
					carry=1;
					j++;
					}
				else
					break;
				}
			}
		bn_add(r,r,t1);
		t1p[i]=0;
		}
	r->top=max+1;
	bn_fix_top(r);
	bn_set_tos(tos);
	}
#endif

/* r must be different to a and b */
int bn_mul(r, a, b)
BIGNUM *r;
BIGNUM *a;
BIGNUM *b;
	{
	register int i,j;
	int max;
	BN_ULONG *ap,*rp;
	BN_ULONG c1,c2,l,h,m;
	BN_ULONG l2;

	max=(a->top+b->top+1);
	if (bn_expand(r,(max+1)*BITS2) == NULL) return(0);
	bn_zero(r);
	r->top=max;
	for (i=0; i<max; i++)
		r->d[i]=0;
	ap=a->d;
	rp=r->d;
	r->neg=a->neg^b->neg;

	for (i=0; i<b->top; i++)
		{
		m=b->d[i];
		c1=c2=0;
		rp= &(r->d[i]);
		for (j=0; j<a->top; j++)
			{
#ifndef LONGLONG
			BN_ULONG T2,r2,r3;
			
			r3=m&MASK2l;
			T2=(m>>BITS4)&MASK2l;
			l =ap[j]&MASK2l;
			h =(ap[j]>>BITS4)&MASK2l;

			r2=T2*l;
			l =r3*l;
			r3=r3*h;
			h= T2*h;

			h+=((r2>>BITS4)&MASK2l)+((r3>>BITS4)&MASK2l);
			r2=((r2&MASK2l)<<BITS4);
			r3=((r3&MASK2l)<<BITS4);
			
			l+=r2;   if ((l&MASK2) < r2) h++;
			l+=r3;   if ((l&MASK2) < r3) h++;
			l+= *rp; if ((l&MASK2) < *rp) h++;
#else
			BN_ULLONG t;

			t=(BN_ULLONG)m*ap[j]+ *rp;
			l=(BN_ULONG)t&MASK2;
			h=(t>>BITS2)&MASK2;
#endif
			l&=MASK2;
			h&=MASK2;
			l2=(l+c1)&MASK2;
			*(rp++)=l2&MASK2;
			c1=(h+c2)&MASK2;
			if (l2 < l) c1++;
			c2=(c1 < h)?1:0;
			}
		/* As was pointed out by Paul Riethmuller <par@sequent.com>
		 * the following is not needed because I am 'working'
		 * my way 'up' the words.  I suppose I just got
		 * carried away when I wrote the code orignally, or was
		 * it just because I was paranoid about covering all
		 * posibilities :-) */
		*rp = c1;

#ifdef undef
		if (c1 | c2)
			{
			for (;;)
				{
				l=(*rp+c1)&MASK2;
				*(rp++)=l;
				if (l < c1)
					{
					c1=(c2+1)&MASK2;
					c2=(c1 == 0)?1:0;
					}
				else
					{
					c1=c2;
					c2=0;
					}
				if ((c1|c2) == 0) break;
				}
			}
#endif
		}
	bn_fix_top(r);
	return(1);
	}

int bn_lshift1(r, a)
BIGNUM *r;
BIGNUM *a;
	{
	register BN_ULONG *ap,*rp,t,c;
	int i;

	if (r != a)
		{
		r->neg=a->neg;
		if (bn_expand(r,(a->top+1)*BITS2) == NULL) return(0);
		r->top=a->top;
		}
	else
		{
		if (bn_expand(r,(a->top+1)*BITS2) == NULL) return(0);
		}
	ap=a->d;
	rp=r->d;
	c=0;
	for (i=0; i<a->top; i++)
		{
		t= *(ap++);
		*(rp++)=((t<<1)|c)&MASK2;
		c=(t & TBIT)?1:0;
		}
	if (c)
		{
		*rp=1;
		r->top++;
		}
	return(1);
	}

int bn_rshift1(r, a)
BIGNUM *r;
BIGNUM *a;
	{
	BN_ULONG *ap,*rp,t,c;
	int i;

	if (bn_is_zero(a))
		{
		bn_zero(r);
		return(1);
		}
	if (a != r)
		{
		if (bn_expand(r,a->top*BITS2) == NULL) return(0);
		r->top=a->top;
		r->neg=a->neg;
		}
	ap=a->d;
	rp=r->d;
	c=0;
	for (i=a->top-1; i>=0; i--)
		{
		t=ap[i];
		rp[i]=((t>>1)&MASK2)|c;
		c=(t&1)?TBIT:0;
		}
	bn_fix_top(r);
	return(1);
	}

int bn_lshift(r, a, n)
BIGNUM *r;
BIGNUM *a;
int n;
	{
	int i,nw,lb,rb;
	BN_ULONG *t,*f;
	BN_ULONG l;

	r->neg=a->neg;
	if (bn_expand(r,(a->top*BITS2)+n) == NULL) return(0);
	nw=n/BITS2;
	lb=n%BITS2;
	rb=BITS2-lb;
	f=a->d;
	t=r->d;
	t[a->top+nw]=0;
	if (lb == 0)
		for (i=a->top-1; i>=0; i--)
			t[nw+i]=f[i];
	else
		for (i=a->top-1; i>=0; i--)
			{
			l=f[i];
			t[nw+i+1]|=(l>>rb)&MASK2;
			t[nw+i]=(l<<lb)&MASK2;
			}
	for (i=0; i<nw; i++)
		t[i]=0;
	r->top=a->top+nw+1;
	bn_fix_top(r);
	return(1);
	}

int bn_rshift(r, a, n)
BIGNUM *r;
BIGNUM *a;
int n;
	{
	int i,nw,lb,rb;
	BN_ULONG *t,*f;
	BN_ULONG l;

	r->neg=a->neg;
	nw=n/BITS2;
	rb=n%BITS2;
	lb=BITS2-rb;
	if (nw > a->top)
		{
		bn_zero(r);
		return(1);
		/*abort();/**/
		}
	if (bn_expand(r,(a->top-nw+1)*BITS2) == NULL) return(0);
	f=a->d;
	t=r->d;
	if (rb == 0)
		for (i=nw; i<a->top; i++)
			t[i-nw]=f[i];
	else
		{
		l=f[nw];
		for (i=nw; i<a->top; i++)
			{
			t[i-nw] =(l>>rb)&MASK2;
			l=f[i+1];
			t[i-nw]|=(l<<lb)&MASK2;
			}
		}
	r->top=a->top-nw;
	t[r->top]=0;
	bn_fix_top(r);
	return(1);
	}

#ifdef undef
void bn_rshift(r,a,n)
BIGNUM *r,*a;
int n;
	{
	int i;

	if (n) bn_rshift1(r,a);
	for (i=1; i<n; i++)
		bn_rshift1(r,r);
	}
#endif

#ifdef undef
void bn_gcd(r,in_a,in_b)
BIGNUM *r,*in_a,*in_b;
	{
	BIGNUM *a,*b,*t;
	int tos;

	tos=bn_get_tos();
	a=bn_get_reg();
	b=bn_get_reg();
	bn_copy(a,in_a);
	bn_copy(b,in_b);

	if (bn_cmp(a,b) < 0) { t=a; a=b; b=t; }
	t=euclid(a,b);
	r=bn_copy(r,t);
	bn_set_tos(tos);
	}
#endif


#ifdef undef
static BIGNUM *euclid(a,b)
BIGNUM *a,*b;
	{
	BIGNUM *t;
	int shifts=0;
	int i;

	for (;;)
		{
		if (bn_is_zero(b))
			break;

		if ((a->d[0]&1))	/* a is odd */
			{
			if (b->d[0]&1)  /* b is odd */
				{
				bn_sub(a,a,b);
				bn_rshift1(a,a);
				if (bn_cmp(a,b) < 0)
					{ t=a; a=b; b=t; }
				}
			else		/* a odd - b even */
				{
				bn_rshift1(b,b);
				if (bn_cmp(a,b) < 0)
					{ t=a; a=b; b=t; }
				}
			}
		else			/* a is even */
			{
			if (b->d[0]&1)	/* b id odd */
				{
				bn_rshift1(a,a);
				if (bn_cmp(a,b) < 0)
					{ t=a; a=b; b=t; }
				}
			else		/* a even - b even */
				{
				bn_rshift1(a,a);
				bn_rshift1(b,b);
				shifts++;
				}
			}
		}
	if (shifts)
		bn_lshift(a,a,shifts);
	return(a);
	}
#endif

int bn_clear_bit(a, n)
BIGNUM *a;
int n;
	{
	int i,j;

	i=n/BITS2;
	j=n%BITS2;
	if (a->top <= i) return(0);

	return(a->d[i]&(~(1L<<j)));
	}

int bn_is_bit_set(a, n)
BIGNUM *a;
int n;
	{
	int i,j;

	i=n/BITS2;
	j=n%BITS2;
	if (a->top <= i) return(0);
	return((a->d[i]&(1L<<j))?1:0);
	}

/* rem != m */
int bn_mod(rem, m, d)
BIGNUM *rem;
BIGNUM *m;
BIGNUM *d;
	{
	int i,nm,nd,tos;
	BIGNUM *div;

	if (bn_Ucmp(m,d) < 0)
		return((bn_copy(rem,m) == NULL)?0:1);
	tos=bn_get_tos();
	div=bn_get_reg();
	if (div == NULL) return(0);

	if (!bn_copy(rem,m)) return(0);

	nm=bn_num_bits(rem);
	nd=bn_num_bits(d);
	if (!bn_lshift(div,d,nm-nd)) return(0);
	for (i=nm-nd; i>=0; i--)
		{
		if (bn_cmp(rem,div) >= 0)
			{
			if (!bn_sub(rem,rem,div)) return(0);
			}
		if (!bn_rshift1(div,div)) return(0);
		}
	bn_set_tos(tos);
	return(1);
	}

#ifdef undef
void bn_mod2(rem,m,d)
BIGNUM *rem,*m,*d;
	{
	int i,j,nr,nm,nd,x;
	int tos;
	static BIGNUM *shifts[BITS2];
	BN_ULONG *shiftp[BITS2];

	bn_copy(rem,m);
	if (bn_cmp(m,d) < 0)
		return;

	tos=bn_get_tos();
	for (i=0; i<BITS2; i++)
		shifts[i]=bn_get_reg();

	nm=bn_num_bits(rem);
	nd=bn_num_bits(d);

	bn_lshift(shifts[0],d,nm-nd);
	for (i=1; i<BITS2; i++) bn_rshift1(shifts[i],shifts[i-1]);
	for (i=0; i<BITS2; i++) shiftp[i]=shifts[i]->d;

	x=0;
	for (i=nm-nd; i>=0; i--)
		{
		if (bn_cmp(rem,shifts[x]) >= 0)
			bn_sub(rem,rem,shifts[x]);

		shifts[x]->top--;
		shifts[x]->d++;
		x=(x+1)%BITS2;
		}
	for (i=0; i<BITS2; i++) shifts[i]->d=shiftp[i];
	bn_set_tos(tos);
	}
#endif

static int mod_init=1;
static int mod_bits,mod_shift;
static BIGNUM *mod_value;
static BIGNUM *mod_shifts[BITS2];
static BN_ULONG *mod_shiftp[BITS2];
static int mod_top[BITS2];

int bn_MOD_init(d, max_bits)
BIGNUM *d;
int max_bits;
	{
	int i;

	if (mod_init)
		{
		mod_init=0;
		for (i=0; i<BITS2; i++)
			{
			mod_shifts[i]=bn_new();
			if (mod_shifts[i] == NULL)
				{ return(0); }
			}
		mod_value=bn_new();
		if (mod_value == NULL)
			{ return(0); }
		}
	if (bn_copy(mod_value,d) == NULL) return(0);
	mod_bits=bn_num_bits(d);
	mod_shift=max_bits-mod_bits;

	if (!bn_lshift(mod_shifts[0],d,mod_shift)) return(0);
	for (i=1; i<BITS2; i++)
		if (!bn_rshift1(mod_shifts[i],mod_shifts[i-1])) return(0);
	for (i=0; i<BITS2; i++)
		{
		mod_shiftp[i]=mod_shifts[i]->d;
		mod_top[i]=mod_shifts[i]->top;
		}
	return(1);
	}

int bn_MOD(ret, m, d)
BIGNUM *ret;
BIGNUM *m;
BIGNUM *d;
	{
	int i,j,nm,nd,x;
	int w;

	if (bn_copy(ret,m) == NULL) return(0);
	if (bn_cmp(m,mod_value) < 0)
		return(1);
	nm=bn_num_bits(m);
	nd=mod_bits;
	i=nm-nd;

	j=mod_shift-i;	/* take off what we are shifted, how far do we need to go back */
	w=j/BITS2;	/* number of words to jump back */
	x=j%BITS2;	/* which shift to start on */

	if (w != 0)
		for (i=0; i<BITS2; i++) 
			{
			j=w+(x>i);
			mod_shifts[i]->top-=j;
			mod_shifts[i]->d+=j;
			}
	
	for (i=nm-nd; i>=0; i--)
		{
		if (bn_cmp(ret,mod_shifts[x]) >= 0)
			{
			if (!bn_sub(ret,ret,mod_shifts[x])) return(0);
			}

		mod_shifts[x]->top--;
		mod_shifts[x]->d++;
		x=(x+1)%BITS2;
		}
	for (i=0; i<BITS2; i++)
		{
		mod_shifts[i]->d=mod_shiftp[i];
		mod_shifts[i]->top=mod_top[i];
		}
	return(1);
	}

int bn_div(div, rem, m, d)
BIGNUM *div;
BIGNUM *rem;
BIGNUM *m;
BIGNUM *d;
	{
	int i,nm,nd;
	BIGNUM *D;
	int tos;

	if (bn_is_zero(d))
		{
		RSA_errno=RSA_ERR_DIV_BY_ZERO;
		return(0);
		}

	if (bn_cmp(m,d) < 0)
		{
		if (rem != NULL)
			{ if (bn_copy(rem,m) == NULL) return(0); }
		if (div != NULL) bn_zero(div);
		return(1);
		}

	tos=bn_get_tos();
	D=bn_get_reg();
	if (div == NULL) div=bn_get_reg();
	if (rem == NULL) rem=bn_get_reg();
	if ((D == NULL) || (div == NULL) || (rem == NULL))
		return(0);

	nd=bn_num_bits(d);
	nm=bn_num_bits(m);
	if (bn_copy(D,d) == NULL) return(0);
	if (bn_copy(rem,m) == NULL) return(0);

	/* The next 2 are needed so we can do a div->d[0]|=1 later
	 * since bn_lshift1 will only work once there is a value :-) */
	bn_zero(div);
	div->top=1;

	if (!bn_lshift(D,D,nm-nd)) return(0);
	for (i=nm-nd; i>=0; i--)
		{
		if (!bn_lshift1(div,div)) return(0);
		if (bn_cmp(rem,D) >= 0)
			{
			div->d[0]|=1;
			if (!bn_sub(rem,rem,D)) return(0);
			}
/* CAN IMPROVE */
		if (!bn_rshift1(D,D)) return(0);
		}
	div->neg=m->neg^d->neg;
	bn_set_tos(tos);
	return(1);
	}

#ifdef undef
void bn_mul0(r,a,b)
BIGNUM *r,*a,*b;
   {
   int i,j,k,max,nr,na,nb,mask;
   BIGNUM *tmp;
   static int init=1;
   static BIGNUM *shifts[BITS2];
   BN_ULONG *shiftp[BITS2];
   BN_ULLONG l;
	BN_ULONG *rp,*ap,*bp,*sp;

   if (bn_cmp(a,b) < 0)
           {
           tmp=a;
           a=b;
           b=tmp;
           }

   if (init)
           {
           init=0;
           for (i=0; i<BITS; i++)
                   {
                   shifts[i]=bn_new();
                   }
           }

   max=(a->top+b->top+1);
   bn_expand(r,(max+1)*BITS2);
   bn_zero(r);

   na=bn_num_bits(a);
   nb=bn_num_bits(b);
   bn_copy(shifts[0],a);
   for (i=1; i<BITS2; i++) bn_lshift1(shifts[i],shifts[i-1]);
   for (i=0; i<BITS2; i++) shiftp[i]=shifts[i]->d;
	ap=a->d;
	bp=b->d;
	rp=r->d;

   mask=1;
   for (i=0; i<BITS2; i++)
           {
           for (j=0; j<b->top; j++)
                   {
                   if (bp[j]&mask)
                           {
				sp=shifts[i]->d;
                           for (k=0; k<shifts[i]->top; k++)
                                   rp[j+k]+=sp[k];
                           }
                   }
		mask<<=1;
           }

   for (i=0; i<max; i++)
           {
		l=rp[i];
           if (l > MASK2)
                   {
                   rp[i+1]+=(l>>BITS2);
                   rp[i]=l&MASK2;
                   rp[i]=l&MASK2;
                   }
           }
   while ((rp[max] == 0) && (max > 0))
           max--;
   r->top=max+1;
   r->numbits=r->top*BITS2;
	r->neg=a->neg^b->neg;
	}
#endif

BN_ULONG bn_mod_word(a, w)
BIGNUM *a;
unsigned long w;
	{
	BN_ULONG ret;
	int i;

	ret=0;
	for (i=a->top-1; i>=0; i--)
		{
#ifndef LONGLONG
		ret=((ret<<BITS4)|((a->d[i]>>BITS4)&MASK2l))%w;
		ret=((ret<<BITS4)|(a->d[i]&MASK2l))%w;
#else
		ret=(((BN_ULLONG)ret<<BITS2)|a->d[i])%w;
#endif
		}
	return(ret);
	}

int bn_add_word(a, w)
BIGNUM *a;
unsigned long w;
	{
	BN_ULONG l;
	int i;

	if (bn_expand(a,a->top*BITS2+1) == NULL) return(0);
	i=0;
	for (;;)
		{
		l=(a->d[i]+w)&MASK2;
		a->d[i]=l;
		if (w > l)
			w=1;
		else
			break;
		i++;
		}
	if (i >= a->top)
		a->top++;
	return(1);
	}


int bn_mul_mod(ret, a, b, m)
BIGNUM *ret;
BIGNUM *a;
BIGNUM *b;
BIGNUM *m;
	{
	BIGNUM *t;
	int tos,r=1;

	tos=bn_get_tos();
	t=bn_get_reg();
	if (t == NULL) return(0);
	if (!bn_mul(t,a,b)) { r=0; goto err; }
	if (!bn_mod(ret,t,m)) { r=0; goto err; }
err:
	bn_set_tos(tos);
	return(r);
	}

#ifndef RECP_MUL_MOD
/* this one works */
int bn_mod_exp(r,a,p,m)
BIGNUM *r,*a,*p,*m;
	{
	int tos,i,bits;
	BIGNUM *v=NULL,*tmp=NULL;

	tos=bn_get_tos();
	if (v == NULL) v=bn_get_reg();
	if (tmp == NULL) tmp=bn_get_reg();
	if ((v == NULL) || (tmp == NULL)) goto err;
	if (bn_copy(v,a) == NULL) goto err;
	bits=bn_num_bits(p);

	if (p->d[0]&1)
		{ if (bn_copy(r,a) == NULL) goto err; }
	else	{ if (bn_one(r) == NULL) goto err; }

	if (!bn_MOD_init(m,m->top*3*BITS2)) goto err;
	for (i=1; i<bits; i++)
		{
		if (!bn_mul(tmp,v,v)) goto err;
		if (!bn_MOD(v,tmp,m)) goto err;
		if (bn_is_bit_set(p,i))
			{
			if (!bn_mul(tmp,r,v)) goto err;
			if (!bn_MOD(r,tmp,m)) goto err;
			}
		}
	bn_set_tos(tos);
	return(1);
err:
	bn_set_tos(tos);
	return(0);
	}

#else
int bn_mod_exp(r, a, p, m)
BIGNUM *r;
BIGNUM *a;
BIGNUM *p;
BIGNUM *m;
	{
	int tos,nb,i,bits;
	BIGNUM *v=NULL,*tmp=NULL;
	BIGNUM *d=NULL;

	tos=bn_get_tos();
	if (v == NULL) v=bn_get_reg();
	if (tmp == NULL) tmp=bn_get_reg();
	if (d == NULL) d=bn_get_reg();
	if ((v == NULL) || (tmp == NULL) || (d == NULL)) goto err;
	if (!bn_mod(v,a,m)) goto err;
	bits=bn_num_bits(p);

	if (p->d[0]&1)
		{ if (!bn_mod(r,a,m)) goto err; }
	else	{ if (!bn_one(r)) goto err; }

	nb=bn_reciprical(d,m);
	if (nb == -1) goto err;
	for (i=1; i<bits; i++)
		{
		if (!bn_modmul_recip(v,v,v,m,d,nb)) goto err;
		if (bn_is_bit_set(p,i))
			{ if (!bn_modmul_recip(r,r,v,m,d,nb)) goto err; }
		}
	bn_set_tos(tos);
	return(1);
err:
	bn_set_tos(tos);
	return(0);
	}
#endif

#ifdef undef
void bn_modmul(ret,a,b,m)
BIGNUM *ret,*a,*b,*m;
	{
	int i,bb;
	BN_ULONG mask;
	int word;

	bn_zero(ret);

	bb=bn_num_bits(b);
	word=(bb-1)/BITS2;
	mask=1<<((bb-1)%BITS2);

	for (i=bb-1; i>=0; i--)
		{
		bn_lshift1(ret,ret);
		if (bn_cmp(ret,m) >= 0)
			bn_sub(ret,ret,m);
		if (b->d[word] & mask)
			{
			bn_add(ret,ret,a);
			if (bn_cmp(ret,m) >= 0)
				bn_sub(ret,ret,m);
			}

		mask>>=1;
		if (!mask)
			{
			word--;
			mask=TBIT;
			}
		}
	}
#endif

#ifdef undef
void bn_modexp(r,a,p,m)
BIGNUM *r,*a,*p,*m;
	{
	int i,bits,tos;
	BIGNUM *v[2],*R[2];
	int vn=0,rn=0;

	tos=bn_get_tos();
	v[0]=bn_get_reg();
	v[1]=bn_get_reg();
	R[0]=bn_get_reg();
	R[1]=bn_get_reg();
	bn_mod(v[vn],a,m);
	bits=bn_num_bits(p);

	if (p->d[0]&1)
		bn_copy(R[rn],a);
	else	bn_one(R[rn]);

	for (i=1; i<=bits; i++)
		{
		bn_modmul(v[!vn],v[vn],v[vn],m);
		vn= !vn;
		if (bn_is_bit_set(p,i))
			{
			bn_modmul(R[!rn],R[rn],v[vn],m);
			rn= !rn;
			}
		}
	bn_copy(r,R[rn]);
	bn_set_tos(tos);
	}
#endif

int bn_modmul_recip(r, x, y, m, i, nb)
BIGNUM *r;
BIGNUM *x;
BIGNUM *y;
BIGNUM *m;
BIGNUM *i;
int nb;
	{
	int tos,j;
	BIGNUM *a,*b,*c,*d;

	tos=bn_get_tos();
	a=bn_get_reg();
	b=bn_get_reg();
	c=bn_get_reg();
	d=bn_get_reg();
	if ((a == NULL) || (b == NULL) || (c == NULL) || (d == NULL))
		goto err;

	if (!bn_mul(a,x,y)) goto err;
	if (!bn_rshift(d,a,nb-1)) goto err;
	if (!bn_mul(b,d,i)) goto err;
	if (!bn_rshift(c,b,nb-1)) goto err;
	if (!bn_mul(b,m,c)) goto err;
	if (!bn_sub(r,a,b)) goto err;
	j=0;
	while (bn_cmp(r,m) >= 0)
		{
		if (j++ > 2)
			{
			RSA_errno=RSA_ERR_BAD_BN_MODMUL_RECIP;
			goto err;
			}
		if (!bn_sub(r,r,m)) goto err;
		}

	bn_set_tos(tos);
	return(1);
err:
	bn_set_tos(tos);
	return(0);
	}

int bn_reciprical(r, m)
BIGNUM *r;
BIGNUM *m;
	{
	int nm,tos;
	BIGNUM *t;

	tos=bn_get_tos();
	t=bn_get_reg();
	if (t == NULL) goto err;

	if (!bn_one(t)) goto err;
	nm=bn_num_bits(m);
	if (!bn_lshift(t,t,nm*2)) goto err;

	if (!bn_div(r,NULL,t,m)) goto err;
	bn_set_tos(tos);
	return(nm+1);
err:
	bn_set_tos(tos);
	return(-1);
	}

