/* X509_D2i.c */
/* Copyright (C) 1995 Eric Young (eay@mincom.oz.au).
 * All rights reserved.
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.
 * See the COPYRIGHT file in the SSLeay distribution for more details.
 */

#include <stdio.h>
#include "crypto.h"
#include "Buffer.h"
#include "der.h"
#include "bn.h"
#include "X509.h"

#define READ_CHUNK	2048

int D2i_X509_ALGOR(a, pp)
X509_ALGOR *a;
unsigned char **pp;
	{
	unsigned char *p;
	unsigned long len;
	int i,tag,class;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_SEQUENCE)
		{
		DER_errno=DER_ERR_EXPECTING_A_SEQUENCE_IN_D2I_X509_ALGOR;
		return(0);
		}

	i=D2i_DER_OBJECT(a->algorithm,&p);
	if (!i) return(0);

	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_NULL)
		{
		DER_errno=DER_ERR_EXPECTING_A_NULL_IN_D2I_X509_ALGOR;
		return(0);
		}
	*pp=p;
	return(1);
	}

int D2i_X509_NAME(a, pp)
X509_NAME *a;
unsigned char **pp;
	{
	unsigned char *p,*pt,*ps,*ptmp;
	unsigned long len;
	int i,num_obj,tag,class;
	DER_OBJECT o;
	DER_BIT_STRING *bs;
#ifdef DG_GCC_BUG
	unsigned char *bug;
#endif

	o.num=0;
	o.values=NULL;
	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_SEQUENCE)
		{
		DER_errno=DER_ERR_EXPECTING_A_SEQUENCE_IN_D2I_X509_NAME;
		return(0);
		}
	ps=p;
	pt=p+len;
	num_obj=0;
	while (p < pt)
		{
		DER_get_object(&p,&len,&tag,&class);
		if (tag != DER_SET)
			{
			DER_errno=DER_ERR_EXPECTING_A_SET_IN_D2I_X509_NAME;
			return(0);
			}
		p+=len;
		num_obj++;
		}
	if (a->num != 0)
		{
		for (i=0; i<a->num; i++)
			{
			if (a->objects[i] != NULL)
				DER_OBJECT_free(a->objects[i]);
			if (a->values[i] != NULL) 
				DER_BIT_STRING_free(a->values[i]);
			}
		free(a->objects);
		free(a->types);
		free(a->values);
		a->objects=NULL;
		a->types=NULL;
		a->values=NULL;
		a->num=0;
		}

	a->num=num_obj;
	a->objects=(DER_OBJECT **)malloc(sizeof(DER_OBJECT *)*num_obj);
	a->types=(int *)malloc(sizeof(int)*num_obj);
	a->values=(DER_BIT_STRING **)malloc(sizeof(DER_BIT_STRING *)*num_obj);
	if ((a->objects == NULL) || (a->values == NULL) || (a->types == NULL))
		{
		DER_errno=DER_ERR_OUT_OF_MEM;
		return(0);
		}
	for (i=0; i<num_obj; i++)
		{
		a->objects[i]=NULL;
		a->types[i]=0;
		a->values[i]=NULL;
		}
	p=ps;
	num_obj=0;
	while (p < pt)
		{
		DER_get_object(&p,&len,&tag,&class);
		if (tag != DER_SET) /* test not needed */
			{
			DER_errno=DER_ERR_EXPECTING_A_SET_IN_D2I_X509_NAME;
			return(0);
			}
		DER_get_object(&p,&len,&tag,&class);
		if (tag != DER_SEQUENCE) /* test not needed */
			{
			DER_errno=DER_ERR_EXPECTING_A_SEQUENCE_IN_D2I_X509_NAME;
			return(0);
			}

		a->objects[num_obj]=(DER_OBJECT *)DER_OBJECT_new();
		if (a->objects[num_obj] == NULL) return(0);
		if (!D2i_DER_OBJECT(a->objects[num_obj],&p)) return(0);

		DER_get_object(&p,&len,&tag,&class);
		a->types[num_obj]=tag;
		bs=DER_BIT_STRING_new();
		if (bs == NULL) return(0);
		bs->length=len;
		bs->data=(unsigned char *)malloc(len+1);
		if (bs->data == NULL)
			{ DER_errno=DER_ERR_OUT_OF_MEM; return(0); }
		memcpy(bs->data,p,len);
		p+=len;
		bs->data[len]='\0';
		a->values[num_obj]=bs;

		num_obj++;
#ifdef DG_GCC_BUG
		bug+=(int)p;
#endif
		}
	*pp=p;
	return(1);
	}

int D2i_X509_VAL(a, pp)
X509_VAL *a;
unsigned char **pp;
	{
	unsigned char *p;
	unsigned long len;
	int tag,class;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_SEQUENCE)
		{
		DER_errno=DER_ERR_EXPECTING_A_SEQUENCE_IN_D2I_X509_VAL;
		return(0);
		}
	if (!D2i_UTCTime((unsigned char **)&(a->notBefore),&p)) return(0);
	if (!D2i_UTCTime((unsigned char **)&(a->notAfter),&p)) return(0);
	*pp=p;
	return(1);
	}

int D2i_X509_PUBKEY(a, pp)
X509_PUBKEY *a;
unsigned char **pp;
	{
	unsigned char *p;
	unsigned long len;
	int tag,class;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_SEQUENCE)
		{
		DER_errno=DER_ERR_EXPECTING_A_SEQUENCE_IN_D2I_X509_PUBKEY;
		return(0);
		}
	if (!D2i_X509_ALGOR(a->algor,&p)) return(0);
	if (!D2i_DER_BIT_STRING(a->public_key,&p)) return(0);
	*pp=p;
	return(1);
	}

int D2i_X509_SIG(a, pp)
X509_SIG *a;
unsigned char **pp;
	{
	unsigned char *p;
	unsigned long len;
	int tag,class;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_SEQUENCE)
		{
		DER_errno=DER_ERR_EXPECTING_A_SEQUENCE_IN_D2I_X509_SIG;
		return(0);
		}
	if (!D2i_X509_ALGOR(a->algor,&p)) return(0);
	if (!D2i_OCTET_STRING(a->digest,&p)) return(0);
	*pp=p;
	return(1);
	}

int D2i_X509_CINF(a, pp)
X509_CINF *a;
unsigned char **pp;
	{
	unsigned char *p;
	unsigned long len;
	int tag,class;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_SEQUENCE)
		{
		DER_errno=DER_ERR_EXPECTING_A_SEQUENCE_IN_D2I_X509_CINF;
		return(0);
		}
/*	D2i_INTEGER(&(a->version),&p); /* seems to be missing */
	if (!D2i_INTEGER(a->serialNumber,&p)) return(0);
	if (!D2i_X509_ALGOR(a->signature,&p)) return(0);
	if (!D2i_X509_NAME(a->issuer,&p)) return(0);
	if (!D2i_X509_VAL(a->validity,&p)) return(0);
	if (!D2i_X509_NAME(a->subject,&p)) return(0);
	if (!D2i_X509_PUBKEY(a->key,&p)) return(0);
	*pp=p;
	return(1);
	}

int D2i_X509_file(file, a)
char *file;
X509 *a;
	{
	int i;
	FILE *in;

	in=fopen(file,"r");
	if (in == NULL)
		{
		DER_errno=DER_ERR_UNABLE_TO_OPEN_FILE;
		return(0);
		}
	i=D2i_X509_fp(in,a);
	fclose(in);
	return(i);
	}

int D2i_X509_fp(in, a)
FILE *in;
X509 *a;
	{
	BUFFER *b;
	int i,lenbuf=0;

	b=buffer_new();
	if (b == NULL)
		{ DER_errno=DER_ERR_OUT_OF_MEM; return(0); }
	if (!buffer_grow(b,READ_CHUNK))
		{ DER_errno=DER_ERR_OUT_OF_MEM; return(0); }
	
	for (;;)
		{
		i=fread(&(b->data[lenbuf]),1,READ_CHUNK,in);
		if (i <= 0) break;
		lenbuf+=i;
		if (!buffer_grow(b,lenbuf+READ_CHUNK))
			{
			DER_errno=DER_ERR_OUT_OF_MEM;
			return(0);
			}
		}
	if (lenbuf < 10)
		{
		buffer_free(b);
		DER_errno=DER_ERR_FILE_TOO_SMALL_IN_D2I_X509_FP;
		return(0);
		}
	i=D2i_X509(a,lenbuf,(unsigned char *)b->data);
	buffer_free(b);
	return(i);
	}

int D2i_X509(a, lenbuf, p)
X509 *a;
int lenbuf;
unsigned char *p;
	{
	unsigned long len;
	int tag,class;

	DER_get_object(&p,&len,&tag,&class);
	len=DER_object_size(1,(int)len,tag);

	if (lenbuf < len)
		{
		DER_errno=DER_ERR_NOT_ENOUGH_DATA_PASSED_IN_D2I_X509;
		return(0);
		} 
	if (tag != DER_SEQUENCE)
		{
		DER_errno=DER_ERR_EXPECTING_A_SEQUENCE_IN_D2I_X509;
		return(0);
		}
	if (!D2i_X509_CINF(a->cert_info,&p)) return(0);
	if (!D2i_X509_ALGOR(a->sig_alg,&p)) return(0);
	if (!D2i_DER_BIT_STRING(a->signature,&p)) return(0);
	return(1);
	}

int D2i_RSAPrivateKey_fp(in, rsa)
FILE *in;
RSA *rsa;
	{
	int i,lenbuf=0;
	BUFFER *b;
	unsigned char *pp;

	b=buffer_new();
	if (b == NULL)
		{ DER_errno=DER_ERR_OUT_OF_MEM; return(0); }
	if (!buffer_grow(b,READ_CHUNK))
		{ DER_errno=DER_ERR_OUT_OF_MEM; return(0); }
	
	for (;;)
		{
		i=fread(&(b->data[lenbuf]),1,READ_CHUNK,in);
		if (i <= 0) break;
		lenbuf+=i;
		if (!buffer_grow(b,(long)lenbuf+READ_CHUNK))
			{
			DER_errno=DER_ERR_OUT_OF_MEM;
			return(0);
			}
		}
	if (lenbuf < 10)
		{
		buffer_free(b);
		DER_errno=DER_ERR_FILE_TOO_SMALL_IN_D2I_RSAPRIVATEKEY_FP;
		return(0);
		}
	pp=(unsigned char *)b->data;
	i=D2i_RSAPrivateKey(rsa,&pp);
	buffer_free(b);
	return(i);
	}

int D2i_RSAPrivateKey(rsa, pp)
RSA *rsa;
unsigned char **pp;
	{
	unsigned char *p;
	unsigned long len;
	int tag,class;
	DER_BIT_STRING bs;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_SEQUENCE)
		{
		DER_errno=DER_ERR_EXPECTING_A_SEQUENCE_IN_D2I_RSAPRIVATEKEY;
		return(0);
		}
	bs.length=0; bs.data=NULL;
	if (!D2i_INTEGER(&bs,&p)) return(0); /* version */
	if (!D2i_INTEGER(&bs,&p)) return(0); /* n */
	rsa->n=bn_bin2bn(bs.length,bs.data,rsa->n);
	if (rsa->n == NULL) { DER_errno=DER_ERR_RSA; return(0); }

	if (!D2i_INTEGER(&bs,&p)) return(0); /* e */
	rsa->e=bn_bin2bn(bs.length,bs.data,rsa->e);
	if (rsa->e == NULL) { DER_errno=DER_ERR_RSA; return(0); }

	if (!D2i_INTEGER(&bs,&p)) return(0); /* d */
	rsa->d=bn_bin2bn(bs.length,bs.data,rsa->d);
	if (rsa->d == NULL) { DER_errno=DER_ERR_RSA; return(0); }

	if (!D2i_INTEGER(&bs,&p)) return(0); /* p */
	rsa->p=bn_bin2bn(bs.length,bs.data,rsa->p);
	if (rsa->p == NULL) { DER_errno=DER_ERR_RSA; return(0); }

	if (!D2i_INTEGER(&bs,&p)) return(0); /* q */
	rsa->q=bn_bin2bn(bs.length,bs.data,rsa->q);
	if (rsa->q == NULL) { DER_errno=DER_ERR_RSA; return(0); }

	if (!D2i_INTEGER(&bs,&p)) return(0); /* dmp1 */
	rsa->dmp1=bn_bin2bn(bs.length,bs.data,rsa->dmp1);
	if (rsa->dmp1 == NULL) { DER_errno=DER_ERR_RSA; return(0); }

	if (!D2i_INTEGER(&bs,&p)) return(0); /* dmq1 */
	rsa->dmq1=bn_bin2bn(bs.length,bs.data,rsa->dmq1);
	if (rsa->dmq1 == NULL) { DER_errno=DER_ERR_RSA; return(0); }

	if (!D2i_INTEGER(&bs,&p)) return(0); /* iqmp */
	rsa->iqmp=bn_bin2bn(bs.length,bs.data,rsa->iqmp);
	if (rsa->iqmp == NULL) { DER_errno=DER_ERR_RSA; return(0); }

	free(bs.data);
	*pp=p;
	return(1);
	}

int D2i_RSAPublicKey(rsa, pp)
RSA *rsa;
unsigned char **pp;
	{
	unsigned char *p;
	unsigned long len;
	int tag,class;
	DER_BIT_STRING bs;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_SEQUENCE)
		{
		DER_errno=DER_ERR_EXPECTING_A_SEQUENCE_IN_D2I_RSAPUBLICKEY;
		return(0);
		}
	bs.length=0; bs.data=NULL;
	if (!D2i_INTEGER(&bs,&p)) return(0); /* n */
	rsa->n=bn_bin2bn(bs.length,bs.data,rsa->n);
	if (rsa->n == NULL) { DER_errno=DER_ERR_RSA; return(0); }

	if (!D2i_INTEGER(&bs,&p)) return(0); /* e */
	rsa->e=bn_bin2bn(bs.length,bs.data,rsa->e);
	if (rsa->e == NULL) { DER_errno=DER_ERR_RSA; return(0); }

	free(bs.data);
	*pp=p;
	return(1);
	}
