/* PEM.c */
/* Copyright (C) 1995 Eric Young (eay@mincom.oz.au).
 * All rights reserved.
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.
 * See the COPYRIGHT file in the SSLeay distribution for more details.
 */

#include <stdio.h>
#include "crypto.h"
#include "der.h"
#include "X509.h"
#include "PEM.h"
#include "PEM_locl.h"
#include "Buffer.h"
#include "des.h"
#include "md2.h"
#include "md5.h"

static char *version="\0PEM part of SSLeay v 0.4.3 15/06/95";

#define MIN_LENGTH	4
#define DES_COUNT	1
#define conv_bin2ascii(a)       (data_bin2ascii[(a)&0x3f])
#define conv_ascii2bin(a)       (data_ascii2bin[(a)&0x7f])

/* 64 char lines
 * pad input with 0
 * left over chars are set to =
 * 1 byte  => xx==
 * 2 bytes => xxx=
 * 3 bytes => xxxx
 */
#define BIN_PER_LINE    (64/4*3)
#define CHUNKS_PER_LINE (64/4)
#define CHAR_PER_LINE   (64+1)

#ifdef PROTO
static int PEM_bin2ascii(int len, unsigned char *f, unsigned char *t);
static int PEM_ascii2bin(unsigned char *f, unsigned char *t);
static void PEM_proc_type(char *buf, int type);
static void PEM_dek_info(char *buf, int type, long int len, char *str);
static void PEM_make_key_des_md5(int strl, char *str, int count, des_cblock (*iv), des_cblock (*key));
static void PEM_make_key_des_md2(int strl, char *str, int count, des_cblock (*iv), des_cblock (*key));
static int PEM_des_cbc_encrypt(unsigned char *input, unsigned char *output, long int length, struct des_ks_struct *ks, des_cblock (*iv), int encrypt);
static int def_callback(char *buf, int num, int w);
#else
static int PEM_bin2ascii();
static int PEM_ascii2bin();
static void PEM_proc_type();
static void PEM_dek_info();
static void PEM_make_key_des_md5();
static void PEM_make_key_des_md2();
static int PEM_des_cbc_encrypt();
static int def_callback();
#endif

static int (*callback)()=def_callback;
static unsigned char data_bin2ascii[65]="ABCDEFGHIJKLMNOPQRSTUVWXYZ\
abcdefghijklmnopqrstuvwxyz0123456789+/";
static unsigned char data_ascii2bin[256]={
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0xFF,0xFF,0xFF,0x3E,0xFF,0xFF,0xFF,0x3F,
	0x34,0x35,0x36,0x37,0x38,0x39,0x3A,0x3B,
	0x3C,0x3D,0xFF,0xFF,0xFF,0x00,0xFF,0xFF,
	0xFF,0x00,0x01,0x02,0x03,0x04,0x05,0x06,
	0x07,0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,
	0x0F,0x10,0x11,0x12,0x13,0x14,0x15,0x16,
	0x17,0x18,0x19,0xFF,0xFF,0xFF,0xFF,0xFF,
	0xFF,0x1A,0x1B,0x1C,0x1D,0x1E,0x1F,0x20,
	0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,
	0x29,0x2A,0x2B,0x2C,0x2D,0x2E,0x2F,0x30,
	0x31,0x32,0x33,0xFF,0xFF,0xFF,0xFF,0xFF,
	};

static int def_callback(buf, num, w)
char *buf;
int num;
int w;
	{
	int i,j;

	for (;;)
		{
		i=des_read_pw_string(buf,num,"Enter password:",w);
		if (i != 0)
			{
			memset(buf,0,num);
			return(-1);
			}
		j=strlen(buf);
		if (j < MIN_LENGTH)
			{
			fprintf(stderr,"password too short, needs to be at least %d chars\n",MIN_LENGTH);
			}
		else
			break;
		}
	return(j);
	}

void PEM_set_getkey_callback(a)
int (*a)();
	{
	callback=a;
	}

static void PEM_proc_type(buf, type)
char *buf;
int type;
	{
	char *str;

	if (type == PEM_TYPE_ENCRYPTED)
		str="ENCRYPTED";
	else if (type == PEM_TYPE_ENCRYPTED)
		str="MIC-ONLY";
	else if (type == PEM_TYPE_MIC_ONLY)
		str="MIC-CLEAR";
	else
		str="BAD-TYPE";
		
	strcat(buf,"Proc-Type: 4,");
	strcat(buf,str);
	strcat(buf,"\n");
	}

static void PEM_dek_info(buf, type, len, str)
char *buf;
int type;
long int len;
char *str;
	{
	static unsigned char map[17]="0123456789ABCDEF";
	char *s;
	long i;
	int j;

	if (type == PEM_DEK_DES_CBC)
		s=PEM_STRING_DES_CBC;
	else if (type == PEM_DEK_DES_EDE)
		s=PEM_STRING_DES_EDE;
	else if (type == PEM_DEK_DES_ECB)
		s=PEM_STRING_DES_ECB;
	else if (type == PEM_DEK_RSA)
		s=PEM_STRING_RSA;
	else if (type == PEM_DEK_RSA_MD2)
		s=PEM_STRING_RSA_MD2;
	else if (type == PEM_DEK_RSA_MD5)
		s=PEM_STRING_RSA_MD5;
	else	s=PEM_STRING_BAD_DAK;
	strcat(buf,"DEK-Info: ");
	strcat(buf,s);
	strcat(buf,",");
	j=strlen(buf);
	for (i=0; i<len; i++)
		{
		buf[j+i*2]  =map[(str[i]>>4)&0x0f];
		buf[j+i*2+1]=map[(str[i]   )&0x0f];
		}
	buf[j+i*2]='\n';
	buf[j+i*2+1]='\0';
	}

int PEM_write_X509(fp, x)
FILE *fp;
X509 *x;
	{
	int i;
	unsigned char *p,*buf;

	i=i2D_X509(x,NULL);
	if (i <= 0)
		{
		PEM_errno=PEM_ERR_X509;
		return(0);
		}
	buf=(unsigned char *)malloc(i);
	if (buf == NULL)
		{
		PEM_errno=PEM_ERR_MALLOC_FAILURE;
		return(0);
		}
	p=buf;
	i=i2D_X509(x,&p);
	i=PEM_write(fp,"X509 CERTIFICATE","",i,buf);
	free(buf);
	return(i);
	}

int PEM_read_X509(fp, x)
FILE *fp;
X509 *x;
	{
	int i;
	char *name,*header;
	unsigned char *data;
	long len;

	if (x == NULL)
		{
		x=X509_new();
		if (x == NULL)
			{
			PEM_errno=PEM_ERR_X509;
			return(0);
			}
		}

	for (;;)
		{
		if (!PEM_read(fp,&name,&header,&len,&data))
			return(0);
		if (strcmp(name,"X509 CERTIFICATE") == 0) break;
		free(name);
		free(header);
		free(data);
		}
	if (!PEM_do_header(header,len,data)) return(0);
	i=D2i_X509(x,len,data);
	free(name);
	free(header);
	free(data);
	if (i <= 0)
		{
		PEM_errno=PEM_ERR_DER;
		return(0);
		}
	return(1);
	}

int PEM_read_RSA(fp, x)
FILE *fp;
RSA *x;
	{
	int i;
	char *name,*header;
	unsigned char *data;
	unsigned char *p;
	long len;

	if (x == NULL)
		{
		x=RSA_new();
		if (x == NULL)
			{
			PEM_errno=PEM_ERR_RSA;
			return(0);
			}
		}

	for (;;)
		{
		if (!PEM_read(fp,&name,&header,&len,&data))
			return(0);
		if (strcmp(name,"RSA PRIVATE KEY") == 0) break;
		free(name);
		free(header);
		free(data);
		}
	if (!PEM_do_header(header,len,data)) return(0);
	p=(unsigned char *)data;
	i=D2i_RSAPrivateKey(x,&p);
	free(name);
	free(header);
	free(data);
	if (i <= 0)
		{
		PEM_errno=PEM_ERR_DER;
		return(0);
		}
	return(1);
	}

int PEM_write_RSA(fp, r, enc, klen, kstr)
FILE *fp;
RSA *r;
int enc;
int klen;
char *kstr;
	{
	unsigned char *str;
	unsigned char *p;
	int size=0,i;
	char buf[1024],buff[512],*pwd,salt[8];
	C_Block key,iv;
	des_key_schedule ks;
	
	size=i2D_RSAPrivateKey(r,NULL);
	if (size <= 0) { PEM_errno=PEM_ERR_X509; return(0); }
	str=(unsigned char *)malloc(size+8);
	if (str == NULL) return(0);
	p=str;
	i=i2D_RSAPrivateKey(r,&p);
	if (!i) { PEM_errno=PEM_ERR_X509; return(0); }

	if (enc && (kstr == NULL))
		{
		klen=(*callback)(buf,1024,1);
		if (klen <= 0)
			{
			memset(buf,0,1024);
			memset(str,0,size);
			free(str);
			PEM_errno=PEM_ERR_READ_KEY;
			return(0);
			}
		pwd=buf;
		}
	else
		pwd=kstr;

	if (enc)
		{
		MD5_rand_seed(size,str); /**/
		MD5_rand(8,iv);
		PEM_make_key_des_md5(klen,pwd,DES_COUNT,
			(des_cblock *)&(iv[0]),
			(des_cblock *)&(key[0]));
		if (pwd == buf) memset(buf,0,1024);
		set_key((des_cblock *)&(key[0]),ks);

		buff[0]='\0';
		PEM_proc_type(buff,PEM_TYPE_ENCRYPTED);
		PEM_dek_info(buff,PEM_DEK_DES_CBC,8,(char *)iv);
	
		i=PEM_des_cbc_encrypt(str,str,
			size,ks,(C_Block *)&(iv[0]),DES_ENCRYPT);
		}
	else
		buff[0]='\0';
	i=PEM_write(fp,"RSA PRIVATE KEY",buff,i,str);

	if (str != NULL) memset(str,0,size);
	free(str);
	memset(salt,0,8);
	memset(key,0,8);
	memset(ks,0,sizeof(ks));
	memset(iv,0,8);
	return(i);
	}

int PEM_do_header(header, len, data)
char *header;
int len;
unsigned char *data;
	{
	int i,v;
	C_Block key,iv;
	des_key_schedule ks;
	char buf[1024];

	if ((header == NULL) || (*header == '\0') || (*header == '\n'))
		return(1);
	if (strncmp(header,"Proc-Type: ",11) != 0)
		{
		PEM_errno=PEM_ERR_NOT_PROC_TYPE;
		return(0);
		}
	header+=11;
	if (*header != '4') return(0); header++;
	if (*header != ',') return(0); header++;
	if (strncmp(header,"ENCRYPTED",9) != 0)
		{
		PEM_errno=PEM_ERR_NOT_ENCRYPTED;
		return(0);
		}
	for (; (*header != '\n') && (*header != '\0'); header++)
		;
	if (*header == '\0')
		{
		PEM_errno=PEM_ERR_SHORT_HEADER;
		return(0);
		}
	header++;
	if (strncmp(header,"DEK-Info: ",10) != 0)
		{
		PEM_errno=PEM_ERR_NOT_DEK_INFO;
		return(0);
		}
	header+=10;
	if (strncmp(header,"DES-CBC,",8) == 0)
		{
		header+=8;
		for (i=0; i<8; i++) iv[i]=0;
		for (i=0; i<16; i++)
			{
			if ((*header >= '0') && (*header <= '9'))	v=*header-'0';
			else if ((*header >= 'A') && (*header <= 'F'))	v=*header-'A'+10;
			else if ((*header >= 'a') && (*header <= 'f'))	v=*header-'a'+10;
			else
				{
				PEM_errno=PEM_ERR_BAD_CBC_IV_CHARS;
				return(0);
				}
			header++;
			iv[i/2]|=v<<((!(i&1))*4);
			}

		if ((*callback)(buf,1024,0) <= 0)
			{
			PEM_errno=PEM_ERR_BAD_READ_PW_STRING;
			return(0);
			}
		PEM_make_key_des_md5(strlen(buf),buf,DES_COUNT,
			(C_Block *)&(iv[0]),
			(C_Block *)&(key[0]));
		memset(buf,0,1024);
		set_key((des_cblock *)&(key[0]),ks);
		memset(key,0,8);
		i=PEM_des_cbc_encrypt(data,data,
			(long)len,ks,(C_Block *)&(iv[0]),DES_DECRYPT);
		memset(iv,0,8);
		memset(ks,0,sizeof(ks));
		if (i == -1) return(0);
		}
	else
		{
		PEM_errno=PEM_ERR_UNSUPPORTED_ENCRYPTION;
		return(0);
		}
	return(1);
	}

static int PEM_bin2ascii(len, f, t)
int len;
unsigned char *f;
unsigned char *t;
	{
	int r,i,j,chunks;
	unsigned long l;

	i=len/BIN_PER_LINE;
	j=len%BIN_PER_LINE;
	if (j != 0) j=((j+2)/3)*4+1;
	r=(i*CHAR_PER_LINE+j);
	if (t == NULL) return(r);

	chunks=0;
	for (;;)
		{
		if (len >= 3)
			{
			l= (f[0]<<16)|(f[1]<<8)|f[2];
			*(t++)=conv_bin2ascii(l>>18);
			*(t++)=conv_bin2ascii(l>>12);
			*(t++)=conv_bin2ascii(l>> 6);
			*(t++)=conv_bin2ascii(l    );
			chunks++;
			f+=3;
			}
		else
			{
			l=0;
			l=(f[0]<<16);
			if (len == 2) l|=(f[1]<<8);

			*(t++)=conv_bin2ascii(l>>18);
			*(t++)=conv_bin2ascii(l>>12);
			*(t++)=(len == 1)?'=':conv_bin2ascii(l>> 6);
			*(t++)='=';
			chunks++;
			}
		if (chunks == CHUNKS_PER_LINE)
			{
			*(t++)='\n';
			chunks=0;
			}
		if (len <= 3) break;
		len-=3;
		}
	if (chunks != 0) *(t++)='\n';
	*(t++)='\0';
	return(r);
	}

static int PEM_ascii2bin(f, t)
unsigned char *f;
unsigned char *t;
	{
	int i,j,chunks,err,c,sub;
	unsigned long l,len;

	err=len=0;
	chunks=0;
	for (;;)
		{
		i=0;
		sub=0;
		l=0;
		for (i=0; i<4; i++) /* get 4 chars */
			{
			c= *(f++);
			j=conv_ascii2bin(c);
			if (j == 0xFF)
				{
				err=1;
				PEM_errno=PEM_ERR_BASE64_BAD_INPUT_CHAR;
				break;
				}
			if (c == '=') sub++;
			l<<=6;
			l|=j;
			}
		if (err) break;
		if (++chunks > CHUNKS_PER_LINE)
			{
			err=1;
			PEM_errno=PEM_ERR_BASE64_LINE_TOO_LONG;
			break;
			}
		if (t != NULL)
			{
			*(t++)=(unsigned char)(l>>16)&0xff;
			*(t++)=(unsigned char)(l>> 8)&0xff;
			*(t++)=(unsigned char)(l    )&0xff;
			len+=3;
			}
		if (sub) break;
		if (*f == '\n')
			{
			if (chunks == CHUNKS_PER_LINE)
				{
				f++;
				chunks=0;
				}
			else	break;
			}
		if (*f == '\0') break;
		}
	if (err) return(0);
	return(len-sub);
	}

int PEM_write(fp, name, header, len, data)
FILE *fp;
char *name;
char *header;
long int len;
unsigned char *data;
	{
	int i;
	unsigned char *buf;
	
	fprintf(fp,"-----BEGIN %s-----\n",name);
	fputs(header,fp);
	fputc('\n',fp);

	i=PEM_bin2ascii(len,data,NULL);
	buf=(unsigned char *)malloc(i+1);
	if (buf == NULL) return(0);
	i=PEM_bin2ascii(len,data,buf);
	i=fwrite(buf,1,i,fp);
	free(buf);
	fprintf(fp,"-----END %s-----\n",name);
	return(i);
	}

int PEM_read(fp, name, header, len, data)
FILE *fp;
char **name;
char **header;
long int *len;
unsigned char **data;
	{
	int end=0,i,j,hl;
	char buf[256];
	BUFFER *nameB;
	BUFFER *headerB;
	BUFFER *dataB;
	
	nameB=buffer_new();
	headerB=buffer_new();
	dataB=buffer_new();
	if ((nameB == NULL) || (headerB == NULL) || (dataB == NULL))
		{
		PEM_errno=PEM_ERR_MALLOC_FAILURE;
		return(0);
		}

	for (;;)
		{
		buf[0]='\0';
		fgets(buf,256,fp);
		if (buf[0] == '\0')
			{
			PEM_errno=PEM_ERR_NO_START_LINE;
			goto err;
			}
		if (strncmp(buf,"-----BEGIN ",11) == 0)
			{
			i=strlen(&(buf[11]));
			if (strncmp(&(buf[11+i-6]),"-----\n",i-6) != 0)
				continue;
			if (!buffer_grow(nameB,i+9))
				{
				PEM_errno=PEM_ERR_MALLOC_FAILURE;
				goto err;
				}
			strncpy(nameB->data,&(buf[11]),i-6);
			nameB->data[i-6]='\0';
			break;
			}
		}
	hl=0;
	if (!buffer_grow(headerB,256))
		{ PEM_errno=PEM_ERR_MALLOC_FAILURE; goto err; }
	headerB->data[0]='\0';
	for (;;)
		{
		buf[0]='\0';
		fgets(buf,256,fp);
		if (buf[0] == '\0') break;
		if (buf[0] == '\n') break;
		i=strlen(buf);
		if (!buffer_grow(headerB,hl+i+9))
			{ PEM_errno=PEM_ERR_MALLOC_FAILURE; goto err; }
		strncpy(&(headerB->data[hl]),buf,i);
		headerB->data[hl+i]='\0';
		hl+=i;
		}

	j=0;
	if (!buffer_grow(dataB,1024))
		{ PEM_errno=PEM_ERR_MALLOC_FAILURE; goto err; }
	dataB->data[0]='\0';
	for (;;)
		{
		buf[0]='\0';
		fgets(buf,265,fp);
		if (buf[0] == '\0') break;
		i=strlen(buf);
		if (i != 65) end=1;
		if (strncmp(buf,"-----END ",9) == 0)
			break;
		if (i > 65) break;
		if (!buffer_grow(dataB,i+j+9))
			{ PEM_errno=PEM_ERR_MALLOC_FAILURE; goto err; }
		strncpy(&(dataB->data[j]),buf,i);
		dataB->data[j+i]='\0';
		j+=i;
		if (end)
			{
			buf[0]='\0';
			fgets(buf,265,fp);
			break;
			}
		}
	i=strlen(nameB->data);
	if (	(strncmp(buf,"-----END ",9) != 0) ||
		(strncmp(nameB->data,&(buf[9]),i) != 0) ||
		(strncmp(&(buf[9+i]),"-----\n",6) != 0))
		{
		PEM_errno=PEM_ERR_BAD_END_LINE;
		goto err;
		}

	i=PEM_ascii2bin((unsigned char *)dataB->data,
		(unsigned char *)dataB->data);
	dataB->data[i]='\0';

	if (i == 0) goto err;
	*name=nameB->data;
	*header=headerB->data;
	*data=(unsigned char *)dataB->data;
	*len=i;
	free(nameB);
	free(headerB);
	free(dataB);
	return(1);
err:
	buffer_free(nameB);
	buffer_free(headerB);
	buffer_free(dataB);
	return(0);
	}

/* pkcs-5 pbeWithMD5AndDESCBC { 1 2 840 113549 1 5 1 } */
static void PEM_make_key_des_md5(strl, str, count, iv, key)
int strl;
char *str;
int count;
des_cblock (*iv);
des_cblock (*key);
	{
	MD5_CTX c;
	unsigned char md[MD5_DIGEST_LENGTH];
	int i;

	MD5Init(&c);
	MD5Update(&c,(unsigned char *)str,strl);
	MD5Update(&c,(unsigned char *)iv,8);
	MD5Final(&(md[0]),&c);
	for (i=1; i<count; i++)
		MD5(MD5_DIGEST_LENGTH,&(md[0]),&(md[0]));
	memcpy(key,&(md[0]),DES_KEY_SZ);
	memset(&(md[0]),0,MD5_DIGEST_LENGTH);
	des_set_odd_parity(key);
	}

/* pkcs-5 pbeWithMD2AndDESCBC { 1 2 840 113549 1 5 3 } */
static void PEM_make_key_des_md2(strl, str, count, iv, key)
int strl;
char *str;
int count;
des_cblock (*iv);
des_cblock (*key);
	{
	MD2_CTX c;
	unsigned char md[MD2_DIGEST_LENGTH];
	int i;

	MD2Init(&c);
	MD2Update(&c,(unsigned char *)str,strl);
	MD2Update(&c,(unsigned char *)iv,8);
	MD2Final(&(md[0]),&c);
	for (i=1; i<count; i++)
		MD2(MD5_DIGEST_LENGTH,&(md[0]),&(md[0]));
	memcpy(key,&(md[0]),DES_KEY_SZ);
	memset(&(md[0]),0,MD2_DIGEST_LENGTH);
	des_set_odd_parity(key);
	}

/* must have room for padding of upto 8 bytes */
static int PEM_des_cbc_encrypt(input, output, length, ks, iv, encrypt)
unsigned char *input;
unsigned char *output;
long int length;
struct des_ks_struct *ks;
des_cblock (*iv);
int encrypt;
	{
	int i,v;

	if (encrypt)
		{
		v=8-length%8;
		for (i=0; i < v; i++)
			input[length+i]=v;
		des_cbc_encrypt((des_cblock*)input,(des_cblock *)output,
			length+v,ks,iv,encrypt);
		return(length+v);
		}
	else
		{
		des_cbc_encrypt((des_cblock *)input,(des_cblock *)output,
			length,ks,iv,encrypt);
		v=output[length-1];
		if (v > 8) return(-1);
		for (i=0; i<v; i++)
			if (output[length-1-i] != v)
				{
				PEM_errno=PEM_ERR_BAD_CBC_DECRYPT;
				return(-1);
				}
		return(length-v);
		}
	}

