/* lhash.c */
/* Copyright (C) 1995 Eric Young (eay@mincom.oz.au).
 * All rights reserved.
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.
 * See the COPYRIGHT file in the SSLeay distribution for more details.
 */

static char *version="\0lhash part of SSLeay v 0.4.3 15/06/95";

/* Code for dynamic hash table routines
 * Author - Eric Young v 1.7
 *
 * 1.7 eay - Removed the fputs() for realloc failures - the code
 *           should silently tolerate them.  I have also fixed things
 *           lint complained about 04/05/95
 *
 * 1.6 eay - Fixed an invalid pointers in contract/expand 27/07/92
 *
 * 1.5 eay - Fixed a misuse of realloc in expand 02/03/1992
 *
 * 1.4 eay - Fixed lh_doall so the function can call lh_delete 28/05/91
 *
 * 1.3 eay - Fixed a few lint problems 19/3/1991
 *
 * 1.2 eay - Fixed lh_doall problem 13/3/1991
 *
 * 1.1 eay - Added lh_doall
 *
 * 1.0 eay - First version
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "lhash.h"

#define MIN_NODES	16
#define UP_LOAD		(2*LH_LOAD_MULT) /* load times 256  (default 2) */
#define DOWN_LOAD	(LH_LOAD_MULT)   /* load times 256  (default 1) */

#ifdef PROTO
static void expand(LHASH *lh);
static void contract(LHASH *lh);
static LHASH_NODE **getrn(LHASH *lh, char *data, unsigned long *rhash);
#else
static void expand();
static void contract();
static LHASH_NODE **getrn();
#endif

LHASH *lh_new(h, c)
unsigned long (*h)();
int (*c)();
	{
	LHASH *ret;
	int i;

	if ((ret=(LHASH *)malloc(sizeof(LHASH))) == NULL)
		goto err0;
	if ((ret->b=(LHASH_NODE **)malloc(sizeof(LHASH_NODE *)*MIN_NODES)) == NULL)
		goto err1;
	for (i=0; i<MIN_NODES; i++)
		ret->b[i]=NULL;
	ret->comp=(c == NULL)?strcmp:c;
	ret->hash=(h == NULL)?lh_strhash:h;
	ret->num_nodes=MIN_NODES/2;
	ret->num_alloc_nodes=MIN_NODES;
	ret->p=0;
	ret->pmax=MIN_NODES/2;
	ret->up_load=UP_LOAD;
	ret->down_load=DOWN_LOAD;
	ret->num_items=0;

	ret->num_expands=0;
	ret->num_expand_reallocs=0;
	ret->num_contracts=0;
	ret->num_contract_reallocs=0;
	ret->num_hash_calls=0;
	ret->num_comp_calls=0;
	ret->num_insert=0;
	ret->num_replace=0;
	ret->num_delete=0;
	ret->num_no_delete=0;
	ret->num_retreve=0;
	ret->num_retreve_miss=0;
	ret->num_hash_comps=0;

	return(ret);
err1:
	free((char *)ret);
err0:
	return(NULL);
	}

void lh_free(lh)
LHASH *lh;
	{
	free((char *)lh->b);
	free((char *)lh);
	}

char *lh_insert(lh, data)
LHASH *lh;
char *data;
	{
	unsigned long hash;
	LHASH_NODE *nn,**rn;
	char *ret;

	if (lh->up_load <= (lh->num_items*LH_LOAD_MULT/lh->num_nodes))
		expand(lh);

	rn=getrn(lh,data,&hash);

	if (*rn == NULL)
		{
		if ((nn=(LHASH_NODE *)malloc(sizeof(LHASH_NODE))) == NULL)
			return(NULL);
		nn->data=data;
		nn->next=NULL;
#ifndef NO_HASH_COMP
		nn->hash=hash;
#endif
		*rn=nn;
		ret=NULL;
		lh->num_insert++;
		lh->num_items++;
		}
	else /* replace same key */
		{
		ret= (*rn)->data;
		(*rn)->data=data;
		lh->num_replace++;
		}
	return(ret);
	}

char *lh_delete(lh, data)
LHASH *lh;
char *data;
	{
	unsigned long hash;
	LHASH_NODE *nn,**rn;
	char *ret;

	rn=getrn(lh,data,&hash);

	if (*rn == NULL)
		{
		lh->num_no_delete++;
		return(NULL);
		}
	else
		{
		nn= *rn;
		*rn=nn->next;
		ret=nn->data;
		free((char *)nn);
		lh->num_delete++;
		}

	lh->num_items--;
	if ((lh->num_nodes > MIN_NODES) &&
		(lh->down_load >= (lh->num_items*LH_LOAD_MULT/lh->num_nodes)))
		contract(lh);

	return(ret);
	}

char *lh_retrieve(lh, data)
LHASH *lh;
char *data;
	{
	unsigned long hash;
	LHASH_NODE **rn;
	char *ret;

	rn=getrn(lh,data,&hash);

	if (*rn == NULL)
		{
		lh->num_retreve_miss++;
		return(NULL);
		}
	else
		{
		ret= (*rn)->data;
		lh->num_retreve++;
		}
	return(ret);
	}

void lh_doall(lh, func)
LHASH *lh;
void (*func)();
	{
	unsigned int i;
	LHASH_NODE *a,*n;

	for (i=0; i<lh->num_nodes; i++)
		{
		a=lh->b[i];
		while (a != NULL)
			{
			/* 28/05/91 - eay - n added so items can be deleted
			 * via lh_doall */
			n=a->next;
			func(a->data);
			a=n;
			}
		}
	}

static void expand(lh)
LHASH *lh;
	{
	LHASH_NODE **n,**n1,**n2,*np;
	int p,i,j;
	unsigned long hash,nni;

	lh->num_nodes++;
	lh->num_expands++;
	p=lh->p++;
	n1= &(lh->b[p]);
	n2= &(lh->b[p+lh->pmax]);
	*n2=NULL;	 /* 27/07/92 - eay - undefined pointer bug */
	nni=lh->num_alloc_nodes;
	
	for (np= *n1; np != NULL; )
		{
#ifndef NO_HASH_COMP
		hash=np->hash;
#else
		hash=(*(lh->hash))(np->data);
		lh->num_hash_calls++;
#endif
		if ((hash%nni) != p)
			{ /* move it */
			*n1= (*n1)->next;
			np->next= *n2;
			*n2=np;
			}
		else
			n1= &((*n1)->next);
		np= *n1;
		}

	if ((lh->p) >= lh->pmax)
		{
		j=lh->num_alloc_nodes*2;
		n=(LHASH_NODE **)realloc((char *)lh->b,
			(unsigned int)sizeof(LHASH_NODE *)*j);
		if (n == NULL)
			{
/*			fputs("realloc error in lhash",stderr); /**/
			lh->p=0;
			return;
			}
		/* else */
		for (i=lh->num_alloc_nodes; i<j; i++)	/* 26/02/92 eay */
			n[i]=NULL;			/* 02/03/92 eay */
		lh->pmax=lh->num_alloc_nodes;
		lh->num_alloc_nodes=j;
		lh->num_expand_reallocs++;
		lh->p=0;
		lh->b=n;
		}
	}

static void contract(lh)
LHASH *lh;
	{
	LHASH_NODE **n,*n1,*np;

	np=lh->b[lh->p+lh->pmax-1];
	lh->b[lh->p+lh->pmax-1]=NULL; /* 24/07-92 - eay - weird but :-( */
	if (lh->p == 0)
		{
		n=(LHASH_NODE **)realloc((char *)lh->b,
			(unsigned int)(sizeof(LHASH_NODE *)*lh->pmax));
		if (n == NULL)
			{
/*			fputs("realloc error in lhash",stderr); /**/
			return;
			}
		lh->num_contract_reallocs++;
		lh->num_alloc_nodes/=2;
		lh->pmax/=2;
		lh->p=lh->pmax-1;
		lh->b=n;
		}
	else
		lh->p--;

	lh->num_nodes--;
	lh->num_contracts++;

	n1=lh->b[lh->p];
	if (n1 == NULL)
		lh->b[lh->p]=np;
	else
		{
		while (n1->next != NULL)
			n1=n1->next;
		n1->next=np;
		}
	}

static LHASH_NODE **getrn(lh, data, rhash)
LHASH *lh;
char *data;
unsigned long *rhash;
	{
	LHASH_NODE **ret,*n1;
	unsigned long hash,nn;
	int (*cf)();

	hash=(*(lh->hash))(data);
	lh->num_hash_calls++;
	*rhash=hash;

	nn=hash%lh->pmax;
	if (nn < lh->p)
		nn=hash%lh->num_alloc_nodes;

	cf=lh->comp;
	ret= &(lh->b[nn]);
	for (n1= *ret; n1 != NULL; n1=n1->next)
		{
#ifndef NO_HASH_COMP
		lh->num_hash_comps++;
		if (n1->hash != hash)
			{
			ret= &(n1->next);
			continue;
			}
#endif
		lh->num_comp_calls++;
		if ((*cf)(n1->data,data) == 0)
			break;
		ret= &(n1->next);
		}
	return(ret);
	}

void lh_stats(lh, out)
LHASH *lh;
FILE *out;
	{
	fprintf(out,"num_items             = %ld\n",lh->num_items);
	fprintf(out,"num_nodes             = %ld\n",lh->num_nodes);
	fprintf(out,"num_alloc_nodes       = %ld\n",lh->num_alloc_nodes);
	fprintf(out,"num_expands           = %ld\n",lh->num_expands);
	fprintf(out,"num_expand_reallocs   = %ld\n",lh->num_expand_reallocs);
	fprintf(out,"num_contracts         = %ld\n",lh->num_contracts);
	fprintf(out,"num_contract_reallocs = %ld\n",lh->num_contract_reallocs);
	fprintf(out,"num_hash_calls        = %ld\n",lh->num_hash_calls);
	fprintf(out,"num_comp_calls        = %ld\n",lh->num_comp_calls);
	fprintf(out,"num_insert            = %ld\n",lh->num_insert);
	fprintf(out,"num_replace           = %ld\n",lh->num_replace);
	fprintf(out,"num_delete            = %ld\n",lh->num_delete);
	fprintf(out,"num_no_delete         = %ld\n",lh->num_no_delete);
	fprintf(out,"num_retreve           = %ld\n",lh->num_retreve);
	fprintf(out,"num_retreve_miss      = %ld\n",lh->num_retreve_miss);
	fprintf(out,"num_hash_comps        = %ld\n",lh->num_hash_comps);
	fprintf(out,"num_alloc_nodes       = %ld\n",lh->num_alloc_nodes);
#ifdef DEBUG
	fprintf(out,"p                     = %ld\n",lh->p);
	fprintf(out,"pmax                  = %ld\n",lh->pmax);
	fprintf(out,"up_load               = %ld\n",lh->up_load);
	fprintf(out,"down_load             = %ld\n",lh->down_load);
#endif
	}

void lh_node_stats(lh, out)
LHASH *lh;
FILE *out;
	{
	LHASH_NODE *n;
	unsigned long i,num;

	for (i=0; i<lh->num_nodes; i++)
		{
		for (n=lh->b[i],num=0; n != NULL; n=n->next)
			num++;
		fprintf(out,"node %6d -> %3d\n",i,num);
		}
	}

void lh_node_usage_stats(lh, out)
LHASH *lh;
FILE *out;
	{
	LHASH_NODE *n;
	unsigned long i,num;
	unsigned long total=0,n_used=0;

	for (i=0; i<lh->num_nodes; i++)
		{
		for (n=lh->b[i],num=0; n != NULL; n=n->next)
			num++;
		if (num != 0)
			{
			n_used++;
			total+=num;
			}
		}
	fprintf(out,"%d nodes used out of %ld\n",n_used,lh->num_nodes);
	fprintf(out,"%d items\n",total);
	if (n_used == 0) return;
	fprintf(out,"load %d.%02d  actual load %d.%02d\n",
		(int)(total/lh->num_nodes),
		(int)((total%lh->num_nodes)*100/lh->num_nodes),
		(int)(total/n_used),
		(int)((total%n_used)*100/n_used));
	}

/*
static unsigned long lh_strhash(str)
char *str;
	{
	int i,l;
	unsigned long ret=0;
	unsigned short *s;

	l=(strlen(str)+1)/2;
	s=(unsigned short *)str;
	for (i=0; i<l; i++)
		ret^=(s[i]<<(i&0x0f));
	return(ret);
	} /**/

unsigned long lh_strhash(c)
char *c;
	{
	unsigned long ret=0;
	int i=0;

	if ((c == NULL) || (*c == '\0'))
		return(ret);
	while (*c)
		{
		ret^=(((unsigned long)(*c)) <<i)|((*c)<<(i&3));
		i+= *(c++);
		i%=24;
		}
	return(ret);
	}


