/* der.c */
/* Copyright (C) 1995 Eric Young (eay@mincom.oz.au).
 * All rights reserved.
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.
 * See the COPYRIGHT file in the SSLeay distribution for more details.
 */

#include <stdio.h>
#include "X509.h"
#include "der.h"

static char *version="\0DER part of SSLeay v 0.4.2 06/06/95";

static unsigned long der_get_length();
static void der_put_length();
static void der_print_info();

int DER_get_object(pp,plength,ptag,pclass)
unsigned char **pp;
unsigned long *plength;
int *ptag,*pclass;
	{
	int i,ret;
	unsigned long l;
	unsigned char *p= *pp;
	int tag,class;

	ret=(*p&DER_CONSTRUCTED);
	class=(*p&DER_PRIVATE);
	i= *p&DER_PRIMATIVE_TAG;
	if (i == DER_PRIMATIVE_TAG)
		{		/* high-tag */
		p++;
		l=0;
		while (*p&0x80)
			{
			l<<=7;
			l|= *(p++)&0x7f;
			}
		l<<=7;
		l|= *(p++)&0x7f;
		tag=l;
		}
	else
		{ /* constructed */
		tag=i;
		p++;
		}
	*pp=p;
	*ptag=tag;
	*pclass=class;
	*plength=der_get_length(pp);
	return(ret);
	}


int DER_object_size(constructed,length,tag)
int constructed;
int length,tag;
	{
	int ret;

	ret=length;
	ret++;
	if (tag >= 31)
		{
		while (tag > 0)
			{
			tag>>=7;
			ret++;
			}
		}
	if ((length == 0) && constructed)
		ret+=2;
	ret++;
	if (length > 127)
		{
		while (length > 0)
			{
			length>>=8;
			ret++;
			}
		}
	return(ret);
	}

/* class 0 is constructed */
void DER_put_object(pp,constructed,length,tag,class)
unsigned char **pp;
int constructed;
int length,tag,class;
	{
	unsigned char *p= *pp;
	int i;

	i=(constructed)?DER_CONSTRUCTED:0;
	i|=(class&DER_PRIVATE);
	if (tag < 31)
		*(p++)=i|(tag&DER_PRIMATIVE_TAG);
	else
		{
		*(p++)=i|DER_PRIMATIVE_TAG;
		while (tag > 0x3f)
			{
			*(p++)=(tag&0x3f)|0x80;
			tag>>=7;
			}
		*(p++)=(tag&0x3f);
		}
	if (constructed && (length == 0))
		*(p++)=0x80; /* der_put_length would output 0 instead */
	else
		der_put_length(&p,length);
	*pp=p;
	}

static void der_put_length(pp,length)
unsigned char **pp;
int length;
	{
	unsigned char *p= *pp;
	int i,l;
	if (length < 0x7f)
		*(p++)=length;
	else
		{
		l=length;
		for (i=0; l > 0; i++)
			l>>=8;
		*(p++)=i|0x80;
		l=i;
		while (i-- > 0)
			{
			p[i]=length&0xff;
			length>>=8;
			}
		p+=l;
		}
	*pp=p;
	}

static unsigned long der_get_length(pp)
unsigned char **pp;
	{
	unsigned char *p= *pp;
	unsigned long ret=0;
	int i;

	i= *p&0x7f;
	if (*(p++) & 0x80)
		{
		while (i-- > 0)
			{
			ret<<=8;
			ret|= *(p++);
			}
		}
	else
		ret=i;
	*pp=p;
	return(ret);
	}

int i2D_DER_BIT_STRING(a,pp)
DER_BIT_STRING *a;
unsigned char **pp;
	{
	int ret,i,j,r,bits;
	unsigned char *p;

	/* our bit strings are always a multiple of 8 :-) */
	bits=0;
	ret=1+a->length;
	r=DER_object_size(0,ret,DER_DER_BIT_STRING);
	if (pp == NULL) return(r);
	p= *pp;

	DER_put_object(&p,0,ret,DER_DER_BIT_STRING,DER_UNIVERSAL);
	if (bits == 0)
		j=0;
	else	j=8-bits;
	*(p++)=j;
	for (i=0; i<a->length; i++)
		*(p++)=a->data[i];
	if (a->length > 0) p[-1]&=(0xff<<j);
	*pp= p;
	return(r);
	}

int i2D_OCTET_STRING(a,pp)
DER_BIT_STRING *a;
unsigned char **pp;
	{
	int ret,i,r;
	unsigned char *p;

	ret=a->length;
	r=DER_object_size(0,ret,DER_OCTET_STRING);
	if (pp == NULL) return(r);
	p= *pp;

	DER_put_object(&p,0,ret,DER_OCTET_STRING,DER_UNIVERSAL);
	for (i=0; i<a->length; i++)
		*(p++)=a->data[i];
	*pp= p;
	return(r);
	}

int i2D_UTCTime(a,pp)
unsigned char *a;
unsigned char **pp;
	{
	int ret,r;
	unsigned char *p;

	ret=strlen(a);
	r=DER_object_size(0,ret,DER_UTCTIME);
	if (pp == NULL) return(r);
	p= *pp;

	DER_put_object(&p,0,ret,DER_UTCTIME,DER_UNIVERSAL);
	memcpy(p,a,ret);
	p+=ret;
	*pp=p;
	return(r);
	}

int i2D_DER_OBJECT(a,pp)
DER_OBJECT *a;
unsigned char **pp;
	{
	int i,j,r,n,ret,f;
	unsigned long l;
	unsigned char *p;

	ret=0;
	if (a->num >= 2) ret=1;
	for (i=2; i<a->num; i++)
		{
		l=a->values[i];
		do	{
			ret++;
			l>>=7;
			} while (l);
		}
	r=DER_object_size(0,ret,DER_DER_OBJECT_IDENTIFIER);
	if (pp == NULL) return(r);

	p= *pp;
	DER_put_object(&p,0,ret,DER_DER_OBJECT_IDENTIFIER,DER_UNIVERSAL);

	if (a->num == 0)
		*(p++)=(unsigned char)(a->values[0]*40);
	else if (a->num >= 2)
		{
		*(p++)=(unsigned char)(a->values[0]*40+((a->values[1])%40));
		for (i=2; i<a->num; i++)
			{
			l=a->values[i];
			j=0;
			do	{
				j++;
				l>>=7;
				} while (l);
			l=a->values[i];
			n=j; /* n is number of bytes */
			f=1;
			for (j--; j >= 0; j--)
				{
				p[j]=(unsigned char)(l&0x7f);
				l>>=7;
				if (f)
					f=0;
				else
					p[j]|=0x80;
				}
			p+=n;
			}
		}
	*pp=p;
	return(r);
	}

int i2D_INTEGER(a,pp)
DER_BIT_STRING *a;
unsigned char **pp;
	{
	int pad=0,ret,r;
	unsigned char *p;

	ret=a->length;
	if ((ret > 0) && (a->data[0]&0x80))
		pad=1;
	/* add one if top bit is set - we are only saveing +ve numbers */
	ret+=pad;
	r=DER_object_size(0,ret,DER_INTEGER);
	if (pp == NULL) return(r);
	p= *pp;

	DER_put_object(&p,0,ret,DER_INTEGER,DER_UNIVERSAL);
	if (pad) *(p++)=0;
	memcpy(p,a->data,a->length);
	p+=a->length;
	*pp=p;
	return(r);
	}

void i2f_DER_BIT_STRING(fp,a)
FILE *fp;
DER_BIT_STRING *a;
	{
	int i;
	static char *h="0123456789ABCDEF";

	if (a->length == 0)
		fprintf(fp,"= NULL");
	else
		{
		fprintf(fp,"%c ",(a->length > 35)?'=':'-');
		for (i=0; i<a->length; i++)
			{
			if ((i != 0) && (i%35 == 0))
				{
				if (i+35 >= a->length)
					fputs("\n- ",fp);
				else
					fputs("\n= ",fp);
				}
			fputc((h[((unsigned char)a->data[i]>>4)&0x0f]),fp);
			fputc((h[((unsigned char)a->data[i]   )&0x0f]),fp);
			}
		}
	fprintf(fp,"\n");
	}

void i2f_DER_OBJECT(fp,a)
FILE *fp;
DER_OBJECT *a;
	{
	int i;

	fprintf(fp,"= ");
	for (i=0; i<a->num; i++)
		fprintf(fp,"%d%s",a->values[i],((i+1) == a->num)?"":" ");
	fprintf(fp,"\n");
	}

#ifdef undef
void der_set_object(a,d)
DER_OBJECT *a;
unsigned long *d;
	{
	int i,n;

	for (n=0; d[n]; n++)
		;
	if (a->num < n)
		{
		if (a->values == NULL)
			a->values=(unsigned long *)malloc(
				sizeof(unsigned long)*n);
		else
			a->values=(unsigned long *)realloc(a->values,
				sizeof(unsigned long)*n);
		}
	a->num=n;
	for (i=0; i<n; i++)
		a->values[i]=d[i];
	}
#endif

int D2i_INTEGER(bs,pp)
DER_BIT_STRING *bs;
unsigned char **pp;
	{
	unsigned char *p,*s;
	unsigned long len;
	int tag,class;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_INTEGER)
		{
		DER_errno=DER_ERR_EXPECTING_AN_INTEGER_IN_D2I_INTEGER;
		return(0);
		}
	if (len > 0)
		{
		s=(unsigned char *)malloc(len);
		if (s == NULL)
			{
			DER_errno=DER_ERR_OUT_OF_MEM;
			return(0);
			}
		memcpy(s,p,len);
		p+=len;
		}
	else
		s=NULL;

	bs->length=len;
	if (bs->data != NULL) free(bs->data);
	bs->data=s;
	*pp=p;
	return(1);
	}

#ifdef undef
int D2i_BN_INTEGER(max,d,pp)
int max;
unsigned char *d;
unsigned char **pp;
	{
	unsigned char *p;
	unsigned long len;
	int tag,class;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
#ifdef DER_DEBUG
	fprintf(stderr,"D2i_INTEGER len=%d ",len);
	der_print_info(stderr,tag,class);
#endif
	if (tag != DER_INTEGER)
		{
		fprintf(stderr,"expecting an INTEGER in D2i_INTEGER\n");
		exit(1);
		}
	if (max > len) max=len;
	memcpy(d,p,max);
	p+=len;
	*pp=p;
	return(max);
	}
#endif

int D2i_DER_OBJECT(o,pp)
DER_OBJECT *o;
unsigned char **pp;
	{
	unsigned char *p;
	unsigned long len,ret;
	int tag,class;
	int i,n;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_DER_OBJECT_IDENTIFIER)
		{
		DER_errno=DER_ERR_EXPECTING_AN_OBJECT_IDENTIFIER_IN_D2I_DER_OBJECT;
		return(0);
		}
	n=0;
	if (len >= 1) n=2;
	for (i=1; i<len; i++)
		if (!(p[i]&0x80)) n++;
	o->num=n;
	if (o->values != NULL) free(o->values);
	o->values=(unsigned long *)malloc(sizeof(unsigned long)*n);
	if (o->values == NULL)
		{
		DER_errno=DER_ERR_OUT_OF_MEM;
		return(0);
		}
	if (len >= 1)
		{
		o->values[0]=(unsigned long)p[0]/40;
		o->values[1]=(unsigned long)p[0]%40;
		}
	p++;
	
	for (i=2; i<n; i++)
		{
		ret=0;
		do	{
			ret=(ret<<7)|(*p&0x7f);
			} while (*(p++) & 0x80);
		o->values[i]=ret;
		}
	*pp=p;
	return(1);
	}

int D2i_PRINTABLESTRING(sp,pp)
unsigned char **sp;
unsigned char **pp;
	{
	unsigned char *p,*s;
	unsigned long len;
	int tag,class;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_PRINTABLESTRING)
		{
		DER_errno=DER_ERR_EXPECTING_A_PRINTABLESTRING_IN_D2I_PRINTABLESTRING;
		return(0);
		}
	if (*sp != NULL) free(*sp);
	s=(unsigned char *)malloc(len+1);
	if (s == NULL)
		{
		DER_errno=DER_ERR_OUT_OF_MEM;
		return(0);
		}
	memcpy(s,p,len);
	p+=len;

	s[len]='\0';
	*sp=s;
	*pp=p;
	return(1);
	}

D2i_DER_BIT_STRING(bp,pp)
DER_BIT_STRING *bp;
unsigned char **pp;
	{
	unsigned char *p,*s;
	unsigned long len;
	int tag,class;
	int i;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_DER_BIT_STRING)
		{
		DER_errno=DER_ERR_EXPECTING_A_DER_BIT_STRING_IN_D2i_DER_BIT_STRING;
		return(0);
		}
	i= *(p++);
	if (len-- > 1) /* using one because of the bits left byte */
		{
		s=(unsigned char *)malloc(len);
		if (s == NULL)
			{
			DER_errno=DER_ERR_OUT_OF_MEM;
			return(0);
			}
		memcpy(s,p,len);
		s[len-1]&=(0xff<<i);
		p+=len;
		}
	else
		s=NULL;

	bp->length=len;
	if (bp->data != NULL) free(bp->data);
	bp->data=s;
	*pp=p;
	return(1);
	}

int D2i_OCTET_STRING(bp,pp)
DER_BIT_STRING *bp;
unsigned char **pp;
	{
	unsigned char *p,*s;
	unsigned long len;
	int tag,class;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_OCTET_STRING)
		{
		DER_errno=DER_ERR_EXPECTING_AN_OCTET_STRING_IN_D2i_OCTET_STRING;
		return(0);
		}
	if (len > 0)
		{
		s=(unsigned char *)malloc(len);
		if (s == NULL)
			{
			DER_errno=DER_ERR_OUT_OF_MEM;
			return(0);
			}
		memcpy(s,p,len);
		p+=len;
		}
	else
		s=NULL;

	bp->length=len;
	if (bp->data != NULL) free(bp->data);
	bp->data=s;
	*pp=p;
	return(1);
	}

int D2i_UTCTime(sp,pp)
unsigned char **sp;
unsigned char **pp;
	{
	unsigned char *p,*s;
	unsigned long len;
	int tag,class;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	if (tag != DER_UTCTIME)
		{
		DER_errno=DER_ERR_EXPECTING_A_DER_UTCTIME_IN_D2I_UTCTIME;
		return(0);
		}
	if (*sp != NULL) free(*sp);
	s=(unsigned char *)malloc(len+1);
	if (s == NULL)
		{
		DER_errno=DER_ERR_OUT_OF_MEM;
		return(0);
		}
	memcpy(s,p,len);
	p+=len;

	s[len]='\0';
	*sp=s;
	*pp=p;
	return(1);
	}

static void der_print_info(file,tag,class)
FILE *file;
int tag,class;
	{
	if (class & DER_CONSTRUCTED)
		fprintf(file,"construct: ");
	else
		fprintf(file,"primative: ");
	if ((class & DER_PRIVATE) == DER_PRIVATE)
		fprintf(file,"private   : ");
	else if ((class & DER_CONTEXT_SPECIFIC) == DER_CONTEXT_SPECIFIC)
		fprintf(file,"context   : ");
	else if ((class & DER_APPLICATION) == DER_APPLICATION)
		fprintf(file,"applicaion: ");
	else if ((class & DER_UNIVERSAL) == DER_UNIVERSAL) /* lint complains */
		fprintf(file,"universal : ");

	if (tag == DER_EOC)
		fprintf(file,"EOC");
	else if (tag == DER_INTEGER)
		fprintf(file,"INTEGER");
	else if (tag == DER_DER_BIT_STRING)
		fprintf(file,"DER_BIT_STRING");
	else if (tag == DER_OCTET_STRING)
		fprintf(file,"OCTET_STRING");
	else if (tag == DER_NULL)
		fprintf(file,"NULL");
	else if (tag == DER_DER_OBJECT_IDENTIFIER)
		fprintf(file,"DER_OBJECT_IDENTIFIER");
	else if (tag == DER_SEQUENCE)
		fprintf(file,"SEQUENCE");
	else if (tag == DER_SET)
		fprintf(file,"SET");
	else if (tag == DER_PRINTABLESTRING)
		fprintf(file,"PRINTABLESTRING");
	else if (tag == DER_T61STRING)
		fprintf(file,"T61STRING");
	else if (tag == DER_IA5STRING)
		fprintf(file,"IA5STRING");
	else if (tag == DER_UTCTIME)
		fprintf(file,"UTCTIME");
	else	fprintf(file,"unknown");
	fprintf(file,"\n");
	}

static void der_parse2();

void DER_parse(fp,pp)
FILE *fp;
unsigned char *pp;
	{
	der_parse2(fp,&pp,0);
	}

static void der_parse2(fp,pp,depth)
FILE *fp;
unsigned char **pp;
int depth;
	{
	unsigned char *p,*ep;
	unsigned long len;
	int tag,class;
	int i;

	p= *pp;
	DER_get_object(&p,&len,&tag,&class);
	for (i=0; i<depth; i++) fputc(' ',fp);
	fprintf(fp,"depth=%d len=%3d ",depth,len);
	der_print_info(fp,tag,class);
	if (class & DER_CONSTRUCTED)
		{
		ep=p+len;
		while (p < ep)
			der_parse2(fp,&p,depth+1);
		}
	else
		p+=len;
	*pp=p;
	}
