/* server.c */
/* Copyright (C) 1995 Eric Young (eay@mincom.oz.au).
 * All rights reserved.
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.
 * See the COPYRIGHT file in the SSLeay distribution for more details.
 */

#include <stdio.h>
#include <netdb.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <sys/time.h> /* Needed under linux for FD_XXX */
#include <sys/errno.h>
#include <netinet/in.h>
#include "net.h"
#include "X509.h"
#include "ssl.h"

static int init_server();
int do_accept();
void do_server();
int verify_callback();

extern int errno;
void body();

#define BUFSIZZ	10*1024
static int accept_socket=-1;

#define TEST_CERT	"server.PEM"

int verify=SSL_VERIFY_NONE;
int verify_depth=0;
int verify_error=VERIFY_OK;
char *cert_file=TEST_CERT,*key_file=TEST_CERT;

void usage()
	{
	fprintf(stderr,"usage: server args\n");
	fprintf(stderr,"\n");
	fprintf(stderr," -port arg     - port to connect to (default is %d\n",PORT);
	fprintf(stderr," -verify arg   - turn on peer certificate verification\n");
	fprintf(stderr," -Verify arg   - turn on peer certificate verification, must have a cert.\n");
	fprintf(stderr," -cert arg     - certificate file to use, PEM format assumed\n");
	fprintf(stderr,"                 (default is %s)\n",TEST_CERT);
	fprintf(stderr," -key arg      - RSA file to use, PEM format assumed, in cert file if\n");
	fprintf(stderr,"                 not specified (default is %s)\n",TEST_CERT);
	fprintf(stderr," -CApath arg   - PEM format directory of CA's\n");
	fprintf(stderr," -CAfile arg   - PEM format file of CA's\n");
	}

main(argc,argv)
int argc;
char *argv[];
	{
	int port=PORT;
	char *CApath=NULL,*CAfile=NULL;
	int badop=0;

	argc--;
	argv++;

	while (argc >= 1)
		{
		if	(strcmp(*argv,"-port") == 0)
			{
			if (argc-- < 1) goto bad;
			port=atoi(*(++argv));
			if (port == 0) goto bad;
			}
		else if	(strcmp(*argv,"-verify") == 0)
			{
			verify=SSL_VERIFY_PEER;
			if (argc-- < 1) goto bad;
			verify_depth=atoi(*(++argv));
			fprintf(stderr,"verify depth is %d\n",verify_depth);
			}
		else if	(strcmp(*argv,"-Verify") == 0)
			{
			verify=SSL_VERIFY_PEER|SSL_VERIFY_FAIL_IF_NO_PEER_CERT;
			if (argc-- < 1) goto bad;
			verify_depth=atoi(*(++argv));
			fprintf(stderr,"verify depth is %d, must return a certificate\n",verify_depth);
			}
		else if	(strcmp(*argv,"-cert") == 0)
			{
			if (argc-- < 1) goto bad;
			cert_file= *(++argv);
			}
		else if	(strcmp(*argv,"-key") == 0)
			{
			if (argc-- < 1) goto bad;
			key_file= *(++argv);
			}
		else if	(strcmp(*argv,"-CApath") == 0)
			{
			if (argc-- < 1) goto bad;
			CApath= *(++argv);
			}
		else if	(strcmp(*argv,"-CAfile") == 0)
			{
			if (argc-- < 1) goto bad;
			CAfile= *(++argv);
			}
		else
			{
			fprintf(stderr,"unknown option %s\n",*argv);
			badop=1;
			break;
			}
		argc--;
		argv++;
		}
	if (badop)
		{
bad:
		usage();
		exit(1);
		}

	SSL_debug("server.log");
	if ((!X509_load_verify_locations(CAfile,CApath)) ||
		(!X509_set_default_verify_paths()))
		{
		fprintf(stderr,"X509_load_verify_locations error:%s\n",
			X509_error_string(X509_errno));
		exit(1);
		}

	do_server(SERVICE,port,NULL);
	return(0);
	}

void body(hostname,s)
char *hostname;
int s;
	{
	unsigned char buf[BUFSIZZ];
	fd_set readfds;
	int width;
	int i;
	SSL *con;

	fprintf(stderr,"CONNECTED\n");
	con=(SSL *)SSL_new();
	SSL_set_fd(con,s);
	SSL_set_verify(con,verify,verify_callback);

	if (SSL_use_RSAPrivateKey_file(con,cert_file,X509_FILETYPE_PEM) <= 0)
		{
		fprintf(stderr,"unable to set public key\n");
		fprintf(stderr,"%s\n",SSL_error_string(SSL_errno));
		exit(1);
		}
	if (key_file == NULL) key_file=cert_file;
	if (SSL_use_certificate_file(con,key_file,X509_FILETYPE_PEM) <= 0)
		{
		fprintf(stderr,"unable to set certificate\n");
		fprintf(stderr,"%s\n",SSL_error_string(SSL_errno));
		exit(1);
		}

	SSL_set_verify(con,verify,verify_callback);

	if (!SSL_accept(con))
		{
		fprintf(stderr,"ERROR\n");
		if (verify_error != VERIFY_OK)
			{
			fprintf(stderr,"verify error:%s\n",
				X509_verify_error_string(verify_error));
			}
		else
			{
			fprintf(stderr,"SSL error:function is:%s\n",
				SSL_error_func_string(SSL_errno));
			fprintf(stderr,"         :error is   :%s\n",
				SSL_error_string(SSL_errno));
			}
		SSL_free(con);
		close(s);
		return;
		}
	printf("CIPHER is %s\n",SSL_get_cipher(con));

	width=s+1;
	for (;;)
		{
		FD_ZERO(&readfds);
		FD_SET(fileno(stdin),&readfds);
		FD_SET(s,&readfds);
		select(width,&readfds,NULL,NULL,NULL);
		if (FD_ISSET(fileno(stdin),&readfds))
			{
			i=read(fileno(stdin),buf,BUFSIZZ);
			if ((i <= 0) || (buf[0] == 'Q'))
				{
				fprintf(stderr,"DONE\n");
				close(s);
				shutdown(accept_socket,2);
				exit(0);
				}
			SSL_write(con,buf,i);
			}
		if (FD_ISSET(s,&readfds))
			{
			i=SSL_read(con,buf,BUFSIZZ);
			if (i<=0)
				{
				fprintf(stderr,"DONE\n");
				break;
				}
			write(fileno(stdout),buf,i);
			}
		}
	SSL_free(con);
	}

void close_accept_socket()
	{
	if (accept_socket >= 0)
		shutdown(accept_socket,2);
	}

void do_server(server,port,ret)
char *server;
int port;
int *ret;
	{
	int sock;
	char *name;

	if (!init_server(&accept_socket,server,port)) exit(1);

	if (ret != NULL)
		{
		*ret=accept_socket;
		return;
		}
	for (;;)
		{
		fprintf(stderr,"ACCEPT\n");
		if (do_accept(accept_socket,&sock,&name) == 0)
			exit(1);
		body(name,sock);
		free(name);
		close(sock);
		}
	}

static int init_server(sock,service,port)
int *sock;
char *service;
int port;
	{
	struct servent ser;
	struct servent *serv;
	struct sockaddr_in server;
	int s,i;
	struct linger ling;

	if ((serv=getservbyname(service,PROTO)) == NULL)
		{
		serv=&ser;
		ser.s_port=htons(port);
		}

	server.sin_family=AF_INET;
	server.sin_port=serv->s_port;
	server.sin_addr.s_addr=INADDR_ANY;

	s=socket(AF_INET,SOCK_STREAM,0);
	if (s == -1)
		return(0);
	if (bind(s,(struct sockaddr *)&server,sizeof(server)) == -1)
		{
		perror("bind");
		return(0);
		}
	if (listen(s,5) == -1)
			return(0);
	ling.l_onoff=1;
	ling.l_linger=0;
	i=setsockopt(s,SOL_SOCKET,SO_LINGER,(char *)&ling,sizeof(ling));
	if (i < 0) { perror("linger"); exit(0); }

	i=0;
	*sock=s;
	return(1);
	}

int do_accept(acc_sock,sock,host)
int acc_sock;
int *sock;
char **host;
	{
	int ret,i;
	struct hostent *h1,*h2;
	static struct sockaddr_in from;
	int len;
	struct linger ling;

redoit:
	memset((char *)&from,0,sizeof(from));
	len=sizeof(from);
	ret=accept(acc_sock,(struct sockaddr *)&from,&len);
	if (ret == -1)
		{
		if (errno == EINTR)
			{
			/*check_timeout();/**/
			goto redoit;
			}
		perror("accept");
		return(0);
		}
	
	ling.l_onoff=1;
	ling.l_linger=0;
	i=setsockopt(ret,SOL_SOCKET,SO_LINGER,(char *)&ling,sizeof(ling));
	if (i < 0) { perror("linger"); exit(0); }
	i=0;
	i=setsockopt(ret,SOL_SOCKET,SO_KEEPALIVE,(char *)&i,sizeof(i));
	if (i < 0) { perror("keepalive"); exit(0); }

	h1=gethostbyaddr((char *)&from.sin_addr.s_addr,
		sizeof(from.sin_addr.s_addr),AF_INET);
	if (h1 == NULL)
		{
		fprintf(stderr,"bad gethostbyaddr\n");
		return(0);
		}
	if ((*host=(char *)malloc(strlen(h1->h_name)+1)) == NULL)
		{
		perror("malloc");
		return(0);
		}
	strcpy(*host,h1->h_name);

	h2=gethostbyname(h1->h_name);
	i=0;
	if (h2->h_addrtype != AF_INET)
		{
		fprintf(stderr,"gethostbyname addr is not AF_INET\n");
		return(0);
		}
	while (h2->h_addr_list[i])
		{
		if (memcmp(h2->h_addr_list[i],(char *)&from.sin_addr.s_addr,
			sizeof(from.sin_addr.s_addr)) == 0)
			{
			*sock=ret;
			return(1);
			}
		}
	return(0);
	}

/* duplicated in server.c and client.c */
int verify_callback(ok,xs,xi,depth,error)
int ok;
char *xs,*xi; /* should be X509 * but we can just have them as char *. */
int depth,error;
	{
	char *s;

	s=(char *)X509_oneline_X509_NAME(X509_get_subject_name(xs));
	if (s == NULL)
		{
		fprintf(stderr,"error:%s\n",X509_error_string(X509_errno));
		return(0);
		}
	fprintf(stderr,"depth=%d %s\n",depth,s);
	free(s);
	if (error == VERIFY_ERR_UNABLE_TO_GET_ISSUER)
		{
		s=(char *)X509_oneline_X509_NAME(X509_get_issuer_name(xs));
		if (s == NULL)
			{
			fprintf(stderr,"verify error:%s\n",
				X509_error_string(X509_errno));
			return(0);
			}
		fprintf(stderr,"issuer= %s\n",s);
		free(s);
		}
	if (!ok)
		{
		fprintf(stderr,"verify error:num=%d:%s\n",error,
			X509_verify_error_string(error));
		if (verify_depth <= depth)
			{
			ok=1;
			verify_error=VERIFY_OK;
			}
		else
			{
			ok=0;
			verify_error=error;
			}
		}
	fprintf(stderr,"verify return:%d\n",ok);
	return(ok);
	}

