/* x509.c */
/* Copyright (C) 1995 Eric Young (eay@mincom.oz.au).
 * All rights reserved.
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.
 * See the COPYRIGHT file in the SSLeay distribution for more details.
 */

#include <stdio.h>
#include "der.h"
#include "bn.h"
#include "PEM.h"
#include "X509.h"

#define	POSTFIX	".serial"

#define FORMAT_UNDEF	0
#define FORMAT_DER	1
#define FORMAT_TEXT	2
#define FORMAT_PEM	3

char *usage[]={
"usage: x509 args\n",
"\n",
" -inform arg     - input format - default PEM (one of DER, TXT or PEM)\n",
" -outform arg    - output format - default PEM\n",
" -keyform arg    - rsa key format - default PEM\n",
" -CAform arg     - CA format - default PEM\n",
" -CAkeyform arg  - CA key format - default PEM\n",
" -in arg         - input file - default stdin\n",
" -out arg        - output file - default stdout\n",
" -serial         - print serial number value\n",
" -hash           - print hash value\n",
" -subject        - print subject DN\n",
" -issuer         - print issuer DN\n",
" -startdate      - notBefore field\n",
" -enddate        - notAfter field\n",
" -noout          - no certificate output\n",
" -signkey arg    - set issuer to subject and sign with private key and\n",
"                   put the public in cert.\n",
" -CA arg         - set the CA certificate, must be PEM format.\n",
" -CAkey arg      - set the CA RSA key, must be PEM format, if this arg\n",
"                   missing, it is asssumed to be in the CA file.\n",
" -CAcreateserial - create serial number file if it does not exist\n",
NULL
};

static int callb();
RSA *load_key();
X509 *load_cert();

main(argc,argv)
int argc;
char *argv[];
	{
	X509 *x,*xca;
	RSA *rsa,*CArsa;
	int i,num,badops=0;
	FILE *out=NULL;
	int informat,outformat,keyformat,CAformat,CAkeyformat;
	char *infile=NULL,*outfile=NULL,*keyfile=NULL,*CAfile=NULL;
	char *CAkeyfile=NULL,*str=NULL;
	int serial=0,hash=0,subject=0,issuer=0,startdate=0,enddate=0;
	int noout=0,sign_flag=0,CA_flag=0,CA_createserial=0;
	char **pp;

	informat=FORMAT_PEM;
	outformat=FORMAT_PEM;
	keyformat=FORMAT_PEM;
	CAformat=FORMAT_PEM;
	CAkeyformat=FORMAT_PEM;

	argc--;
	argv++;
	num=0;
	while (argc >= 1)
		{
		if 	(strcmp(*argv,"-inform") == 0)
			{
			if (argc-- < 1) goto bad;
			informat=str2fmt(*(++argv));
			}
		else if (strcmp(*argv,"-outform") == 0)
			{
			if (argc-- < 1) goto bad;
			outformat=str2fmt(*(++argv));
			}
		else if (strcmp(*argv,"-keyform") == 0)
			{
			if (argc-- < 1) goto bad;
			keyformat=str2fmt(*(++argv));
			}
		else if (strcmp(*argv,"-CAform") == 0)
			{
			if (argc-- < 1) goto bad;
			CAformat=str2fmt(*(++argv));
			}
		else if (strcmp(*argv,"-CAkeyform") == 0)
			{
			if (argc-- < 1) goto bad;
			CAformat=str2fmt(*(++argv));
			}
		else if (strcmp(*argv,"-in") == 0)
			{
			if (argc-- < 1) goto bad;
			infile= *(++argv);
			}
		else if (strcmp(*argv,"-out") == 0)
			{
			if (argc-- < 1) goto bad;
			outfile= *(++argv);
			}
		else if (strcmp(*argv,"-signkey") == 0)
			{
			if (argc-- < 1) goto bad;
			keyfile= *(++argv);
			sign_flag= ++num;
			}
		else if (strcmp(*argv,"-CA") == 0)
			{
			if (argc-- < 1) goto bad;
			CAfile= *(++argv);
			CA_flag= ++num;
			}
		else if (strcmp(*argv,"-CAkey") == 0)
			{
			if (argc-- < 1) goto bad;
			CAkeyfile= *(++argv);
			}
		else if (strcmp(*argv,"-serial") == 0)
			serial= ++num;
		else if (strcmp(*argv,"-hash") == 0)
			hash= ++num;
		else if (strcmp(*argv,"-subject") == 0)
			subject= ++num;
		else if (strcmp(*argv,"-issuer") == 0)
			issuer= ++num;
		else if (strcmp(*argv,"-startdate") == 0)
			startdate= ++num;
		else if (strcmp(*argv,"-enddate") == 0)
			enddate= ++num;
		else if (strcmp(*argv,"-noout") == 0)
			noout= ++num;
		else if (strcmp(*argv,"-CAcreateserial") == 0)
			CA_createserial= ++num;
		else
			{
			fprintf(stderr,"unknown option %s\n",*argv);
			badops=1;
			break;
			}
		argc--;
		argv++;
		}

	if (badops)
		{
bad:
		for (pp=usage; (*pp != NULL); pp++)
			fprintf(stderr,*pp);
		exit(1);
		}

	if (!X509_set_default_verify_paths())
		exit(1);

	if ((CAkeyfile == NULL) && (CA_flag) && (CAformat == FORMAT_PEM))
		{ CAkeyfile=CAfile; }
	else if ((CA_flag) && (CAkeyfile == NULL))
		{
		fprintf(stderr,"need to specify a CAkey if using the CA command\n");
		exit(1);
		}

	x=load_cert(infile,informat);
	if (CA_flag) xca=load_cert(CAfile,CAformat);

	if (num)
		{
		for (i=1; i<=num; i++)
			{
			if (issuer == i)
				{
				str=X509_oneline_X509_NAME(X509_get_issuer_name(x));
				if (str == NULL)
					{
					fprintf(stderr,"unable to get issuer Name from certificate\n");
					fprintf(stderr,"%s\n",
						X509_error_string(X509_errno));
					exit(1);
					}
				fprintf(stdout,"issuer= %s\n",str);
				free(str);
				}

			if (subject == i) 
				{
				str=X509_oneline_X509_NAME(X509_get_subject_name(x));
				if (str == NULL)
					{
					fprintf(stderr,"unable to get subject Name from certificate\n");
					fprintf(stderr,"%s\n",
						X509_error_string(X509_errno));
					exit(1);
					}
				fprintf(stdout,"subject=%s\n",str);
				free(str);
				}
			if (serial == i)
				{
				fprintf(stdout,"serial ");
				i2f_DER_BIT_STRING(stdout,x->cert_info->serialNumber);
				}
			if (hash == i)
				{
				fprintf(stdout,"%08x\n",
					X509_subject_name_hash(x));
				}
			if (startdate == i)
				{
				fprintf(stdout,"notBefore=%s\n",
					x->cert_info->validity->notBefore);
				}
			if (enddate == i)
				{
				fprintf(stdout,"notAfter =%s\n",
					x->cert_info->validity->notAfter);
				}

			/* should be in the library */
			if (sign_flag == i)
				{
				fprintf(stderr,"Getting Private key\n");
				rsa=load_key(keyfile,keyformat);
				sign(x,rsa);
				}
			if (CA_flag == i)
				{
				fprintf(stderr,"Getting CA Private Key\n");
				if (CAkeyfile != NULL)
					CArsa=load_key(CAkeyfile,CAkeyformat);
				certify(CAfile,x,xca,CArsa,CA_createserial);
				}
			}
		}

	if (noout) goto end;

	if (outfile == NULL)
		out=stdout;
	else
		{
		out=fopen(outfile,"w");
		if (out == NULL) { perror(outfile); exit(1); }
		}

	if 	(outformat == FORMAT_DER)
		i=i2D_X509_fp(out,x);
	else if (outformat == FORMAT_TEXT)
		{ i2f_X509(out,x); i=1; }
	else if (outformat == FORMAT_PEM)
		i=PEM_write_X509(out,x);
	else	{
		fprintf(stderr,"bad output format specified for outfile\n");
		exit(1);
		}
	if (!i) { fprintf(stderr,"unable to write certificate\n"); exit(1); }
	fclose(out);
end:
	exit(0);
	return(0);
	}

int str2fmt(s)
char *s;
	{
	if 	((*s == 'D') || (*s == 'd'))
		return(FORMAT_DER);
	else if ((*s == 'T') || (*s == 't'))
		return(FORMAT_TEXT);
	else if ((*s == 'P') || (*s == 'p'))
		return(FORMAT_PEM);
	else
		return(FORMAT_UNDEF);
	}

int certify(CAfile,x,xca,rsa,create)
char *CAfile;
X509 *x,*xca;
RSA *rsa;
int create;
	{
	FILE *io;
	char *buf,buf2[1024];
	BIGNUM *serial;
	DER_BIT_STRING *bs;

	buf=(char *)malloc(bn_num_bytes(rsa->n)*2+
		strlen(CAfile)+strlen(POSTFIX)+1);
	if (buf == NULL) { fprintf(stderr,"out of mem\n"); return(0); }
	strcpy(buf,CAfile);
	strcat(buf,POSTFIX);
	serial=bn_new();
	bs=DER_BIT_STRING_new();
	if ((serial == NULL) || (bs == NULL))
		{
		fprintf(stderr,"%s\n",RSA_error_string(RSA_errno));
		return(0);
		}

	io=fopen(buf,"r");
	if (io == NULL)
		{
		if (!create)
			{
			perror(buf);
			exit(0);
			}
		else
			bn_zero(serial);
		}
	else 
		{
		if (!f2i_DER_BIT_STRING(io,bs,1024,buf2))
			{
			fprintf(stderr,"unable to load serial number from %s\n",buf);
			fprintf(stderr,"%s\n",DER_error_string(DER_errno));
			fclose(io);
			return(0);
			}
		else
			{
			serial=bn_bin2bn(bs->length,bs->data,serial);
			if (serial == NULL)
				{
				fprintf(stderr,"error converting bin 2 bn");
				return(0);
				}
			}
		}
	fclose(io);

	if (!bn_add_word(serial,1))
		{ fprintf(stderr,"add_word failure\n"); return(0); }
	bs->data=(unsigned char *)buf2;
	bs->length=bn_bn2bin(serial,bs->data);

	io=fopen(buf,"w");
	if (io == NULL)
		{
		fprintf(stderr,"error attempting to write serial number file\n");
		perror(buf);
		exit(1);
		}
	i2f_DER_BIT_STRING(io,bs);
	fclose(io);
	
	if (!X509_add_cert(x)) return(0);
	if (!X509_verify(x,callb)) return(0);

	/* don't free this X509 struct or bad things will happen
	 * unless you put issuer first :-) */
	x->cert_info->issuer=xca->cert_info->subject;
	if (x->cert_info->validity->notBefore == NULL)
		free(x->cert_info->validity->notBefore);
	x->cert_info->validity->notBefore=(char *)malloc(100);
	x->cert_info->serialNumber=bs;

	if (x->cert_info->validity->notBefore == NULL)
		{
		fprintf(stderr,"out of mem\n");
		exit(0);
		}

	X509_gmtime(x->cert_info->validity->notBefore,0);
	X509_gmtime(x->cert_info->validity->notAfter,60*60*24*28);
	if (!X509_sign_md5_with_rsa(x,rsa))
		{
		fprintf(stderr,"%d\n",X509_error_string(X509_errno));
		exit(1);
		}
	return(1);
	}

static int callb(ok,xs,xi,depth,error)
int ok;
X509 *xs,*xi;
int depth,error;
	{
	if ((!ok) || (depth > 0))
		{
		printf("error with certificate - error %d at depth %d\n%s\n",
			error,depth,X509_verify_error_string(error));
			return(ok);
		}
	return(ok);
	}

RSA *load_key(file,format)
char *file;
int format;
	{
	FILE *key;
	RSA *rsa;
	int i;

	key=fopen(file,"r");
	if (key == NULL) { perror(file); exit(1); }
	rsa=RSA_new();
	if (rsa == NULL) { fprintf(stderr,"%s\n",RSA_error_string(RSA_errno)); }
	if	(format == FORMAT_DER)
		i=D2i_RSAPrivateKey_fp(key,rsa);
	else if (format == FORMAT_TEXT)
		i=f2i_RSAPrivateKey(key,rsa);
	else if (format == FORMAT_PEM)
		i=PEM_read_RSA(key,rsa);
	else
		{
		fprintf(stderr,"bad input format specified for key\n");
		exit(1);
		}
	fclose(key);
	if (!i)
		{
		fprintf(stderr,"unable to load Private Key\n");
		exit(1);
		}
	return(rsa);
	}

X509 *load_cert(file,format)
char *file;
int format;
	{
	X509 *x;
	FILE *cert;
	int i;

	if (file == NULL)
		cert=stdin;
	else
		{
		cert=fopen(file,"r");
		if (cert == NULL) { perror(file); exit(1); }
		}
	x=X509_new();
	if (x == NULL)
		{ fprintf(stderr,"%s\n",X509_error_string(X509_errno)); }
	if 	(format == FORMAT_DER)
		i=D2i_X509_fp(cert,x);
	else if (format == FORMAT_TEXT)
		i=f2i_X509(cert,x);
	else if (format == FORMAT_PEM)
		i=PEM_read_X509(cert,x);
	else	{
		fprintf(stderr,"bad input format specified for input cert\n");
		exit(1);
		}
	if (!i)
		{
		if (format == FORMAT_DER)
			fprintf(stderr,"%s\n",DER_error_string(DER_errno));
		else if (format == FORMAT_TEXT)
			fprintf(stderr,"%s\n",DER_error_string(DER_errno));
		else if (format == FORMAT_PEM)
			fprintf(stderr,"%s\n",PEM_error_string(PEM_errno));
		fprintf(stderr,"unable to load certificate\n");
		exit(1);
		}
	fclose(cert);
	return(x);
	}

int sign(x,rsa)
X509 *x;
RSA *rsa;
	{
	int j;
	unsigned char *p;
	/* X509_NAME *n; */

	/* n=x->cert_info->subject; */
	/* don't free this X509 struct or bad
	 * things will happen unless you put
	 * n back first :-) */
	x->cert_info->issuer=x->cert_info->subject;
	j=i2D_RSAPublicKey(rsa,NULL);
	p=x->cert_info->key->public_key->data=(unsigned char *)malloc(j+10);
	if (p == NULL) { fprintf(stderr,"out of memory\n"); exit(1); }

	x->cert_info->key->public_key->length=j;
	if (!i2D_RSAPublicKey(rsa,&p))
		{
		fprintf(stderr,"%s\n", X509_error_string(X509_errno));
		exit(1);
		}
	if (!X509_sign_md5_with_rsa(x,rsa))
		{
		fprintf(stderr,"RSA sign:%s\n", X509_error_string(X509_errno));
		exit(1);
		}
	return(1);
	}

