/*
 * Copyright (c) 1989 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 */

/*
 * Copyright (c) 2000 Qualcomm Incorporated.  All rights reserved.
 * The file License.txt specifies the terms for use, modification,
 * and redistribution.
 */
 
/*
 * QPOPPER.
 *
 * Revisions:
 *
 *  06/10/00 [rg]
 *           - Add return value to pop_exit and ring.
 *           - Added irix to HPUX voidstar define.
 *
 *  06/05/00 [rg]
 *           - Be less chatty in banner when SHY defined.
 *
 *  05/02/00 [rg]
 *           - Use both ferror() and errno for I/O errors on client
 *             input stream.
 *
 *  04/28/00 [rg]
 *           - Use ferror() on p->input instead of errno for I/O errors
 *             on client input stream.
 *
 *  03/07/00 [rg]
 *           - Added trace call when ready for user input.
 *           - Changed AUTH to SPEC_POP_AUTH.
 *
 *  12/03/99 [rg]
 *           - Added POP* parameter to tgets and myfgets.
 *           - Added logging for I/O errors and discarded input to myfgets.
 *           - Added errno to POP EOF -ERR message.
 *
 *  04/04/98 [py]
 *           Made pop_exit(), callstack.
 *
 */

#include "config.h"

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <setjmp.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#if HAVE_STRINGS_H
#  include <strings.h>
#endif

#if HAVE_UNISTD_H
#  include <unistd.h>
#endif /* HAVE_UNISTD_H */

#if HAVE_SYS_UNISTD_H
#  include <sys/unistd.h>
#endif /* HAVE_SYS_UNISTD_H */

#ifdef SPEC_POP_AUTH
#  ifdef HAVE_SYS_SECURITY_H 
#    include <sys/security.h>
#  endif
#  ifdef HAVE_PROTO_H
#    include <prot.h>
#  endif
#endif /* SPEC_POP_AUTH */

#include "popper.h"
#include "misc.h"

#ifndef HAVE_STRERROR
  char * strerror();
#endif

extern  state_table *   pop_get_command();
int poptimeout = 0;

volatile int hangup = FALSE;

int catchSIGHUP ( SIGPARAM );

int     pop_timeout = POP_TIMEOUT;
int     login_delay = -1, 
        expire      = -1;

#ifdef _DEBUG
  POP *global_debug_p = NULL;
#endif


/* 
 *  popper: Handle a Post Office Protocol version 3 session
 */
int
qpopper ( argc, argv )
int       argc;
char        **  argv;
{
    POP                 p;
    state_table     *   s;
    char                message [ MAXLINELEN ];
    char            *   tgets();
    
    /*
     * seed random with the current time to nearest second 
     */
    srandom ( (unsigned int) time ( (TIME_T *) 0 ) );

#if defined(POPSCO) || defined(SPEC_POP_AUTH)
#  ifdef HAVE_SET_AUTH_PARAMETERS
    (void) set_auth_parameters ( argc, argv );
#  endif /* HAVE_SET_AUTH_PARAMETERS */
#endif /* POPSCO or SPEC_POP_AUTH */

#ifdef AUX
    (void) set42sig();
#endif /* AUX */

/* 
 * Set umask for better security 
 */
#ifdef BINMAIL_IS_SETGID
    umask ( 0007 );        /* Trust the mail delivery group */
#else
    umask ( 0077 );        /* Trust no-one */
#endif

    if ( signal (SIGHUP,  VOIDSTAR catchSIGHUP ) == SIG_ERR ||
         signal (SIGPIPE, VOIDSTAR catchSIGHUP ) == SIG_ERR  ) {
        EXIT ( 1 ); 
    }

#ifdef _DEBUG
    global_debug_p = &p;
#endif

/*  
 * Start things rolling 
 */
    if ( pop_init ( &p, argc, argv ) != POP_SUCCESS )
        EXIT ( 1 );

/*  
 * Tell the user that we are listenting 
 */
    { /* local env */
    char                myname [ 128 ];

#ifdef _DEBUG
    strcpy ( p.myhost, "DEBUG" );
#endif /* _DEBUG */

#ifdef APOP
    sprintf ( p.md5str, "<%lu.%ld@%s>", 
              (long) getpid(), (long) time ( (TIME_T *) 0 ), p.myhost );
#else
    p.md5str[0] = '\0';
#endif /* APOP */

#ifdef _DEBUG
    strcpy ( myname, "local host" );
#else
    strcpy ( myname, p.myhost );
#endif /* _DEBUG */

#ifdef SHY
    /* 
     * say as little as possible.  According to RFC 1939 all we need is
     * "+OK" and APOP string, but we'll be a little chatty and add "ready".
     */
    pop_msg ( &p, POP_SUCCESS, HERE,
              "ready  %s",
              p.md5str );
#else
    pop_msg ( &p, POP_SUCCESS, HERE, 
              "QPOP%.*s%s (version %s) at %s starting.  %s",
              (strlen(BANNERSFX)>0 ? 1 : 0), " ",
              BANNERSFX,
              VERSION, 
              myname,
              p.md5str );
#endif /* SHY */

    } /* local env */

/*
 *      State loop.  The POP server is always in a particular state in 
 *      which a specific suite of commands can be executed.  The following 
 *      loop reads a line from the client, gets the command, and processes 
 *      it in the current context (if allowed) or rejects it.  This continues 
 *      until the client quits or an error occurs. 
 */

    for ( p.CurrentState  = auth1;
          p.CurrentState != halt && p.CurrentState != error;
        ) {
#ifdef SETPROCTITLE
        setproctitle ( "%s@%s [%s]: cmd read", p.user, p.client, p.ipaddr );
#endif
        DEBUG_LOG3 ( &p, "Qpopper ready for input from %s at %s [%s]",
                     ( *p.user == '\0' ? "(null)" : p.user ),
                     p.client, p.ipaddr );

        if ( hangup ) {
            pop_exit ( &p, HANGUP );
        } 
        else if ( tgets ( &p, message, MAXLINELEN, 
                          p.input, pop_timeout ) == NULL ) {
            pop_exit ( &p, (poptimeout) ? TIMEOUT :  ABORT );
        } 
        else if ( StackSize ( &(p.InProcess) ) ) { 
            int i;
            FP f;
            p.inp_buffer = message;
            for ( f = GetTop  (&(p.InProcess), &i ); 
                  f; 
                  f = GetNext (&(p.InProcess), &i ) 
                ) {
                  (*f)(&p);
            }
        } 
        else if ( ( s = pop_get_command ( &p, message ) ) != NULL )  {
            if ( s->function != NULL ) {

#ifdef    SETPROCTITLE
                int i;
                char command [ 10 ];

                for ( i = 0; s->command [ i ]; i++ )
                    command [ i ] = toupper ( s->command [ i ] );
                command [ i ] = 0;
                setproctitle ( "%s@%s [%s]: %s",
                               p.user, p.client, p.ipaddr, command );
#endif /* SETPROCTITLE */

/* 
 * honor a halt or error state from internal command processing 
 */

                p.CurrentState = ( p.CurrentState != halt  ) && 
                                 ( p.CurrentState != error )
                                 ? s->result [ (*s->function) ( &p ) ]
                                 : p.CurrentState;
                DEBUG_LOG1 ( &p, "p.CurrentState now %s",
                             get_state_name ( p.CurrentState ) );
            } /* s->function != NULL */
            else {
/* 
 * No function no problem: [actually it fails] 
 */
                p.CurrentState = s->success_state;
                pop_msg ( &p, POP_SUCCESS, HERE, NULL );
            }
        } /* s = pop_get_command(...) != NULL */
    } /* for state loop */

/*  
 * Say goodbye to the client 
 */
    pop_msg ( &p, POP_SUCCESS, HERE, "Pop server at %s signing off.",
#ifdef _DEBUG
              "local host"
#else
              p.myhost
#endif
            );
/*  
 * Log the end of activity 
 */
    DEBUG_LOG4 ( &p, "(v%s) Ending request from \"%s\" at (%s) %s",
                 VERSION, p.user, p.client, p.ipaddr );

    /*
     * If requested, write timing record
     */
#ifdef DO_TIMING
    pop_log ( &p, POP_INFO, HERE,
              "(v%s) Timing for %s@%s (%s) auth=%u init=%u clean=%u",
              VERSION, p.user, p.client, 
              ( p.CurrentState == error ? "error" : "normal" ),
              p.login_time, p.init_time, p.clean_time );
#endif /* DO_TIMING */

/*  
 * Stop logging 
 */
    closelog();
    return ( 0 );
}

jmp_buf env;

/*
 *  There seems to be a problem with the AIX fgets.  This is suppose to
 *  fix it.
 */
char *
myfgets ( p, str, size, fp )
POP      *p;
char        *str;
int               size;
FILE                   *fp;
{
    char *cp;
    char ch;
    int nbytes   = 0;
    int found_nl = 0;

    cp = str;

    while ( --size > 0 ) {
        if ( ( nbytes = read ( fileno(fp), cp, 1 ) ) <= 0 )
            break;

        if ( *cp == '\n' ) {
            *++cp = '\0';
            found_nl++;
            break;
        }
        ++cp;
    }

    if ( ( nbytes <= 0 ) || ( cp == str ) ) {
        if ( nbytes == 0 ) {
            pop_log ( p, POP_NOTICE, HERE, "EOF from %s at %s (%s): "
                      "[%i] %i (%s); %d (%s)",
                      p->user, p->ipaddr, p->client, 
                      nbytes, errno, STRERROR(errno),
                      ferror(fp), STRERROR(ferror(fp)) );
        } else {
            pop_log ( p, POP_NOTICE, HERE, "I/O Error from %s at %s (%s): "
                      "[%i] %i (%s); %d (%s)",
                      p->user, p->ipaddr, p->client, 
                      nbytes, errno, STRERROR(errno),
                      ferror(fp), STRERROR(ferror(fp)) );
        }
        return ( NULL );
    } else {
        if ( !found_nl ) {
            unsigned lost = 0;

            while ( ( read ( fileno(fp), &ch, 1 ) == 1 ) && ch != '\n' ) {
                lost++;
            }
            if ( lost > 0 ) {
                pop_log ( p, POP_DEBUG, HERE, 
                          "line too long; %u character(s) discarded from "
                          "%s at %s (%s)",
                          lost, p->user, p->ipaddr, p->client );
            }
        }
        return ( str );
    }
}

/*
 * fgets, but with a timeout
 */
char *tgets ( p, str, size, fp, timeout )
POP  *p;
char *str;
int   size;
FILE *fp;
int   timeout;
{
  int ring();
  (void) signal ( SIGALRM, VOIDSTAR ring );
  alarm ( timeout );
  if ( setjmp ( env ) )
    str = NULL;
  else
    str = myfgets ( p, str, size, fp );
/*  str = fgets(str,size,fp); */
  alarm  ( 0 );
  signal ( SIGALRM, SIG_DFL );
  return ( str );
}

int 
ring ( SIGPARAM )
{
  poptimeout = 1;
  longjmp ( env, 1 );
  return POP_FAILURE;
}
  

#ifdef STRNCASECMP
/*
 *  Perform a case-insensitive string comparision
 */
strncasecmp(str1,str2,len)
register char   *   str1;
register char   *   str2;
register int        len;
{
    register int    i;
    char            a,
                    b;

    for (i=len-1;i>=0;i--){
        a = str1[i];
        b = str2[i];
        if (isupper(a)) a = tolower(str1[i]);
        if (isupper(b)) b = tolower(str2[i]);
        if (a > b) return (1);
        if (a < b) return(-1);
    }
    return(0);
}
#endif


int 
catchSIGHUP ( SIGPARAM )
{

    extern volatile int hangup;

    hangup = TRUE ;

    /* This should not be a problem on BSD systems */
    signal ( SIGHUP,   VOIDSTAR catchSIGHUP );
    signal ( SIGPIPE,  VOIDSTAR catchSIGHUP );
    return 0;
}


#ifndef HAVE_STRERROR
char *
strerror(e)
        int e;
{
    if(e < sys_nerr)
        return(sys_errlist[e]);
    else
        return("unknown error");
}
#endif

#ifdef POPSCO
/*
 * Ftruncate() for non-BSD systems.
 *
 * This module gives the basic functionality for ftruncate() which
 * truncates the given file descriptor to the given length.
 * ftruncate() is a Berkeley system call, but does not exist in any
 * form on many other versions of UNIX such as SysV. Note that Xenix
 * has chsize() which changes the size of a given file descriptor,
 * so that is used if M_XENIX is defined.
 *
 * Since there is not a known way to support this under generic SysV,
 * there is no code generated for those systems.
 *
 * SPECIAL NOTE: On Xenix, using this call in the BSD library
 * will REQUIRE the use of -lx for the extended library since chsize()
 * is not in the standard C library.
 *
 * By Marc Frajola, 3/27/87
 */

#include <fcntl.h>

ftruncate(fd,length)
    int fd;                     /* File descriptor to truncate */
    off_t length;               /* Length to truncate file to */
{
    int status;                 /* Status returned from truncate proc */

    status = chsize(fd,length);
/*
    status = -1;
    NON-XENIX SYSTEMS CURRENTLY NOT SUPPORTED
*/

    return(status);
}
#endif

#ifdef NEED_FTRUNCATE
/* ftruncate emulations that work on some System V's.
   This file is in the public domain.  */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <fcntl.h>

#ifdef F_CHSIZE

int
ftruncate (fd, length)
     int fd;
     off_t length;
{
  return fcntl (fd, F_CHSIZE, length);
}

#else /* not F_CHSIZE */
#ifdef F_FREESP

/* By William Kucharski <kucharsk@netcom.com>.  */

#include <sys/stat.h>
#include <errno.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

int
ftruncate (fd, length)
     int fd;
     off_t length;
{
  struct flock fl;
  struct stat filebuf;

  if (fstat (fd, &filebuf) < 0)
    return -1;

  if (filebuf.st_size < length)
    {
      /* Extend file length. */
      if (lseek (fd, (length - 1), SEEK_SET) < 0)
        return -1;

      /* Write a "0" byte. */
      if (write (fd, "", 1) != 1)
        return -1;
    }
  else
    {

      /* Truncate length. */

      fl.l_whence = 0;
      fl.l_len = 0;
      fl.l_start = length;
      fl.l_type = F_WRLCK;      /* write lock on file space */

      /* This relies on the *undocumented* F_FREESP argument to fcntl,
         which truncates the file so that it ends at the position
         indicated by fl.l_start.  Will minor miracles never cease?  */

      if (fcntl (fd, F_FREESP, &fl) < 0)
        return -1;
    }

  return 0;
}

#else /* not F_CHSIZE nor F_FREESP */
#ifdef HAVE_CHSIZE

int
ftruncate (fd, length)
     int fd;
     off_t length;
{
  return chsize (fd, length);
}

#else /* not F_CHSIZE nor F_FREESP nor HAVE_CHSIZE */

#include <errno.h>
#ifndef errno
extern int errno;
#endif

int
ftruncate (fd, length)
     int fd;
     off_t length;
{
  errno = EIO;
  return -1;
}

#endif /* not HAVE_CHSIZE */
#endif /* not F_FREESP */
#endif /* not F_CHSIZE */
#endif /* NEED_FTRUNCATE */

int
pop_exit ( p, e )
    POP         *p;
    EXIT_REASON e;
{
    switch ( e ) {
    case HANGUP :
        pop_msg ( p, POP_FAILURE, HERE, "POP hangup from %s", p->myhost );
        break;
    case TIMEOUT :
        DEBUG_LOG1 ( p, "Timed out %d", pop_timeout );
        pop_msg ( p, POP_FAILURE, HERE, "POP timeout from %s", p->myhost ); 
        break;
    case ABORT:
        pop_msg ( p, POP_FAILURE, HERE,
                  "POP EOF or I/O Error: %i (%s); %d (%s)",
                   errno, STRERROR(errno),
                   ferror(p->input), STRERROR(ferror(p->input)) );
        break;
    } /* switch(e) */

    if ( p->xmitting ) 
        pop_xmit_clean ( p );

#ifndef NOUPDATEONABORT
    if ( (p->CurrentState != auth1) && (p->CurrentState != auth2)
                                    && !pop_updt(p) )
        pop_msg ( p, POP_FAILURE, HERE, "POP mailbox update for %s failed!",
                  p->user );
#else
    if ( (p->CurrentState != auth1) && (p->CurrentState != auth2)
                                    && !pop_restore(p) )
        pop_msg ( p, POP_FAILURE, HERE, "POP mailbox restoration for %s failed!",
                  p->user );
#endif
    p->CurrentState = error;
    return POP_FAILURE;
}

char *
get_state_name ( state curState )
{
    static char buf [ 31 ] = "";

    switch ( curState )
    {
        case auth1:
            strcpy ( buf, "auth1" );
            break;
        case auth2:
            strcpy ( buf, "auth2" );
            break;
        case trans:
            strcpy ( buf, "trans" );
            break;
        case update:
            strcpy ( buf, "update" );
            break;
        case halt:
            strcpy ( buf, "halt" );
            break;
        case error:
            strcpy ( buf, "error" );
            break;
        default:
            sprintf ( buf, "** %d **", curState );
    }
    
    return buf;
}

