/*
 * Copyright (c) 1989 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 */


/*
 * Copyright (c) 2000 Qualcomm Incorporated.  All rights reserved.
 * The file License.txt specifies the terms for use, modification,
 * and redistribution.
 *
 * Revisions:
 *     03/16/00  [rcg]
 *              - Brought HPUX11 into the HPUX fold.
 *
 *     03/09/00  [rcg]
 *              - Changed AUTH to SPEC_POP_AUTH to avoid conflicts with
 *                RPC .h files.
 *
 *     03/07/00  [rcg]
 *              - Updated authentication OK message to account for hidden
 *                messages.
 *
 *     03/06/00  [rcg]
 *              - Added AIX patch contributed by Nik Conwell.
 *
 *     02/10/00  [rcg]
 *              - Fixed typo causing auth failures when CHECK_SHELL defined.
 *
 *     02/09/00  [rcg]
 *              - Added extended response codes.
 */


#include <config.h>
#include <stdio.h>
#include <sys/types.h>
#include <string.h>

#if HAVE_STRINGS_H
#  include <strings.h>
#endif

#include <pwd.h>
#include "popper.h"

#define SLEEP_SECONDS 10


/* This error message is vague on purpose to help improve
   security at the inconvience of administrators and users */

char    *pwerrmsg = "[AUTH] Password supplied for \"%s\" is incorrect.";


/*------------------------------------- NONAUTHFILE */
#ifdef NONAUTHFILE
int
checknonauthfile ( user ) 
     char *user;
{
    char buf [ MAXUSERNAMELEN +1 ];
    FILE *fp;
    char cool = 0;

    fp = fopen ( NONAUTHFILE, "r" );
    if ( fp != NULL ) {
        while ( fgets ( buf, MAXUSERNAMELEN +1, fp ) ) {
            buf [ strlen(buf) -1 ] = '\0';
            if ( !strcmp ( buf, user ) ) {
                fclose ( fp );
                return ( -1 );
            }
        }

        fclose ( fp );
    }
    return ( 0 );
}
#endif /* NONAUTHFILE */

/*------------------------------------- AUTHFILE */
#ifdef AUTHFILE
int
checkauthfile ( user ) 
char *user;
{
    char buf [ MAXUSERNAMELEN +1 ];
    FILE *fp;
    char cool = 0;

    fp = fopen ( AUTHFILE, "r" );
    if ( fp != NULL ) {
        while ( fgets ( buf, MAXUSERNAMELEN +1, fp ) ) {
            buf [ strlen(buf) -1 ] = '\0';
            if ( !strcmp ( buf, user ) ) {
                fclose ( fp );
                return ( 0 );
            }
        }

        fclose ( fp );
    }
    return ( -1 );
}
#endif /* AUTHFILE */

int 
auth_user_kerberos (p, pw)
POP     *   p;
struct passwd *pw;
{
/*------------------------------------- KERBEROS */
#ifdef KERBEROS
    char lrealm[REALM_SZ];
    int status;
    struct passwd *pwp;
 
    status = krb_get_lrealm ( lrealm, 1 );
    if ( status == KFAILURE  ) {
        pop_log ( p, POP_WARNING, HERE, "%s: (%s.%s@%s) %s", 
                  p->client, kdata.pname, 
                  kdata.pinst, kdata.prealm, krb_err_txt[status] );
        return ( pop_msg ( p, POP_FAILURE, HERE,
                "[AUTH] Kerberos error:  \"%s\".", krb_err_txt[status] ) );
    }

#  ifdef KUSEROK
    if ( kuserok ( &kdata, p->user ) ) {
        pop_log ( p, POP_WARNING, HERE, 
                  "%s: (%s.%s@%s): not in %s's ACL.",
                  p->client, kdata.pname, kdata.pinst, 
                  kdata.prealm, p->user );
        return ( pop_msg ( p, POP_FAILURE, HERE, "[AUTH] Not in %s's ACL.", 
                           p->user ) );
    }
#  else /* not KUSEROK */
    if ( strcmp ( kdata.prealm, lrealm ) )  {
         pop_log ( p, POP_WARNING, HERE,
                   "%s: (%s.%s@%s) realm not accepted.", 
                   p->client, kdata.pname, kdata.pinst, kdata.prealm );
         return ( pop_msg ( p, POP_FAILURE, HERE,
                            "[AUTH] Kerberos realm \"%s\" not accepted.", 
                            kdata.prealm ) );
    }

    if ( strcmp ( kdata.pinst, "" ) ) {
        pop_log ( p, POP_WARNING, HERE, 
                 "%s: (%s.%s@%s) instance not accepted.", 
                 p->client, kdata.pname, kdata.pinst, kdata.prealm );
        return ( pop_msg ( p, POP_FAILURE, HERE,
                 "[AUTH] Must use null Kerberos(tm) instance -- \"%s.%s\" not accepted.",
                 kdata.pname, kdata.pinst ) );
    }
#  endif /* KUSEROK */
    return ( POP_SUCCESS );
#else   /* Kerberos not defined */
    return ( pop_msg ( p, POP_FAILURE, HERE,
             "[SYS/PERM] Kerberos failure: Qpopper has not been compiled with -DKERBEROS" ) );
#endif  /* KERBEROS */
}


/*------------------------------------- SPEC_POP_AUTH */
#ifdef SPEC_POP_AUTH
    char *crypt();

/*----------------------------------------------- SUNOS4 and not ISC (not PAM) */
#  if defined(SUNOS4) && !defined(ISC) && !defined(USE_PAM)

#  include <sys/label.h>
#  include <sys/audit.h>
#  include <pwdadj.h>

static int
auth_user ( p, pw )
POP     *   p;
struct passwd *pw;
{
    struct passwd_adjunct *pwadj;

    /*  
     * Look for the user in the shadow password file 
     */
    pwadj = getpwanam ( p->user );
    if ( pwadj == NULL ) {
        sleep  ( SLEEP_SECONDS );
        return ( pop_msg ( p, POP_FAILURE, HERE,
                "[AUTH] (shadow) Password supplied for \"%s\" is empty.", 
                p->user ) );
    } else {
        pw->pw_passwd = (char *)strdup(pwadj->pwa_passwd);
    }

    /*  
     * We don't accept connections from users with null passwords 
     *
     *  Compare the supplied password with the password file entry 
     */
    if ( (  pw->pw_passwd == NULL ) || 
         ( *pw->pw_passwd == '\0' ) ||
           strcmp ( crypt ( p->pop_parm[1], pw->pw_passwd ), pw->pw_passwd ) 
       ) {
        sleep  ( SLEEP_SECONDS );
        return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
    }

    return ( POP_SUCCESS );
}

#  endif  /* defined(SUNOS4) && !defined(ISC) */

/*----------------------------------------------- SOLARIS2 or AUX or UXPDS (not PAM) */
#  if !defined(USE_PAM) && (defined(SOLARIS2) || defined(AUX) || defined(UXPDS))

#    ifdef HAVE_SHADOW_H
#      include <shadow.h>
#    endif /* HAVE_SHADOW_H */

static int
auth_user ( p, pw )
POP     *   p;
struct passwd *pw;
{
    register struct spwd * pwd;
    long today;

    /*  
     * Look for the user in the shadow password file 
     */
    pwd = getspnam ( p->user );
    if ( pwd == NULL ) {
        if ( !strcmp ( pw->pw_passwd, "x" ) ) {      /* This my be a YP entry */
            sleep  ( SLEEP_SECONDS );
            return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
        }
    } else {
        today = (long)time((time_t *)NULL)/24/60/60;

        /* 
         * Check for expiration date 
         */
        if ( pwd->sp_expire > 0 && today > pwd->sp_expire ) {
            sleep  ( SLEEP_SECONDS );
            return ( pop_msg ( p, POP_FAILURE, HERE,
                               "[AUTH] \"%s\": account expired.", p->user ) );
        }

        /* 
         * Check if password is valid 
         */
        if ( pwd->sp_max > 0 && today > pwd->sp_lstchg + pwd->sp_max ) {
            sleep  ( SLEEP_SECONDS );
            return ( pop_msg ( p, POP_FAILURE, HERE,
                               "[AUTH] \"%s\": account expired.", p->user ) );
        }

        pw->pw_passwd = (char *)strdup(pwd->sp_pwdp);
        endspent();
    }

    /*  
     * We don't accept connections from users with null passwords 
     *
     *  Compare the supplied password with the password file entry 
     */
    if ( (  pw->pw_passwd == NULL ) || 
         ( *pw->pw_passwd == '\0' ) ||
           strcmp ( crypt ( p->pop_parm[1], pw->pw_passwd ), pw->pw_passwd ) 
       ) {
        sleep  ( SLEEP_SECONDS );
        return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
    }

    return ( POP_SUCCESS );
}

#  endif  /* SOLARIS2 || AUX || UXPDS */

/*----------------------------------------------- PIX or ISC */
#  if defined(PTX) || defined(ISC)

#    ifdef HAVE_SHADOW_H
#      include <shadow.h>
#    endif /* HAVE_SHADOW_H */

static int
auth_user ( p, pw )
POP     *   p;
struct passwd *pw;
{
    register struct spwd * pwd;
    long today;

    /*  
     * Look for the user in the shadow password file 
     */
    pwd = getspnam ( p->user );
    if ( pwd == NULL ) {
        if ( !strcmp ( pw->pw_passwd, "x" ) ) {      /* This my be a YP entry */
            sleep  ( SLEEP_SECONDS );
            return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
        }
    } else {
        pw->pw_passwd = (char *)strdup(pwd->sp_pwdp);
    }

    /*  
     * We don't accept connections from users with null passwords 
     *
     *  Compare the supplied password with the password file entry 
     */
    if ( (  pw->pw_passwd == NULL ) || 
         ( *pw->pw_passwd == '\0' ) ||
           strcmp ( crypt ( p->pop_parm[1], pw->pw_passwd ), pw->pw_passwd ) 
       ) {
        sleep  ( SLEEP_SECONDS );
        return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
    }

    return ( POP_SUCCESS );
}

#  endif  /* PTX || ISC */

/*----------------------------------------------- POPSCO or HPUX */
#  if defined(POPSCO) || defined(HPUX)

#    ifdef POPSCO
#      include <sys/security.h>
#      include <sys/audit.h>
#    else /* must be HPUX */
#      include <hpsecurity.h>
#    endif /* POPSCO */

#  include <prot.h>

#  define PASSWD(p)       p->ufld.fd_encrypt

static int
auth_user ( p, pw )
POP     *   p;
struct passwd *pw;
{
    register struct pr_passwd *pr;

    pr = getprpwnam ( p->user );
    if ( pr == NULL ) {
        if ( !strcmp ( pw->pw_passwd, "x" ) ) {
            sleep  ( SLEEP_SECONDS );
            return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
        }

    /*  
     * We don't accept connections from users with null passwords 
     */
    if ( (  pw->pw_passwd == NULL ) || 
         ( *pw->pw_passwd == '\0' ) ||
            ( strcmp ( bigcrypt ( p->pop_parm[1], pw->pw_passwd ), pw->pw_passwd ) &&
              strcmp ( crypt    ( p->pop_parm[1], pw->pw_passwd ), pw->pw_passwd )
            ) 
        ) {
        sleep  ( SLEEP_SECONDS );
        return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
        }
    } else {
        /* 
         * We don't accept connections from users with null passwords 
         *
         * Compare the supplied password with the password file entry 
         */
        if ( ( PASSWD(pr) == NULL ) || ( *PASSWD(pr) == '\0' ) ) {
            sleep  ( SLEEP_SECONDS );
            return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
        }
        
        if ( strcmp ( bigcrypt ( p->pop_parm[1], PASSWD(pr) ), PASSWD(pr) ) &&
             strcmp ( crypt    ( p->pop_parm[1], PASSWD(pr) ), PASSWD(pr) )  ) {
            sleep  ( SLEEP_SECONDS );
            return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
        }
    }

    return ( POP_SUCCESS );
}

#  endif  /* POPSCO || HPUX */

/*----------------------------------------------- ULTRIX */
#  ifdef ULTRIX

#    include <auth.h>

static int
auth_user ( p, pw )
struct passwd  *   pw;
POP     *   p;
{
    AUTHORIZATION *auth, *getauthuid();

    auth = getauthuid ( pw->pw_uid );
    if ( auth == NULL ) {
        if ( !strcmp(pw->pw_passwd, "x") ) {      /* This my be a YP entry */
            sleep  ( SLEEP_SECONDS );
            return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
        }
    } else {
        pw->pw_passwd = (char *)strdup(auth->a_password);
    }

    /*  
     * We don't accept connections from users with null passwords 
     *
     *  Compare the supplied password with the password file entry 
     */
    if ( (  pw->pw_passwd == NULL ) || 
         ( *pw->pw_passwd == '\0' )  {
        sleep ( SLEEP_SECONDS );
        return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
    }

    if ( strcmp ( crypt16 ( p->pop_parm[1], pw->pw_passwd ), pw->pw_passwd ) &&
         strcmp ( crypt   ( p->pop_parm[1], pw->pw_passwd ), pw->pw_passwd )  ) {
        sleep  ( SLEEP_SECONDS );
        return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
    }

    return ( POP_SUCCESS );
}

#  endif  /* ULTRIX */

/*----------------------------------------------- OSF1 */
/* this is a DEC Alpha / Digital Unix system */
#  ifdef OSF1

#    include <sys/types.h>
#    include <sys/security.h>
#    include <prot.h>

#    define   PASSWD(p)   (p->ufld.fd_encrypt)

static int
auth_user ( p, pw )
POP     *   p;
struct passwd *pw;
{
    register struct pr_passwd *pr;

    pr = getprpwnam ( p->user );
    if ( pr == NULL ) {
        if ( !strcmp ( pw->pw_passwd, "x" ) ) {      /* This my be a YP entry */
            sleep  ( SLEEP_SECONDS );
            return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
        }
    } else {
        pw->pw_passwd = (char *)strdup(PASSWD(pr));
    }

    /*  
     * We don't accept connections from users with null passwords 
     *
     *  Compare the supplied password with the password file entry 
     */
    if ( (  pw->pw_passwd == NULL ) || 
         ( *pw->pw_passwd == '\0' )  ) {
        sleep  ( SLEEP_SECONDS );
        return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
    }

    if ( strcmp ( bigcrypt ( p->pop_parm[1], pw->pw_passwd ), pw->pw_passwd ) &&
         strcmp ( crypt    ( p->pop_parm[1], pw->pw_passwd ), pw->pw_passwd )  ) {
        sleep  ( SLEEP_SECONDS );
        return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
    }

    return ( POP_SUCCESS );
}

#  endif        /* OSF1 */

/*----------------------------------------------- UNIXWARE */
#  ifdef UNIXWARE

#    include <shadow.h>

static int
auth_user ( p, pw )
struct passwd  *   pw;
POP     *   p;
{
    register struct spwd * pwd;
    long today;

    /*  
     * Look for the user in the shadow password file 
     */
    pwd = getspnam ( p->user );
    if ( pwd == NULL ) {
        if ( !strcmp ( pw->pw_passwd, "x" ) ) {      /* This my be a YP entry */
            sleep  ( SLEEP_SECONDS );
            return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
        }
    } else {
        today = (long)time((time_t *)NULL)/24/60/60;

        /* 
         * Check for expiration date 
         */
        if ( pwd->sp_expire > 0 && today > pwd->sp_expire ) {
            sleep  ( SLEEP_SECONDS );
            return ( pop_msg ( p, POP_FAILURE, HERE,
                              "[AUTH] \"%s\": account expired.", p->user ) );
        }

        /* 
         * Check if password is valid 
         */
        if ( pwd->sp_max > 0 && today > pwd->sp_lstchg + pwd->sp_max ) {
            sleep  ( SLEEP_SECONDS );
            return ( pop_msg ( p, POP_FAILURE, HERE,
                               "[AUTH] \"%s\": account expired.", p->user ) );
        }

        pw->pw_passwd = (char *)strdup(pwd->sp_pwdp);
        endspent();
    }

    /*  
     * We don't accept connections from users with null passwords 
     *
     *  Compare the supplied password with the password file entry 
     */
    if ( (  pw->pw_passwd == NULL ) || 
         ( *pw->pw_passwd == '\0' ) ||
         strcmp ( crypt ( p->pop_parm[1], pw->pw_passwd ), pw->pw_passwd ) 
       ) {
        sleep  ( SLEEP_SECONDS );
        return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
    }

    return ( POP_SUCCESS );
}

#  endif  /* UNIXWARE */

/*----------------------------------------------- LINUX (not PAM) */
#  if defined(LINUX) && !defined(USE_PAM)


#    ifdef HAVE_SHADOW_H
#      include <shadow.h>
#    endif /* HAVE_SHADOW_H */

static int
auth_user ( p, pw )
POP     *   p;
struct passwd *pw;
{
    register struct spwd * pwd;
    long today;

    /*  
     * Look for the user in the shadow password file 
     */
    pwd = getspnam ( p->user );
    if ( pwd == NULL ) {
        if ( !strcmp ( pw->pw_passwd, "x" ) ) {      /* This my be a YP entry */
            sleep  ( SLEEP_SECONDS );
            return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
        }
    } else {
        today = (long)time((time_t *)NULL)/24/60/60;

        /* 
         * Check for expiration date 
         */
        if ( pwd->sp_expire > 0 && today > pwd->sp_expire ) {
            sleep  ( SLEEP_SECONDS );
            return ( pop_msg ( p, POP_FAILURE, HERE,
                               "[AUTH] \"%s\": account expired.", p->user ) );
        }

        /* 
         * Check if password is valid 
         */
        if ( pwd->sp_max > 0 && today > pwd->sp_lstchg + pwd->sp_max ) {
            sleep  ( SLEEP_SECONDS );
            return ( pop_msg ( p, POP_FAILURE, HERE,
                               "[AUTH] \"%s\": account expired.", p->user ) );
        }

        pw->pw_passwd = (char *)strdup(pwd->sp_pwdp);
        endspent();
    }

    /*  
     * We don't accept connections from users with null passwords 
     *
     *  Compare the supplied password with the password file entry
     *
     *  Kernels older than 2.0.x need pw_encrypt() for shadow support 
     */
    if ( (  pw->pw_passwd == NULL ) || 
         ( *pw->pw_passwd == '\0' ) || 
         ( strcmp ( crypt      ( p->pop_parm[1], pw->pw_passwd ), pw->pw_passwd ) &&
#    ifdef HAVE_PW_ENCRYPT
           strcmp ( pw_encrypt ( p->pop_parm[1], pw->pw_passwd ), pw->pw_passwd )
#    else /* not HAVE_PW_ENCRYPT */
           strcmp ( crypt      ( p->pop_parm[1], pw->pw_passwd ), pw->pw_passwd )
#    endif /* HAVE_PW_ENCRYPT */
         )
       ) {
        sleep  ( SLEEP_SECONDS);
        return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
    }

    return ( POP_SUCCESS );
}

#  endif  /* LINUX */

/*----------------------------------------------- AIX  */
#  ifdef AIX

/*
 * AIX specific special authentication
 *
 * authenticate() is a standard (AIX) subroutine for password authentication
 * It's located in libs.a.    nik@bu.edu   11/19/1998 
 */

#    include <stddef.h>

static int
auth_user ( POP *p, struct passwd *pw )
{
  int reenter   = 0;
  char *message = 0;

  /* 
   * Nb. authenticate is probably going to blow away struct passwd *pw
   * static area.  Luckily, nobody who got it before auth_user() is expecting it
   * to still be valid after the auth_user() call. 
   */

  if ( authenticate ( p->user, p->pop_parm[1], &reenter, &message) == 0 ) {
      return ( POP_SUCCESS );
  } else {
      sleep  ( SLEEP_SECONDS );
      return ( pop_msg ( p, POP_FAILURE, HERE, message, p->user ) );
  }
}

#  endif /* AIX */

/*----------------------------------------------- PAM */
/* Code based on PAM patch contributed by German Poo.  
 * Patched by Kenneth Porter.
 */
#  ifdef USE_PAM
#    ifdef HAVE_SECURITY_PAM_APPL_H
#      include <security/pam_appl.h>
#    endif /* HAVE_SECURITY_PAM_APPL_H */

/* 
 * Some globals to make it easier to communicate between functions 
 */
static int   gp_errcode   = 0;
static char *GP_ERRSTRING =
            "[AUTH] PAM authentication failed for user \"%.100s\": %.128s (%d)";

static int 
PAM_qpopper_conv ( int num_msg, 
                   const struct pam_message **msg, 
                   struct pam_response **resp, 
                   void *appdata_ptr )
{
    int                   replies  = 0;
    struct pam_response  *reply    = NULL;
    POP                  *p        = appdata_ptr;


#   define COPY_STRING(s) (s) ? strdup(s) : NULL

    DEBUG_LOG1 ( p, "PAM_qpopper_conv: num_msg=%i", num_msg );

    reply = (struct pam_response*) malloc ( sizeof (struct pam_response) * num_msg );
    if ( reply == NULL ) 
        return PAM_CONV_ERR;

    for ( replies = 0; replies < num_msg; replies++ ) {
        DEBUG_LOG2 ( p, "PAM_qpopper_conv: msg_style[%i]=%i", 
                     replies, msg[replies]->msg_style );
        switch (msg[replies]->msg_style) {
        
        case PAM_PROMPT_ECHO_ON: /* assume it wants user name */
            reply[replies].resp_retcode = PAM_SUCCESS;
            reply[replies].resp = COPY_STRING(p->user);
            /* PAM frees resp */
            break;
            
        case PAM_PROMPT_ECHO_OFF: /* assume it wants password */
            reply[replies].resp_retcode = PAM_SUCCESS;
            reply[replies].resp = COPY_STRING(p->pop_parm[1]);
            /* PAM frees resp */
            break;
        
        case PAM_TEXT_INFO:
        case PAM_ERROR_MSG:
            reply[replies].resp_retcode = PAM_SUCCESS;
            reply[replies].resp = NULL;
            break;
            
        default:
            free (reply);
            gp_errcode = 1;
            return PAM_CONV_ERR;
        } /* switch */
    } /* for */
    
    *resp = reply;
    return PAM_SUCCESS;    
}
 
static struct pam_conv PAM_conversation = {
    &PAM_qpopper_conv,
    NULL
};

static int
auth_user(p, pw)
POP     * p;
struct passwd *pw;
{
    pam_handle_t    *pamh;
    int              pamerror;
    int              erc;
    const char      *errmsg;

    /* 
     * Let conv function access POP structure 
     */
    PAM_conversation.appdata_ptr = p;

    pamerror = pam_start ( USE_PAM, p->user, &PAM_conversation, &pamh );
    DEBUG_LOG3 ( p, "pam_start (service name %s) returned %i; gp_errcode=%i", 
                 USE_PAM, pamerror, gp_errcode );
    if ( ( gp_errcode != 0 ) || ( pamerror != PAM_SUCCESS ) ) {
        pam_end ( pamh, 0 );
        sleep   ( SLEEP_SECONDS );
        erc = pamerror ? pamerror : gp_errcode;
        return  ( pop_msg ( p, POP_FAILURE, HERE, GP_ERRSTRING, p->user,
                            pam_strerror(NULL, erc), erc ) );
    }
    pamerror = pam_authenticate ( pamh, 0 );
    DEBUG_LOG2 ( p, "pam_authenticate returned %i; gp_errcode=%i", 
                 pamerror, gp_errcode );
    if ( ( gp_errcode != 0 ) || ( pamerror != PAM_SUCCESS ) ) {
        erc = pamerror ? pamerror : gp_errcode;
        errmsg = pam_strerror ( pamh, erc );
        pam_end ( pamh, 0 );
        sleep   ( SLEEP_SECONDS );
        return  ( pop_msg ( p, POP_FAILURE, HERE, GP_ERRSTRING, 
                            p->user, errmsg, erc ) );
    }
    pamerror = pam_acct_mgmt ( pamh, 0 );
    DEBUG_LOG2 ( p, "pam_acct_mgmt returned %i; gp_errcode=%i", 
                 pamerror, gp_errcode );
    if ( ( gp_errcode != 0 ) || ( pamerror != PAM_SUCCESS ) ) {
        erc = pamerror ? pamerror : gp_errcode;
        errmsg = pam_strerror ( pamh, erc );
        pam_end ( pamh, 0 );
        sleep   ( SLEEP_SECONDS );
        return  ( pop_msg ( p, POP_FAILURE, HERE, GP_ERRSTRING, 
                            p->user, errmsg, erc ) );
    }
    pamerror = pam_setcred ( pamh, PAM_ESTABLISH_CRED );
    DEBUG_LOG2 ( p, "pam_setcred returned %i; gp_errcode=%i", 
                 pamerror, gp_errcode );
    if ( ( gp_errcode != 0 ) || ( pamerror != PAM_SUCCESS ) ) {
        erc = pamerror ? pamerror : gp_errcode;
        errmsg = pam_strerror ( pamh, erc );
        pam_end ( pamh, 0 );
        sleep   ( SLEEP_SECONDS );
        return  ( pop_msg ( p, POP_FAILURE, HERE, GP_ERRSTRING, 
                            p->user, errmsg, erc ) );
    }
    pam_end ( pamh, PAM_SUCCESS );
    
    return POP_SUCCESS;
}
#  endif /* USE_PAM */

/*------------------------------------- not SPEC_POP_AUTH */
#else   /* not SPEC_POP_AUTH */

char *crypt();

static int
auth_user ( p, pw )
POP     *   p;
struct passwd  *   pw;
{
    /*  
     * We don't accept connections from users with null passwords 
     *
     *  Compare the supplied password with the password file entry 
     */
    if ( (  pw->pw_passwd == NULL ) || 
         ( *pw->pw_passwd == '\0' ) ||
            strcmp ( crypt ( p->pop_parm[1], pw->pw_passwd ), pw->pw_passwd )
       ) {
        sleep ( SLEEP_SECONDS );
        return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
    }

    return ( POP_SUCCESS );
}

#endif  /* SPEC_POP_AUTH */


/* 
 *  pass:   Obtain the user password from a POP client
 */

#ifdef SECURENISPLUS
#  include <rpc/rpc.h>
#  include <rpc/key_prot.h>
#endif /* SECURENISPLUS */

int 
pop_pass ( p )
POP     *  p;
{
    struct passwd pw, *pwp;

#ifdef CHECK_SHELL
    char *getusershell();
    void endusershell();
    char *shell;
    char *cp;
    int shellvalid;
#endif /* CHECK_SHELL */

#ifdef SECURENISPLUS
    UID_T uid_save;
    char net_name  [ MAXNETNAMELEN ],
         secretkey [ HEXKEYBYTES + 1];

    *secretkey = '\0';
#endif /* SECURENISPLUS */

#ifdef NONAUTHFILE
    /* 
     * Is the user not authorized to use POP? 
     */
    if ( checknonauthfile ( p->user ) != 0 ) {
        DEBUG_LOG2 ( p, "User %.128s in nonauthfile %.256s",
                     p->user, NONAUTHFILE );
        sleep  ( SLEEP_SECONDS );
        return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
    }
#endif /* NONAUTHFILE */

#ifdef AUTHFILE
    /* 
     * Is the user authorized to use POP? 
     */
    if ( checkauthfile ( p->user ) != 0 ) {
        DEBUG_LOG2 ( p, "User %.128s not in authfile %.256s",
                     p->user, AUTHFILE );
        sleep  ( SLEEP_SECONDS );
        return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
    }
#endif /* AUTHFILE */

    /*  
     * Look for the user in the password file 
     */
    pwp = getpwnam ( p->user );
    if ( pwp == NULL ) {
        DEBUG_LOG1 ( p, "User %.128s not in passwd file",
                     p->user );
        sleep ( SLEEP_SECONDS );
        return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
    }

    pw = *pwp;
    pw.pw_dir = strdup(pwp->pw_dir);
    
    DEBUG_LOG5 ( p, "getpwnam returned %p (copied to %p); "
                    "home (%p) = (%d) '%s'",
                     pwp, &pw, pw.pw_dir, 
                     strlen(pw.pw_dir), pw.pw_dir );

#ifdef SECURENISPLUS
    /*  
     * we must do this keyserv stuff (as well as auth_user()!) as the user 
     */
    uid_save = geteuid();
    seteuid ( pw.pw_uid );

    /*  
     * see if DES keys are already known to the keyserv(1m) 
     */
    if ( ! key_secretkey_is_set() ) {
        /*  
         * keys are not known, so we must get the DES keys
         * and register with the keyserv(1m) 
         */
        getnetname ( net_name );

        if ( getpublickey ( net_name, secretkey ) ) {
            if ( strlen ( p->pop_parm[1] ) > 8 ) 
                ( p->pop_parm [1] ) [8] = '\0';

            if ( ! getsecretkey ( net_name, secretkey, p->pop_parm[1] ) ||
                 *secretkey == '\0' ) {
                sleep  ( SLEEP_SECONDS );
                free   ( pw.pw_dir );
                return ( pop_msg ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
            }

            key_setsecret ( secretkey );
            memset ( secretkey, '\0', sizeof(secretkey) );
        } else {
            /* 
             * if there are no keys defined, we assume that password entry
             * either resides in /etc/shadow or "root" has access to the
             * corresponding NIS+ entry 
             */
            seteuid ( 0 );
        }
    }
#endif /* SECURENISPLUS */

#ifdef BLOCK_UID
    if ( pw.pw_uid <= BLOCK_UID ) {
        free ( pw.pw_dir );
        return ( pop_msg ( p, POP_FAILURE, HERE,
                           "[AUTH] Access is blocked for UIDs below %d", 
                           BLOCK_UID ) );
    }
#endif /* BLOCK_UID */

#ifdef CHECK_SHELL
    /*  
     * Disallow anyone who does not have a standard shell as returned by
     * getusershell(), unless the sys admin has included the wildcard
     * shell in /etc/shells.  (default wildcard - /POPPER/ANY/SHELL)
     */
    shell = pw.pw_shell;
    if ( shell == NULL || *shell == 0 ) {
        /* 
         * You can default the shell, but I don't think it's a good idea 
         */
        /* 
         shell = "/usr/bin/sh"; 
         */
        free ( pw.pw_dir );
        return ( pop_msg ( p, POP_FAILURE, HERE, 
                           "[AUTH] No user shell defined" ) );
    }
    
    for ( shellvalid = 0; 
          shellvalid == 0 && ( cp = getusershell() ) != NULL;
        )
        if ( !strcmp ( cp, WILDCARD_SHELL ) || 
             !strcmp ( cp, shell )           )
             shellvalid = 1;
    endusershell();

    if ( shellvalid == 0 ) {
        free ( pw.pw_dir );
        return ( pop_msg ( p, POP_FAILURE, HERE, 
                           "[AUTH] \"%s\": shell not found.", 
                           p->user ) );
    }
#endif /* CHECK_SHELL */

    if ( ( p->kerberos ? auth_user_kerberos ( p, pw ) 
                       : auth_user          ( p, pwp ) 
         ) != POP_SUCCESS 
       ) {
        pop_log ( p, POP_PRIORITY, HERE,
                 "[AUTH] Failed attempted login to %s from host (%s) %s",
                  p->user, p->client, p->ipaddr );
        free   ( pw.pw_dir );
        sleep  ( SLEEP_SECONDS );
        return ( POP_FAILURE );
    }

#ifdef SECURENISPLUS
    seteuid ( uid_save );
#endif /* SECURENISPLUS */

    /*  
     * Make a temporary copy of the user's maildrop 
     * and set the group and user id 
     * and get information about the maildrop 
     */
    if ( pop_dropcopy ( p, &pw ) != POP_SUCCESS ) {
        free   ( pw.pw_dir );
        sleep  ( SLEEP_SECONDS );
        return ( POP_FAILURE );
        }

    /*  
     * Initialize the last-message-accessed number 
     */
    p->last_msg = 0;

    p->AuthState = pass;   /* plain or kerberos authenticated successfully */
    
    /* 
     * Authorization completed successfully 
     */

#ifdef LOG_LOGIN
    pop_log ( p, POP_PRIORITY, HERE,
              "(v%s) POP login by user \"%s\" at (%s) %s",
              VERSION, p->user, p->client, p->ipaddr );
#endif /* LOG_LOGIN */

    free ( pw.pw_dir );
    return ( pop_msg ( p, POP_SUCCESS, HERE,
                       "%s has %d visible message%s (%d hidden) in %d octets.",
                        p->user,
                        p->visible_msg_count,
                        p->visible_msg_count == 1 ? "" : "s",
                        (p->msg_count - p->visible_msg_count),
                        p->drop_size ) );
}

