// view.cpp : implementation of the CWpyScrollView class
//
//Python is Copyright 1991-1997 by Stichting Mathematisch Centrum,
//Amsterdam, The Netherlands.  See their copyright notice.

//This code is copyright 1994-1997 by James C. Ahlstrom.  See the
//copyright notice in "copyrite.jim" or e-mail jim@interet.com.
// NO WARRANTIES AT ALL.  USE AT YOUR OWN RISK.


#include "stdafx.h"
#include "app.h"

#include "doc.h"
#include "view.h"
#include "controls.h"
#include "globals.h"
#include "util.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

static void RecordNewView(CView *view)
{
#ifndef NETSCAPE_PI	// We are not in a Netscape Plugin
	PyObject * pyview = PyEval_CallObject (theApp.m_python_view, NULL);
	if (!pyview || pyview == Py_None) {
		ASSERT(0);
		AfxMessageBox("Can not create view");
	}
	else
		theApp.RecordObjects(view, pyview, TRUE);
#endif
}

/////////////////////////////////////////////////////////////////////////////
// CWpyScrollView

IMPLEMENT_DYNCREATE(CWpyScrollView, CScrollView)

BEGIN_MESSAGE_MAP(CWpyScrollView, CScrollView)
	//{{AFX_MSG_MAP(CWpyScrollView)
	ON_WM_SIZE()
	ON_WM_CREATE()
	ON_WM_VSCROLL()
	ON_WM_HSCROLL()
	ON_WM_CHAR()
	ON_WM_LBUTTONDOWN()
	ON_WM_MOUSEMOVE()
	ON_WM_LBUTTONUP()
	ON_WM_DESTROY()
	ON_WM_LBUTTONDBLCLK()
	ON_WM_RBUTTONDBLCLK()
	ON_WM_RBUTTONDOWN()
	ON_WM_RBUTTONUP()
	ON_WM_ERASEBKGND()
	ON_WM_KEYDOWN()
	//}}AFX_MSG_MAP
	// Standard printing commands
	ON_COMMAND(ID_FILE_PRINT, CScrollView::OnFilePrint)
#ifdef MS_WIN32
	ON_COMMAND(ID_FILE_PRINT_DIRECT, CScrollView::OnFilePrint)
#endif
	ON_COMMAND(ID_FILE_PRINT_PREVIEW, CScrollView::OnFilePrintPreview)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CWpyScrollView construction/destruction

CWpyScrollView::CWpyScrollView()
{
	is_printing = 0;
	RecordNewView(this);
	m_DrawItems.SetSize(0, 1024);
}

CWpyScrollView::~CWpyScrollView()
{
	theApp.UnRecordCpp(this, TRUE);	// Remove map to Python object.
	CObject *obj;
	POSITION pos = m_DeleteList.GetHeadPosition();
	while (pos){	// Delete all objects on delete list.
		obj = m_DeleteList.GetNext(pos);
		theApp.UnRecordCpp(obj, TRUE);
		delete obj;
	}
	m_DeleteList.RemoveAll();
	int i, num = m_DrawItems.GetSize();
	for (i = 0; i < num; i++)
		delete m_DrawItems[i];
	m_DrawItems.RemoveAll();
}

HWND CWpyScrollView::UnsubclassWindow()
{
	ASSERT(::IsWindow(m_hWnd));

	// set WNDPROC back to original value
	WNDPROC* lplpfn = GetSuperWndProcAddr();
	SetWindowLong(m_hWnd, GWL_WNDPROC, (LONG)*lplpfn);
	*lplpfn = NULL;

	// and Detach the HWND from the CWnd object
	return Detach();
}

#ifdef MS_WIN16
WNDPROC* CWpyScrollView::GetSuperWndProcAddr()
{   
    static WNDPROC pfnSuper;   // place to store window proc
    return &pfnSuper;
}
#endif

/////////////////////////////////////////////////////////////////////////////
// CWpyScrollView implementation


/////////////////////////////////////////////////////////////////////////////
// CWpyScrollView drawing

// Class CDrawItem: An object drawn on a view

CDrawItem::CDrawItem(PyObject * obj)
{
	long x, y, w, h, t;
	GetAttrLong(obj, "wpyLocX", &x);
	GetAttrLong(obj, "wpyLocY", &y);
	GetAttrLong(obj, "wpySizeX", &w);
	GetAttrLong(obj, "wpySizeY", &h);
	m_rect = CRect((int)x, (int)y, (int)(x + w), (int)(y + h));
	m_pyobj = obj;
	char * pt = GetAttrString(obj, "wpyType");
	ASSERT(pt);
	if (pt)
		m_type = *pt;
	else
		m_type = '?';	// Unknown type
	if (m_type == 'l') { 	// Line: Is a point at the upper left corner?
		x = GetAttrLong(obj, "wpyPointAtUL", &t);
		ASSERT(x);
		m_pointAtUL = (BOOL)t;
		x = GetAttrLong(obj, "wpyWidth", &t);
		ASSERT(x);
		m_lineWidth = (int)t;
	}
	Py_INCREF(m_pyobj);
}

CDrawItem::~CDrawItem()
{
	Py_DECREF(m_pyobj);
}

// End of CDrawItem

int CWpyScrollView::FindDrawIndex(PyObject * obj)
{
	int i, num = m_DrawItems.GetSize();
	CDrawItem * item;
	for (i = 0; i < num; i++){
		item = (CDrawItem *)m_DrawItems[i];
		if (item->m_pyobj == obj)	// Linear search - need a dict?
			return i;
	}
	return -1;
}

void CWpyScrollView::MoveDrawn()
{
	int i, count;
	count = m_DrawItems.GetSize();
	for (i = 0; i < count; i++) {
		CDrawItem * item = (CDrawItem *)m_DrawItems[i];
		long x, y, dx, dy;
		PyObject * pyobj = item->m_pyobj;
		GetAttrLong(pyobj, "wpyLocX", &x);
		GetAttrLong(pyobj, "wpyLocY", &y);
		dx = x - item->m_rect.left;
		dy = y - item->m_rect.top;
		if (dx || dy) {
			InvalidateRect(item->m_rect - GetDeviceScrollPosition());
			item->m_rect.OffsetRect(dx, dy);
			InvalidateRect(item->m_rect - GetDeviceScrollPosition());
		}
	}
}

void CWpyScrollView::Redraw(PyObject * obj)
{
	int i, i1, i2;
 	if (obj == Py_None) {	// Redraw everything
		i1 = 0;
		i2 = m_DrawItems.GetSize();
	}
	else {		// just redraw one item
		i1 = FindDrawIndex(obj);
		if (i1 < 0)
			return;
		i2 = i1 + 1;
	}
	for (i = i1; i < i2; i++) {
		CDrawItem * item = (CDrawItem *)m_DrawItems[i];
		InvalidateRect(item->m_rect - GetDeviceScrollPosition());
		long x, y, w, h;
		PyObject * pyobj = item->m_pyobj;
		GetAttrLong(pyobj, "wpyLocX", &x);
		GetAttrLong(pyobj, "wpyLocY", &y);
		GetAttrLong(pyobj, "wpySizeX", &w);
		GetAttrLong(pyobj, "wpySizeY", &h);
		if (item->m_type == 'T') {	// TextOut()
			PyObject_SetAttrString(pyobj, "wpyOffset", PyInt_FromLong(-1L));
			if (!w)
				w = 32000;
		}
		if (!w && item->m_type == 't')	// DrawText() fit-to-text
			w = 32000;
		item->m_rect.SetRect((int)x, (int)y, (int)(x + w), (int)(y + h));
		InvalidateRect(item->m_rect - GetDeviceScrollPosition());
	}
}

static PyObject * GetDrawnObj(int method, int x, int y, CObArray& drawItems)
{
// Search for drawn objects:  method==0, return the object at (x,y);
// method==1, return a list of objects at (x,y); method==2, return
// a list of all objects ignoring (x,y).
	PyObject * pylist = 0;
	if (method)	// Return a list instead of a single item
		pylist = PyList_New(0);
	int i, num = drawItems.GetSize();
	if (method == 2) {
		for (i = 0; i < num; i++) {
			CDrawItem * item = (CDrawItem *)drawItems[i];
			PyObject * pyobj = item->m_pyobj;
			if (PyList_Append(pylist, pyobj) != 0)
				break;
		}
		return pylist;
	}

	CPoint point = CPoint(x, y);
	for (i = 0; i < num; i++) {
		CDrawItem * item = (CDrawItem *)drawItems[i];
		CRect rect = item->m_rect;
		if (!rect.PtInRect(point))
			continue;
		PyObject * pyobj = item->m_pyobj;
		if (method) {
			if (item->m_type == 'l') {	//Check distance from line
				double x0, y0, x1, y1;
				if (item->m_pointAtUL) {
					x0 = item->m_rect.left;	// point at upper left
					y0 = item->m_rect.top;
					x1 = item->m_rect.right;
					y1 = item->m_rect.bottom;
				}
				else {
					x0 = item->m_rect.left;	// point at lower left
					y0 = item->m_rect.bottom;
					x1 = item->m_rect.right;
					y1 = item->m_rect.top;
				}
				double vx, vy, t1, squared_dist;
				vx = x1 - x0;
				vy = y1 - y0;
				t1 = vx * vx + vy * vy;
				if (t1 < 0.1)
					continue;
				t1 = ((x - x0) * vx + (y - y0) * vy) / t1;
				vx = x0 + vx * t1;
				vy = y0 + vy * t1;
				squared_dist  = (vx - x) * (vx - x) + (vy - y) * (vy - y);
				if (squared_dist > item->m_lineWidth * item->m_lineWidth / 4.0 + 1)
					continue;
			}
			if (PyList_Append(pylist, pyobj) != 0)
				break;
		}
		else {
			if (item->m_type == 'l')	// Do not return drawn lines
				continue;
			Py_INCREF(pyobj);
			return pyobj;
		}
	}
	if (method)
		return pylist;
 	Py_INCREF(Py_None);
	return Py_None;
}

PyObject * CWpyScrollView::ScrollDrawnObjs (long x1, long y1, long x2, long y2, long dx, long dy)
{	// Move all drawn objs touching the rectangle by (dx, dy)
	int i, num = m_DrawItems.GetSize();
	long x, y, w, h;
	PyObject *pyobj, *v;
	CRect rect(x1, y1, x2, y2);
	CRect test;
	for (i = 0; i < num; i++) {
		CDrawItem * item = (CDrawItem *)m_DrawItems[i];
		if (!test.IntersectRect(rect, item->m_rect))
			continue;
		InvalidateRect(item->m_rect - GetDeviceScrollPosition());
		pyobj = item->m_pyobj;
		GetAttrLong(pyobj, "wpyLocX", &x);
		GetAttrLong(pyobj, "wpyLocY", &y);
		x += dx;
		y += dy;
		v = PyInt_FromLong(x);
		PyObject_SetAttrString(pyobj, "wpyLocX", v);
		v = PyInt_FromLong(y);
		PyObject_SetAttrString(pyobj, "wpyLocY", v);
		GetAttrLong(pyobj, "wpySizeX", &w);
		GetAttrLong(pyobj, "wpySizeY", &h);
		item->m_rect.SetRect((int)x, (int)y, (int)(x + w), (int)(y + h));
		InvalidateRect(item->m_rect - GetDeviceScrollPosition());
	}
	return NULL;
}

PyObject * CWpyScrollView::GetDrawnObjs(int method, int x, int y)
{
	return ::GetDrawnObj(method, x, y, m_DrawItems);
}

void CWpyScrollView::RecordDraw(PyObject * obj)
{
	if (!is_printing) {	// Do not alter list during printing.
		CDrawItem * item = new CDrawItem(obj);
		m_DrawItems.Add(item);
		InvalidateRect(item->m_rect - GetDeviceScrollPosition());
	}
}

void CWpyScrollView::DestroyAllDrawn()
{
	int i, num = m_DrawItems.GetSize();
	for (i = 0; i < num; i++)
		delete m_DrawItems[i];
	m_DrawItems.RemoveAll();
	if(num)
		InvalidateRect(NULL);
}

void CWpyScrollView::DestroyDrawn(PyObject * obj)
{
	if (!is_printing){
		int index = FindDrawIndex(obj);
		if (index >= 0){
			CDrawItem * item = (CDrawItem *)m_DrawItems[index];
			InvalidateRect(item->m_rect - GetDeviceScrollPosition());
			m_DrawItems.RemoveAt(index);
			delete item;
		}
	}
}

static void CheckBrush(PyObject * pyobj, PyObject ** cur_brush, CDC *pDC)
{	// See if the current brush is correct for drawing.
	PyObject * pybrush = PyObject_GetAttrString(pyobj, "wpyBrush");
	ASSERT(pybrush);
	CBrush * pBrush;
	if (pybrush && pybrush != *cur_brush &&	// Must change brush
		(pBrush = (CBrush *)theApp.GetCppObject(pybrush))){
			pDC->SelectObject(pBrush);
			*cur_brush = pybrush;
	}
	Py_XDECREF(pybrush);
}

static void CheckPen(PyObject * pyobj, PyObject ** cur_pen, CDC *pDC)
{	// See if the current pen is correct for drawing.
	PyObject * pypen = PyObject_GetAttrString(pyobj, "wpyPen");
	ASSERT(pypen);
	CPen * pPen;
	if (pypen && pypen != *cur_pen &&	// Must change pen
		(pPen = (CPen *)theApp.GetCppObject(pypen))){
			pDC->SelectObject(pPen);
			*cur_pen = pypen;
	}
	Py_XDECREF(pypen);
}

static void SetupDc(CDC * pDC, PyObject * pydc)	// Set attributes of DC
{
	PyObject * v = PyInt_FromLong((long)pDC);
	PyObject_SetAttrString(pydc, "wpyphysDC", v);
	v = PyInt_FromLong((long)pDC->GetDeviceCaps(HORZRES));
	PyObject_SetAttrString(pydc, "wpySizeX", v);
	v = PyInt_FromLong((long)pDC->GetDeviceCaps(VERTRES));
	PyObject_SetAttrString(pydc, "wpySizeY", v);
#ifdef MS_WIN32
	v = PyInt_FromLong((long)pDC->GetDeviceCaps(PHYSICALWIDTH));
	PyObject_SetAttrString(pydc, "wpyPageX", v);
	v = PyInt_FromLong((long)pDC->GetDeviceCaps(PHYSICALHEIGHT));
	PyObject_SetAttrString(pydc, "wpyPageY", v);
	v = PyInt_FromLong((long)pDC->GetDeviceCaps(PHYSICALOFFSETX));
	PyObject_SetAttrString(pydc, "wpyOffsetX", v);
	v = PyInt_FromLong((long)pDC->GetDeviceCaps(PHYSICALOFFSETY));
	PyObject_SetAttrString(pydc, "wpyOffsetY", v);
#else
	POINT pt;
	pt.x = 2550;
	pt.y = 3300;
	pDC->Escape(GETPHYSPAGESIZE, 0, NULL, &pt);
	v = PyInt_FromLong(pt.x);
	PyObject_SetAttrString(pydc, "wpyPageX", v);
	v = PyInt_FromLong(pt.y);
	PyObject_SetAttrString(pydc, "wpyPageY", v);
	pt.x = pt.y = 60;
	pDC->Escape(GETPRINTINGOFFSET, 0, NULL, &pt);
	v = PyInt_FromLong(pt.x);
	PyObject_SetAttrString(pydc, "wpyOffsetX", v);
	v = PyInt_FromLong(pt.y);
	PyObject_SetAttrString(pydc, "wpyOffsetY", v);
#endif
}

static void WpyOnDraw(CView* This, CDC* pDC, CObArray& drawItems, int& is_printing)
{
	int num = drawItems.GetSize();
	if (pDC->IsPrinting()){
		is_printing = 1;
		PyObject * pyobj = theApp.GetPythonObject(This);
		ASSERT(pyobj);
		PyObject * pydc = PyObject_GetAttrString(pyobj, "wpyPrinterDC");
		ASSERT(pydc);
		SetupDc(pDC, pydc);
		int save_dc = pDC->SaveDC();
		theApp.SetUpDC(pDC, pydc);
		PyObject * v = Py_BuildValue("(Oi)", pydc, 1);
		theApp.CallPython(This, "WpyPhysOnDraw", v);
		Py_DECREF(v);
		pDC->RestoreDC(save_dc);
		Py_DECREF(pydc);
		is_printing = 0;
		return;
	}
	if (num == 0){	// Call Python to refresh draw list
		PyObject * pyobj = theApp.GetPythonObject(This);
		ASSERT(pyobj);
		PyObject * pydc = PyObject_GetAttrString(pyobj, "wpyDC");
		ASSERT(pydc);
		PyObject * v = PyInt_FromLong((long)pDC);
		PyObject_SetAttrString(pydc, "wpyphysDC", v);
		int save_dc = pDC->SaveDC();
		theApp.SetUpDC(pDC, pydc);
		RECT r;
		This->GetClientRect(&r);
		v = PyInt_FromLong((long)r.right);
		PyObject_SetAttrString(pydc, "wpySizeX", v);
		v = PyInt_FromLong((long)r.bottom);
		PyObject_SetAttrString(pydc, "wpySizeY", v);
		v = Py_BuildValue("(Oi)", pydc, 0);
		theApp.CallPython(This, "WpyPhysOnDraw", v);
		Py_DECREF(v);
		pDC->RestoreDC(save_dc);
		Py_DECREF(pydc);
	}
	// Redraw from the list
	int i;
	PyObject * pyfont, *pyback, * cur_font = 0;
	PyObject * cur_pen = 0;
	PyObject * cur_brush = 0;
	PyObject * pycolor, * cur_color = 0;
	CWpyFont * pWpyFont;
	for (i = 0; i < num; i++){
		CDrawItem * item = (CDrawItem *)drawItems[i];
		CRect rect = item->m_rect;
		if (!pDC->RectVisible(rect))
			continue;
		PyObject * pyobj = item->m_pyobj;
		switch (item->m_type){
			case 'a':	// Draw arc
				CheckPen(pyobj, &cur_pen, pDC);
				CheckBrush(pyobj, &cur_brush, pDC);
				WpyArcInt(pDC, pyobj);
				break;
			case 'e':	// Draw ellipse
				CheckPen(pyobj, &cur_pen, pDC);
				CheckBrush(pyobj, &cur_brush, pDC);
				WpyEllipseInt(pDC, pyobj);
				break;
			case 'i':	// Draw image
				WpyDrawImageInt(pDC, pyobj);
				break;
			case 'l':	// Draw line
				CheckPen(pyobj, &cur_pen, pDC);
				WpyLineToInt(pDC, pyobj);
				break;
			case 'p':	// Draw polygon
				CheckPen(pyobj, &cur_pen, pDC);
				CheckBrush(pyobj, &cur_brush, pDC);
				WpyPolygonInt(pDC, pyobj);
				break;
			case 'r':	// Draw rectangle
				CheckPen(pyobj, &cur_pen, pDC);
				CheckBrush(pyobj, &cur_brush, pDC);
				WpyRectangleInt(pDC, pyobj);
				break;
			case 't':	// Draw Text
			case 'T':	// Text Out
				pyfont = PyObject_GetAttrString(pyobj, "wpyFont");
				ASSERT(pyfont);
				if (pyfont && pyfont != cur_font &&	// Must change font
					(pWpyFont = (CWpyFont *)theApp.GetCppObject(pyfont))){
						pDC->SelectObject(&pWpyFont->m_ScreenFont);
						cur_font = pyfont;
				}
				pycolor = PyObject_GetAttrString(pyobj, "wpyTextColor");
				ASSERT(pycolor);
				if (pycolor && pycolor != cur_color){	// Must change color
					PyObject * py = PyTuple_GetItem(pycolor, 0);
					ASSERT(py);
					long r = PyInt_AsLong(py);
					py = PyTuple_GetItem(pycolor, 1);
					ASSERT(py);
					long g = PyInt_AsLong(py);
					py = PyTuple_GetItem(pycolor, 2);
					ASSERT(py);
					long b = PyInt_AsLong(py);
					pDC->SetTextColor(RGB(r, g, b));
					cur_color = pycolor;
				}
				long sizex;	// Update for zero sizex
				GetAttrLong(pyobj, "wpySizeX", &sizex);
				pyback = PyObject_GetAttrString(pyobj, "wpyTextBkColor");
				int did_color = 0;
				COLORREF old;
				if (pyback && PyTuple_Check(pyback)){	// Must change background color
					did_color = 1;
					PyObject * py = PyTuple_GetItem(pyback, 0);
					ASSERT(py);
					long r = PyInt_AsLong(py);
					py = PyTuple_GetItem(pyback, 1);
					ASSERT(py);
					long g = PyInt_AsLong(py);
					py = PyTuple_GetItem(pyback, 2);
					ASSERT(py);
					long b = PyInt_AsLong(py);
					old = pDC->SetBkColor(RGB(r, g, b));
				}
				else {
					PyErr_Clear();
				}
				if (item->m_type == 't'){
					WpyDrawTextInt(pDC, pyobj, 0);
				}
				else {		// Use TextOut()
					char * text = GetAttrString(pyobj, "wpyText");
					ASSERT(text);
					long offset;
					GetAttrLong(pyobj, "wpyOffset", &offset);
					if (offset < 0) {	// Recalculate text size
						char * justify = GetAttrString(pyobj, "wpyJustify");
						ASSERT(justify);
						long width;
						GetAttrLong(pyobj, "wpySizeX", &width);
						offset = WpyFixTextOut(pyobj, pDC, text, width, justify);
					}
					pDC->TextOut(rect.left + offset, rect.top, text, strlen(text));
				}
				if (did_color)
					pDC->SetBkColor(old);
				if (!sizex) {	// wpySizeX has been recalculated
					GetAttrLong(pyobj, "wpySizeX", &sizex);
					item->m_rect.right = item->m_rect.left + sizex;
				}
				Py_XDECREF(pyback);
				Py_DECREF(pyfont);
				Py_DECREF(pycolor);
				break;
		}
	}
}

void CWpyScrollView::OnDraw(CDC* pDC)
{
	WpyOnDraw(this, pDC, m_DrawItems, is_printing);
}

static BOOL EraseBkgnd(CObject * This, CDC* pDC) 
{
	PyObject * pyobj = theApp.GetPythonObject(This);
	PyObject * pybrush = PyObject_GetAttrString(pyobj, "wpyBrush");
	CBrush * pBrush;
	if (pybrush){
		if (pBrush = (CBrush *)theApp.GetCppObject(pybrush)){
			CBrush* pOldBrush = pDC->SelectObject(pBrush);
			CRect rect;
			pDC->GetClipBox(&rect);     // Erase the area needed
			pDC->PatBlt(rect.left, rect.top, rect.Width(), rect.Height(), PATCOPY);
			pDC->SelectObject(pOldBrush);
			Py_DECREF(pybrush);
			return TRUE;
		}
	Py_DECREF(pybrush);
	}
	else
		PyErr_Clear();
	return FALSE;
}

BOOL CWpyScrollView::OnEraseBkgnd(CDC* pDC) 
{
	return ::EraseBkgnd(this, pDC) || CScrollView::OnEraseBkgnd(pDC);
}

/////////////////////////////////////////////////////////////////////////////
// CWpyScrollView printing

static void PreparePrinting(CObject * This, CPrintInfo* pInfo)
{
	PyObject * pyview = theApp.GetPythonObject(This);
	ASSERT(pyview);
	theApp.CallPython(pyview, "OnPreparePrinting");	
	long nMaxPage;
	if (!GetAttrLong(pyview, "wpyPrintMaxPage", &nMaxPage))
		nMaxPage = 0;
	if (nMaxPage > 0)
		pInfo->SetMaxPage(nMaxPage);
}

BOOL CWpyScrollView::OnPreparePrinting(CPrintInfo* pInfo)
{
	PreparePrinting(this, pInfo);
	return DoPreparePrinting(pInfo);
}

static void BeginPrinting(CObject * This, CDC* pDC, CPrintInfo* pInfo)
{
	PyObject * pyview = theApp.GetPythonObject(This);
	ASSERT(pyview);
	PyObject * pydc = PyObject_GetAttrString(pyview, "wpyPrinterDC");
	ASSERT(pydc);
	SetupDc(pDC, pydc);
	int save_dc = pDC->SaveDC();
	theApp.SetUpDC(pDC, pydc);
	PyObject * args = Py_BuildValue("(O)", pydc);
	theApp.CallPython(pyview, "OnBeginPrinting", args);
	Py_DECREF(args);
	Py_DECREF(pydc);
	pDC->RestoreDC(save_dc);
	long nMaxPage;
	if (!GetAttrLong(pyview, "wpyPrintMaxPage", &nMaxPage))
		nMaxPage = 1;
	if (nMaxPage <= 0)
		nMaxPage = 1;
	pInfo->SetMaxPage(nMaxPage);
}

void CWpyScrollView::OnBeginPrinting(CDC* pDC, CPrintInfo* pInfo)
{
	BeginPrinting(this, pDC, pInfo);
}

void EndPrinting(CObject * This, CDC* pDC, CPrintInfo* pInfo)
{
	PyObject * pyview = theApp.GetPythonObject(This);
	ASSERT(pyview);
	PyObject * pydc = PyObject_GetAttrString(pyview, "wpyPrinterDC");
	ASSERT(pydc);
	SetupDc(pDC, pydc);
	int save_dc = pDC->SaveDC();
	theApp.SetUpDC(pDC, pydc);
	PyObject * args = Py_BuildValue("(O)", pydc);
	theApp.CallPython(pyview, "OnEndPrinting", args);
	Py_DECREF(args);
	Py_DECREF(pydc);
	pDC->RestoreDC(save_dc);
}

void CWpyScrollView::OnEndPrinting(CDC* pDC, CPrintInfo* pInfo)
{
	EndPrinting(this, pDC, pInfo);
}

static void PrepareDC(CObject * This, CDC* pDC, CPrintInfo* pInfo) 
{
	if (pInfo) {	// We are printing
		PyObject * pyview = theApp.GetPythonObject(This);
		ASSERT(pyview);
		PyObject * pydc = PyObject_GetAttrString(pyview, "wpyPrinterDC");
		ASSERT(pydc);
		SetupDc(pDC, pydc);
		int save_dc = pDC->SaveDC();
		theApp.SetUpDC(pDC, pydc);
		PyObject * args = Py_BuildValue("(Ol)", pydc, (long)pInfo->m_nCurPage);
		PyObject * ret = theApp.CallPythonRet(pyview, "OnPrepareDC", args);
		Py_DECREF(args);
		Py_DECREF(pydc);
		pDC->RestoreDC(save_dc);
		if (ret && PyInt_Check(ret) && PyInt_AsLong(ret) == 0)
			pInfo->m_bContinuePrinting = 0;
		Py_XDECREF(ret);
	}
}

void CWpyScrollView::OnPrepareDC(CDC* pDC, CPrintInfo* pInfo) 
{
	CScrollView::OnPrepareDC(pDC, pInfo);
	PrepareDC(this, pDC, pInfo);
}

static void XxPrint(CObject * This, CDC* pDC, CPrintInfo* pInfo) 
{
	PyObject * pyview = theApp.GetPythonObject(This);
	ASSERT(pyview);
	PyObject * pydc = PyObject_GetAttrString(pyview, "wpyPrinterDC");
	ASSERT(pydc);
	SetupDc(pDC, pydc);
	int save_dc = pDC->SaveDC();
	theApp.SetUpDC(pDC, pydc);
	PyObject * args = Py_BuildValue("(Ol)", pydc, (long)pInfo->m_nCurPage);
	PyObject * ret = theApp.CallPythonRet(pyview, "WpyPhysOnPrint", args);
	Py_DECREF(args);
	Py_DECREF(pydc);
	if (ret && PyInt_Check(ret) && PyInt_AsLong(ret) == 0)
		pInfo->m_bContinuePrinting = 0;
	Py_XDECREF(ret);
	pDC->RestoreDC(save_dc);
}

void CWpyScrollView::OnPrint(CDC* pDC, CPrintInfo* pInfo) 
{
	XxPrint(this, pDC, pInfo);
}

/////////////////////////////////////////////////////////////////////////////
// CWpyScrollView diagnostics

#ifdef _DEBUG
void CWpyScrollView::AssertValid() const
{
	CScrollView::AssertValid();
}

void CWpyScrollView::Dump(CDumpContext& dc) const
{
	CScrollView::Dump(dc);
}

CWpythonDoc* CWpyScrollView::GetDocument() // non-debug version is inline
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CWpythonDoc)));
	return (CWpythonDoc*)m_pDocument;
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CWpyScrollView message handlers

void CWpyScrollView::OnInitialUpdate() 
{
	DestroyAllDrawn();
	SetScrollSizes(MM_TEXT, CSize(0, 0));
	CDocument * doc = GetDocument();
	PyObject * pydoc, * args;
	if (doc && (pydoc = theApp.GetPythonObject(doc)))
	  args = Py_BuildValue("(O)", pydoc);
	else
	  args = Py_BuildValue("(O)", Py_None);
	theApp.CallPython(this, "WpyphysOnInitialUpdate", args);
	Py_DECREF(args);
	CScrollView::OnInitialUpdate();
}

void CWpyScrollView::OnSize(UINT nType, int cx, int cy) 
{
	CScrollView::OnSize(nType, cx, cy);
	PyObject * ret = Py_BuildValue("(iii)", nType, cx, cy);
	theApp.CallPython(this, "WpyPhysOnSize", ret);
	Py_DECREF(ret);
}

int CWpyScrollView::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CScrollView::OnCreate(lpCreateStruct) == -1)
		return -1;
	CFrameWnd * frame = GetParentFrame();
	PyObject * pyframe, * args;
	if (frame && (pyframe = theApp.GetPythonObject(frame)))
	  args = Py_BuildValue("(O)", pyframe);
	else
	  args = Py_BuildValue("(O)", Py_None);
	theApp.CallPython(this, "WpyphysOnCreate", args);
	Py_DECREF(args);
	return 0;
}

void CWpyScrollView::OnVScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
	if (pScrollBar)		// Message came from a control
		WpyOnScroll(nSBCode, nPos, pScrollBar);
	else{
		CScrollView::OnVScroll(nSBCode, nPos, pScrollBar);
		int pos = GetScrollPos(SB_VERT);
		PyObject * ret = Py_BuildValue("(i)", pos);
		theApp.CallPython(this, "WpyPhysOnVScroll", ret);
		Py_DECREF(ret);
	}
}

void CWpyScrollView::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
	if (pScrollBar)	// Message came from a control
		WpyOnScroll(nSBCode, nPos, pScrollBar);
	else{
		CScrollView::OnHScroll(nSBCode, nPos, pScrollBar);
		int pos = GetScrollPos(SB_HORZ);
		PyObject * ret = Py_BuildValue("(i)", pos);
		theApp.CallPython(this, "WpyPhysOnHScroll", ret);
		Py_DECREF(ret);
	}
}

void CWpyScrollView::OnChar(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	if (WpyOnChar(this, nChar, nRepCnt, nFlags))
		CScrollView::OnChar(nChar, nRepCnt, nFlags);
}

void CWpyScrollView::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	if (WpyOnVChar(this, nChar, nRepCnt, nFlags))
		CScrollView::OnKeyDown(nChar, nRepCnt, nFlags);
}

void CWpyScrollView::OnLButtonDown(UINT nFlags, CPoint point) 
{
	WpyMouse(this, nFlags, point, "OnLButtonDown");
	CScrollView::OnLButtonDown(nFlags, point);
}

void CWpyScrollView::OnMouseMove(UINT nFlags, CPoint point) 
{
	if (!nFlags){	// Only send MouseMove when flags is set
		CScrollView::OnMouseMove(nFlags, point);
		return;
	}
	WpyMouse(this, nFlags, point, "OnMouseMove");
}

void CWpyScrollView::OnLButtonUp(UINT nFlags, CPoint point) 
{
	WpyMouse(this, nFlags, point, "OnLButtonUp");
	CScrollView::OnLButtonUp(nFlags, point);
}

void CWpyScrollView::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
	WpyMouse(this, nFlags, point, "OnLButtonDblClk");
	CScrollView::OnLButtonDblClk(nFlags, point);
}

void CWpyScrollView::OnRButtonDblClk(UINT nFlags, CPoint point) 
{
	WpyMouse(this, nFlags, point, "OnRButtonDblClk");
	CScrollView::OnRButtonDblClk(nFlags, point);
}

void CWpyScrollView::OnRButtonDown(UINT nFlags, CPoint point) 
{
	WpyMouse(this, nFlags, point, "OnRButtonDown");
	CScrollView::OnRButtonDown(nFlags, point);
}

void CWpyScrollView::OnRButtonUp(UINT nFlags, CPoint point) 
{
	WpyMouse(this, nFlags, point, "OnRButtonUp");
	CScrollView::OnRButtonUp(nFlags, point);
}

void CWpyScrollView::OnDestroy() 
{
	CScrollView::OnDestroy();
	theApp.CallPython(this, "WpyPhysOnDestroy");
}

BOOL CWpyScrollView::OnCommand(WPARAM wParam, LPARAM lParam) 
{
	if (theApp.PyOnCommand(wParam, lParam))
		return TRUE;
	return CScrollView::OnCommand(wParam, lParam);
}

/////////////////////////////////////////////////////////////////////////////
// CWpythonEditView

IMPLEMENT_DYNCREATE(CWpythonEditView, CEditView)

CWpythonEditView::CWpythonEditView()
{
	RecordNewView(this);
}

CWpythonEditView::~CWpythonEditView()
{
	theApp.UnRecordCpp(this, TRUE);	// Remove map to Python object.
	CObject *obj;
	POSITION pos = m_DeleteList.GetHeadPosition();
	while (pos){	// Delete all objects on delete list.
		obj = m_DeleteList.GetNext(pos);
		theApp.UnRecordCpp(obj, TRUE);
		delete obj;
	}
	m_DeleteList.RemoveAll();	
}

HWND CWpythonEditView::UnsubclassWindow()
{
	ASSERT(::IsWindow(m_hWnd));

	// set WNDPROC back to original value
	WNDPROC* lplpfn = GetSuperWndProcAddr();
	SetWindowLong(m_hWnd, GWL_WNDPROC, (LONG)*lplpfn);
	*lplpfn = NULL;

	// and Detach the HWND from the CWnd object
	return Detach();
}

#ifdef MS_WIN16
WNDPROC* CWpythonEditView::GetSuperWndProcAddr()
{   
    static WNDPROC pfnSuper;   // place to store window proc
    return &pfnSuper;
}
#endif


BEGIN_MESSAGE_MAP(CWpythonEditView, CEditView)
	//{{AFX_MSG_MAP(CWpythonEditView)
	ON_WM_CHAR()
	ON_WM_DESTROY()
	ON_WM_CREATE()
	ON_WM_RBUTTONDBLCLK()
	ON_WM_RBUTTONDOWN()
	ON_WM_RBUTTONUP()
	ON_WM_SIZE()
	ON_WM_ERASEBKGND()
	ON_WM_KEYDOWN()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


/////////////////////////////////////////////////////////////////////////////
// CWpythonEditView drawing

BOOL CWpythonEditView::OnEraseBkgnd(CDC* pDC) 
{
	// TODO: Add your message handler code here and/or call default
	
	return ::EraseBkgnd(this, pDC) || CEditView::OnEraseBkgnd(pDC);
}

/////////////////////////////////////////////////////////////////////////////
// CWpythonEditView printing

BOOL CWpythonEditView::OnPreparePrinting(CPrintInfo* pInfo)
{
	PreparePrinting(this, pInfo);
	return DoPreparePrinting(pInfo);
}

void CWpythonEditView::OnBeginPrinting(CDC* pDC, CPrintInfo* pInfo)
{
	BeginPrinting(this, pDC, pInfo);
}

void CWpythonEditView::OnEndPrinting(CDC* pDC, CPrintInfo* pInfo)
{
	EndPrinting(this, pDC, pInfo);
}

void CWpythonEditView::OnPrepareDC(CDC* pDC, CPrintInfo* pInfo) 
{
	CView::OnPrepareDC(pDC, pInfo);
	PrepareDC(this, pDC, pInfo);
}

void CWpythonEditView::OnPrint(CDC* pDC, CPrintInfo* pInfo) 
{
	XxPrint(this, pDC, pInfo);
}

/////////////////////////////////////////////////////////////////////////////
// CWpythonEditView diagnostics

#ifdef _DEBUG
void CWpythonEditView::AssertValid() const
{
	CEditView::AssertValid();
}

void CWpythonEditView::Dump(CDumpContext& dc) const
{
	CEditView::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CWpythonEditView message handlers

void CWpythonEditView::OnInitialUpdate() 
{
	CDocument * doc = GetDocument();
	PyObject * pydoc, * args;
	if (doc && (pydoc = theApp.GetPythonObject(doc)))
	  args = Py_BuildValue("(O)", pydoc);
	else
	  args = Py_BuildValue("(O)", Py_None);
	theApp.CallPython(this, "WpyphysOnInitialUpdate", args);
	Py_DECREF(args);
	CEditView::OnInitialUpdate();
}

void CWpythonEditView::OnChar(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	if (WpyOnChar(this, nChar, nRepCnt, nFlags))
		CEditView::OnChar(nChar, nRepCnt, nFlags);
}

void CWpythonEditView::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	if (WpyOnVChar(this, nChar, nRepCnt, nFlags))
		CEditView::OnKeyDown(nChar, nRepCnt, nFlags);
}

void CWpythonEditView::OnDestroy() 
{
	CEditView::OnDestroy();
	theApp.CallPython(this, "WpyPhysOnDestroy");
}

int CWpythonEditView::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CEditView::OnCreate(lpCreateStruct) == -1)
		return -1;
	CFrameWnd * frame = GetParentFrame();
	PyObject * pyframe, * args;
	if (frame && (pyframe = theApp.GetPythonObject(frame)))
	  args = Py_BuildValue("(O)", pyframe);
	else
	  args = Py_BuildValue("(O)", Py_None);
	theApp.CallPython(this, "WpyphysOnCreate", args);
	Py_DECREF(args);
	return 0;
}

void CWpythonEditView::OnRButtonDblClk(UINT nFlags, CPoint point) 
{
	WpyMouse(this, nFlags, point, "OnRButtonDblClk");
	CEditView::OnRButtonDblClk(nFlags, point);
}

void CWpythonEditView::OnRButtonDown(UINT nFlags, CPoint point) 
{
	WpyMouse(this, nFlags, point, "OnRButtonDown");
	CEditView::OnRButtonDown(nFlags, point);
}

void CWpythonEditView::OnRButtonUp(UINT nFlags, CPoint point) 
{
	WpyMouse(this, nFlags, point, "OnRButtonUp");
	CEditView::OnRButtonUp(nFlags, point);
}

void CWpythonEditView::OnSize(UINT nType, int cx, int cy) 
{
	CEditView::OnSize(nType, cx, cy);
	PyObject * ret = Py_BuildValue("(iii)", nType, cx, cy);
	theApp.CallPython(this, "WpyPhysOnSize", ret);
	Py_DECREF(ret);
}
/////////////////////////////////////////////////////////////////////////////
// CWpythonView

IMPLEMENT_DYNCREATE(CWpythonView, CView)

CWpythonView::CWpythonView()
{
	is_printing = 0;
	m_IsSubclassed = 0;
	RecordNewView(this);
	m_DrawItems.SetSize(0, 1024);
}

CWpythonView::~CWpythonView()
{
	CObject *obj;
	POSITION pos = m_DeleteList.GetHeadPosition();
	while (pos){	// Delete all objects on delete list.
		obj = m_DeleteList.GetNext(pos);
		theApp.UnRecordCpp(obj, TRUE);
		delete obj;
	}
	m_DeleteList.RemoveAll();
	int i, num = m_DrawItems.GetSize();
	for (i = 0; i < num; i++)
		delete m_DrawItems[i];
	m_DrawItems.RemoveAll();
}

BOOL CWpythonView::SubclassWindow(HWND hWnd)
{	// Extra logic to prevent multiple subclass
	if (m_IsSubclassed)
		return 0;
	else {
		m_IsSubclassed = CWnd::SubclassWindow(hWnd);
		if (m_IsSubclassed)
			theApp.m_pNetscapeView = this;
		return m_IsSubclassed;
	}
}

HWND CWpythonView::UnsubclassWindow()
{	// Extra logic to prevent multiple unsubclass
	if (!m_IsSubclassed)
		return 0;
	m_IsSubclassed = 0;

	if (theApp.m_pNetscapeView == this)
		theApp.m_pNetscapeView = 0;

	ASSERT(::IsWindow(m_hWnd));

	// set WNDPROC back to original value
	WNDPROC* lplpfn = GetSuperWndProcAddr();
	SetWindowLong(m_hWnd, GWL_WNDPROC, (LONG)*lplpfn);
	*lplpfn = NULL;

	// and Detach the HWND from the CWnd object
	return Detach();
}

#ifdef MS_WIN16
WNDPROC* CWpythonView::GetSuperWndProcAddr()
{   
    static WNDPROC pfnSuper;   // place to store window proc
    return &pfnSuper;
}
#endif


BEGIN_MESSAGE_MAP(CWpythonView, CView)
	//{{AFX_MSG_MAP(CWpythonView)
	ON_WM_CHAR()
	ON_WM_HSCROLL()
	ON_WM_VSCROLL()
	ON_WM_LBUTTONDBLCLK()
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_MOUSEMOVE()
	ON_WM_RBUTTONDBLCLK()
	ON_WM_RBUTTONDOWN()
	ON_WM_RBUTTONUP()
	ON_WM_ERASEBKGND()
	ON_WM_INITMENUPOPUP()
	ON_WM_KEYDOWN()
	//}}AFX_MSG_MAP
#ifdef NETSCAPE_PI
	ON_WM_MOUSEACTIVATE()
#endif
END_MESSAGE_MAP()


/////////////////////////////////////////////////////////////////////////////
// CWpythonView drawing

int CWpythonView::FindDrawIndex(PyObject * obj)
{
	int i, num = m_DrawItems.GetSize();
	CDrawItem * item;
	for (i = 0; i < num; i++){
		item = (CDrawItem *)m_DrawItems[i];
		if (item->m_pyobj == obj)	// Linear search - need a dict?
			return i;
	}
	return -1;
}

void CWpythonView::MoveDrawn()
{
	int i, count;
	count = m_DrawItems.GetSize();
	for (i = 0; i < count; i++) {
		CDrawItem * item = (CDrawItem *)m_DrawItems[i];
		long x, y, dx, dy;
		PyObject * pyobj = item->m_pyobj;
		GetAttrLong(pyobj, "wpyLocX", &x);
		GetAttrLong(pyobj, "wpyLocY", &y);
		dx = x - item->m_rect.left;
		dy = y - item->m_rect.top;
		if (dx || dy) {
			InvalidateRect(item->m_rect);
			item->m_rect.OffsetRect(dx, dy);
			InvalidateRect(item->m_rect);
		}
	}
}

void CWpythonView::Redraw(PyObject * obj)
{
	int i, i1, i2;
 	if (obj == Py_None) {	// Redraw everything
		i1 = 0;
		i2 = m_DrawItems.GetSize();
	}
	else {		// just redraw one item
		i1 = FindDrawIndex(obj);
		if (i1 < 0)
			return;
		i2 = i1 + 1;
	}
	for (i = i1; i < i2; i++) {
		CDrawItem * item = (CDrawItem *)m_DrawItems[i];
		InvalidateRect(item->m_rect);
		long x, y, w, h;
		PyObject * pyobj = item->m_pyobj;
		GetAttrLong(pyobj, "wpyLocX", &x);
		GetAttrLong(pyobj, "wpyLocY", &y);
		GetAttrLong(pyobj, "wpySizeX", &w);
		GetAttrLong(pyobj, "wpySizeY", &h);
		if (item->m_type == 'T') {	// TextOut()
			PyObject_SetAttrString(pyobj, "wpyOffset", PyInt_FromLong(-1L));
			if (!w)
				w = 32000;
		}
		if (!w && item->m_type == 't')	// DrawText() fit-to-text
			w = 32000;
		item->m_rect.SetRect((int)x, (int)y, (int)(x + w), (int)(y + h));
		InvalidateRect(item->m_rect);
	}
}

PyObject * CWpythonView::GetDrawnObjs(int method, int x, int y)
{
	return ::GetDrawnObj(method, x, y, m_DrawItems);
}

void CWpythonView::RecordDraw(PyObject * obj)
{
	if (!is_printing) {	// Do not alter list during printing.
		CDrawItem * item = new CDrawItem(obj);
		m_DrawItems.Add(item);
		InvalidateRect(item->m_rect);
	}
}

void CWpythonView::DestroyAllDrawn()
{
	int i, num = m_DrawItems.GetSize();
	for (i = 0; i < num; i++)
		delete m_DrawItems[i];
	m_DrawItems.RemoveAll();
	if (num)
		InvalidateRect(NULL);
}

void CWpythonView::DestroyDrawn(PyObject * obj)
{
	if (!is_printing){
		int index = FindDrawIndex(obj);
		if (index >= 0){
			CDrawItem * item = (CDrawItem *)m_DrawItems[index];
			InvalidateRect(item->m_rect);
			m_DrawItems.RemoveAt(index);
			delete item;
		}
	}
}

void CWpythonView::OnDraw(CDC* pDC)
{
	WpyOnDraw(this, pDC, m_DrawItems, is_printing);
}

/////////////////////////////////////////////////////////////////////////////
// CWpythonView diagnostics

#ifdef _DEBUG
void CWpythonView::AssertValid() const
{
	CView::AssertValid();
}

void CWpythonView::Dump(CDumpContext& dc) const
{
	CView::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CWpythonView message handlers

void CWpythonView::OnBeginPrinting(CDC* pDC, CPrintInfo* pInfo) 
{
	// TODO: Add your specialized code here and/or call the base class
	
	CView::OnBeginPrinting(pDC, pInfo);
}

void CWpythonView::OnEndPrinting(CDC* pDC, CPrintInfo* pInfo) 
{
	// TODO: Add your specialized code here and/or call the base class
	
	CView::OnEndPrinting(pDC, pInfo);
}

BOOL CWpythonView::OnCommand(WPARAM wParam, LPARAM lParam) 
{
	if (theApp.PyOnCommand(wParam, lParam))
		return TRUE;
	return CView::OnCommand(wParam, lParam);
}

void CWpythonView::OnChar(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	if (WpyOnChar(this, nChar, nRepCnt, nFlags))
		CView::OnChar(nChar, nRepCnt, nFlags);
}

void CWpythonView::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	if (WpyOnVChar(this, nChar, nRepCnt, nFlags))
		CView::OnKeyDown(nChar, nRepCnt, nFlags);
}

void CWpythonView::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
	if (pScrollBar)	// Message came from a control
		WpyOnScroll(nSBCode, nPos, pScrollBar);
	else{
		CView::OnHScroll(nSBCode, nPos, pScrollBar);
		int pos = GetScrollPos(SB_HORZ);
		PyObject * ret = Py_BuildValue("(i)", pos);
		theApp.CallPython(this, "WpyPhysOnHScroll", ret);
		Py_DECREF(ret);
	}
}

void CWpythonView::OnVScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
	if (pScrollBar)		// Message came from a control
		WpyOnScroll(nSBCode, nPos, pScrollBar);
	else{
		CView::OnVScroll(nSBCode, nPos, pScrollBar);
		int pos = GetScrollPos(SB_VERT);
		PyObject * ret = Py_BuildValue("(i)", pos);
		theApp.CallPython(this, "WpyPhysOnVScroll", ret);
		Py_DECREF(ret);
	}
}

void CWpythonView::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
	WpyMouse(this, nFlags, point, "OnLButtonDblClk");
	CView::OnLButtonDblClk(nFlags, point);
}

void CWpythonView::OnLButtonDown(UINT nFlags, CPoint point) 
{
	WpyMouse(this, nFlags, point, "OnLButtonDown");
	CView::OnLButtonDown(nFlags, point);
}

void CWpythonView::OnLButtonUp(UINT nFlags, CPoint point) 
{
	WpyMouse(this, nFlags, point, "OnLButtonUp");
	CView::OnLButtonUp(nFlags, point);
	theApp.FlushMessages();
}

void CWpythonView::OnMouseMove(UINT nFlags, CPoint point) 
{
	if (!nFlags){	// Only send MouseMove when flags is set
		CView::OnMouseMove(nFlags, point);
		return;
	}
	WpyMouse(this, nFlags, point, "OnMouseMove");
}

void CWpythonView::OnRButtonDblClk(UINT nFlags, CPoint point) 
{
	WpyMouse(this, nFlags, point, "OnRButtonDblClk");
	CView::OnRButtonDblClk(nFlags, point);
}

void CWpythonView::OnRButtonDown(UINT nFlags, CPoint point) 
{
	WpyMouse(this, nFlags, point, "OnRButtonDown");
	CView::OnRButtonDown(nFlags, point);
}

void CWpythonView::OnRButtonUp(UINT nFlags, CPoint point) 
{
	WpyMouse(this, nFlags, point, "OnRButtonUp");
	CView::OnRButtonUp(nFlags, point);
}

BOOL CWpythonView::OnCmdMsg( UINT nID, int nCode, void* pExtra, 
	AFX_CMDHANDLERINFO* pHandlerInfo )
{
	if (nID >= COMMAND_BASE_SEQ && nID <= COMMAND_MAX_SEQ) switch(nCode){
	case CN_COMMAND:
		theApp.CallPython(nID, "WpyPhysOnCommand");
		return( TRUE );
	case CN_UPDATE_COMMAND_UI:
		// Not called !!
		theApp.PyOnCommandUI( (CCmdUI*)pExtra );
		return( TRUE );
	}
	return CView::OnCmdMsg( nID, nCode, pExtra, pHandlerInfo);
}

#ifdef NETSCAPE_PI
int CWpythonView::OnMouseActivate(CWnd* pDesktopWnd, UINT nHitTest, UINT message)
{
	return CWnd::OnMouseActivate(pDesktopWnd, nHitTest, message);
}
#endif

void CWpythonView::OnInitMenuPopup(CMenu * pMenu, UINT u, BOOL b)
{	// OnCommandUI messages not generated!
	CView::OnInitMenuPopup(pMenu, u, b);
	UINT i, num = pMenu->GetMenuItemCount();
	for (i = 0; i < num; i++){
		UINT id = pMenu->GetMenuItemID(i);
		PyObject * pPyObj;
		if (id > 0 && (pPyObj = theApp.GetPythonControl(id))){
			long value;
			if (GetAttrLong(pPyObj, "wpyphysMenuFlags", &value)){
				if (value & MF_GRAYED)
					pMenu->EnableMenuItem(i, MF_GRAYED | MF_BYPOSITION);
				else
					pMenu->EnableMenuItem(i, MF_ENABLED | MF_BYPOSITION);
				if (value & WPY_MF_CHECKBOX){
					if (value & MF_CHECKED)
						pMenu->CheckMenuItem(i, MF_CHECKED | MF_BYPOSITION);
					else
						pMenu->CheckMenuItem(i, MF_UNCHECKED | MF_BYPOSITION);
				}
				else if (value & WPY_MF_RADIOBUTTON){
					if (value & MF_CHECKED)
						pMenu->CheckMenuItem(i, MF_CHECKED | MF_BYPOSITION);
					else
						pMenu->CheckMenuItem(i, MF_UNCHECKED | MF_BYPOSITION);
				}
			}
		}
	}
}

BOOL CWpythonView::OnEraseBkgnd(CDC* pDC) 
{
	// TODO: Add your message handler code here and/or call default
	
	return ::EraseBkgnd(this, pDC) || CView::OnEraseBkgnd(pDC);
}
