open Py_types
open Py_exceptions

let count_pc s = 
  let count = ref 0 in
  let n = String.length s in
  let i = ref 0 in
  while !i < n do
    if s.[!i] = '%' then
    if !i+1 < n then
    if s.[!i+1] <> '%'
    then incr count
    else incr i;
    incr i
  done;
  !count
;;

let py_format fmt v =
let m = count_pc fmt in
if m = 0 then fmt
else let v' = ref begin 
  match v with
  | PyDictionary d -> v
  | _ ->
    if m = 1 then PyTuple [v] (* single, non sequence, argument *)
    else try PyTuple (Py_util.list_of_sequence v) (* sequence argument *)
    with NonSequence -> raise (ValueError "In fmt % expr, expr must be dictionary or sequence when no of % > 1")
end
in
  let get_integer k =
    match !v' with
    | PyTuple (PyInt i :: t) -> v' := PyTuple t; i
    | PyTuple (_ :: _) -> raise (TypeError "Integer required for format")
    | PyTuple([]) -> raise (ValueError "Out of values for format") 
    | PyDictionary d -> 
      begin match d#get_item k with
      | Some (PyInt i) -> i
      | None -> raise (KeyError ("in A % D form",k))
      | _ -> raise (TypeError "Integer required for format")
      end 
    | _ -> raise (Failure "get_integer in py_format")
  and get_float k =
    match !v' with
    | PyTuple(PyFloat f :: t) -> v' := PyTuple t; f
    | PyTuple (_ :: _) -> raise (ValueError "Float required for format")
    | PyTuple ([]) -> raise (ValueError "Out of values for format") 
    | PyDictionary d -> 
      begin match d#get_item k with
      | Some (PyFloat i) -> i
      | None -> raise (KeyError ("in A % D form",k))
      | _ -> raise (TypeError "Float required for format")
      end 
    | _ -> raise (Failure "get_float in py_format")
and get_string k =
    match !v' with
    | PyTuple (PyString s :: t) -> v' := PyTuple t; s
    | PyTuple (_ :: _) -> raise (ValueError "String required for format")
    | PyTuple ([]) -> raise (ValueError "Out of values for format") 
    | PyDictionary d -> 
      begin match d#get_item k with
      | Some (PyString i) -> i
      | None -> raise (KeyError ("in A % D form",k))
      | _ -> raise (TypeError "String required for format")
      end 
    | _ -> raise (Failure "get_string in py_format")
in
  let i = ref 0 
  and n = String.length fmt
  and s = ref "" 
  in let process_escape () =
    (* format control flags and variables *)
    let minimum_width = ref 0
    and precision = ref 0
    and zero_padding = ref false
    and hash_convert = ref false
    and left_adjust = ref false
    and leave_space = ref false
    and show_sign = ref false
    and dictionary_key = ref PyInitial

    (* parsing subroutines *)
    in let rec get_leading_flags () = 
      match fmt.[!i] with
      | '#' -> hash_convert := true; incr i; get_leading_flags()
      | '0' -> zero_padding := true; incr i; get_leading_flags()
      | '-' -> left_adjust := true; incr i; get_leading_flags()
      | ' ' -> leave_space := true; incr i; get_leading_flags()
      | '+' -> show_sign := true ; incr i; get_leading_flags()
      | _ -> ()
    and get_width () =
      match fmt.[!i] with
      | '0'..'9' as ch -> 
        minimum_width := !minimum_width * 10 + (Char.code ch) - (Char.code '0');
        incr i;
        get_width ()
      | '*' -> minimum_width := get_integer !dictionary_key
      | _ -> ()
    and get_precision () =
      if fmt.[!i] = '.'
      then begin
        incr i;
        match fmt.[!i] with
        | '0'..'9' as ch -> 
          precision := !precision * 10 + (Char.code ch) - (Char.code '0');
          incr i;
          get_precision ()
        | '*' -> precision := get_integer !dictionary_key
        | _ -> ()
      end
    and get_dictionary_key () =
      match fmt.[!i] with 
      | '(' -> 
        incr i;
        let key = ref "" in 
        while fmt.[!i] <> ')' do 
          key := !key ^ (String.make 1 fmt.[!i]); 
          incr i 
        done;
        dictionary_key := PyString !key;
        incr i
      | _ -> ()
    and get_trailing_flags () =
      match fmt.[!i] with
      | 'h'
      | 'l'
      | 'L'
      | 'q'
      | 'Z'
        -> incr i
      | _ -> ()
    and get_code () =
      begin match fmt.[!i] with
      | 's' -> 
        let s' = get_string !dictionary_key in  
        let n = String.length s' in
        if n < !minimum_width then
        begin 
          if !left_adjust 
          then s := !s ^ s' ^ (String.make (!minimum_width - n) ' ')
          else s := !s ^ (String.make (!minimum_width - n) ' ') ^ s'
        end
        else s := !s ^ s'
        
      | 'e' | 'E'  -> 
        (* incomplete: a hack! *)
        let f = get_float !dictionary_key in 
        s := !s ^ (string_of_float f)
      | 'f' -> 
        let f = get_float !dictionary_key in
        s := !s ^ (string_of_float f)
      | 'g' -> 
        let f = get_float !dictionary_key in
        s := !s ^ (string_of_float f)
      | 'c' -> 
        (* currently: utf8 of the integer
           should be: single byte hex, use code 'U'? for
           utf8
        *)
        let c = get_integer !dictionary_key in 
        s := !s ^ (Py_string.utf8_of_int c)

      (* integers *)
      | 'i' | 'd'  -> 
        let i = get_integer !dictionary_key in
        let negative = i < 0 
        and s' = ref (string_of_int (abs i)) in 
        let n = String.length !s' in
        let sign = 
          if negative then "-"
          else if !show_sign then "+"
          else if !leave_space then " "
          else ""
        in let n' = n + String.length sign
        in let pad = !minimum_width - n'
        in 
          if pad <= 0 (* no padding *)
          then s := !s ^ sign ^ !s'
          else if !left_adjust (* pad right blank *)
          then s := !s ^ sign ^ !s' ^ (String.make pad ' ')
          else if !zero_padding (* pad left zero *)
          then s := !s ^ sign ^ (String.make pad '0') ^ !s'
          else (* pad left blank *)
            s:= !s ^ (String.make pad ' ') ^ sign ^ !s'
        
          
      | 'o' 
      | 'u'
      | 'x'
      | 'X' -> raise (NotImplemented "ouxX formats")

      | '%' -> s := !s ^ (String.make 1 '%')
      | x -> s := !s ^ (String.make 1 x)
      end;
      incr i
    in
      (* process escape *)
      get_dictionary_key();
      get_leading_flags();
      get_width();
      get_precision();
      get_trailing_flags();
      get_code()
    (* end process escape *)
  in while !i < n do
    match fmt.[!i] with
    | '%' -> incr i; process_escape ()
    | ch -> s:= !s ^ (String.make 1 ch); incr i
  done;
  !s
;;


