open Py_types
open Py_exceptions
open Py_builtins_util

(* shared *)

(* convert file kind to integer flag bits *)
let cvt_mode x = match x with 
  | Unix.S_REG ->  0
  | Unix.S_DIR ->  0x4000
  | Unix.S_CHR ->  0x2000
  | Unix.S_BLK ->  0x6000
  | Unix.S_LNK ->  0x8000 
  | Unix.S_FIFO -> 0x1000
  | Unix.S_SOCK -> 0xc000

module File_stats = struct
open Unix
let pystats file_stats = 
  PyTuple [
    PyInt (cvt_mode file_stats.st_kind) (* st_mode *);
    PyInt file_stats.st_ino      (* st_ino *);
    PyInt file_stats.st_dev      (* st_dev *);
    PyInt file_stats.st_nlink    (* st_nlink *);
    PyInt file_stats.st_uid      (* st_uid *);
    PyInt file_stats.st_gid      (* st_gid *);
    PyInt  file_stats.st_size    (* st_size *);
    PyFloat file_stats.st_atime  (* st_atime *);
    PyFloat file_stats.st_mtime  (* st_mtime *);
    PyFloat file_stats.st_ctime  (* st_ctime *)
  ]
end

module Timing = struct
open Unix
let mktimes times now = 
   PyTuple [
    PyFloat times.tms_utime;
    PyFloat times.tms_stime;
    PyFloat times.tms_cutime;
    PyFloat times.tms_cstime;
    PyFloat now
  ]
end

let py_posix_chdir
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  begin match arg e 0 with
  | PyString s -> Unix.chdir s
  | _ -> raise (TypeError "chdir requires string argument")
  end
  ;
  PyNone

let py_posix_chmod
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let filename, mode =
    begin match arg e 0 with 
    | PyString s -> s
    | _ -> raise (TypeError "chmod requires string for filename")
    end
  ,
    begin match arg e 1 with 
    | PyInt i -> i
    | _ -> raise (TypeError "chmod requires integer for mode")
    end
  in
    Unix.chmod filename mode
    ;
    PyNone

let py_posix_chown
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 3;
  let filename, uid, gid =
    begin match arg e 0 with 
    | PyString s -> s
    | _ -> raise (TypeError "chown requires string for filename")
    end
  ,
    begin match arg e 1 with 
    | PyInt i -> i
    | _ -> raise (TypeError "chown requires integer for uid")
    end
  ,
    begin match arg e 2 with 
    | PyInt i -> i
    | _ -> raise (TypeError "chown requires integer for gid")
    end
  in
    Unix.chown filename uid gid
    ;
    PyNone


(* py_posix_close: use file_close *)

let py_posix_dup
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | PyFile f -> PyFile (Unix.dup f)
  | _ -> raise (TypeError "dup requires native file object")
  
let py_posix_dup2
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let fd1, fd2 =
    begin match arg e 0 with
    | PyFile f -> f
    | _ -> raise (TypeError "dup2 requires native file object (arg1)")
    end
  ,
    begin match arg e 1 with
    | PyFile f -> f
    | _ -> raise (TypeError "dup2 requires native file object (arg2)")
    end
  in 
    Unix.dup2 fd1 fd2
    ;
    PyNone
  
let py_posix_execv
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  print_endline "WARNING: KILLING VIPER WITH EXECV!!"; flush stdout;
  let program, arguments =
    begin match arg e 0 with
    | PyString s -> s
    | _ -> raise (TypeError "program (arg1) of execv must be string")
    end
  ,
     Array.of_list
     begin
       List.map 
       begin fun x -> 
         match x with 
         | PyString s -> s 
         | _ -> raise (TypeError "element of args (arg2) of execv must be string")
       end
       (Py_util.list_of_sequence (arg e 1))
     end
   in Unix.execv program arguments
   ;
   PyNone

let py_posix_execve
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 3;
  print_endline "WARNING: KILLING VIPER WITH EXECVE!!"; flush stdout;
  let program, arguments, environment =
    begin match arg e 0 with
    | PyString s -> s
    | _ -> raise (TypeError "program (arg1) of execve must be string")
    end
  ,
     Array.of_list
     begin
       List.map 
       begin fun x -> 
         match x with 
         | PyString s -> s 
         | _ -> raise (TypeError "element of args (arg2) of execve must be string")
       end
       (Py_util.list_of_sequence (arg e 1))
     end
   , 
     let d = begin match arg e 2 with
     | PyDictionary d -> d
     | _ -> raise (TypeError "Dictionary required for arg3 of execve")
     end
     in let envlist = ref [] in
     d#iter
     begin fun k v ->
       let k', v' = 
         begin match k with 
         | PyString s -> s
         | _ -> raise (TypeError "String required in dictionary key of arg3 of execve")
         end
       ,
         begin match v with 
         | PyString s -> s
         | _ -> raise (TypeError "String required in dictionary value of arg3 of execve")
         end
       in envlist := (k' ^ "=" ^ v') :: !envlist
     end
     ;
     (Array.of_list !envlist)
   in Unix.execve program arguments environment
   ;
   PyNone


(* Note: unlike posix._exit, this function flushes pending I/O
  this is a bug, since posix._exit is intended to close child
  subprocesses and threads.
*)

let py_posix_exit
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | PyInt n -> exit n
  | _ -> raise (TypeError "Arg of exit must be integer")
  ;
  PyNone

let py_posix_fdopen
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  raise (NotImplemented "posix.fdopen");
  PyNone

let py_posix_fork
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  PyInt (Unix.fork ())
  
let py_posix_fstat
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let file = match arg e 0 with
  | PyFile f -> f
  | _ -> raise (TypeError "posix.fstat requires native file argument")
  in File_stats.pystats (Unix.fstat file)

let py_posix_ftruncate
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let file, size = 
    begin match arg e 0 with
    | PyFile f -> f
    | _ -> raise (TypeError "posix.ftruncate requires native file for arg1")
    end
  ,
    begin match arg e 1 with
    | PyInt i -> i
    | _ -> raise (TypeError "posix.ftruncate requires int for arg2")
    end
  in 
    Unix.ftruncate file size
    ; 
    PyNone
  
let py_posix_getcwd
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  PyString (Unix.getcwd ())

let py_posix_getegid
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  PyInt (Unix.getegid ())

let py_posix_geteuid
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  PyInt (Unix.geteuid ())

let py_posix_getgid
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  PyInt (Unix.getgid ())

let py_posix_getpgrp
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  raise (NotImplemented "Bad Python Documentation for getpgrp")

let py_posix_getpid
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  PyInt (Unix.getpid ())

let py_posix_getppid
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  PyInt (Unix.getppid ())

let py_posix_getuid
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  PyInt (Unix.getuid ())

let py_posix_kill
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let pid,signal = 
    begin match arg e 0 with
    | PyInt i -> i
    | _ -> raise (TypeError "posix.kill requires int pid")
    end
  ,
    begin match arg e 1 with
    | PyInt i -> i
    | _ -> raise (TypeError "posix.kill requires int signal number")
    end
  in 
    Unix.kill pid signal
    ;
    PyNone

let py_posix_link
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let src,dst = 
    begin match arg e 0 with
    | PyString s -> s
    | _ -> raise (TypeError "posix.link requires string src")
    end
  ,
    begin match arg e 1 with
    | PyString s -> s
    | _ -> raise (TypeError "posix.link requires string dst")
    end
  in 
    Unix.link src dst
    ;
    PyNone

let py_posix_listdir
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  let dirname = match arg e 0 with
  | PyString s -> s
  | _ -> raise (TypeError "posix.listdir requires string pathname")
  in 
    let dh = Unix.opendir dirname
    and dl = ref [] 
  in 
    begin try dl := PyString (Unix.readdir dh) :: !dl
    with End_of_file -> Unix.closedir dh
    end
  ;
    PyMutableList (Varray.of_list !dl)

(* Note: this function returns some kind of value,
  I do not know what it is (not documented in ocaml doco)
  or if python does (not documented in python doco)
*)
let py_posix_lseek
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 3;
  let file,pos,how = 
    begin match arg e 0 with
    | PyFile f -> f
    | _ -> raise (TypeError "posix.lseek requires native file arg1")
    end
  ,
    begin match arg e 1 with
    | PyInt i -> i
    | _ -> raise (TypeError "posix.lseek requires int position")
    end
  ,
    begin match arg e 2 with
    | PyInt 0 -> Unix.SEEK_SET
    | PyInt 1 -> Unix.SEEK_CUR
    | PyInt 2 -> Unix.SEEK_END
    | PyInt _ -> raise (ValueError "posix.lseek 'how' must be 0,1 or 2")
    | _ -> raise (TypeError "posix.lseek requires int position")
    end
  in 
    PyInt (Unix.lseek file pos how)

let py_posix_lstat
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let file = match arg e 0 with
  | PyString f -> f
  | _ -> raise (TypeError "posix.lstat requires string filename argument")
  in File_stats.pystats (Unix.lstat file)

let py_posix_mkfifo
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atleast e 1;
  atmost e 2;
  let path, mode = 
    begin match arg e 0 with
    | PyString s -> s
    | _ -> raise (TypeError "posix.mkfifo requires string path arg1")
    end
  ,
    begin match optional e 1 with
    | Some x -> 
      begin match x with
      | PyInt i -> i
      | _ -> raise (TypeError "posix.mkfifo mode must be integer or omitted")
      end
    | None -> 0x1b6 (* octal 666 *)
    end
  in
    Unix.mkfifo path mode
    ;
    PyNone
  
let py_posix_mkdir
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atleast e 1;
  atmost e 2;
  let path, mode = 
    begin match arg e 0 with
    | PyString s -> s
    | _ -> raise (TypeError "posix.mkdir requires string path arg1")
    end
  ,
    begin match optional e 1 with
    | Some x -> 
      begin match x with
      | PyInt i -> i
      | _ -> raise (TypeError "posix.mkdir mode must be integer or omitted")
      end
    | None -> 0x1b6 (* octal 666 *)
    end
  in
    Unix.mkdir path mode
    ;
    PyNone
 
let py_posix_nice
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let niceness = match arg e 0 with
  | PyInt i -> i
  | _ -> raise (TypeError "posix.nice requires integer niceness")
  in 
    PyInt (Unix.nice niceness)

(* USE py_file_open *)
(*
let py_posix_open
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atleast e 2;
  atmost e 3;
*)

let py_posix_pipe
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  let i,o = Unix.pipe() in
  PyTuple [PyFile i; PyFile o]
  
let py_posix_plock
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  raise (NotImplemented "Posix.plock is not available in Viper")

(* Note: os.popen returns a Python file object, not a native one! *)
let py_posix_popen
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let cmd =
    match arg e 0 with
    | PyString s -> s
    | _ -> raise (TypeError "posix.popen requires string command")
  in 
    match arg e 1 with
    | PyString "r" -> PyFile (Unix.descr_of_in_channel (Unix.open_process_in cmd))
    | PyString "w" -> PyFile (Unix.descr_of_out_channel (Unix.open_process_out cmd))
    | PyString _ -> raise (ValueError "posix.popen requires mode 'r' or 'w'")
    | _ -> raise (TypeError "posix.popen requires string mode 'r' or 'w'")

let py_posix_putenv
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let varname, value =
    begin match arg e 0 with
    | PyString s -> s
    | _ -> raise (TypeError "posix.putenv requires string for varname")
    end
  ,
    begin match arg e 1 with
    | PyString s -> s
    | _ -> raise (TypeError "posix.putenv requires string for value")
    end
  in 
    Unix.putenv varname value
    ;
    PyNone
    
(* NotImplemented: correct strerror. We use a hack, just print the
   error number as a string. We can't use the ocaml Unix.strerror function,
   because it requires a variant of type Unix.error, and we have an integer.
   To do this right, we need a message table. There's no point at all
   using mapping the integer to a the ocaml Unix.error type so as to call
   Unix.strerror, we might as well map the integer directly to the string.
   [Note that the reverse mapping, from the ocaml variant to an integer
   is a pain, but is necessary for Python compatibility]
*)

let py_posix_strerror
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | PyInt i ->
    PyString ("POSIX or BSD ERROR " ^ (string_of_int i) ^ 
      " [Sorry, correct message not yet implemented]")
  | _ -> raise (TypeError "posix.strerror requires integer errno")
 
(* read: Use py_file_read *)

let py_posix_readlink
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | PyString s -> PyString (Unix.readlink s)
  | _ -> raise (TypeError "posix.readlink requires string pathname")

(* remove: use py_posix_unlink *)

let py_posix_rmdir
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | PyString s -> Unix.rmdir s; PyNone
  | _ -> raise (TypeError "posix.rmdir requires string pathname")

let py_posix_setgid
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | PyInt gid -> Unix.setgid gid; PyNone
  | _ -> raise (TypeError "posix.setgid requires integer gid")
  
let py_posix_setpgrp
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  raise (NotImplemented "setpgrp not supported");
  PyNone
  
let py_posix_setpgid
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  raise (NotImplemented "posix.setpgid");
  PyNone
  

let py_posix_setsid
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  PyInt (Unix.setsid ())

let py_posix_setuid
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | PyInt uid -> Unix.setuid uid; PyNone
  | _ -> raise (TypeError "posix.setuid requires integer uid")

let py_posix_stat
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let file = match arg e 0 with
  | PyString f -> f
  | _ -> raise (TypeError "posix.stat requires string filename argument")
  in File_stats.pystats (Unix.stat file)


let py_posix_symlink
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let src,dst = 
    begin match arg e 0 with
    | PyString s -> s
    | _ -> raise (TypeError "posix.link requires string src")
    end
  ,
    begin match arg e 1 with
    | PyString s -> s
    | _ -> raise (TypeError "posix.link requires string dst")
    end
  in 
    Unix.symlink src dst
    ;
    PyNone

let int_of_process_status x =
  match x with 
  | Unix.WEXITED i -> 256 * i
  | Unix.WSIGNALED i -> i
  | Unix.WSTOPPED i -> i

let py_posix_system
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  match arg e 0 with
  | PyString cmd -> PyInt (int_of_process_status (Unix.system cmd))
  | _ -> raise (TypeError "posix.system requires string argument")

let py_posix_tcgetpgrp
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  raise (NotImplemented "posix.tcgetpgrp")
  ;
  PyNone

let py_posix_tcsetpgrp
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  raise (NotImplemented "posix.tcsetpgrp")
  ;
  PyNone

let py_posix_times
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  let times = Unix.times ()
  and now = Unix.time () in
  Timing.mktimes times now
  
let py_posix_umask
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | PyInt i -> PyInt (Unix.umask i)
  | _ -> raise (TypeError "posix.umask requires integer argument")

let py_posix_uname
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  raise (NotImplemented "posix.uname")
  ;
  PyNone

let py_posix_unlink
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | PyString s -> Unix.unlink s; PyNone
  | _ -> raise (TypeError "posix.unlink/remove requires string pathname")


let py_posix_utime
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let path, (access, modif) = 
    begin match arg e 0 with
    | PyString f -> f
    | _ -> raise (TypeError "posix.utime requires string pathname for arg 1")
    end
  ,
    begin match arg e 1 with
    | PyTuple [PyFloat a; PyFloat m] -> a,m
    | _ -> raise (TypeError "posix.utime requires tuple of two floats (access, modification) for arg 2")
    end
  in
    Unix.utimes path access modif
    ;
    PyNone

let py_posix_wait
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  let pid, status = Unix.wait ()
  in PyTuple [PyInt pid; PyInt (int_of_process_status status)]


let py_posix_waitpid
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let pid, options =
    begin match arg e 0 with
    | PyInt pid -> pid
    | _ -> raise (TypeError "posix.wait_pid requires integer pid for arg 1")
    end
  ,
    begin match arg e 1 with
    | PyInt 0 -> []
    | PyInt 1 -> [Unix.WNOHANG]
    | PyInt 2 -> [Unix.WUNTRACED]
    | PyInt 3 -> [Unix.WNOHANG; Unix.WUNTRACED]
    | _ -> raise (TypeError "posix.wait_pid requires integer 0,1,2 or 3 for options (arg 2)")
    end
  in let pid,status = Unix.waitpid options pid
  in PyTuple [PyInt pid; PyInt (int_of_process_status status)]

(* write: use file_write  *)

(* ---------------------------------------------------- *)
(* This painful routine converts the nice ocaml unix error codes
   back to integers
*)
let get_posix_errcode ocaml_error = 
  match ocaml_error with
  | Unix.E2BIG -> 7
  | Unix.EACCES -> 13
  | Unix.EAGAIN -> 11
  | Unix.EBADF -> 9
  | Unix.EBUSY -> 16
  | Unix.ECHILD -> 10
  | Unix.EDEADLK -> 35
  | Unix.EDOM -> 33
  | Unix.EEXIST -> 17
  | Unix.EFAULT -> 14
  | Unix.EFBIG -> 27
  | Unix.EINTR -> 4
  | Unix.EINVAL -> 22
  | Unix.EIO -> 5
  | Unix.EISDIR -> 21
  | Unix.EMFILE -> 24
  | Unix.EMLINK -> 31
  | Unix.ENAMETOOLONG -> 36
  | Unix.ENFILE -> 23
  | Unix.ENODEV -> 19
  | Unix.ENOENT -> 2
  | Unix.ENOEXEC -> 8
  | Unix.ENOLCK -> 37
  | Unix.ENOMEM -> 12
  | Unix.ENOSPC -> 28 
  | Unix.ENOSYS -> 38
  | Unix.ENOTDIR -> 20
  | Unix.ENOTEMPTY -> 39
  | Unix.ENOTTY -> 25
  | Unix.ENXIO -> 6
  | Unix.EPERM -> 1
  | Unix.EPIPE -> 32
  | Unix.ERANGE -> 34
  | Unix.EROFS -> 30
  | Unix.ESPIPE -> 29
  | Unix.ESRCH -> 3
  | Unix.EXDEV -> 18
  | Unix.EWOULDBLOCK -> 11
  | Unix.EINPROGRESS -> 115
  | Unix.EALREADY -> 114
  | Unix.ENOTSOCK -> 88
  | Unix.EDESTADDRREQ -> 89
  | Unix.EMSGSIZE -> 90
  | Unix.EPROTOTYPE -> 91
  | Unix.ENOPROTOOPT -> 92
  | Unix.EPROTONOSUPPORT -> 93
  | Unix.ESOCKTNOSUPPORT -> 94
  | Unix.EOPNOTSUPP -> 95
  | Unix.EPFNOSUPPORT -> 96
  | Unix.EAFNOSUPPORT -> 97
  | Unix.EADDRINUSE -> 98
  | Unix.EADDRNOTAVAIL -> 99
  | Unix.ENETDOWN -> 100
  | Unix.ENETUNREACH -> 101
  | Unix.ENETRESET -> 102
  | Unix.ECONNABORTED -> 103
  | Unix.ECONNRESET -> 104
  | Unix.ENOBUFS -> 105
  | Unix.EISCONN -> 106
  | Unix.ENOTCONN -> 107
  | Unix.ESHUTDOWN -> 108
  | Unix.ETOOMANYREFS -> 109
  | Unix.ETIMEDOUT -> 110
  | Unix.ECONNREFUSED -> 111
  | Unix.EHOSTDOWN -> 112
  | Unix.EHOSTUNREACH -> 113
  | Unix.ELOOP -> 40
  | Unix.EUNKNOWNERR i -> i




