open Py_types
open Py_exceptions
open Py_builtins_util

(* regular expression flags used by Viper: these must agree with
   the values in pcre.py *)

let pcre_CASELESS       =   0x0001
and pcre_MULTILINE      =   0x0002
and pcre_DOTALL         =   0x0004
and pcre_EXTENDED       =   0x0008
and pcre_ANCHORED       =   0x0010
and pcre_DOLLAR_ENDONLY =   0x0020
and pcre_EXTRA          =   0x0040
and pcre_NOTBOL         =   0x0080
and pcre_NOTEOL         =   0x0100
and pcre_UNGREEDY       =   0x0200
and pcre_NOTEMPTY       =   0x0400

;;

(* compilation flags: convert bitmask into bitmask *) 
let ravel_cflags (flags:int) : Pcre.icflag =
  let flag_list = ref [] in
  if (flags land pcre_CASELESS) <> 0 then flag_list := Pcre.CASELESS :: !flag_list;
  if (flags land pcre_MULTILINE) <> 0 then flag_list := Pcre.MULTILINE :: !flag_list;
  if (flags land pcre_DOTALL) <> 0 then flag_list := Pcre.DOTALL :: !flag_list;
  if (flags land pcre_EXTENDED) <> 0 then flag_list := Pcre.EXTENDED :: !flag_list;
  if (flags land pcre_ANCHORED) <> 0 then flag_list := Pcre.C_ANCHORED :: !flag_list;
  if (flags land pcre_DOLLAR_ENDONLY) <> 0 then flag_list := Pcre.DOLLAR_ENDONLY :: !flag_list;
  if (flags land pcre_EXTRA) <> 0 then flag_list := Pcre.EXTRA  :: !flag_list;
  if (flags land pcre_UNGREEDY) <> 0 then flag_list := Pcre.UNGREEDY :: !flag_list;
  Pcre.cflags !flag_list
;;


let py_re_compile
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atleast e 1;
  atmost e 2;
  match arg e 0 with 
  | PyString s ->
    let flags = 
      begin match optional e 1 with
      | Some flags' -> 
        begin match flags' with
        | PyInt iflags -> iflags
        | _ -> raise (TypeError "re_compile requires flags to be an integer")
        end
      | None -> 0
      end 
    in begin 
      let iflags = ravel_cflags flags in
      try PyRegexp (Pcre.compile iflags None s)
      with Pcre.BadPattern (s',i) ->
        raise (ValueError ("Bad regexp: '" ^ s ^"'-->" ^ s' ^ " position " ^ (string_of_int i)))
    end
  | _ -> raise (TypeError "re_compile requires string (regexp) argument")
;;

(* runtime flags: convert bitmask into bitmask *) 
let ravel_rflags (flags:int) : Pcre.irflag =
  let flag_list = ref [] in
  if flags land pcre_ANCHORED <> 0 then flag_list := Pcre.R_ANCHORED :: !flag_list;
  if flags land pcre_NOTBOL <> 0 then flag_list := Pcre.NOTBOL :: !flag_list;
  if flags land pcre_NOTEOL <> 0 then flag_list := Pcre.NOTEOL :: !flag_list;
  if flags land pcre_NOTEMPTY <> 0 then flag_list := Pcre.NOTEMPTY :: !flag_list;
  Pcre.rflags !flag_list
;;


let py_re_match
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 5;
  let flags = match arg e 0 with 
    | PyInt i ->  ravel_rflags i 
    | _ -> raise (TypeError "re_match flags (arg 1) must be integer")
  and regexp = match arg e 1 with
    | PyRegexp r -> r
    | _ -> raise (TypeError "re_match regexp (arg 2) must be regexp")
  and startpos = match arg e 2 with
    | PyInt i -> i 
    | _ -> raise (TypeError "re_match startpos (arg 3) must be integer")
  and endpos = match arg e 3 with
    | PyInt i -> i 
    | _ -> raise (TypeError "re_match endpos (arg 4) must be integer")
  and data = match arg e 4 with
    | PyString s -> s
    | _ -> raise (TypeError "re_match data (arg 5) must be string")
  in
  try 
    let data' = String.sub data 0 endpos in
    let offsets = Pcre.pcre_exec flags regexp startpos data' in
    let pairs = ref [] in
    let index = ref (Array.length offsets) in
    while !index <> 0 do
      pairs := PyTuple [PyInt offsets.(!index-2); PyInt offsets.(!index-1)]  :: !pairs;
      decr index; decr index
    done;
    PyTuple !pairs 
  with Not_found -> PyNone
;;

