open Py_types
open Py_exceptions
open Py_mtypes

(* NOTE: although the constructor accepts the name, directory and
  filename for the module, it does not store this information
  in the dictionary, and the getattr method does not do
  any special loolup to find it either. Neither does the
  module environment. 

  The name, filename, and directory attributes must be manually
  stored in the dictionary.
*)
  
class py_module name' directory' filename' dictionary' i2n n2i env' = 
  object(self)
    val super : environment_t option = env'
    val dict : Py_dict.py_dictionary = dictionary'
    val name : string list  = name'
    val filename : string = filename'
    val file_directory : string = directory'
    val name_to_index: int VarMap.t = n2i
    val index_to_name: string array  = i2n
    val vars = Array.create (Array.length i2n) PyInitial

    (* initialise the variable array *)
    initializer
      dict#iter
      begin fun key value -> 
        match key with PyString name ->
        begin try 
          let i = Py_mtypes.VarMap.find name name_to_index in
          vars.(i) <- value
        with Not_found -> ()
        end
        | _ -> ()
      end

    method get_name = name
    method get_dotted_name = Py_module_file.path_dot_join name
    method get_filename = Some filename
    method get_file_directory = Some file_directory
    method get_dictionary = (dict :> dictionary_t)
    method get_environment = super 
    method get_vars = vars

    (* This routine is used for module.attribute lookup,
       and so does not look in the module's environment *)
    method get_attr k = 
      match k with 
      | PyString name ->
        if name = "__dict__" then Some (PyDictionary dict )
        else if VarMap.mem name name_to_index
        then Some vars.(VarMap.find name name_to_index)
        else dict#get_item k 
      | _ -> dict#get_item k 

    method set_attr k v =
      match k with 
      | PyString name ->
        if Py_mtypes.VarMap.mem name name_to_index
        then begin 
          vars.(Py_mtypes.VarMap.find name name_to_index) <- v; 
          true
        end
        else dict#set_item k v
      | _ -> dict#set_item k v

    (* Python compatible has_attr *)
    method has_attr k = 
      match self#get_attr k with 
      | Some _ -> true 
      | None -> false

    method del_attr k = 
      match k with
      | PyString name ->
        if Py_mtypes.VarMap.mem name name_to_index
        then begin 
          vars.(Py_mtypes.VarMap.find name name_to_index) <- PyInitial;
          true
        end
        else dict#del_item k
      | _ -> dict#del_item k
       
    (* tests *)
    method can_del_attr k = dict#has_item k
    method can_get_attr k = 
      match self#get_attr k with Some _ -> true | None -> false
    method can_set_attr (k:expr_t) (v:expr_t) = true 
    method can_set_any_attr (k:expr_t) = true 
    method get_index_to_global_name = index_to_name
    method get_global_name_to_index = name_to_index

    method get_indexed level i = 
      if level = 0 
      then vars.(i)
      else match super with
      | Some parent -> parent #get_indexed (level-1) i
      | None -> raise (Failure "Indexed addressing failed")

    method set_indexed level i (v:expr_t) = 
      if level = 0
      then vars.(i) <- v
      else match super with
      | Some parent -> parent#set_indexed (level-1) i v
      | None -> raise (Failure "Indexed addressing failed")
      
    method del_indexed level i =
      if level = 0
      then vars.(i) <- PyInitial
      else match super with 
      | Some parent -> parent#del_indexed level i
      | None -> raise (Failure "Indexed adressing failed")

  end

(* this environment is used for _defining_ modules,
   and also for lookup from functions, when local
   lookup fails *)

(* Note: there are no special attributes here, not even __dict__ *)
class py_module_environment (modul':module_t) = 
  object (self)
    val modul = modul'
    val dict = modul'#get_dictionary
    val super = modul'#get_environment
    val vars = modul'#get_vars

    val name_to_index = modul'#get_global_name_to_index

    (* get underlying module *)
    method get_module = modul

    (* get an attribute *)
    method get_attr k =
      match k with 
      | PyString name -> 
        if Py_mtypes.VarMap.mem name name_to_index
        then Some vars.(Py_mtypes.VarMap.find name name_to_index)
        else if dict#has_item k
        then  dict#get_item k
        else begin match super with
        | Some parent -> parent#get_attr k
        | None -> None
        end
      | _ -> 
        if dict#has_item k
        then  dict#get_item k
        else begin match super with
        | Some parent -> parent#get_attr k
        | None -> None
        end

    (* set an attribute *)
    (* NOTE: we don't bother to keep the dictionary synched with
       indexed access. This means 'locals' will fail to show the
       correct dictionary, as in Python. The reason we don't,
       is that changes to the dictionary will not be reflected
       in the fast load array anyhow. This could be fixed,
       by storing to the dictionary as well as the array,
       and also by returning a n object attached to the
       array, instead of a dictionary (but with the same interface
     *)
    method set_attr k v = 
      match k with 
      | PyString name ->
        if Py_mtypes.VarMap.mem name name_to_index
        then begin 
          vars.(Py_mtypes.VarMap.find name name_to_index) <- v; 
          true
        end
        else dict#set_item k v
      | _ -> dict#set_item k v

    method del_attr k = dict # del_item k

    (* Python compatible has_attr *)
    method has_attr k = 
      match self#get_attr k with 
      | Some _ -> true 
      | None -> false

    (* test which methods will work *)
    method can_get_attr k = self#has_attr k
    method can_set_attr (k:expr_t) (v:expr_t) = true
    method can_set_any_attr (k:expr_t) = true
    method can_del_attr (k:expr_t) = true (* a lie for the moment *)
  
   (* Do indexed addressing *)
    method get_indexed level i = 
      if level = 0 
      then vars.(i)
      else match super with
      | Some parent -> parent #get_indexed (level-1) i
      | None -> raise (Failure "Indexed adressing failed")

    method set_indexed level i (v:expr_t) = 
      if level = 0
      then vars.(i) <- v
      else match super with
      | Some parent -> parent#set_indexed (level-1) i v
      | None -> raise (Failure "Indexed adressing failed")

    method del_indexed level i =
      if level = 0
      then vars.(i) <- PyInitial
      else match super with
      | Some parent -> parent#del_indexed level i
      | None -> raise (Failure "Indexed adressing failed")

    (* retrieve locals and globals dictionaries *)
    method get_locals = dict
    method get_globals = dict
    method keys = dict # keys
  end

