# 1 "py_lex.mll"
 
open Py_parse
open Py_string

(* 'zero_is_octal' determines the treatment of
   zero prefixed digits strings.

   If true (default), Python and C rules apply,
   and a zero prefixed digit string is treated
   as octal. Viper standard is decimal.
   A warning is issued in either case.
*)
   
type warning_mode = WarnAlways | WarnOnce | WarnNever;;
let octal_warning_mode = ref WarnOnce;;
let octal_warning_count = ref 0;;
let zero_is_octal = ref true;;
let set_zero_is_octal v = zero_is_octal := v;;
let set_octal_warning_mode x =
  octal_warning_mode :=
    begin 
      if x > 0 then WarnAlways
      else if x = 0 then WarnOnce
      else WarnNever
    end
;;

let do_octal_warning () = 
  !octal_warning_mode = WarnAlways
  or (!octal_warning_mode = WarnOnce && !octal_warning_count = 0)
;;

let comment_nesting_level = ref 0;;
let comment_text = ref "";;

let lexeme = Lexing.lexeme;;

let buf_pos = ref 0;;
let last_buf_pos = ref 0;;
let line_no = ref 0;;
let filename = ref "Unknown filename";;

let incr_lex_counters lexbuf =
  line_no := !line_no + 1;
  last_buf_pos := !buf_pos;
  buf_pos := Lexing.lexeme_end lexbuf
;;

let reset filename' =
  buf_pos :=0;
  last_buf_pos :=0;
  line_no := 0;
  filename := filename';
  comment_nesting_level := 0
;;

let set_buf_pos x = buf_pos := x;;
let get_buf_pos () = !buf_pos;;

let substr = String.sub
let len = String.length

(* special hackery to adjust the line count when a newline is
present _inside_ a string token -- allowed in some python strings *)
let lfcount s = 
  let n = ref 0 in
  for i = 0 to (String.length s) - 1 do
    if s.[i] = '\n' then incr n
  done;
  !n
;;

let adj s = line_no := !line_no + (lfcount s);;

let decode_qstring s = adj s; let n = len s in unescape (substr s 0 (n-1)) 
let decode_dqstring s = adj s; let n = len s in unescape (substr s 0 (n-1)) 
let decode_qqqstring s = adj s; let n = len s in unescape (substr s 0 (n-3)) 
let decode_dddstring s = adj s; let n = len s in unescape (substr s 0 (n-3)) 

let decode_raw_qstring s = adj s; let n = len s in substr s 0 (n-1) 
let decode_raw_dqstring s = adj s; let n = len s in substr s 0 (n-1) 
let decode_raw_qqqstring s = adj s; let n = len s in substr s 0 (n-3) 
let decode_raw_dddstring s = adj s; let n = len s in substr s 0 (n-3) 

let lex_tables = {
  Lexing.lex_base = 
   "\000\000\040\000\080\000\134\000\173\000\217\000\089\001\135\001\
    \022\000\240\000\071\002\179\255\040\000\181\255\023\000\012\000\
    \002\000\255\255\014\000\015\000\038\000\237\255\236\255\070\000\
    \072\000\232\255\068\000\218\000\082\000\126\001\199\001\017\000\
    \231\255\219\000\018\000\028\000\150\002\007\003\235\255\207\255\
    \234\255\053\000\219\255\218\255\055\000\217\255\056\000\180\255\
    \196\255\198\255\199\255\003\000\079\000\082\003\081\000\185\255\
    \004\000\183\255\211\255\060\000\191\255\214\255\194\255\061\000\
    \212\255\213\255\192\255\195\255\228\001\159\003\006\001\238\255\
    \244\255\016\001\004\002\169\003\239\255\014\002\228\003\042\002\
    \225\002\238\003\235\002\196\003\041\004\121\000\009\004\242\255\
    \240\000\072\004\102\004\127\004\188\004\246\255\211\004\136\001\
    \238\001\250\255\255\002\243\004\156\004\248\255\016\005\162\001\
    \077\000\252\255\157\000\254\255\201\255\035\005\062\005\205\255\
    \203\255\193\255\206\255\204\255\208\255\202\255\084\000\189\255\
    \197\255\200\255\094\000\187\255\181\255\245\000\255\255\134\000\
    \099\000\124\000\253\255\255\255\198\000\255\255\072\001\210\001\
    \231\002\055\003\130\003\131\005\161\000\166\005\201\005\172\000\
    \246\005\199\000\203\000\025\006\060\006\204\000\191\006\234\006\
    \205\000\026\007\239\000\245\000\067\007\108\007\001\000\144\007\
    \002\000\201\007\003\000\073\008\004\000\027\008\252\000\155\008\
    \007\000\203\008\010\001\250\008\027\000\018\001\030\009\029\000\
    \020\001\041\000\077\009\032\001\160\009\043\000\205\009\035\001\
    \253\009\046\000\037\001\042\010\048\000\040\001\052\000\091\010\
    \074\000\209\010\075\000\254\010\076\000\046\011\077\000";
  Lexing.lex_backtrk = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\073\000\255\255\076\000\076\000\
    \067\000\255\255\035\000\030\000\065\000\255\255\255\255\027\000\
    \025\000\255\255\026\000\034\000\028\000\014\000\012\000\022\000\
    \255\255\031\000\033\000\032\000\002\000\002\000\255\255\255\255\
    \255\255\040\000\255\255\255\255\029\000\255\255\039\000\255\255\
    \255\255\255\255\255\255\071\000\069\000\002\000\255\255\255\255\
    \255\255\255\255\255\255\046\000\255\255\255\255\255\255\045\000\
    \255\255\255\255\255\255\255\255\015\000\012\000\255\255\255\255\
    \255\255\255\255\255\255\015\000\255\255\255\255\015\000\255\255\
    \255\255\015\000\255\255\255\255\014\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\010\000\255\255\255\255\255\255\006\000\
    \255\255\255\255\255\255\008\000\255\255\255\255\255\255\004\000\
    \255\255\255\255\255\255\255\255\255\255\015\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\073\000\255\255\003\000\
    \003\000\003\000\255\255\255\255\001\000\255\255\000\000\001\000\
    \255\255\000\000\255\255\001\000\001\000\001\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\001\000\001\000\001\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\001\000\001\000\255\255\
    \255\255\001\000\001\000\255\255\255\255\001\000\001\000\001\000\
    \001\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\001\000\001\000\001\000\001\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\001\000\
    \001\000\255\255\255\255\001\000\001\000\255\255\255\255";
  Lexing.lex_default = 
   "\195\000\191\000\178\000\165\000\161\000\157\000\148\000\139\000\
    \132\000\105\000\011\000\000\000\255\255\000\000\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\000\000\000\000\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\000\000\000\000\
    \000\000\255\255\000\000\000\000\255\255\000\000\255\255\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\000\000\
    \255\255\000\000\000\000\255\255\000\000\000\000\000\000\255\255\
    \000\000\000\000\000\000\000\000\255\255\255\255\255\255\000\000\
    \000\000\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\000\000\255\255\000\000\000\000\255\255\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\000\000\
    \000\000\000\000\255\255\000\000\000\000\255\255\000\000\255\255\
    \255\255\255\255\000\000\000\000\136\000\000\000\136\000\136\000\
    \136\000\136\000\136\000\142\000\255\255\142\000\142\000\255\255\
    \142\000\255\255\255\255\142\000\151\000\255\255\151\000\151\000\
    \255\255\151\000\255\255\255\255\151\000\159\000\159\000\159\000\
    \159\000\163\000\163\000\163\000\163\000\169\000\255\255\169\000\
    \169\000\169\000\255\255\169\000\169\000\255\255\169\000\169\000\
    \255\255\169\000\182\000\255\255\182\000\182\000\182\000\255\255\
    \182\000\182\000\255\255\182\000\182\000\255\255\182\000\193\000\
    \193\000\193\000\193\000\197\000\197\000\197\000\197\000";
  Lexing.lex_trans = 
   "\107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\000\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \133\000\124\000\134\000\135\000\122\000\056\000\057\000\133\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\125\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \125\000\065\000\133\000\121\000\120\000\118\000\067\000\061\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\058\000\059\000\107\000\196\000\179\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \116\000\111\000\050\000\114\000\049\000\048\000\054\000\180\000\
    \055\000\060\000\066\000\119\000\107\000\103\000\103\000\107\000\
    \123\000\112\000\113\000\117\000\192\000\115\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\108\000\
    \131\000\107\000\130\000\166\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \167\000\103\000\103\000\142\000\181\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\142\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\103\000\103\000\107\000\
    \131\000\142\000\137\000\138\000\133\000\142\000\151\000\151\000\
    \104\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\168\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\151\000\126\000\133\000\126\000\127\000\125\000\151\000\
    \255\255\255\255\255\255\255\255\255\255\107\000\169\000\255\255\
    \062\000\162\000\109\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\109\000\109\000\109\000\169\000\125\000\255\255\063\000\
    \064\000\065\000\128\000\255\255\169\000\255\255\169\000\129\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \255\255\255\255\182\000\255\255\107\000\182\000\255\255\182\000\
    \255\255\074\000\182\000\074\000\255\255\158\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\255\255\255\255\255\255\255\255\000\000\096\000\
    \255\255\000\000\131\000\000\000\137\000\138\000\000\000\000\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\000\000\107\000\000\000\149\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \150\000\000\000\000\000\000\000\000\000\000\000\255\255\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \000\000\000\000\107\000\000\000\140\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\000\000\
    \000\000\141\000\000\000\000\000\068\000\255\255\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \085\000\000\000\086\000\070\000\000\000\000\000\255\255\000\000\
    \071\000\000\000\087\000\000\000\000\000\088\000\000\000\000\000\
    \000\000\000\000\103\000\103\000\097\000\000\000\089\000\000\000\
    \107\000\255\255\000\000\000\000\131\000\090\000\137\000\138\000\
    \085\000\000\000\086\000\000\000\000\000\000\000\000\000\098\000\
    \071\000\000\000\087\000\000\000\000\000\088\000\105\000\000\000\
    \255\255\000\000\000\000\000\000\097\000\068\000\089\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\106\000\000\000\000\000\000\000\000\000\107\000\000\000\
    \000\000\000\000\000\000\000\000\070\000\000\000\105\000\000\000\
    \000\000\071\000\000\000\072\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\073\000\000\000\
    \000\000\079\000\000\000\000\000\000\000\000\000\076\000\000\000\
    \000\000\071\000\000\000\072\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \255\255\000\000\000\000\000\000\000\000\000\000\076\000\000\000\
    \012\000\013\000\000\000\013\000\014\000\080\000\000\000\080\000\
    \000\000\255\255\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\000\000\000\000\000\000\012\000\
    \015\000\016\000\017\000\000\000\018\000\019\000\020\000\021\000\
    \022\000\023\000\024\000\025\000\026\000\027\000\028\000\029\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\031\000\032\000\033\000\034\000\035\000\000\000\255\255\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\037\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\038\000\039\000\040\000\041\000\036\000\042\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\037\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\043\000\044\000\045\000\046\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \000\000\000\000\255\255\000\000\000\000\000\000\000\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\131\000\000\000\137\000\138\000\053\000\000\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\000\000\000\000\000\000\
    \000\000\051\000\000\000\000\000\000\000\000\000\052\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\131\000\000\000\137\000\138\000\000\000\000\000\047\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\000\000\000\000\000\000\000\000\053\000\000\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\131\000\000\000\137\000\138\000\
    \000\000\000\000\000\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\000\000\000\000\000\000\
    \000\000\053\000\000\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\068\000\000\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\000\000\070\000\000\000\000\000\255\255\
    \000\000\071\000\000\000\072\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\076\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\073\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \077\000\071\000\000\000\072\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\076\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \000\000\079\000\000\000\000\000\000\000\000\000\076\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \076\000\099\000\099\000\099\000\099\000\099\000\099\000\099\000\
    \099\000\099\000\099\000\083\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\082\000\076\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\068\000\
    \076\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\000\000\000\000\000\000\000\000\000\000\
    \100\000\000\000\000\000\000\000\000\000\000\000\070\000\000\000\
    \000\000\000\000\000\000\071\000\000\000\087\000\000\000\000\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \090\000\091\000\091\000\091\000\091\000\091\000\091\000\000\000\
    \000\000\000\000\000\000\071\000\000\000\087\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\092\000\
    \000\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\000\000\000\000\
    \000\000\000\000\000\000\093\000\099\000\099\000\099\000\099\000\
    \099\000\099\000\099\000\099\000\099\000\099\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\094\000\000\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\000\000\000\000\
    \000\000\000\000\000\000\093\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\000\000\000\000\000\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\099\000\099\000\099\000\099\000\099\000\
    \099\000\099\000\099\000\099\000\099\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\000\000\000\000\000\000\000\000\000\000\101\000\
    \099\000\099\000\099\000\099\000\099\000\099\000\099\000\099\000\
    \099\000\099\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\102\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\109\000\109\000\109\000\109\000\000\000\000\000\101\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \079\000\000\000\000\000\000\000\000\000\076\000\109\000\109\000\
    \109\000\109\000\109\000\109\000\109\000\109\000\109\000\109\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\110\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\076\000\255\255\000\000\
    \146\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\147\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \000\000\255\255\000\000\143\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\000\000\
    \144\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\255\255\000\000\146\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\147\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \000\000\255\255\255\255\145\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\000\000\
    \131\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\255\255\255\255\143\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\144\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\155\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\156\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\152\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\153\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\155\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\156\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\000\000\154\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\255\255\000\000\255\255\000\000\
    \000\000\131\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\000\000\
    \152\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\153\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\131\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\131\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \000\000\000\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \160\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\160\000\000\000\000\000\000\000\
    \131\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\164\000\255\255\000\000\
    \170\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\171\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\000\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\255\255\000\000\000\000\000\000\
    \131\000\000\000\000\000\000\000\000\000\000\000\000\000\172\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\164\000\255\255\000\000\
    \173\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\174\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\255\255\000\000\
    \170\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\171\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\175\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\255\255\000\000\173\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\174\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\172\000\
    \000\000\255\255\000\000\176\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\000\000\
    \131\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\000\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\175\000\000\000\
    \255\255\000\000\183\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\184\000\000\000\000\000\000\000\
    \000\000\255\255\177\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\255\255\000\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\185\000\000\000\255\255\000\000\186\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\187\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\000\000\
    \255\255\000\000\183\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\184\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\000\000\188\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\000\000\
    \255\255\000\000\186\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \000\000\000\000\000\000\000\000\187\000\000\000\000\000\000\000\
    \000\000\185\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\255\255\000\000\189\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\255\255\255\255\000\000\000\000\
    \000\000\131\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\188\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\131\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\190\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\194\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\000\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\131\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\131\000\000\000\000\000\
    \000\000\000\000\255\255\000\000\000\000\194\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\000\000\
    \255\255\000\000\000\000\000\000\000\000\131\000\000\000\000\000\
    \000\000\000\000\198\000\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\198\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255";
  Lexing.lex_check = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \008\000\014\000\008\000\008\000\016\000\051\000\056\000\000\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\012\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \012\000\015\000\001\000\018\000\019\000\020\000\031\000\034\000\
    \002\000\002\000\002\000\002\000\002\000\002\000\002\000\002\000\
    \002\000\035\000\035\000\002\000\000\000\002\000\002\000\002\000\
    \002\000\002\000\002\000\002\000\002\000\002\000\002\000\002\000\
    \002\000\002\000\002\000\002\000\002\000\002\000\002\000\002\000\
    \023\000\026\000\041\000\024\000\044\000\046\000\052\000\002\000\
    \054\000\059\000\063\000\118\000\028\000\104\000\104\000\000\000\
    \122\000\026\000\026\000\023\000\001\000\024\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\028\000\
    \127\000\003\000\128\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\129\000\001\000\
    \003\000\085\000\085\000\140\000\002\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\143\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\106\000\106\000\002\000\
    \132\000\145\000\132\000\132\000\004\000\146\000\149\000\152\000\
    \085\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\003\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\154\000\009\000\005\000\009\000\009\000\125\000\155\000\
    \000\000\158\000\160\000\162\000\164\000\003\000\166\000\168\000\
    \033\000\004\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\170\000\125\000\008\000\033\000\
    \033\000\033\000\009\000\172\000\173\000\175\000\176\000\009\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \001\000\177\000\179\000\181\000\004\000\183\000\185\000\186\000\
    \188\000\070\000\189\000\070\000\190\000\005\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\192\000\194\000\196\000\198\000\255\255\088\000\
    \002\000\255\255\134\000\255\255\134\000\134\000\255\255\255\255\
    \005\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\255\255\006\000\255\255\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \006\000\255\255\255\255\255\255\255\255\255\255\003\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \255\255\255\255\007\000\255\255\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\255\255\
    \255\255\007\000\255\255\255\255\029\000\004\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \029\000\255\255\029\000\029\000\255\255\255\255\132\000\255\255\
    \029\000\255\255\029\000\255\255\255\255\029\000\255\255\255\255\
    \255\255\255\255\103\000\103\000\095\000\255\255\029\000\255\255\
    \006\000\005\000\255\255\255\255\135\000\029\000\135\000\135\000\
    \029\000\255\255\029\000\255\255\255\255\255\255\255\255\095\000\
    \029\000\255\255\029\000\255\255\255\255\029\000\103\000\255\255\
    \009\000\255\255\255\255\255\255\095\000\030\000\029\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\103\000\255\255\255\255\255\255\255\255\007\000\255\255\
    \255\255\255\255\255\255\255\255\030\000\255\255\103\000\255\255\
    \255\255\030\000\255\255\030\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\096\000\096\000\
    \096\000\096\000\096\000\096\000\096\000\096\000\030\000\255\255\
    \255\255\068\000\255\255\255\255\255\255\255\255\068\000\255\255\
    \255\255\030\000\255\255\030\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \134\000\255\255\255\255\255\255\255\255\255\255\068\000\255\255\
    \010\000\010\000\255\255\010\000\010\000\079\000\255\255\079\000\
    \255\255\006\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\255\255\255\255\255\255\010\000\
    \010\000\010\000\010\000\255\255\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\255\255\007\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \255\255\255\255\135\000\255\255\255\255\255\255\255\255\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\136\000\255\255\136\000\136\000\036\000\255\255\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\255\255\255\255\255\255\
    \255\255\037\000\255\255\255\255\255\255\255\255\037\000\098\000\
    \098\000\098\000\098\000\098\000\098\000\098\000\098\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\137\000\255\255\137\000\137\000\255\255\255\255\010\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\255\255\255\255\255\255\255\255\037\000\255\255\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\138\000\255\255\138\000\138\000\
    \255\255\255\255\255\255\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\255\255\255\255\255\255\
    \255\255\053\000\255\255\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\069\000\255\255\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\255\255\069\000\255\255\255\255\136\000\
    \255\255\069\000\255\255\069\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\075\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\069\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \075\000\069\000\255\255\069\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\075\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \255\255\078\000\255\255\255\255\255\255\255\255\078\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\137\000\
    \081\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\078\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\081\000\078\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\084\000\
    \081\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\255\255\255\255\255\255\255\255\255\255\
    \086\000\255\255\255\255\255\255\255\255\255\255\084\000\255\255\
    \255\255\255\255\255\255\084\000\255\255\084\000\255\255\255\255\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\138\000\255\255\255\255\255\255\255\255\255\255\
    \084\000\089\000\089\000\089\000\089\000\089\000\089\000\255\255\
    \255\255\255\255\255\255\084\000\255\255\084\000\090\000\090\000\
    \090\000\090\000\090\000\090\000\090\000\090\000\090\000\090\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\089\000\
    \255\255\089\000\089\000\089\000\089\000\089\000\089\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \091\000\091\000\091\000\091\000\091\000\091\000\255\255\255\255\
    \255\255\255\255\255\255\091\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\091\000\255\255\
    \091\000\091\000\091\000\091\000\091\000\091\000\255\255\255\255\
    \255\255\255\255\255\255\091\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\092\000\092\000\092\000\
    \092\000\092\000\092\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\094\000\094\000\094\000\094\000\
    \094\000\094\000\255\255\255\255\255\255\092\000\092\000\092\000\
    \092\000\092\000\092\000\099\000\099\000\099\000\099\000\099\000\
    \099\000\099\000\099\000\099\000\099\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\094\000\094\000\094\000\094\000\
    \094\000\094\000\255\255\255\255\255\255\255\255\255\255\099\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\102\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\099\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\109\000\109\000\109\000\109\000\255\255\255\255\099\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \109\000\255\255\255\255\255\255\255\255\109\000\110\000\110\000\
    \110\000\110\000\110\000\110\000\110\000\110\000\110\000\110\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\109\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\255\255\109\000\139\000\255\255\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\255\255\255\255\139\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\255\255\
    \255\255\141\000\255\255\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\255\255\255\255\
    \141\000\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\255\255\255\255\142\000\255\255\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\255\255\255\255\142\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\144\000\144\000\
    \144\000\144\000\144\000\144\000\144\000\144\000\144\000\255\255\
    \255\255\144\000\139\000\144\000\144\000\144\000\144\000\144\000\
    \144\000\144\000\144\000\144\000\144\000\144\000\144\000\144\000\
    \144\000\144\000\144\000\144\000\144\000\144\000\255\255\255\255\
    \144\000\147\000\147\000\147\000\147\000\147\000\147\000\147\000\
    \147\000\147\000\255\255\255\255\147\000\141\000\147\000\147\000\
    \147\000\147\000\147\000\147\000\147\000\147\000\147\000\147\000\
    \147\000\147\000\147\000\147\000\147\000\147\000\147\000\147\000\
    \147\000\255\255\255\255\147\000\148\000\148\000\148\000\148\000\
    \148\000\148\000\148\000\148\000\148\000\148\000\255\255\148\000\
    \142\000\148\000\148\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\148\000\148\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\148\000\148\000\148\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\148\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\144\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\139\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \147\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\141\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\148\000\255\255\255\255\255\255\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\142\000\150\000\255\255\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\150\000\255\255\
    \255\255\255\255\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\255\255\151\000\144\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\151\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\147\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\255\255\153\000\255\255\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\255\255\255\255\148\000\255\255\150\000\255\255\
    \255\255\153\000\255\255\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\255\255\156\000\255\255\
    \156\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\151\000\156\000\255\255\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\255\255\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\255\255\255\255\157\000\255\255\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\153\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \255\255\255\255\159\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\150\000\
    \255\255\255\255\156\000\255\255\255\255\255\255\255\255\255\255\
    \157\000\161\000\161\000\161\000\161\000\161\000\161\000\161\000\
    \161\000\161\000\255\255\161\000\161\000\161\000\161\000\161\000\
    \161\000\161\000\161\000\161\000\161\000\161\000\161\000\161\000\
    \161\000\161\000\161\000\161\000\161\000\161\000\161\000\161\000\
    \161\000\255\255\151\000\157\000\159\000\255\255\255\255\255\255\
    \161\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\159\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\153\000\165\000\165\000\165\000\165\000\165\000\
    \165\000\165\000\165\000\165\000\255\255\161\000\165\000\255\255\
    \165\000\165\000\165\000\165\000\165\000\165\000\165\000\165\000\
    \165\000\165\000\165\000\165\000\165\000\165\000\165\000\165\000\
    \165\000\165\000\165\000\255\255\255\255\165\000\255\255\255\255\
    \255\255\255\255\255\255\156\000\255\255\255\255\255\255\255\255\
    \161\000\163\000\163\000\163\000\163\000\163\000\163\000\163\000\
    \163\000\163\000\255\255\163\000\163\000\163\000\163\000\163\000\
    \163\000\163\000\163\000\163\000\163\000\163\000\163\000\163\000\
    \163\000\163\000\163\000\163\000\163\000\163\000\163\000\163\000\
    \163\000\255\255\255\255\255\255\157\000\255\255\255\255\255\255\
    \163\000\255\255\255\255\255\255\255\255\255\255\255\255\165\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \159\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\165\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\255\255\163\000\167\000\255\255\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\255\255\255\255\167\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \163\000\161\000\255\255\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\255\255\255\255\169\000\255\255\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\169\000\169\000\169\000\169\000\169\000\
    \169\000\169\000\169\000\255\255\255\255\169\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\167\000\
    \255\255\255\255\171\000\171\000\171\000\171\000\171\000\171\000\
    \171\000\171\000\171\000\255\255\255\255\171\000\255\255\171\000\
    \171\000\171\000\171\000\171\000\171\000\171\000\171\000\171\000\
    \171\000\171\000\171\000\171\000\171\000\171\000\171\000\171\000\
    \171\000\171\000\167\000\165\000\171\000\255\255\174\000\174\000\
    \174\000\174\000\174\000\174\000\174\000\174\000\174\000\169\000\
    \255\255\174\000\255\255\174\000\174\000\174\000\174\000\174\000\
    \174\000\174\000\174\000\174\000\174\000\174\000\174\000\174\000\
    \174\000\174\000\174\000\174\000\174\000\174\000\255\255\255\255\
    \174\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\163\000\169\000\255\255\255\255\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\171\000\255\255\
    \178\000\255\255\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\178\000\255\255\255\255\255\255\
    \255\255\171\000\174\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\167\000\255\255\174\000\255\255\255\255\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\178\000\255\255\180\000\255\255\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\180\000\
    \255\255\255\255\255\255\169\000\178\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\255\255\255\255\
    \182\000\255\255\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\182\000\255\255\255\255\255\255\
    \255\255\255\255\171\000\255\255\180\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\255\255\255\255\
    \184\000\255\255\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\255\255\174\000\180\000\
    \255\255\255\255\255\255\255\255\184\000\255\255\255\255\255\255\
    \255\255\182\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\255\255\255\255\187\000\255\255\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\255\255\255\255\182\000\178\000\255\255\255\255\
    \255\255\187\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\184\000\255\255\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\255\255\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\255\255\184\000\191\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\187\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \180\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\187\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\191\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\182\000\255\255\255\255\
    \255\255\193\000\193\000\193\000\193\000\193\000\193\000\193\000\
    \193\000\193\000\191\000\193\000\193\000\193\000\193\000\193\000\
    \193\000\193\000\193\000\193\000\193\000\193\000\193\000\193\000\
    \193\000\193\000\193\000\193\000\193\000\193\000\193\000\193\000\
    \193\000\255\255\255\255\193\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\184\000\195\000\195\000\
    \195\000\195\000\195\000\195\000\195\000\195\000\195\000\255\255\
    \195\000\195\000\195\000\195\000\195\000\195\000\195\000\195\000\
    \195\000\195\000\195\000\195\000\195\000\195\000\195\000\195\000\
    \195\000\195\000\195\000\195\000\195\000\195\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\195\000\255\255\255\255\
    \255\255\255\255\187\000\255\255\255\255\193\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\255\255\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\255\255\255\255\
    \193\000\255\255\255\255\255\255\255\255\197\000\255\255\255\255\
    \255\255\255\255\195\000\191\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\195\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\197\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\197\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\193\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\195\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\197\000"
}

let rec parse_qstring lexbuf = parse_qstring_rec lexbuf 0
and parse_qstring_rec lexbuf state =
  match Lexing.engine lex_tables state lexbuf with
    0 -> (
# 176 "py_lex.mll"
            STRING (decode_qstring (lexeme lexbuf)) )
  | 1 -> (
# 177 "py_lex.mll"
      ERRORTOKEN ("' string"))
  | n -> lexbuf.Lexing.refill_buff lexbuf; parse_qstring_rec lexbuf n

and parse_dqstring lexbuf = parse_dqstring_rec lexbuf 1
and parse_dqstring_rec lexbuf state =
  match Lexing.engine lex_tables state lexbuf with
    0 -> (
# 180 "py_lex.mll"
             STRING (decode_dqstring (lexeme lexbuf)) )
  | 1 -> (
# 181 "py_lex.mll"
      ERRORTOKEN ("\" string"))
  | n -> lexbuf.Lexing.refill_buff lexbuf; parse_dqstring_rec lexbuf n

and parse_qqqstring lexbuf = parse_qqqstring_rec lexbuf 2
and parse_qqqstring_rec lexbuf state =
  match Lexing.engine lex_tables state lexbuf with
    0 -> (
# 184 "py_lex.mll"
              STRING (decode_qqqstring (lexeme lexbuf)) )
  | 1 -> (
# 185 "py_lex.mll"
      ERRORTOKEN ("''' string"))
  | n -> lexbuf.Lexing.refill_buff lexbuf; parse_qqqstring_rec lexbuf n

and parse_dddstring lexbuf = parse_dddstring_rec lexbuf 3
and parse_dddstring_rec lexbuf state =
  match Lexing.engine lex_tables state lexbuf with
    0 -> (
# 188 "py_lex.mll"
              STRING (decode_dddstring (lexeme lexbuf)) )
  | 1 -> (
# 189 "py_lex.mll"
      ERRORTOKEN ("\"\"\" string"))
  | n -> lexbuf.Lexing.refill_buff lexbuf; parse_dddstring_rec lexbuf n

and parse_raw_qstring lexbuf = parse_raw_qstring_rec lexbuf 4
and parse_raw_qstring_rec lexbuf state =
  match Lexing.engine lex_tables state lexbuf with
    0 -> (
# 192 "py_lex.mll"
                STRING (decode_raw_qstring (lexeme lexbuf)) )
  | 1 -> (
# 193 "py_lex.mll"
      ERRORTOKEN ("raw ' string"))
  | n -> lexbuf.Lexing.refill_buff lexbuf; parse_raw_qstring_rec lexbuf n

and parse_raw_dqstring lexbuf = parse_raw_dqstring_rec lexbuf 5
and parse_raw_dqstring_rec lexbuf state =
  match Lexing.engine lex_tables state lexbuf with
    0 -> (
# 196 "py_lex.mll"
                 STRING (decode_raw_dqstring (lexeme lexbuf)) )
  | 1 -> (
# 197 "py_lex.mll"
      ERRORTOKEN ("raw \" string"))
  | n -> lexbuf.Lexing.refill_buff lexbuf; parse_raw_dqstring_rec lexbuf n

and parse_raw_qqqstring lexbuf = parse_raw_qqqstring_rec lexbuf 6
and parse_raw_qqqstring_rec lexbuf state =
  match Lexing.engine lex_tables state lexbuf with
    0 -> (
# 200 "py_lex.mll"
                  STRING (decode_raw_qqqstring (lexeme lexbuf)) )
  | 1 -> (
# 201 "py_lex.mll"
      ERRORTOKEN ("raw ''' string"))
  | n -> lexbuf.Lexing.refill_buff lexbuf; parse_raw_qqqstring_rec lexbuf n

and parse_raw_dddstring lexbuf = parse_raw_dddstring_rec lexbuf 7
and parse_raw_dddstring_rec lexbuf state =
  match Lexing.engine lex_tables state lexbuf with
    0 -> (
# 204 "py_lex.mll"
                  STRING (decode_raw_dddstring (lexeme lexbuf)) )
  | 1 -> (
# 205 "py_lex.mll"
      ERRORTOKEN (lexeme lexbuf))
  | n -> lexbuf.Lexing.refill_buff lexbuf; parse_raw_dddstring_rec lexbuf n

and parse_hash_comment lexbuf = parse_hash_comment_rec lexbuf 8
and parse_hash_comment_rec lexbuf state =
  match Lexing.engine lex_tables state lexbuf with
    0 -> (
# 208 "py_lex.mll"
                     
    incr_lex_counters lexbuf;
    let lex = lexeme lexbuf in
    let n = String.length lex in
    COMMENT_NEWLINE  ((!line_no, !filename), String.sub lex 0 (n-1))
  )
  | 1 -> (
# 214 "py_lex.mll"
      ERRORTOKEN (lexeme lexbuf))
  | n -> lexbuf.Lexing.refill_buff lexbuf; parse_hash_comment_rec lexbuf n

and parse_C_comment lexbuf = parse_C_comment_rec lexbuf 9
and parse_C_comment_rec lexbuf state =
  match Lexing.engine lex_tables state lexbuf with
    0 -> (
# 217 "py_lex.mll"
           
    incr_lex_counters lexbuf;
    comment_text := !comment_text ^ (lexeme lexbuf);
    parse_C_comment lexbuf
  )
  | 1 -> (
# 222 "py_lex.mll"
         
    comment_text := !comment_text ^ (lexeme lexbuf);
    incr comment_nesting_level; 
    parse_C_comment lexbuf 
  )
  | 2 -> (
# 227 "py_lex.mll"
         
    comment_text := !comment_text ^ (lexeme lexbuf);
    decr comment_nesting_level; 
    if !comment_nesting_level > 0 
    then parse_C_comment lexbuf 
    else ()
  )
  | 3 -> (
# 234 "py_lex.mll"
     
    comment_text := !comment_text ^ (lexeme lexbuf);
    parse_C_comment lexbuf 
  )
  | n -> lexbuf.Lexing.refill_buff lexbuf; parse_C_comment_rec lexbuf n

and pre_pylex lexbuf = pre_pylex_rec lexbuf 10
and pre_pylex_rec lexbuf state =
  match Lexing.engine lex_tables state lexbuf with
    0 -> (
# 240 "py_lex.mll"
        parse_hash_comment lexbuf )
  | 1 -> (
# 241 "py_lex.mll"
         
    comment_text := lexeme lexbuf;
    comment_nesting_level := 1; 
    parse_C_comment lexbuf; 
    COMMENT !comment_text  
  )
  | 2 -> (
# 248 "py_lex.mll"
               Py_keywords.map_keywords (lexeme lexbuf)  )
  | 3 -> (
# 250 "py_lex.mll"
                      
  let s = lexeme lexbuf in
  let n = String.length s in
  let d = String.sub s 2 (n-3) in
  LONG (binbig_int_of_string d))
  | 4 -> (
# 256 "py_lex.mll"
            
  let s = lexeme lexbuf in
  let n = String.length s in
  let d = String.sub s 2 (n-2) in
  INTEGER (binint_of_string d))
  | 5 -> (
# 262 "py_lex.mll"
                      
  let s = lexeme lexbuf in
  let n = String.length s in
  let d = String.sub s 2 (n-3) in
  LONG (octbig_int_of_string d))
  | 6 -> (
# 268 "py_lex.mll"
            
  let s = lexeme lexbuf in
  let n = String.length s in
  let d = String.sub s 2 (n-2) in
  INTEGER (octint_of_string d))
  | 7 -> (
# 274 "py_lex.mll"
                      
  let s = lexeme lexbuf in
  let n = String.length s in
  let d = String.sub s 2 (n-3) in
  LONG (decbig_int_of_string d))
  | 8 -> (
# 280 "py_lex.mll"
            
  let s = lexeme lexbuf in
  let n = String.length s in
  let d = String.sub s 2 (n-2) in
  INTEGER (decint_of_string d))
  | 9 -> (
# 286 "py_lex.mll"
                      
  let s = lexeme lexbuf in
  let n = String.length s in
  let d = String.sub s 2 (n-3) in
  LONG (hexbig_int_of_string d))
  | 10 -> (
# 292 "py_lex.mll"
            
  let s = lexeme lexbuf in
  let n = String.length s in
  let d = String.sub s 2 (n-2) in
  INTEGER (hexint_of_string d))
  | 11 -> (
# 298 "py_lex.mll"
                         
  let s = lexeme lexbuf in
  let n = String.length s in
  let d = String.sub s 0 (n-1) in
  LONG (decbig_int_of_string d))
  | 12 -> (
# 304 "py_lex.mll"
               
  let s = lexeme lexbuf in
  let n = String.length s in
  INTEGER (decint_of_string s))
  | 13 -> (
# 309 "py_lex.mll"
                        
  let s = lexeme lexbuf in
  let n = String.length s in
  let d = String.sub s 0 (n-1) in
  if n = 2 then LONG (decbig_int_of_string d)
  else let converter = 
    if !zero_is_octal 
    then begin
      if do_octal_warning()
      then begin
        print_endline ("WARNING: " ^ s ^ " treated as octal (deprecated)");
        incr octal_warning_count
      end
      ;
      octbig_int_of_string
    end else begin
      if do_octal_warning()
      then begin
        print_endline ("WARNING: " ^ s ^ " treated as decimal (standard)");
        incr octal_warning_count
      end
      ;
      decbig_int_of_string
    end
  in LONG (converter d))
  | 14 -> (
# 335 "py_lex.mll"
              
  let s = lexeme lexbuf in
  let n = String.length s in
  if n = 1 then INTEGER 0
  else let converter = 
    if !zero_is_octal 
    then begin 
      if do_octal_warning()
      then begin
        print_endline ("WARNING: " ^ s ^ " treated as octal (deprecated)");
        incr octal_warning_count
      end
      ;
      octint_of_string
    end else begin
      if do_octal_warning()
      then begin
        print_endline ("WARNING: " ^ s ^ " treated as decimal (standard)");
        incr octal_warning_count
      end
      ;
      decint_of_string
    end
  in INTEGER (converter s))
  | 15 -> (
# 360 "py_lex.mll"
                     FLOAT (floating_of_string (lexeme lexbuf)) )
  | 16 -> (
# 362 "py_lex.mll"
                               
  let s = lexeme lexbuf in
  let n = String.length s in
  let d = String.sub s 0 (n-1) in
  COMPLEX (0.0, floating_of_string d) )
  | 17 -> (
# 368 "py_lex.mll"
                             
  let s = lexeme lexbuf in
  let n = String.length s in
  let d = String.sub s 0 (n-1) in
  COMPLEX (0.0, floating_of_string d) )
  | 18 -> (
# 375 "py_lex.mll"
        LPAR )
  | 19 -> (
# 376 "py_lex.mll"
        RPAR )
  | 20 -> (
# 377 "py_lex.mll"
        LSQB )
  | 21 -> (
# 378 "py_lex.mll"
        RSQB )
  | 22 -> (
# 379 "py_lex.mll"
        COLON )
  | 23 -> (
# 380 "py_lex.mll"
        COMMA )
  | 24 -> (
# 381 "py_lex.mll"
        SEMI )
  | 25 -> (
# 382 "py_lex.mll"
        PLUS )
  | 26 -> (
# 383 "py_lex.mll"
        MINUS )
  | 27 -> (
# 384 "py_lex.mll"
        STAR )
  | 28 -> (
# 385 "py_lex.mll"
        SLASH )
  | 29 -> (
# 386 "py_lex.mll"
        VBAR )
  | 30 -> (
# 387 "py_lex.mll"
        AMPER )
  | 31 -> (
# 388 "py_lex.mll"
        LESS )
  | 32 -> (
# 389 "py_lex.mll"
        GREATER )
  | 33 -> (
# 390 "py_lex.mll"
        EQUAL )
  | 34 -> (
# 391 "py_lex.mll"
        DOT )
  | 35 -> (
# 392 "py_lex.mll"
        PERCENT )
  | 36 -> (
# 393 "py_lex.mll"
        BACKQUOTE )
  | 37 -> (
# 394 "py_lex.mll"
        LBRACE )
  | 38 -> (
# 395 "py_lex.mll"
        RBRACE )
  | 39 -> (
# 396 "py_lex.mll"
        TILDE )
  | 40 -> (
# 397 "py_lex.mll"
        CIRCUMFLEX )
  | 41 -> (
# 400 "py_lex.mll"
         EQEQUAL )
  | 42 -> (
# 401 "py_lex.mll"
                 NOTEQUAL )
  | 43 -> (
# 402 "py_lex.mll"
         LESSEQUAL )
  | 44 -> (
# 403 "py_lex.mll"
         GREATEREQUAL )
  | 45 -> (
# 404 "py_lex.mll"
         LEFTSHIFT )
  | 46 -> (
# 405 "py_lex.mll"
         RIGHTSHIFT )
  | 47 -> (
# 406 "py_lex.mll"
         POWER )
  | 48 -> (
# 407 "py_lex.mll"
         SLOSH )
  | 49 -> (
# 408 "py_lex.mll"
         PLUSPLUS )
  | 50 -> (
# 409 "py_lex.mll"
         MINUSMINUS)
  | 51 -> (
# 410 "py_lex.mll"
         PLUSEQUAL)
  | 52 -> (
# 411 "py_lex.mll"
         MINUSEQUAL)
  | 53 -> (
# 412 "py_lex.mll"
         STAREQUAL)
  | 54 -> (
# 413 "py_lex.mll"
         SLASHEQUAL)
  | 55 -> (
# 414 "py_lex.mll"
         PERCENTEQUAL)
  | 56 -> (
# 415 "py_lex.mll"
         CARETEQUAL)
  | 57 -> (
# 416 "py_lex.mll"
         VBAREQUAL)
  | 58 -> (
# 417 "py_lex.mll"
         AMPEREQUAL)
  | 59 -> (
# 418 "py_lex.mll"
         TILDEEQUAL)
  | 60 -> (
# 419 "py_lex.mll"
         COLONEQUAL)
  | 61 -> (
# 420 "py_lex.mll"
         LEFTARROW )
  | 62 -> (
# 421 "py_lex.mll"
         RIGHTARROW )
  | 63 -> (
# 424 "py_lex.mll"
          LEFTSHIFTEQUAL)
  | 64 -> (
# 425 "py_lex.mll"
          RIGHTSHIFTEQUAL)
  | 65 -> (
# 427 "py_lex.mll"
          parse_qstring lexbuf )
  | 66 -> (
# 428 "py_lex.mll"
        parse_qqqstring lexbuf )
  | 67 -> (
# 429 "py_lex.mll"
           parse_dqstring lexbuf )
  | 68 -> (
# 430 "py_lex.mll"
        parse_dddstring lexbuf )
  | 69 -> (
# 432 "py_lex.mll"
                    parse_raw_qstring lexbuf )
  | 70 -> (
# 433 "py_lex.mll"
                  parse_raw_qqqstring lexbuf )
  | 71 -> (
# 434 "py_lex.mll"
                     parse_raw_dqstring lexbuf )
  | 72 -> (
# 435 "py_lex.mll"
                  parse_raw_dddstring lexbuf )
  | 73 -> (
# 437 "py_lex.mll"
            
    let spaces=lexeme lexbuf in
    let column = ref 0 in
    let n = String.length spaces in
    for i=0 to n-1 do match spaces.[i] with
      | '\t' -> column := ((!column + 8) / 8) * 8
      | ' ' -> incr column
      | _ -> raise (Failure "Error in lexer, bad white space character")
    done;
    WHITE !column 
  )
  | 74 -> (
# 449 "py_lex.mll"
           incr_lex_counters lexbuf; NEWLINE (!line_no, !filename) )
  | 75 -> (
# 450 "py_lex.mll"
        ENDMARKER )
  | 76 -> (
# 451 "py_lex.mll"
      ERRORTOKEN (lexeme lexbuf))
  | n -> lexbuf.Lexing.refill_buff lexbuf; pre_pylex_rec lexbuf n

# 453 "py_lex.mll"
 
