open Util
open Py_types
open Py_exceptions
exception Found of string * string * bool

class py_interpreter (sys':module_t) = 
  let bfd:dictionary_t = ((Py_builtins.create_builtins_dictionary ()) :> dictionary_t) in
  let bfm = 
      ((Py_util.module_of_dictionary 
        ["__builtin__"]
        "py_builtins.ml" 
        "<some directory>"
        bfd
        None)
       :> module_t)
  in 
  let be = ((new Py_module.py_module_environment bfm) :> environment_t) in
  let dict = (new Py_dict.py_dictionary :> dictionary_t) in
  let main = 
    ((Py_util.module_of_dictionary 
    ["__main__"]
    "__some_file__"
    "<some directory>"
    dict
    (Some be))
    :> module_t)
  in
     let env = ((new Py_module.py_module_environment main) :> environment_t)
  in

  object (self)
    val sys:module_t = sys'

    val builtin_functions_dictionary:dictionary_t = bfd

    val builtin_functions_module:module_t = bfm
    (* method private get_builtins_module:module_t = builtin_functions_module *)

    val builtin_environment:environment_t = be

    val mutable environment:environment_t = env

    val mutable exc_type = PyNone
    val mutable exc_value = PyNone
    val mutable exc_traceback = PyNone

    val mutable source_reference = (1,"<uninitialised>")
    val mutable exndict = new Py_dict.py_dictionary
    val mutable traceback = []

    (* diagnostic trace countrol *)
    val mutable tracecount = 0
    method get_tracecount = tracecount
    method incr_tracecount = tracecount <- tracecount + 1
    method decr_tracecount = 
      if tracecount > 0 
      then tracecount <- tracecount - 1
   
    (* environment access *)
    method get_sys:module_t = sys
    method get_builtins_dictionary : dictionary_t = builtin_functions_dictionary
    method get_builtins_environment:environment_t = builtin_environment 


    (* get all the modules loaded by executing files *)
    method get_loaded_modules : dictionary_t = 
      match
        match sys#get_attr (PyString "modules") 
        with
          | Some x -> x
          | None -> raise (SystemError "'modules' attribute of sys doesn't exist")
      with 
      | PyDictionary d -> d
      | _ -> raise (SystemError "'modules' entry of sys not dictionary")

    (* get all the modules built in to the interpreter *)
    method get_builtin_modules : dictionary_t = 
      match 
        match sys#get_attr (PyString "builtin_modules") 
        with 
          | Some x -> x
          | None -> raise (SystemError "sys module doesn't contain a 'builtin_modules' attribute")
      with
        | PyDictionary d -> d
        | _ -> raise (TypeError "builtin_modules must be a dictionary")

    (* get an installed module, either builtin or loaded *)
    method private get_installed_module module_name : module_t = 
      let imported_modules = self#get_loaded_modules 
      and builtin_modules = self#get_builtin_modules 
      in if imported_modules#has_item (PyString module_name)
      then match imported_modules#get_item (PyString module_name) with
      | Some (PyModule m) -> m
      | _ -> raise (TypeError "builtin module must be a module")
      else if builtin_modules#has_item (PyString module_name)
      then match builtin_modules#get_item (PyString module_name) with
      | Some (PyModule m) -> m
      | _ -> raise (TypeError "builtin module must be a module")
      else raise Not_found 

    (* get the path for loading modules from sys *)
    method get_path = 
      (* get the 'path' attribute from sys *)
      let path1 = 
        match sys#get_attr (PyString "path") with
        | Some x -> x
        | None  -> raise (SystemError "sys module doesn't contain a 'path' attribute")
      in

      (* get the path from the attribute *)
      let path2 = 
        try begin match path1 with 
        | PyString _ -> [path1]
        | _ -> Py_util.list_of_sequence path1
        end
        with _ -> raise (TypeError "sys.path must be a list of strings")

      (* convert the path to a list of ocaml strings *)
      in let path3 = List.map
        begin fun x -> match x with 
        | PyString dir -> dir
        | _ -> raise (TypeError "sys.path component must be a Pystring")
        end
        path2
      in  path3

    (* get a module known to be loaded from the file system *)
    method private get_module m = 
      let d = self#get_loaded_modules in
      begin match 
        match d#get_item (PyString m)  with
        | Some x-> x
        | None -> raise (SystemError ("module "^m^" doesn't exist in modules dictionary"))
      with
      | PyModule m -> m
      | _ -> raise (SystemError ("Loaded client module "^m^"not accessible"))
      end

    method type_with_name name =
      let type_module = self#get_module "py_types" in
      match type_module#get_attr (PyString name) with
      | Some x -> x
      | None ->
        print_endline ("ERROR: cannot get type object named " ^ name);
        raise (SystemError "Unable to load module 'py_types'")
    
    method type_of_object obj =
     self#type_with_name (Py_datum.get_typename obj) 

    
    initializer 
      (* put exc_info into sys module *)
      begin 
        let py_get_exc_info interp arglist argdict = self#get_exc_info
        in ignore (sys#get_dictionary#set_item
          (PyString "exc_info") 
          (PyNativeFunction ("exc_info", py_get_exc_info))
        )
      end
      ;
      (* put builtins into dictionary of builtin modules *)
      ignore ( self#get_builtin_modules#set_item 
        (PyString "__builtin__") 
        (PyModule bfm)
      )
      ;
      begin (* exceptions *)
        let exn_module = self#install_module "exceptions" 
        in
          exndict <- exn_module#get_dictionary;
          (* copy standard exceptions into the bottom (builtins) dictionary *)
          exndict#iter (fun k v -> 
            match k with 
            | PyString s when (String.length s > 1) && (s.[0] = '_') -> ()
            | _ ->  ignore (builtin_functions_dictionary#set_item k v))
      end
      ;
      begin (* types *)
        let types = self#install_module "py_types" in
        begin 

          begin try (* type function *)
            let type_function = match types#get_attr (PyString "type") with
            | Some x -> x
            | None -> raise (SystemError "type function not defined in py_types")
            in
            ignore (builtin_functions_dictionary#set_item (PyString "type") type_function);
          with x -> 
            print_endline ("!! WARNING: UNABLE TO GET 'type' function from 'py_types'!!"); 
            print_endline ("Exception is: " ^ (Printexc.to_string x));
            flush stdout;
            raise x
          end
          ;
          begin try (* open function *)
            let open_function = match types#get_attr (PyString "open") 
            with Some x -> x
            | None -> raise (SystemError "open function not defined in 'py_types'")
            in
            ignore (builtin_functions_dictionary#set_item (PyString "open") open_function);
          with x -> 
            print_endline ("!! WARNING: UNABLE TO GET 'open' function from 'py_types'!!"); 
            print_endline ("Exception is: " ^ (Printexc.to_string x));
            flush stdout;
            raise x
          end
          ;
          begin try (* set sys std files *)
            let set_std_files = match types#get_attr (PyString "set_std_files") 
            with Some x -> x
            | None -> raise (SystemError "set_std_files function not defined in 'py_types'")
            in
            ignore (
              Py_exec.py_call 
              (self:>interpreter_t) 
              set_std_files 
              []
            )
          with x -> 
            print_endline ("!! WARNING: UNABLE TO GET 'open' function from 'py_types'!!"); 
            print_endline ("Exception is: " ^ (Printexc.to_string x));
            flush stdout;
            raise x
          end
        end
        ;
        ignore (self#install_module "types");
      end
      ;
      begin (* posix *)
        (* We need a HACK now, to make Python's os module work:
          this module assumes 'posix' is a builtin module,
          in Viper, it isn't, it's loaded from the file system.
        *)
        let posix = (self#install_module "posix") in
        ignore ( self#get_builtin_modules#set_item 
          (PyString "posix") 
          (PyModule posix)
        )
      end

    (* this routine loads a submodule sub of a given package pkg, to be used
       only when the original name is pkg.sub, do not use for sibling loads.
    *)
    method private load_component (pkg:module_t) (mname:string) : module_t =
      let fullname = pkg#get_name @ [mname] in
      let dotted_module_name = Py_module_file.path_dot_join fullname in
      try (* first attempt *)
        self#get_installed_module dotted_module_name
      with Not_found -> 
        let parent_directory = 
          match pkg#get_file_directory (* the directory of the packages __init__ module *)
          with Some x -> x | _ -> raise (SystemError "module doing import has no file")
        in
        let filename, directory, ispkg =
          let prefix = parent_directory ^ "/" ^ mname in
          let f1 = prefix ^ ".vy" in
          if Sys.file_exists f1 then f1, parent_directory, false
          else let f2 = prefix ^ ".py" in
          if Sys.file_exists f2 then f2, parent_directory, false
          else let f3 = prefix ^ "/__init__.vy" in
          if Sys.file_exists f3 then f3, parent_directory ^ "/" ^ mname, true
          else let f4 = prefix ^ "/__init__.py" in
          if Sys.file_exists f4 then f4, parent_directory ^ "/" ^ mname, true
          else begin
            raise 
            (
              ImportError 
              (
                "[load_component]: Cannot find file for package submodule " ^ dotted_module_name ^ 
                " in directory " ^ parent_directory
                , 
                None
              )
            )
          end
          in
          let builtins = self#get_builtins_environment in
          let new_d = new Py_dict.py_dictionary in
          let new_m = Py_util.module_of_dictionary fullname directory filename new_d (Some builtins) in
          let new_e = new Py_module.py_module_environment new_m in
          
          ignore (new_d#set_item (PyString "__builtins__") (PyDictionary self#get_builtins_dictionary));
          ignore (new_d#set_item (PyString "__name__") (PyString dotted_module_name));
          ignore (new_d#set_item (PyString "__file__") (PyString filename));
          ignore (new_d#set_item (PyString "__directory__") (PyString directory));
          if ispkg then ignore (new_d#set_item (PyString "__path__") (PyString directory));
          ignore (self#get_loaded_modules# set_item (PyString dotted_module_name) (PyModule new_m));
          begin try 
            Py_exec_module.exec_file 
              (self:>interpreter_t) 
              (Py_exec.py_exec_wrap_exn (self:>interpreter_t) new_e) 
              filename
            ;
            (* install the packages submodule in the __init__ module unconditionally *)
            ignore (pkg#get_dictionary#set_item (PyString mname) (PyModule new_m));
            new_m 
          with x -> 
            print_endline ("load_modules Error executing " ^ filename);
            flush stdout;

            (* if there is an error importing a module, 'uninstall' it *)
            ignore (self#get_loaded_modules# del_item (PyString dotted_module_name));
            raise (ImportError (("Executing file " ^ filename), Some x))
          end;
          
    (* this routine attempts to load a sibling module. There are two cases:
       1) sibling of the main file
       2) sibling of a package module

       These cases are treated the same.

       When a package submodule x imports y, x and y have the same directory,
       and the names share a common prefix, the common parent module name,
       with one exception: if x is the __init__ module, then _it_ is the parent.
    *)

    (* WARNING! THIS ROUTINE DOES NOT CREATE A DICTIONARY ENTRY IN THE IMPORTER
      In particular, if x.y1 imports y2, x will NOT contain a dictionary entry
      for y2, even though y2 is both loaded and a submodule of x. It is hard
      to say if this is a bug: the behaviour is correct for the extened
      interpretation of siblings of the main file.

      Of course, an __init__ module saying 'import y2' will get an entry for y2,
      but there will be no entry from 'from y2 import *' nor if y1 says 'import y2'.
    *)
    method private load_sibling (importer:module_t) (mname:string) : module_t =
      let fullname = 
        match importer#get_attr (PyString "__path__") with
        | Some x -> importer#get_name @ [mname]
        | None -> (list_all_but_last importer#get_name) @ [mname]
      in 
      let dotted_module_name = Py_module_file.path_dot_join fullname in
      try (* first attempt *)
        self#get_installed_module dotted_module_name
      with Not_found -> 
        let parent_directory = 
          match importer#get_file_directory (* the directory of the packages __init__ module *)
          with Some x -> x | _ -> raise (SystemError "module doing import has no file")
        in
        let filename, directory, ispkg =
          let prefix = parent_directory ^ "/" ^ mname in
          let f1 = prefix ^ ".vy" in
          if Sys.file_exists f1 then f1, parent_directory, false
          else let f2 = prefix ^ ".py" in
          if Sys.file_exists f2 then f2, parent_directory, false
          else let f3 = prefix ^ "/__init__.vy" in
          if Sys.file_exists f3 then f3, parent_directory ^ "/" ^ mname, true
          else let f4 = prefix ^ "/__init__.py" in
          if Sys.file_exists f4 then f4, parent_directory ^ "/" ^ mname, true
          else raise Not_found
          in
          let builtins = self#get_builtins_environment in
          let new_d = new Py_dict.py_dictionary in
          let new_m = Py_util.module_of_dictionary fullname directory filename new_d (Some builtins) in
          let new_e = new Py_module.py_module_environment new_m in
          
          ignore (new_d#set_item (PyString "__builtins__") (PyDictionary self#get_builtins_dictionary));
          ignore (new_d#set_item (PyString "__name__") (PyString dotted_module_name));
          ignore (new_d#set_item (PyString "__file__") (PyString filename));
          ignore (new_d#set_item (PyString "__directory__") (PyString directory));
          if ispkg then ignore (new_d#set_item (PyString "__path__") (PyString directory));
          ignore (self#get_loaded_modules# set_item (PyString dotted_module_name) (PyModule new_m));
          begin try 
            Py_exec_module.exec_file 
              (self:>interpreter_t) 
              (Py_exec.py_exec_wrap_exn (self:>interpreter_t) new_e) 
              filename
            ;
            new_m 
          with x -> 
            print_endline ("load_modules Error executing " ^ filename);
            flush stdout;

            (* if there is an error importing a module, 'uninstall' it *)
            ignore (self#get_loaded_modules# del_item (PyString dotted_module_name));
            raise (ImportError (("Executing file " ^ filename), Some x))
          end;
          
    method private install_module mname = self#load_definite_top mname

    (* this routine is used to load a definite top level module,
       it searches the sys.path
    *)

    method private load_definite_top (mname:string) : module_t =
      try self#get_installed_module mname
      with Not_found ->
      try
        flush stdout;
        List.iter 
        begin fun parent_directory ->
          let prefix = parent_directory ^ "/" ^ mname in
          let f1 = prefix ^ ".vy" in
          if Sys.file_exists f1 
          then raise (Found (f1, parent_directory, false))
          else let f2 = prefix ^ ".py" in
          if Sys.file_exists f2 
          then raise (Found (f2, parent_directory, false))
          else let f3 = prefix ^ "/__init__.vy" in
          if Sys.file_exists f3 
          then raise (Found (f3, parent_directory ^ "/" ^ mname, true))
          else let f4 = prefix ^ "/__init__.py" in
          if Sys.file_exists f4 
          then raise (Found (f4, parent_directory ^ "/" ^ mname, true))
        end
        self#get_path
        ;
        raise Not_found
       with 
       | Not_found -> 
          raise (ImportError ("[load_definite_top]: Could not find module " ^ mname ^ " in sys.path", None))
       | Found (filename, directory, ispkg) ->
          let builtins = self#get_builtins_environment in
          let new_d = new Py_dict.py_dictionary in
          let new_m = Py_util.module_of_dictionary [mname] directory filename new_d (Some builtins) in
          let new_e = new Py_module.py_module_environment new_m in
          
          ignore (new_d#set_item (PyString "__builtins__") (PyDictionary self#get_builtins_dictionary));
          ignore (new_d#set_item (PyString "__name__") (PyString mname));
          ignore (new_d#set_item (PyString "__file__") (PyString filename));
          ignore (new_d#set_item (PyString "__directory__") (PyString directory));
          if ispkg then ignore (new_d#set_item (PyString "__path__") (PyString directory));
          ignore (self#get_loaded_modules# set_item (PyString mname) (PyModule new_m));
          begin try 
            Py_exec_module.exec_file 
              (self:>interpreter_t) 
              (Py_exec.py_exec_wrap_exn (self:>interpreter_t) new_e) 
              filename
              ;
            new_m 
          with x -> 
            print_endline ("load_modules Error executing " ^ filename);
            flush stdout;

            (* if there was an error, 'uninstall' the module *)
            ignore (self#get_loaded_modules # del_item (PyString mname));
            raise (ImportError (("Executing file " ^ filename), Some x))
          end

    method private load_maybe_top (importer:module_t) (mname:string) : module_t =
      try self#load_sibling importer mname
      with Not_found -> self#load_definite_top mname

    (* this is the public method, given a module name x.y.z, return a
       pair consisting of the module x and the module z
    *)
    method import (importer:module_t) (module_name:string list) =
      let top_name = List.hd module_name in
      let top = self#load_maybe_top importer top_name in
      let parent = ref top in
      let parent_name = ref [top_name] in
      List.iter
      begin fun name -> 
        parent := (self#load_component !parent name);
        parent_name := !parent_name @ [name]
      end
      (List.tl module_name)
      ;
      top, !parent

    method set_line sr = 
      source_reference <- sr;
      if tracecount > 0 
      then match sr with (i,s) -> 
        print_string "File: "; print_string s; 
        print_string " Line: "; print_string (string_of_int i);
        print_endline "";
        print_string (Py_functions.get_line_of_file s i);
        print_newline()
  
    method push_line =
      traceback <- source_reference :: traceback

    method pop_line =
      if List.length traceback < 1 
      then raise (SystemError "Traceback is empty in pop_line of interpreter");
      source_reference <- List.hd traceback;
      traceback <- List.tl traceback

    method get_traceback = source_reference :: traceback

    method run_file filename : int =
      try
        ignore (environment # set_attr (PyString "__name__") (PyString "__main__"));
        ignore (environment # set_attr (PyString "__file__") (PyString filename));
        Py_exec_module.exec_file (self:>interpreter_t)
          (Py_exec.py_exec (self:>interpreter_t) (environment:>environment_t)) 
          filename
        ;
        0
      with 
      | SystemExit n -> n
      | x -> 
        print_endline ("Error Near line " ^ (string_of_int (fst source_reference)));
        flush stdout;
        self#py_raise x;
        0 (* dummy value *)

    method run_string s : int =
       try
        ignore (environment # set_attr (PyString "__name__") (PyString "__main__"));
        Py_exec_module.exec_string (self:>interpreter_t)
        (Py_exec.py_exec (self:>interpreter_t) (environment:>environment_t)) 
        s "<string>"
        ;
        0
      with
      | SystemExit n -> n
      | x -> 
        print_endline ("Error Near line " ^ (string_of_int (fst source_reference)));
        flush stdout;
        self#py_raise x;
        0 (* dummy value *)

    method set_exc_info cls v tr = 
      exc_type <- cls;
      exc_value <- v;
      exc_traceback <- tr
    
    method clear_exc_info = 
      exc_type <- PyNone;
      exc_value <-  PyNone;
      exc_traceback <- PyNone
      
    method get_exc_info = PyTuple [exc_type; exc_value; exc_traceback]

    (* This function prints the current execution stack of the interpreter.
       This is not the same as the state recorded in the exc_traceback
       variable, which is a copy of the traceback at some point 
       when set_exc_info was called.

       There is a SERIOUS bug in this mechanism at the moment:
       throwing an exception sometimes bypasses the code that
       pops the current execution stack. It is necessary
       to pop the stack symmetrically with pushes, no matter
       whether an exception is thrown or not. It is also necessary
       to set the exc_info with set_exc_info, whenever an
       exception is detected, before doing the pop, to record
       where the exception was thrown from. 

       Even worse, when destructors are allowed, unwinding
       the stack will have to decrement reference counts to
       ensure they're invoked. In other words, ocaml exceptions
       must be trapped much earlier than I'm currently trapping
       them: before an (python) exceuction frame is dropped
       from the stack. Currently, there is no object representing
       an exceution frame.


       THE WHOLE OF THIS MECHANISM MUST BE REDESIGNED.
       
    *)

    
    method print_tb = 
      print_string (Py_functions.repr (PyTraceback (source_reference::traceback)))

    method print_exc_tb = 
      print_string (Py_functions.repr exc_traceback)

    method py_raise (x:exn) : unit = 

      (* routine to construct a Python exception object from
         its string name, by looking up the exceptions dictionary
     
         If an exception is not found, there are three possible 
         reasons: 
         
         1) the exceptions module was not loaded, or,
         2) it was loaded but corrupted by the user, or,
         3) a bug in Viper due to Skaller's programming

         In all these cases, we just raise a SystemError
         with a suitable message. *)

      let raise_string class_name str = 
        if tracecount > 0 
        then print_endline ("Translating exception '" ^ class_name ^ "', value '" ^ str ^ "'");
        let cls_expr = 
          begin
            match exndict#get_item (PyString class_name)
            with 
            | Some x -> x
            | None -> raise (SystemError ("Exception " ^ class_name ^ " not found in exceptions dictionary"))
          end
        in let cls = match cls_expr with 
        | PyClass cls -> cls
        | _ -> raise (SystemError ("Exception object "^class_name^"not class"))
        in
        let new_d = new Py_dict.py_dictionary
        in  let new_m = Py_util.module_of_dictionary 
          ["dummy"] 
          "dummy" 
          "dummy" 
          new_d 
          (Some builtin_environment) 
        in let inst = 
          try
            Py_exec.create_class_instance 
            (self:>interpreter_t) 
            cls
            [Argument1 (PyString str)] 
          with _ ->  
            raise (SystemError ("Unable to instantiate class exception " ^ class_name))
        in
        self # set_exc_info (PyClass inst#get_class) (PyInstance inst) (PyTraceback self#get_traceback);
        raise (PyException (PyInstanceException inst))

      and raise_OSError errcode funcname filename = 
        if tracecount > 0 
        then print_endline ("Translating exception OSError, function '" ^ funcname ^ "', filename '" ^ filename ^ "'");
        let errdesc = Unix.error_message errcode in
        let errno = Py_posix.get_posix_errcode errcode in
        let arguments = match filename with
        | "" -> [
            Argument1 (PyInt errno);
            Argument1 (PyString (errdesc ^ "[function="^funcname^"]"))
          ]
        | filename -> [
            Argument1 (PyInt errno);
            Argument1 (PyString (errdesc ^ "[function="^funcname^", filename="^filename^"]"));
            Argument1 (PyString filename)
          ]
        in
        let cls_expr = 
          begin
            match exndict#get_item (PyString "OSError")
            with 
            | Some x -> x
            | None -> raise (SystemError ("Exception OSError not found in exceptions dictionary"))
          end
        in let cls = match cls_expr with 
        | PyClass cls -> cls
        | _ -> raise (SystemError ("Exception object OSError not class"))
        in
        let new_d = new Py_dict.py_dictionary
        in  let new_m = Py_util.module_of_dictionary 
          ["dummy"] 
          "dummy" 
          "dummy" 
          new_d 
          (Some builtin_environment) 
        in let inst = 
          try
            Py_exec.create_class_instance 
            (self:>interpreter_t) 
            cls
            arguments
          with _ ->  
            raise (SystemError ("Unable to instantiate class exception OSError"))
        in
        self # set_exc_info (PyClass inst#get_class) (PyInstance inst) (PyTraceback self#get_traceback);
        raise (PyException (PyInstanceException inst))

      and raise_SystemExit errcode = 
        if tracecount > 0 
        then print_endline ("Translating exception SystemExit, code=" ^ (string_of_int errcode));
        let cls_expr = 
          begin
            match exndict#get_item (PyString "SystemExit")
            with 
            | Some x -> x
            | None -> raise (SystemError ("Exception SystemExit not found in exceptions dictionary"))
          end
        in let cls = match cls_expr with 
        | PyClass cls -> cls
        | _ -> raise (SystemError ("Exception object SystemExit not class"))
        in
        let new_d = new Py_dict.py_dictionary
        in  let new_m = Py_util.module_of_dictionary 
          ["dummy"] 
          "dummy" 
          "dummy" 
          new_d 
          (Some builtin_environment) 
        in let inst = 
          try
            Py_exec.create_class_instance 
            (self:>interpreter_t) 
            cls
            [Argument1 (PyInt errcode)] 
          with _ ->  
            raise (SystemError ("Unable to instantiate class exception SystemExit"))
        in
        self # set_exc_info (PyClass inst#get_class) (PyInstance inst) (PyTraceback self#get_traceback);
        raise (PyException (PyInstanceException inst))

      in match x with

      (* Map all exceptions onto Python ones, if possible *)

      (* Internal 'ocaml' exceptions: convert to Python exceptions *)
      | AttributeError (s,k) -> 
        raise_string "AttributeError" (s ^ " " ^ (Py_functions.repr k)) 
      | NameError (s,k) -> 
        raise_string "NameError" (s ^ " " ^ (Py_functions.repr k)) 
      | KeyError (s,k) -> 
        raise_string "KeyError" (s ^ " " ^ (Py_functions.repr k)) 

      | ParseError s -> raise_string "SyntaxError" ("Parser Error " ^ s )
      | LexError s -> raise_string "SyntaxError" ("Lexer Error " ^ s )
      | SyntaxError s -> raise_string "SyntaxError" s 
      | TypeError s -> raise_string "TypeError" s 
      | RangeError s -> raise_string "IndexError" s
      | AssertionError s -> raise_string "AssertionError" s 
      | ImportError (s,x) -> 
        let s' = match x with
        | Some x' -> s ^ ": " ^ (Py_functions.string_of_exception x')
        | None -> s
        in raise_string "ImportError" s'
      | NonSequence -> raise_string "TypeError" "Sequence required"
      | IndexError (s,i,l) -> raise_string "IndexError" 
          (s ^ ", index=" ^ (string_of_int i) ^ ", len=" ^ (string_of_int l))

      (* syntax errors detected dynamically *)
      | NotImplemented s -> raise_string "NotImplementedError" s

      (* Viper programming error -- should never happen *)
      | Failure s -> raise_string "SystemError" ("Internal viper Failure " ^ s)

      | Unix.Unix_error (ecode, fname, param) -> 
        raise_OSError ecode fname param

      (* A Python exception: just raise it as is.
         This can happen when

         1) Client code raised an exception
         2) A downstack evaluation of 'py_raise' has already
            mapped the exception to a Python one *)
      | SystemExit n -> raise_SystemExit n
      | Continuer
      | Breaker
      | Returner _ 
      | PyException _ -> raise x

      (* Woops! Viper has thrown an unrecognized exception.
         Can only happen due to a bug n Viper due to Skaller's programming *)

      | x -> 
        print_endline ("UNKNOWN EXCEPTION " ^ (Printexc.to_string x)); 
        raise (UnknownError x)
  end
;;

let rec get_tail_module_dict dict path = 
  let head = PyString (List.hd path)
  in let theModule = 
    match dict#get_item head with 
    | Some x -> x
    | None -> raise (AttributeError ("Cannot find module in path",head))
  in
  let dict' =  match theModule with 
  | PyModule m -> m#get_dictionary
  | _ -> raise (SystemError "Installed module not module (From .. Import ..)!")
  in
  match path with 
  | h1 :: h2 :: t -> get_tail_module_dict dict' (List.tl path)
  | h :: [] -> dict'
  | [] -> raise (SystemError "in get_tail_module_dict")



