open Py_types
open Util
(* Depends on py_objects *)
(* Used by Py_eval *)


let tuple_like_display before between after each l = 
  if List.length l = 1 
  then begin before(); each (List.hd l); between(); after() end
  else begin display_of_list before between after each l end
;;

(* one origin line number! *)
let get_line_of_file f i =
  let line = ref "<cannot get line from file>" in
  begin try 
    let file = open_in f in
      begin try 
        for i = 1 to i-1 do ignore (input_line file) done;
        line := input_line file
      with _ -> ()
      end;
      close_in file
  with _ -> ()
  end;
  !line
;;

let rec pparameters ps : string = 
  let res = ref "" in
  let p s = res := !res ^ s in
  let p' x () = p x  in
  let pe e = p (repr e) in
  let rec pparam_name n = match n with
  | Param s -> p s
  | Paramtuple tup ->
    p "(";
    List.iter (fun x -> pparam_name x; p ", ") tup;
    p ")"
  in 
  let pparam x = match x with
  | Parameter2 (pn,e) -> pparam_name pn; p " = "; pe e
  | Parameter1 pn -> pparam_name pn
  in
  begin match ps with (op, sp, ssp) ->
      tuple_like_display (p' "") (p' ", ") (p' "") pparam op;
      let comma_needed = ref ((List.length op) <> 0) in
        begin match sp with
        | NoStarParam -> ()
        | StarParam s -> 
          if !comma_needed then p ", ";
          p "*"; p s; 
          comma_needed := true
        end;
        begin match ssp with
        | NoStarStarParam -> ()
        | StarStarParam s -> 
          if !comma_needed then p ", ";
          p "*"; p s; 
        end
  end;
  !res

and _repr exclude (e:expr_t) : string =
  let res = ref "" in
  let p s = res := !res ^ s in
  let p' s () = p s in
  let pe x = res := !res ^ (_repr (e::exclude) x) in
  let psubentry x = match x with 
  | Defsub -> ()
  | Pos e -> pe e
  in let psub x = 
  match x with
  | Ellipsis -> p "[...]"
  | Subscript2 (a,b,c) -> 
    p "["; 
    psubentry a; 
    p ":"; 
    psubentry b; 
    p ":"; 
    psubentry c; 
    p "]"
  | Subscript1 (a,b) -> 
    p "["; 
    psubentry a; 
    p ":"; 
    psubentry b; 
    p "]"
  | Subscript0 a -> 
    p "["; 
    psubentry a; 
    p "]"
  in let pcomparator x = 
    match x with 
    | Less e   -> p " < "; pe e
    | Greater e -> p " > "; pe e
    | LessEqual e -> p " <= "; pe e
    | GreaterEqual e-> p " >= "; pe e
    | Equal e-> p " == "; pe e
    | NotEqual e -> p " != "; pe e
    | Is e -> p " is "; pe e
    | IsNot e-> p " is not "; pe e
    | In e -> p " in "; pe e
    | NotIn e -> p " not in "; pe e

  in let pbinop x = 
    match x with 
    | Add e   -> p " + "; pe e
    | Sub e -> p " - "; pe e
    | Mul e -> p " * "; pe e
    | Div e-> p " / "; pe e
    | Mod e-> p " % "; pe e
    | Asl e -> p " << "; pe e
    | Lsr e -> p " >> "; pe e
    | Pow e -> p " ** "; pe e
  
  and pdictent x = 
    match x with DictEnt (k,v) -> pe k; p ":"; pe v 
  in let parg x =
    match x with 
    | Argument1 v -> pe v
    | Argument2 (n,v) -> p n; p " = "; pe v
  in let ptrailer x = 
    begin match x with
    | Arglist aa -> tuple_like_display (p' "(") (p' ", ") (p' ")") parg aa
    | Dotname n -> p "."; p n 
    | Sublist uu -> List.iter psub uu
    end
  in begin match e with 
    | PyName s -> p s
    | PyVarIndex (level,i) -> p ("<#"^(string_of_int level)^"."^(string_of_int i)^">")
    | PyDict dd ->
       tuple_like_display (p' "{") (p' ", ") (p' "}") pdictent dd
    | PyRepr e -> p "`"; pe e; p "`" 
    | Or ee ->
      tuple_like_display (p' "") (p' " or ")  (p' "") pe ee
    | And ee ->
      tuple_like_display (p' "(") (p' ") and (")  (p' ")") pe ee
    | Not e -> p "not ("; pe e; p")" 
    | Neg e -> p "-("; pe e; p")"  
    | Compare (e, cl) ->
      pe e; List.iter pcomparator cl
    | BitOr ee ->
      tuple_like_display (p' "(") (p' ") | (")  (p' ")") pe ee
    | BitAnd ee ->
      tuple_like_display (p' "(") (p' ") & (")  (p' ")") pe ee
    | BitXor ee -> 
      tuple_like_display (p' "(") (p' ") ^ (")  (p' ")") pe ee
    | Complement e -> p "~("; pe e; p ")"
    | Eval (e, bs) ->
      p "("; pe e; List.iter pbinop bs; p ")"
    | AtomWithTrailers (e, tt) ->
      pe e; List.iter ptrailer tt
    | Lambda (pas, e) ->
      p "lambda ";
      p (pparameters pas);
      p ": ";
      pe e

    (* Datum values *)
    | PyNone -> p "None"
    | PyInitial -> p "<initial>"
    | PyTerminal -> p "<terminal>"
    | PyString s -> p (Py_string.dquote_of_string s)
    | PyInt i -> p (string_of_int i)
    | PyLong i -> p (Big_int.string_of_big_int i)
    | PyRational i -> p (Num.string_of_num i)
    | PyFloat v -> p (string_of_float v)
    | PyComplex (r,i) -> p ("("^(string_of_float r)^"+"^(string_of_float i)^"j)")
    (* algebraic objects *)
    | PyTuple t -> tuple_like_display (p' "(") (p' ", ") (p' ")")  pe t
    | PyList l -> tuple_like_display (p' "[") (p' ", ") (p' "]") pe l
    | IntRange (start,stop,step) -> 
      p "<range>("; p (string_of_int start);
      p ", "; p (string_of_int stop);
      p ", "; p (string_of_int step);
      p ")"

    (* encapsulated objects *)
    | PyMutableList l -> 
      if List.mem e exclude
      then p "[...]"
      else tuple_like_display (p' "[") (p' ", ") (p' "]") pe (Varray.to_list l)
    | PyInstance i -> p ("<instance of " ^ i#get_class#get_name ^ ">")
    | PyClass cls -> p ("<class " ^ cls#get_name ^ ">")
    | PyModule m -> p ("<module " ^ m#get_dotted_name ^ ">")
    | PyFunction m -> p ("<function " ^ m#get_name ^ ">")
    | PyDictionary d -> 
      p "{"; d#iter (fun k v -> pe k; p " : "; pe v; p ","); p "}" 
    | PyBoundMethod (f,e) -> p ("<method " ^ (repr f) ^ " bound to " ^(repr e)^ ">")
    | PyNativeFunction (name,f) -> p ("<native function "^name^">")
    | PyNativeMacro (name,f) -> p ("<native macro "^name^">")
    | PyEnv _ -> p "<environment>"
    | PyInterpreter _ -> p "<interpreter>"

    | PyFile f -> p "<file>"
    | PyRegexp r -> p "<regexp>"

    (* GUI and graphics *)
    | PyWidget r -> p "<widget>"
    | PyFont _ -> p "<font>"
    | PyColor _ -> p "<color>"
    | PyImage _ -> p "<image>"
    | PyCanvas _ -> p "<canvas>"
    | PyGraphicsContext _ -> p "<graphics context>"


    (* extras and weird stuff *)
    | PyTraceback tb ->
      p "Traceback (innermost first)\n";
      let pr_tb_line tbl = 
        match tbl with (i,s) -> 
          p "File: "; p s; 
          p " Line: "; p (string_of_int i);
          p "\n";
          p (get_line_of_file s i);
          p "\n"
      in 
      let rec pr_tb tb' = 
      match tb' with 
      | h :: t -> pr_tb_line h; pr_tb t
      | [] -> ()
      in pr_tb tb
      
    | PyStatement s -> p "<statement>"
    | PyClosure (env,e) -> p "lazy("; pe e; p ")"
    | Unknown -> p "<unknown>"
  end;
  !res

(* NOTE: these functions will NOT hook __str__ or __repr_ methods; 
   Py_builtins.str, repr tries, and calls these functions if they fail;
   these function provide primitive string conversions only.
*)

and repr e  = _repr [] e
and erepr e  = _repr [] e

and str (x:expr_t) : string =
  match x with
  | PyString s -> s
  | _ -> repr x
;;

let callable (x:expr_t) : bool = 
  match x with
  | PyFunction _ -> true
  | PyNativeFunction _ -> true
  | PyClass _ -> true
  | PyBoundMethod _ -> true
  | _ -> false
;;
 
let py_str (x:expr_t): expr_t = PyString (str x);;
let py_repr (x:expr_t): expr_t = PyString (repr x);;
let py_callable (x:expr_t): expr_t = PyInt (if callable x then 1 else 0);;


let rec string_of_exception x =
  match x with
  | Py_exceptions.ParseError s      -> "ParseError: " ^ s
  | Py_exceptions.LexError s        -> "LexError: " ^ s
  | Py_exceptions.TokenError s      -> "TokenError: " ^ s
  | Py_exceptions.SystemError s     -> "SystemError: " ^ s
  | Py_exceptions.ViperError s      -> "ViperError: " ^ s
  | Py_exceptions.NotImplemented s  -> "NotImplemented: " ^ s
  | Py_exceptions.SyntaxError s     -> "SyntaxError: " ^ s
  | Py_exceptions.TypeError s       -> "TypeError: " ^ s
  | Py_exceptions.NonSequence       -> "NonSequence"
  | Py_exceptions.RangeError s      -> "RangeError: " ^ s
  | Py_exceptions.AssertionError s  -> "AssertionError " ^ s

  | Py_exceptions.IndexError (s,ix,n)  -> 
    "IndexError: " ^ s ^ 
    " value " ^ (string_of_int ix) ^ 
    ", max " ^ (string_of_int n)
  | Py_exceptions.AttributeError (s,e)  -> "AttributeError " ^ s ^ ", attribute="^(repr e)
  | Py_exceptions.NameError (s,e)   -> "NameError " ^ s ^ ", name=" ^(repr e)
  | Py_exceptions.KeyError (s,e)    -> "KeyError " ^ s ^ ", key=" ^ (repr e)
  | Py_exceptions.ValueError s      -> "ValueError " ^ s

  | Py_exceptions.ImportError (s,e) -> 
    let s' = match e with
    | Some e' -> s ^ "\n  triggered by \n" ^ (string_of_exception e')
    | None -> s
    in "ImportError: " ^ s'

  | Py_exceptions.PyException e     -> 
    begin match e with
    | PyStringException (s1,s2) -> 
      "Python String Exception: " ^ s1 ^ ": " ^ (str s2)
    | PyInstanceException i ->
      let id = "Python Instance Exception: Class " ^ 
        i#get_class#get_name
      in 
      let d = i#get_dictionary in
      let s = ref "" in
      d#iter
      begin fun k v -> 
        s := !s ^ "  " ^ (str k) ^ " --> " ^ (str v) ^ "\n"
      end;
      id ^ "\n" ^ !s
    end

  | Py_exceptions.UnknownError e    -> "UnknownError " ^ (Printexc.to_string e)
  | _ -> "Ocaml exception " ^ (Printexc.to_string x)
;; 
