open Py_types
open Py_mtypes
open Py_exceptions


class py_function name' parameters' body' glbs i2n n2i env' = 
  object(self)
    val name : string = name'
    val body : statement_t = body'
    val parameters : parameters_t = parameters'
    val env: environment_t = env'

    val globals : string list = glbs 
    val local_name_to_index: int VarMap.t = n2i
    val index_to_local_name: string array  = i2n
    method get_name = name

    method get_attr (attr: expr_t) = Some (PyString name)
    method set_attr (attr: expr_t) (value:expr_t) = false
    method del_attr (attr: expr_t) = false
    method has_attr (attr: expr_t) = false
    method can_del_attr k = env#can_del_attr k
    method can_get_attr k = 
      match self#get_attr k with Some _ -> true | None -> false
    method can_set_attr (k:expr_t) (v:expr_t) = env#can_set_attr k v
    method can_set_any_attr (k:expr_t) = env#can_set_any_attr k


    method get_parameters = parameters 
    method get_code = body 
    method get_environment = env

    (* method get_dictionary = env#top *)
    method get_global_names = globals
    method get_index_to_local_name = index_to_local_name
    method get_local_name_to_index = local_name_to_index
  end

class py_function_environment (f:function_t) (arguments':dictionary_t) =
  object(self)
    val dict = arguments'
    val super = f#get_environment

    val globals = f#get_global_names
    val vars= Array.create (Array.length f#get_index_to_local_name) PyInitial
    val name_to_index = f#get_local_name_to_index
    val func = f

    initializer
      (* map arguments to locals array now *)
       arguments'#iter
       begin fun key value -> 
         match key with PyString name ->
         begin try 
           let i = Py_mtypes.VarMap.find name name_to_index in
           vars.(i) <- value
         with Not_found -> ()
         end
         | _ -> ()
       end

    (* get underlying module *)
    method get_module = super#get_module

    (* get an attribute *)
    method get_attr k =
      match k with 
      | PyString name -> 
        if Py_mtypes.VarMap.mem name name_to_index
        then Some vars.(Py_mtypes.VarMap.find name name_to_index)
        else if dict#has_item k
        then dict#get_item k
        else super#get_attr k
      | _ -> 
        if dict#has_item k
        then dict#get_item k
        else super#get_attr k

    (* set an attribute *)
    (* NOTE: we don't bother to keep the dictionary synched with
       indexed access. This means 'locals' will fail to show the
       correct dictionary, as in Python. The reason we don't,
       is that changes to the dictionary will not be reflected
       in the fast load array anyhow. This could be fixed,
       by storing to the dictionary as well as the array,
       and also by returning a n object attached to the
       array, instead of a dictionary (but with the same interface
     *)
    method set_attr k v = 
      match k with 
      | PyString name ->
        if List.mem name globals 
        then super#set_attr k v
        else if Py_mtypes.VarMap.mem name name_to_index
        then begin 
          vars.(Py_mtypes.VarMap.find name name_to_index) <- v; 
          true
        end
        else dict#set_item k v;
      | _ -> dict#set_item k v;

    method del_attr k = 
      match k with
      | PyString name ->
        if List.mem name globals 
        then super#del_attr k
        else if Py_mtypes.VarMap.mem name name_to_index
        then begin 
          vars.(Py_mtypes.VarMap.find name name_to_index) <- PyInitial;
          true
        end
        else dict # del_item k
      | _ -> dict # del_item k


    (* Python compatible has_attr *)
    method has_attr k = 
      match self#get_attr k with 
      | Some _ -> true 
      | None -> false

    (* test which methods will work *)
    method can_get_attr k = self#has_attr k
    method can_set_attr (k:expr_t) (v:expr_t) = true
    method can_set_any_attr (k:expr_t) = true
    method can_del_attr (k:expr_t) = true (* a lie for the moment *)
  
    (* Do indexed addressing *)
    method get_indexed level i = 
      if level = 0 
      then vars.(i)
      else super#get_indexed (level-1) i

    method set_indexed level i (v:expr_t) = 
      if level = 0
      then vars.(i) <- v
      else super#set_indexed (level-1) i v

    method del_indexed level i =
      if level = 0
      then vars.(i) <- PyInitial
      else super#del_indexed level i

    (* retrieve locals and globals dictionaries *)
    method get_locals = dict
    method get_globals: dictionary_t = raise (Failure "Globals of function environment not implemented")

    method keys = dict#keys
    
  end


