open Py_types
open Py_exceptions
open Py_builtins_util

(* FILE METHODS *)
(*
match k with
      | "closed" -> PyInt (int_of_bool closed)
      | "name" -> PyString filename
      | "mode" -> PyString mode
      | "softspace" -> PyInt (int_of_bool softspace)
      | "close" -> PyNativeFunction py_file_close

      | "flush" -> PyNativeFunction py_file_flush
      | "isatty" -> PyNativeFunction py_file_isatty
      | "fileno" -> PyNativeFunction py_file_fileno
      | "read" -> PyNativeFunction py_file_read
      | "readline" -> PyNativeFunction py_file_readline
      | "readlines" -> PyNativeFunction py_file_readlines
      | "seek" -> PyNativeFunction py_file_seek
      | "tell" -> PyNativeFunction py_file_tell
      | "truncate" -> PyNativeFunction py_file_truncate
      | "write" -> PyNativeFunction py_file_write
      | "writelines" -> PyNativeFunction py_file_writelines

      | _ -> raise (AttributeError ("file object has no attribute '" ^ k ^ "'"))
*) 

let py_file_open
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atleast e 1;
  atmost e 2;
  match arg e 0 with
  | PyString filename ->
    let modestring = 
     match optional e 1 with 
     | None -> "r"
     | Some (PyString s) -> s
     | _ -> raise (TypeError "Optional mode to open function must be a string")
    in
    let mode = match modestring with
    | "r" -> [Unix.O_RDONLY]
    | "w" -> [Unix.O_WRONLY; Unix.O_CREAT; Unix.O_TRUNC]
    | "rw"
    | "wr" -> [Unix.O_RDWR; Unix.O_CREAT]
    | "a" -> [Unix.O_APPEND; Unix.O_CREAT]
    | _ -> raise (ValueError "mode of open must be r,w,rw,wr or a")
    in begin try 
      PyFile (Unix.openfile filename mode 0x1ff) (* octal 777 *)
     with x -> 
     (* print_endline ("File_open: failed, " ^ filename ^ ", mode=" ^ modestring); 
     print_endline (Printexc.to_string x);
     flush stdout;
     *)
     raise x
     end
  | _ -> 
    raise (TypeError "file open function requires string for filename")
;;

let py_file_read
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let f = match arg e 0 with 
    | PyFile f' -> f' 
    | _ -> raise (TypeError "file_read requires file argument")
  in let maxread = 
    match arg e 1 with
    | PyInt i -> 
      if i > 0 
      then i 
      else raise (ValueError "file_read arg 2 must be positive integer")
    | _ -> raise (TypeError "file_read argument 2 must be positive integer")
  in 
    let buffer = String.create maxread in
    let len = Unix.read f buffer 0 maxread in
    PyString (String.sub buffer 0 len)
;;

let py_file_write
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let f = match arg e 0 with 
    | PyFile f' -> f' 
    | _ -> raise (TypeError "file_write requires file argument 1")
  in let s = 
    match arg e 1 with
    | PyString s -> s
    | _ -> raise (TypeError "file.write method argument 2 must be string")
  in let n = String.length s 
  in
    if (Unix.write f s 0 n) <> n
    then raise (SystemError "File write failed? [Expected exception]")
    else PyNone
;;
  
let py_file_close
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let f = arg e 0 in
  begin match f with
  | PyFile f' -> Unix.close f'
  | _ -> 
    raise (TypeError "file close function must be bound to file object")
  end;
  PyNone
;;

  
(* does this work? Probably not :-( *)
let py_file_flush 
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let f = arg e 0 in
  begin match f with
  | PyFile f' -> Unix.tcdrain f' 
  | _ -> 
    raise (TypeError "file flush function must be bound to file object")
  end;
  PyNone
;;

let py_file_get_std_files
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  PyTuple [PyFile Unix.stdin; PyFile Unix.stdout; PyFile Unix.stderr]
;;

