open Py_types
open Py_mtypes
open Py_exceptions
open Py_util
open Util

exception Done
exception NoError
exception Handled

(* ------------------------------------------------------------------*)

(* RAW getattr function *)
(* attr must be evaluated *)
let rec _get_attribute interp obj attr =
  match attr with
  | PyString "__typename__" -> Some (PyString (Py_datum.get_typename obj))
  | _ -> 
    match Py_datum.py_get_attribute obj attr with
    | Some x -> Some x
    | None ->
      let type_object = interp#type_of_object obj in
      if type_object <> obj
      then begin match _get_attribute interp type_object attr with
      | Some value ->
        Some (
          if Py_functions.callable value 
          then PyBoundMethod (value, obj)
          else value
        )
      | None -> None
      end 
      else None

(* enhanced getattr, uses __getattr__ hook *)
(* attr must be evaluated *)
and enhanced_get_attribute interp obj attr =
  match _get_attribute interp obj attr with
  | Some x -> Some x
  | None ->
    match _get_attribute interp obj (PyString "__getattr__") with
    | Some x ->
      if Py_functions.callable x
      then Some (py_call interp x [Argument1 attr])
      else raise (TypeError "__getattr__ must be callable")
    | None -> None

(* client getattr, throws exception if not found *)
and get_attribute interp obj attr =
  match enhanced_get_attribute interp obj attr with
  | Some x -> x
  | None ->
    raise (AttributeError 
      (("Cannot find attribute of " ^ (Py_functions.repr obj)), attr)
    )

and py_istrue interp obj = 
  match obj with
  | PyInstance i ->
    begin try 
      let get_nonzero = get_attribute interp obj (PyString "__nonzero__") in
      if Py_functions.callable get_nonzero
      then 
        let result = py_call interp get_nonzero [] 
        in Py_datum.py_istrue result
      else raise (TypeError ("__nonzero__ method must be callable"))
    with _ -> true 
    end
  | _ -> Py_datum.py_istrue obj

and py_seq_len interp seq = 
  match seq with
  | PyInstance i ->
    let get_len_method =
      try get_attribute interp seq (PyString "__len__")
      with _ -> raise (ValueError "len of instance requires __len__ method") 
    in
      if Py_functions.callable get_len_method
      then py_call interp get_len_method []
      else raise (TypeError ("__len__ method must be callable"))

  | _ -> PyInt (Py_datum.py_seq_len seq)

and py_get_seq_elem interp seq item = 
    match seq with
    | PyInstance i ->
      let get_item_method =
        try get_attribute interp seq (PyString "__getitem__")
        with _ -> raise (ValueError "getitem of instance requires __getitem__ method") 
      in
        if Py_functions.callable get_item_method
        then py_call interp get_item_method [Argument1 item]
        else raise (TypeError ("__getitem__ method must be callable"))
    | _ -> 
      begin match item with
      | PyInt j -> Py_datum.py_get_seq_elem seq j
      | _ -> raise (TypeError "sequence position must be integer")
      end

(* convert [b:e] thing to start,end, where end is one past the end,
  and the range is in the span of the list *)
and resolve_range eval n start finish = 
  let first =
    match start with
    | Pos key -> 
      begin  match eval key with 
      | PyInt j -> if j< 0 then n + j else j
      | _ -> raise (TypeError "Explicit Slice start must be integer")
      end
    | Defsub -> 0
  and last = 
    match finish with
    | Pos key -> 
      begin  match eval key with 
      | PyInt j -> if j< 0 then n + j else j
      | _ -> raise (TypeError "Explicit Slice end must be integer")
      end
    | Defsub -> n
  in ((max 0 first), (min n last))

(* seq is evaluated, start and finish aren't *)
and full_get_slice interp eval seq start finish =
  let len = 
    try 
      match py_seq_len interp seq with 
      | PyInt i -> i
      | _ -> max_int 
    with _ -> max_int 
  in let first, last = 
    resolve_range eval len start finish in
  match seq with
  | PyInstance i ->
    let get_slice_method =
      try get_attribute interp seq (PyString "__getslice__")
      with _ -> raise (TypeError "__getslice__ method required for slicing instance")
    in
      if Py_functions.callable get_slice_method
      then py_call interp  
        get_slice_method [Argument1 (PyInt first); Argument1 (PyInt last)]
      else raise (TypeError "__getslice__ method must be callable")
  | _ -> 
    Py_datum.py_get_seq_slice seq first last

(* ------------------------------------------------------------------*)
and py_exec_wrap_exn interp env s =
  try py_exec interp env s
  with 
  | PyException x -> raise (PyException x) (* transmit untouched *)
  | x -> interp#py_raise x

(* ------------------------------------------------------------------*)
(* GENERAL RULE FOR EVALUATOR: for a function implementing
   a binary operator, the left hand argument is evaluated by the caller,
   while the right hand argument is evaluated by the function.
   
   This rule is broken in a few places! See Notes.

   Note: it is approaching time to use the ocaml type system to
   separate terms from run time objects. They STILL need to be mixed
   to support partial evaluation though.... so perhaps I won't do it
   after all :-)
*)
(* ------------------------------------------------------------------*)
and py_exec 
  (interp: interpreter_t) 
  (env: environment_t) 
  (s:statement_t) : unit =
  let eval = py_eval interp env 
  and istrue = py_istrue interp
  in
  let rec asgn (env:environment_t) expr variable : unit =
    match variable with
    | PyName name -> ignore (env#set_attr (PyString name) expr)
    | PyVarIndex (level, i) -> env#set_indexed level i expr
    | PyTuple _ 
    | PyList _ -> 
      begin let lhs = list_of_lvalue_sequence variable in
         let rhs = 
           try list_of_sequence expr 
           with NonSequence ->
             raise (TypeError "Sequence RHS required in assign")
         in 
           if (List.length rhs) <> (List.length lhs)
           then raise (TypeError "RHS sequence wrong length in assign")
           else
             List.iter
             (fun (value, variable) -> asgn env value variable)
             (List.combine rhs lhs)
      end
    
    (* This is the NASTY case: we could be assigning to a varible
       in another dictionary like  
         x.a = 1
       OR to an element of a list like
         x[i] = 1
       
       In these cases, x can be an expression, which must resolve
       to a module (or class instance) or a list or dictionary,
       respectively

       The cases can be recognized by examining the last trailer.
    *)
    | AtomWithTrailers (atom,trailers) ->
      let leader = list_all_but_last trailers 
      and trailer = list_last trailers in
      begin match trailer with 
      | Arglist _ -> raise (TypeError "Assign to function call")
      | Dotname name -> (* first resolve the leading trailers *)
        let obj = eval (AtomWithTrailers (atom, leader)) in
        begin match obj with
        | PyInstance i -> 
          begin try 
            let setattr_method = 
              try get_attribute interp obj (PyString "__setattr__")
              with _ -> raise Not_found
            in 
              if Py_functions.callable setattr_method
              then ignore (
                py_call interp setattr_method [Argument1 (PyString name); Argument1 expr]
              )
              else raise (TypeError "__setattr__ method must be callable")
          with 
          | Not_found -> ignore (i#set_attr (PyString name) expr)
          end 
        | PyClass c -> ignore (c#get_dictionary#set_item (PyString name) expr)
        | PyModule m -> ignore (m#get_dictionary#set_item (PyString name) expr)
        | _ ->  raise (NotImplemented "In o.a=v, o must be instance, class or module")
        end

      | Sublist subs ->
        let obj = eval (AtomWithTrailers (atom, leader)) in
        begin match obj with
        | PyInstance i -> 
          begin match subs with
          | h::[] -> 
            begin match h with
            | Subscript0 (Pos key) -> 
              let setitem_method = 
                try get_attribute interp obj (PyString "__setitem__")
                with _ -> raise (ValueError "__setitem__ method required for instance[x]=y")
              in 
                if Py_functions.callable setitem_method
                then ignore (
                  py_call interp setitem_method [Argument1 (eval key); Argument1 expr]
                )
                else raise (TypeError "__setitem__ method must be callable")
            | Subscript0 Defsub  -> raise (ViperError "Default single subscript?")
            | Subscript1 (start, finish) -> 
              let setslice_method = 
                try get_attribute interp obj (PyString "__setslice__")
                with _ -> raise (ValueError "__setslice__ method required for instance[x:y]=y")
              in 
                if Py_functions.callable setslice_method
                then begin
                  let first = 
                    match start with
                    | Pos k -> eval k
                    | Defsub -> PyNone
                  and last = 
                    match finish with
                    | Pos k -> eval k
                    | Defsub -> PyNone
                  in ignore (
                    py_call interp setslice_method [Argument1 first; Argument1 last; Argument1 expr]
                  )
                end 
                else raise (TypeError "__setslice__ method must be callable")
            | Subscript2 (_,_,_) -> 
              raise (NotImplemented "Assign to instance with slice with stride a:b:c") 
            | Ellipsis -> raise (NotImplemented "instance[...]=something")
           end
          | _ -> raise (NotImplemented "Assign to instance with multiple subscripts [x,y]") 
          end

        | PyDictionary d -> 
          begin match subs with
          | h::[] -> 
            begin match h with
            | Subscript0 (Pos key) -> ignore (d#set_item (eval key) expr)
            | _ -> raise (NotImplemented "Assign to dictionary with slice notation")
            end
          | _ -> raise (NotImplemented "Assign to dictionary with multiple subscripts") 
          end
        | PyMutableList ls -> 
          begin match subs with
          | h::[] -> 
            begin match h with
            | Subscript0 Defsub -> 
              raise (ViperError "Default single subscript?")

            | Subscript0 (Pos key) -> 
              begin match eval key with
              | PyInt i' -> let n = Varray.length ls in
                let i = if i' < 0 then n + i' else i'
                in if i<0 or i>= n then raise (IndexError ("List index",i,n))
                else Varray.set ls i expr
              | _ -> raise (TypeError "Sequence index must be integer")
              end
            | Subscript1 (start, finish) -> 
              let x = 
                try 
                  match expr with
                  | PyMutableList e' -> e'
                  | _ -> Varray.of_list (Py_util.list_of_sequence expr)
                with NonSequence -> raise (TypeError "x[a:b] = e requires e sequence")
              in let n = Varray.length ls 
              in let (first, last) = resolve_range eval n start finish
              in let m = Varray.length x 
              in let final_length = n - (last - first) + m
              in let result = Array.create final_length PyInitial 
              in
                Array.blit (Varray.get_storage ls) 0 result 0 first;
                Array.blit (Varray.get_storage x) 0 result first m;
                Array.blit (Varray.get_storage ls) last result (first + m) (n - last);
                Varray.assign ls (Varray.ref_array result)
            | Subscript2 (_,_,_) -> 
              raise (NotImplemented "Assign to list with extended slice notation [f:l:s]")
            | Ellipsis -> 
              raise (NotImplemented "Assign to list with ellipsis ...")
            end
          | _ -> raise (NotImplemented "Assign to list with multiple subscripts [x,y]") 
          end
        | _ -> raise (NotImplemented "Assign to atom with subscripts [x,y]; atom must be dictionary or list")
        end
      end
    | _ -> raise (TypeError "Var or Tuple of vars required on LHS of assign")
  in

  (* main routine *)
  let rec exec s : unit =
    begin match s with
    | Empty -> ()
    | While (sr, e, s1, s2)      -> 
      interp#set_line sr;
      begin try
        while istrue (eval e) do
          try exec s1
          with Continuer -> ()
        done;
        raise Done
      with 
        | Breaker -> ()
        | Done -> exec s2
      end

    | For (sr,variables, sequence, s1, s2)  ->
      interp#set_line sr;
      let s = eval sequence in
      begin try
       let n = 
         match py_seq_len interp s with 
         | PyInt i -> i 
         | _ -> raise (ViperError "py_seq_len should return integer") 
       in for i = 0 to n - 1 do
         interp#set_line sr;
         let elem = py_get_seq_elem interp s (PyInt i) in
         asgn env elem variables;
         try exec s1 with Continuer -> ()
       done;
       raise Done
      with 
      | Breaker -> ()
      | Done -> exec s2
      end

    | Def (sr,name,ps,st) -> 
      interp#set_line sr;
      
      let ps' = Py_param.get_parameters_from_parameters eval ps in 
      let glbs = ref [] in
      let rec scan_for_globals statement =
        match statement with
        | Suite ss -> List.iter scan_for_globals ss
        | Global (_, names) ->
          List.iter
          begin fun global_name -> glbs := global_name :: !glbs end
          names
         | _ -> ()
      in 
        scan_for_globals st;
        let i2n = Array.create 0 "" 
        and n2i = VarMap.empty in
        let f = new Py_function.py_function name ps' st !glbs i2n n2i env in
        ignore (env#set_attr (PyString name) (PyFunction (f:>function_t)))

    | Class (sr,name, base_expr, suite) -> 
      interp#set_line sr;

      (* get the bases *)
      let bases = match eval base_expr with
      | PyClass cls -> [cls]
      | PyTuple bases -> 
        List.map
        begin fun x -> match x with 
        | PyClass abase -> (abase :> class_t)
        | _ -> raise (TypeError "Base must be a class")
        end
        bases
      | PyNone -> []
      | _ -> raise (TypeError "Bases must be a tuple or None")
      in

      (* create class object *)
      let i2n = Array.create 0 "" 
      and n2i = VarMap.empty
      and __dict__ = new Py_dict.py_dictionary in
      let theclass = new Py_class.py_class name bases __dict__ i2n n2i env in
      let class_object = PyClass (theclass :> class_t) in

      (* add the class object to the current environment *)
       ignore (env # set_attr (PyString name) class_object);

      (* construct temporary class environment *)
      let class_env = new Py_class.py_class_environment (theclass:>class_t) in
      let class_env' = (class_env :> environment_t) in

      (* execute the body in the class environment *)
      py_exec interp class_env' suite;

    | TryFinally ( s1, s2)  -> 
      begin try py_exec_wrap_exn interp env s1
      with PyException x -> exec s2; raise (PyException x) 
      end;
      exec s2
      
    | TryElse ( s1, hl, s2) -> 
      begin try
        try 
          py_exec_wrap_exn interp env s1;
          raise NoError 
        with PyException x ->
          List.iter
          begin fun xclause -> match xclause with
          | (Except1 (sr, e), handler) ->  
            interp#set_line sr;
            if xcompat x (eval e) then (exec handler; raise Handled)

          | (Except2 (sr, e, target), handler) -> 
            interp#set_line sr;
            if xcompat x (eval e) then begin
              let varname = begin match target with 
              | PyName name -> name
              | _ -> raise (TypeError "Identifier required for exception handler target")
              end 
              in begin match x with 
              | PyStringException (s1, e2) ->
                let env' = new Py_env.py_handler_environment varname e2 env in
                let env'' = (env' :> environment_t) in
                py_exec interp env'' handler; 
                raise Handled

              | PyInstanceException i ->
                let env' = new Py_env.py_handler_environment varname (PyInstance i) env in
                let env'' = (env' :> environment_t) in
                py_exec interp env'' handler; 
                raise Handled
              end
            end
          | (Except0, handler) -> 
            exec handler; raise Handled
          end
          hl;
          raise (PyException x) (* not handled *)
      with 
      | Handled -> interp#clear_exc_info
      | NoError -> exec s2
      end

    | Global _ -> ()
    | Break sr -> interp#set_line sr; raise Breaker
    | Continue sr -> interp#set_line sr; raise Continuer
    | Return (sr,e) -> interp#set_line sr; raise (Returner (eval e))
    | Raise0 sr -> 
      interp#set_line sr;
      let exc = interp # get_exc_info in
      let (cls, value, tr) =
      begin match exc with 
      | PyTuple [cls; value; tr] -> (cls, value, tr)
      | _ -> raise (SystemError "exc_info not tuple")
      end in
      if cls = PyNone then raise (TypeError "Raise outside handler");
      begin match cls with 
      | PyClass x -> 
        begin match value with 
        | PyInstance i -> raise (PyException (PyInstanceException i))
        | _ -> raise (SystemError "Raise0: Class exception value not instance")
        end
      | PyString s ->
        begin match value with 
        | PyString s2 -> raise (PyException (PyStringException (s,value)))
        | _ -> raise (SystemError "Raise0: String exception value not string")
        end
      | _ -> raise (SystemError "Raise0: Exception not class or string")
      end

    | Raise1 (sr, x) -> 
      interp#set_line sr;
      begin match eval x with 
      | PyString s -> 
        interp # set_exc_info (PyString s) (PyString "") (PyTraceback interp#get_traceback);
        raise (PyException (PyStringException (s, (PyString ""))))
      | PyClass cls -> 
        let i = create_class_instance interp cls [] in
         interp # set_exc_info (PyClass cls) (PyInstance i) (PyTraceback interp#get_traceback);
        raise (PyException (PyInstanceException i))
      | PyInstance i -> 
        interp # set_exc_info (PyClass i#get_class) (PyInstance i) (PyTraceback interp#get_traceback);
        raise (PyException (PyInstanceException i))
      | _ -> raise (TypeError "Raise1: argument must be string, class or instance")
      end

    | Raise2 (sr,x1,x2)-> 
      interp#set_line sr;
      let e1 = eval x1 and e2 = eval x2 in
      begin match e1 with
      | PyClass cls ->
        begin match e2 with
        | PyInstance i when isbaseof cls i#get_class -> 
          interp # set_exc_info e1 e2 (PyTraceback interp#get_traceback);
          raise (PyException (PyInstanceException i))
        | _ -> ()
        end;
        let arglist1 = match e2 with 
        | PyTuple ls -> ls
        | PyNone -> []
        | x -> [x]
        in let arglist2 = List.map (fun x -> Argument1 x) arglist1 in
        let i = create_class_instance interp cls arglist2 in
        interp # set_exc_info (PyClass i#get_class) (PyInstance i) (PyTraceback interp#get_traceback);
        raise (PyException (PyInstanceException i))
      | PyString s -> 
        interp # set_exc_info (PyString s) (PyString "") (PyTraceback interp#get_traceback);
        raise (PyException (PyStringException (s, e2))) 
      | _ -> raise (TypeError "Raise2: first arg must be string or class") 
      end

    | Raise3 (sr,x1,x2,tb) -> interp#set_line sr; raise (NotImplemented "Raise3")

    | Print (sr,ls) ->
      interp#set_line sr;
      List.iter
      (fun x -> print_string ((Py_functions.str (eval x)) ^ " "))
      ls;
      print_newline() (* flushes stdout *)

    | PrintComma (sr, ls) ->
      interp#set_line sr;
      List.iter
      (fun x -> print_string ((Py_functions.str (eval x)) ^ " "))
      ls
    | Pass -> ()

    | Exec1 (sr,s) -> 
      interp#set_line sr;
      begin match eval s with 
      | PyString s' -> 
        Py_exec_module.exec_string interp exec (s' ^ "\n") "<string>"
      | PyStatement s' ->
        exec s'
      | _ -> raise (TypeError "Argument to exec must be a string or statement")
      end

    | Exec2 (sr,s,g) -> 
      interp#set_line sr;
      let env = match eval g with 
      | PyDictionary d -> 
        new Py_env.py_python_environment 
        interp#get_builtins_dictionary d d
      | PyEnv env -> env
      | _ -> raise (TypeError "Exec2 second argument must be dictionary or environment")
      in let exec = py_exec interp env in
      begin match eval s with 
      | PyString s' -> 
        Py_exec_module.exec_string interp exec (s' ^ "\n") "<string>"
      | PyStatement s' ->
        py_exec interp env s'
      | _ -> raise (TypeError "Argument to exec must be a string or statement")
      end

    | Exec3 (sr,s,g,l) -> 
      interp#set_line sr;
      let globals = match eval g with 
      | PyDictionary d -> d 
      | _ -> raise (TypeError "Exec3 second argument must be globals dictionary")
      and locals = match eval l with 
      | PyDictionary d -> d 
      | _ -> raise (TypeError "Exec3 third argument must be locals dictionary")
      in let env = new Py_env.py_python_environment 
        interp#get_builtins_dictionary
        globals
        locals
      in let exec = py_exec interp env in
      begin match eval s with 
      | PyString s' -> 
        Py_exec_module.exec_string interp exec (s' ^ "\n") "<string>"
      | PyStatement s' ->
        py_exec interp env s'
      | _ -> raise (TypeError "Argument to exec must be a string or statement")
      end

    | Assert2 (sr,e1,e2) -> 
      interp#set_line sr;
      let e1' = eval e1 in
      if not (istrue(e1'))
      then exec (Raise2 (sr,(PyName "AssertionError"), e2))

    | Assert1 (sr,e) -> 
      interp#set_line sr;
      if not (istrue(eval e))
      then raise (AssertionError (
        "Expression " ^ (Py_functions.repr e) ^
        ", In Line " ^ (string_of_int (fst sr))  ^
        ", File " ^ (snd sr)
        ))
      else ()

    | Suite ( ss1 ) -> List.iter  exec ss1


    | Import (sr,module_names) -> 
      interp#set_line sr;
      let modul = env#get_module in
      List.iter (* for each module being imported *)
      begin fun x -> ignore (
        let top,leaf = interp#import modul x in
        env#set_attr (PyString (List.hd x)) (PyModule top)
      )
      end
      module_names

    | ImportFrom (sr,mpath,namelist) ->
      interp#set_line sr;
      let modul = env#get_module in
      let top,leaf = interp#import modul mpath in
      leaf#get_dictionary#iter
        begin fun k v -> 
          match k with 
          | PyString k' -> 
            if List.mem k' namelist then ignore (env#set_attr k v)
          | _ -> ()
         end 

    | ImportAll (sr,mpath)  ->
      interp#set_line sr;
      let modul = env#get_module in
      let top,leaf = interp#import modul mpath in
      leaf#get_dictionary#iter
        begin fun k v -> 
          match k with
          | PyString s when (String.length s) > 1 && (s.[0] = '_') -> ()
          | _ -> ignore (env#set_attr k v)
        end

    | If (ess, s2)  -> 
      begin try
        List.iter 
        begin fun (sr, c,s) -> 
          interp#set_line sr;
          if istrue (eval c) 
          then begin
            exec s; 
            raise Done
          end
        end 
        ess; 
        exec s2 
      with Done -> ()
      end
    | Assign ( sr, l ) ->
      interp#set_line sr;
      let n = List.length l in 
      let rhs = List.nth l (n-1) in
      let value = eval rhs in
      List.iter (asgn env value) (list_sub l 0 (n-1))

    | PlusEqual (sr,v,e) -> asgn env (Py_datum.py_add (eval v) (eval e)) v 
    | MinusEqual (sr,v,e) -> asgn env (Py_datum.py_sub (eval v) (eval e)) v 
    | StarEqual (sr,v,e) -> asgn env (Py_datum.py_mul (eval v) (eval e)) v 
    | SlashEqual (sr,v,e) -> asgn env (Py_datum.py_div (eval v) (eval e)) v 
    | PercentEqual (sr,v,e) -> asgn env (Py_datum.py_mod (eval v) (eval e)) v 
    | AmperEqual (sr,v,e) -> 
      let e' = 
        match eval e with 
        | PyInt i -> i 
        | _ -> raise (TypeError "&= requires int rhs")
       in asgn env (PyInt (Py_datum.py_bit_and e' (eval v))) v 
    | VbarEqual (sr,v,e) -> 
      let e' = 
        match eval e with 
        | PyInt i -> i 
        | _ -> raise (TypeError "&= requires int rhs")
       in asgn env (PyInt (Py_datum.py_bit_or e' (eval v))) v 

    | CaretEqual (sr,v,e) -> 
      let e' = 
        match eval e with 
        | PyInt i -> i 
        | _ -> raise (TypeError "&= requires int rhs")
       in asgn env (PyInt (Py_datum.py_bit_xor e' (eval v))) v 

    | ColonEqual (sr,v,e) -> asgn env (eval e) v (* same as = *)
    | LeftShiftEqual (sr,v,e) -> asgn env (Py_datum.py_lsl (eval v) (eval e)) v 
    | RightShiftEqual (sr,v,e) -> asgn env (Py_datum.py_lsr (eval v) (eval e)) v 
    | PlusPlus (sr,v) -> asgn env (Py_datum.py_add (eval v) (PyInt 1)) v 
    | MinusMinus (sr,v) -> asgn env (Py_datum.py_sub (eval v) (PyInt 1)) v 
   
    | Expr (sr, e) -> 
      interp#set_line sr;
      ignore (eval e)
    | Del ( sr, tl ) ->
      interp#set_line sr;
      let rec delvars env x = match x with
      | PyName name -> ignore (env # del_attr (PyString name)) (* execute destructors? *)
      | PyTuple ls -> List.iter (delvars env) ls
      | PyList ls -> List.iter (delvars env) ls
      | AtomWithTrailers (atom, trailers) -> 
        let leader = list_all_but_last trailers 
        and trailer = list_last trailers in
        begin match trailer with 
        | Arglist _ -> raise (TypeError "Del function call")
        | Dotname name -> (* first resolve the leading trailers *)
          let obj = eval (AtomWithTrailers (atom, leader)) in
          begin match obj with
          | PyInstance i -> ignore (i#del_attr (PyString name))
          | PyClass c -> ignore (c#get_dictionary#del_item (PyString name))
          | PyModule m -> ignore (m#get_dictionary#del_item (PyString name))
          | _ ->  raise (NotImplemented "In del o.a, o must be instance, class or module")
          end
        | Sublist subs ->
          let obj = eval (AtomWithTrailers (atom, leader)) in
          begin match obj with
          | PyDictionary d -> 
            begin match subs with
            | h::[] -> 
              begin match h with
              | Subscript0 (Pos key) -> ignore (d#del_item key)
              | _ -> raise (NotImplemented "Del dictionary with slice notation")
              end
            | _ -> raise (NotImplemented "Del to dictionary with multiple subscripts") 
            end
          | PyMutableList ls -> 
            begin match subs with
            | h::[] -> 
              begin match h with
              | Subscript0 (Pos key) -> 
                begin match eval key with
                | PyInt i' -> let n = Varray.length ls in
                  let i = if i' < 0 then n + i' else i'
                  in if i<0 or i>= n 
                  then raise (IndexError ("List index",i,n))
                  else Varray.del ls i
                  
                | _ -> raise (TypeError "Sequence index must be integer")
                end
              | Subscript1 (start, finish) -> 
                 let n = Varray.length ls 
                 in let (first, last) = resolve_range eval n start finish
                 in Varray.del_range ls first (last - first)
              | _ -> raise (NotImplemented "Del list with slice notation")
              end
            | _ -> raise (NotImplemented "Del list with multiple subscripts") 
            end
          | _ -> raise (NotImplemented "Del to atom with subscripts; atom must be dictionary or list")
          end
        end
      | _ -> raise (TypeError "Var or Tuple of vars required on RHS of del")
      in delvars env tl
    (* | _ -> raise (SystemError "Unknown statement") *)
    end
  in exec s


(* ------------------------------------------------------------------*)
and py_eval 
  (interp:interpreter_t) 
  (env:environment_t) 
  (expr:expr_t)  
  : expr_t =
  let map = List.map 
  in let fold_left = List.fold_left 
  in let eval: expr_t -> expr_t = py_eval interp env
  in let call = py_call interp
  in let get (name:expr_t) = 
    match env # get_attr name with
    | Some x -> x
    | None -> raise (NameError ("Fetching",name))
  in let iget level i = env # get_indexed level i
  in let rec py_evaluate x op = 
    match op with
    | Add y  -> Py_datum.py_add x (eval y)
    | Sub y  -> Py_datum.py_sub x (eval y)
    | Mul y  -> Py_datum.py_mul x (eval y)
    | Div y  -> Py_datum.py_div x (eval y)
    | Mod y  -> Py_datum.py_mod x (eval y)
    | Asl y  -> Py_datum.py_lsl x (eval y)
    | Lsr y  -> Py_datum.py_lsr x (eval y)
    | Pow y  -> Py_datum.py_pow x (eval y)

  and py_comparison arg op = 
    match op with 
    | Less e -> let e' = eval e in (e',Py_datum.py_less arg e')
    | LessEqual e -> let e' = eval e in (e',Py_datum.py_less_equal arg e')
    | Equal e -> let e' = eval e in (e',Py_datum.py_equal arg e')
    | GreaterEqual e -> let e' = eval e in (e',Py_datum.py_greater_equal arg e')
    | Greater e -> let e' = eval e in (e',Py_datum.py_greater arg e')
    | NotEqual e -> let e' = eval e in (e',Py_datum.py_not_equal arg e')
    | In e -> let e' = eval e in (e',Py_datum.py_in arg (eval e))
    | NotIn e -> let e' = eval e in (e',Py_datum.py_not_in arg e')
    | Is e -> let e' = eval e in (e',Py_datum.py_is arg e')
    | IsNot e -> let e' = eval e in (e',not (Py_datum.py_is arg e'))
 
  (* evaluate python comparison expression, return python value *)
  and py_chain_compare initial clist = 
    let  left = ref initial
    and result = ref true
    and li = ref clist in
    while !result && (List.length !li) > 0 do
      let (e,b) = py_comparison !left (List.hd !li) in
      result := b;
      li := List.tl !li;
      left := e
    done;
    !result

  and convert_subscript 
    (default:int) 
    (bound:int) 
    (value:subscript_entry) 
  : int =
    match value with
    | Defsub -> default
    | Pos x -> 
      match eval x with
      | PyInt y -> if y < 0 && bound >=0 then bound + y else y
      | _ -> raise (TypeError "Subscript must be integer")
  
  and get_sublist li start finish stride = (* used in get_slice *)
    let result = ref [] 
    and n = List.length li 
    in let start' = ref (convert_subscript 0 n start)
    and finish' = ref (convert_subscript n n finish)
    and stride' = convert_subscript 1 (-1) stride
    in 
      if stride' < 0 
      or !start' >= !finish' 
      or !start' >= n
      or !finish' <= 0
      then []
      else begin
        finish' := min !finish' n;
        if !start' < 0 then start' := modulo stride' (- !start');
        if stride' = 1 && !start' <= 0 && !finish' >=n 
        then li
        else (* general calculation *)
          let index = ref !start' in
          while !index < !finish' do
            if !index >= 0 && !index < n 
            then result := !result @ [List.nth li !index]
            else ();
            index := !index + stride'
          done;
          !result
      end

  and get_substring (li:string) start finish stride : string =  (* used in get_slice *)
    let n = String.length li 
    in let start' = ref (convert_subscript 0 n start)
    and finish' = ref (convert_subscript n n finish)
    and stride' = convert_subscript 1 (-1) stride
    in 
      if stride' < 0 
      or !start' >= !finish' 
      or !start' >= n
      or !finish' <= 0
      then "" 
      else begin
        finish' := min !finish' n;
        if !start' < 0 then start' := modulo stride' (- !start');
        if stride' = 1 && !start' <= 0 && !finish' >=n 
        then li
        else 
          if stride' = 1 && 0 < !start' && !start' < !finish'
          then String.sub li !start' (!finish' - !start')
          else 
            let index = ref !start'
            and result = ref "" in
            while !index < !finish' do
              if !index >= 0 && !index < n 
              then result := !result ^ (String.sub li !index 1)
              else ();
              index := !index + stride'
            done;
            !result
      end

  and py_get_subrange start stop step start' stop' step' = (* used in get_slice *)
    let n = max ((stop - start) / step) 0 in
    let first = convert_subscript 0 n start' in
    let last = convert_subscript n n stop' in
    let stride = convert_subscript 1 (-1) step' in
      let start'' = start + step * first
      and stop'' = start + step * last
      and step'' = step * stride 
      in let n' = max ((stop'' - start'') / step'') 0
      in let stop''' = start'' + step'' * n' 
      in IntRange (start'', stop''', step'')
      
  and get_slice atom sub = (* used in get_list_slice *)
    match sub with
    | Ellipsis -> raise (NotImplemented "Ellipsis not understood")
    | Subscript2 (b',e',s') -> 
      begin match atom with
      | PyTuple li -> PyTuple (get_sublist li b' e' s')
      | PyMutableList ar -> 
        let li = Varray.to_list ar in
        let li' = get_sublist li b' e' s' in
        let ar' = Varray.of_list li' in
        PyMutableList ar'
      | PyString li -> PyString (get_substring li b' e' s')
      | IntRange (start,stop,step) -> 
        py_get_subrange start stop step b' e' s'
      | _ -> raise (TypeError "Slice of non-sequence")
      end
    | Subscript1 (b',e') -> full_get_slice interp eval atom b' e'
    | Subscript0 i -> raise (ViperError "Subscript in slice code??")

  (* THIS CODE FINDS object[item] for all cases except object=dictionary *)
  and get_subscript atom i' = (* used by get_list_slice *)
    match i' with
    | Defsub -> raise (SystemError "Parser Error: Default single subscript")
    | Pos j -> py_get_seq_elem interp atom (eval j)

  (* slicing and subscription rules: NASTY!
     If the subscript list consists of a single Subscript0,
     the result is an element of the sequence, except for strings,
     where the result is a one character string.

     Otherwise, the result is a sequence of the same
     kind as the atom: each entry is a slice, even if it
     is a Subscript0.
  *)   
  and get_list_slice atom li = (* used in py_eval_trailer *)
    match li with 
    | [] -> raise (TypeError "Empty subscript list?")
    | [Subscript0 (Pos j)] -> get_subscript atom (Pos j) 
    | h::t -> 
      fold_left 
      (fun x y-> Py_datum.py_add x (get_slice atom y))
      (get_slice atom h) 
      t

  and get_dictionary_item d li = (* used in py_eval_trailer *)
    match li with
    | [] -> raise (TypeError "Empty subscript list?")
    | [Subscript0 (Pos j)] -> 
      let key = eval j in
      begin match d#get_item key with 
      | Some x -> x 
      | None -> raise (KeyError ("Dictionary lookup in subscription", key))
      end
    | h::t -> raise (TypeError "Dictionary Key must be single item")

  and py_eval_trailer atom trailer = (* used in eval *)
  match trailer with
  | Dotname attr -> get_attribute interp atom (PyString attr)
  | Sublist subs -> 
    begin match atom with
    | PyTuple _ 
    | PyList _ 
    | PyMutableList _ 
    | IntRange _ 
    | PyString _ 
    | PyInstance _ -> get_list_slice atom subs 
    | PyDictionary d -> get_dictionary_item d subs
    | _ -> 
      raise (TypeError (
        "Subscription primary must be sequence, got " ^ (Py_functions.repr atom)
      ))
    end
  | Arglist args -> 
    match atom with
    | PyFunction _  
    | PyNativeFunction _
    | PyBoundMethod _
    | PyClass _ ->
      py_call interp atom (eval_arglist interp env args)
    | PyNativeMacro  (name,mac) ->
      py_call interp (PyNativeFunction (name, (mac env))) (eval_arglist interp env args)
    | PyInstance _ -> raise (NotImplemented "__call__ methods")
    | _ -> raise (TypeError "Cannot call object, require function, class or macro")
    
  and eval e' = 
    let istrue = py_istrue interp in
    match e' with 
    | PyStatement st -> 
      begin try py_exec interp env st; PyNone
      with 
      | Returner expr -> expr
      end 

    (* These terms are BOTH raw language terms and evaluated objects,
       so their evaluation is themselves
    *)
    | PyNone 
    | PyString _ 
    | PyInt _ 
    | PyLong _ 
    | PyFloat _
    | PyComplex _ 
      -> e'

    (* These terms are evaluated objects and should not get
       to full eval, directly or indirectly [But they might
       get to a partial eval!]
    *)
    (*
    | IntRange _ 
    | PyRational _ 
    | PyFile _ 
    | PyInitial 
    | PyTerminal
    | PyMutableList _
    | PyInstance _
    | PyDictionary _
    | PyWidget _
    | PyRegexp _
      -> e' 
    *)

    | PyClosure _ -> e' (* not evaluated *)

    | PyName x -> get (PyString x)
    | PyVarIndex (level,i) -> iget level i
    | PyList x -> PyMutableList (Varray.of_list (map eval x))
    | PyTuple x -> PyTuple (map eval x) 
    | PyDict x -> let d = new Py_dict.py_dictionary in 
      List.iter
      (function DictEnt (k,v) -> ignore (d#set_item (eval k) (eval v)))
      x;
      PyDictionary d
    | Or x -> 
      let rec short_or x =
        match x with 
        | h :: t -> 
          let value = eval h in
          if t = [] or (istrue value)
          then value 
          else short_or t
        | [] -> PyNone (* never happens *)
      in short_or x

    | And x -> 
      let rec short_and x =
        match x with 
        | h :: t -> 
          let value = eval h in
          if t = [] or not (istrue value)
          then value 
          else short_and t
        | [] -> PyNone (* never happens *)
      in short_and x

    | Not x -> PyInt (if istrue (eval x) then 0 else 1)
    | Neg x -> Py_datum.py_neg (eval x)
    | Compare (x, cl) -> PyInt (int_of_bool (py_chain_compare (eval x) cl)) 
    | BitOr x -> PyInt (fold_left (fun a b -> Py_datum.py_bit_or a (eval b)) 0 x)
    | BitAnd x -> PyInt (fold_left (fun a b -> Py_datum.py_bit_and a (eval b)) (-1) x)
    | BitXor x -> 
      PyInt begin match x with
      | h :: t -> 
        begin match eval h with
        | PyInt h' -> 
          List.fold_left 
          (fun a b -> Py_datum.py_bit_xor a (eval b)) 
          h' 
          t
        | _ -> raise (TypeError "bad operand(s) for operator ^")
        end
      | _ -> raise (SystemError "Parser error: BitXor list length < 2")
      end
    | Complement x -> PyInt (Py_datum.py_complement (eval x))
    | Eval (x, bl) -> fold_left py_evaluate (eval x) bl 
    | Lambda (ps,x) ->
      let ps' = Py_param.get_parameters_from_parameters eval ps
      and st = Return ((0,"lambda"), x) 
      and glbs = [""]
      and i2n = Array.create 0 ""
      and n2i = Py_mtypes.VarMap.empty
      and env' = (env :> environment_t)
      in let func = new Py_function.py_function "lambda" ps' st glbs i2n n2i env'  
      in let func' = (func :> function_t)
      in PyFunction func'
    
    | AtomWithTrailers (ee, tr) -> fold_left py_eval_trailer (eval ee) tr
    | PyRepr e -> PyString (Py_functions.repr (eval e))
    | _ ->
      print_string "WARNING: Attempt to evaluate object: '"; 
      Py_print.print_expression 0 e';
      print_endline "'";
      raise (TypeError "Evaluate non expression")
  in eval expr


(* Note: generally, function calls shouldn't require any
  environment. Unfortunately, a couple of native functions
  such as 'globals' act on the current environment,
  soi we have to pass an environment in all cases,
  even if it isn't required. This is a pain.
*)

(* Note: we now require the argulist to be evaluated, this function
  will not eval them, [you can use with eval_arglist]
*)
and py_call
  interp atom arglist : expr_t =
  let call = py_call interp in
  match atom with
  | PyClass cls ->  (* create a class instance *)
    interp#push_line;
    let instance_object = 
      create_class_instance interp cls arglist 
    in interp#pop_line;
    (PyInstance instance_object)
  
  | PyBoundMethod (callable,argument) ->
    let selfarg = Argument1 argument
    in call callable (selfarg :: arglist)

  | PyNativeFunction (name,f) -> 
    let alist = ref []
    and adict = new Py_dict.py_dictionary in
    List.iter 
    begin fun arg ->
      match arg with
      | Argument1 v -> alist := !alist @ [v]
      | Argument2 (n,v) -> ignore (adict#set_item (PyString n) (v))
    end
    arglist;
    f interp !alist adict
    
  | PyFunction f -> 
     (* step 1: make parameter slots *)
   begin
     let (ps, star, starstar) = f#get_parameters in
     let n = List.length ps in
     let slots = Array.create  n Unknown in

     (* step 2: copy positional args to slots *)
     let args = ref arglist in
     (let counter = ref 0 in
     (try while !counter < n do
       match !args with
       | Argument1 e :: tl -> 
         slots.(!counter)  <- e;
         args := tl;
         incr counter
       | Argument2 _ :: _ -> raise Breaker
       | [] -> raise Breaker
     done with Breaker -> ()
     )
     );
     
     (* step 3: scan for extra arguments now *)
     let starargs = ref [] in
     let starstarargs = new Py_dict.py_dictionary in
     while List.length !args > 0 do (* each argument *)
       begin match List.hd !args with 
       | Argument1 e -> starargs := !starargs @ [e]
       | Argument2 (k,e) -> 
           let counter2 = ref 0 in
           let parameters = ref ps in
           try 
             while !parameters <> []  do (* each parameter *)
               let parameter = List.hd !parameters in
               let param_name = match parameter with
               | Parameter1 param_name -> param_name
               | Parameter2 (param_name,_) -> param_name
               in
               (match param_name with
               | Param name when name = k -> 
                 if slots.(!counter2) = Unknown
                 then slots.(!counter2) <- e
                 else raise (TypeError "keyword parameter redefined");
                 raise Breaker (* done: found it *)
               | _ -> () (* try with next parameter .. *)
               );
               parameters := List.tl !parameters;
               incr counter2
             done;
             ignore (starstarargs#set_item (PyString k) e) (* not found, add to starstarargs *)
           with Breaker -> ()
       end;
       args := List.tl !args
     done;
     
     (* check if all slots filled *)
     Array.iteri
     (fun i x -> match x with 
       | Unknown -> 
         begin match List.nth ps i with
         | Parameter2 (_, default) -> slots.(i) <- default
         | _ -> raise (TypeError ("Not enough arguments in call to "^f#get_name^": argument " ^(string_of_int (i+1))^ " not filled"))
         end
       | _ -> ()
     )
     slots;
     
     (* check if excesss positional arguments allowed *)
     (match star with 
     | NoStarParam when !starargs <> [] -> 
       let arg_display = 
         List.map
         begin fun arg ->  match arg with
           | Argument1 value -> Py_functions.repr value 
           | Argument2 (name,value) -> name ^ " = " ^ (Py_functions.repr value)
         end 
         arglist
       in
       raise (TypeError (
         "too many arguments in call to " ^ 
         f#get_name ^
         ", requires " ^ (string_of_int n) ^
         ", got " ^ (string_of_int (List.length arglist)) ^
         ": " ^
         (String.concat ", " arg_display)
       ))
     | _ -> ()
     );
     
     (* check if excesss keyword arguments allowed *)
     begin match starstar with 
     | NoStarStarParam when starstarargs#len <> 0 -> 
       raise (TypeError "unused keyword arguments")
     | _ -> ()
     end;
     
     (* create argument-parameter binding frame *)

     let arguments = new Py_dict.py_dictionary in
     Array.iteri
     begin fun i x ->
       let pname = match List.nth ps i with
       | Parameter1 pname -> pname
       | Parameter2 (pname, _) -> pname
       in match pname with 
       | Param name -> ignore (arguments#set_item (PyString name) slots.(i) )
       | Paramtuple ls -> raise (NotImplemented "Tuple parameter binding") 
     end
     slots;

     begin match star with 
     | StarParam name ->  ignore (arguments#set_item (PyString name) (PyTuple !starargs))
     | _ -> ()
     end;
     
     begin match starstar with 
     | StarStarParam name ->  
       ignore (arguments#set_item
         (PyString name)
         (PyDictionary starstarargs)
       )
     | _ -> ()
     end;
     
     begin try 
       interp#push_line;
       (*
       let exec_frame = new py_environment (Some (f#get_environment)) in
       exec_frame#push_dictionary arguments;
       *)
       let exec_frame = new Py_function.py_function_environment f arguments  
       in let exec_frame' = (exec_frame :> environment_t)  
       in
         py_exec interp exec_frame' f#get_code;
         interp#pop_line;
         PyNone
     with | Returner e -> 
       interp#pop_line;
       e 
     end
  end
  | _ -> raise (NotImplemented "Call non-function")

and create_class_instance 
  interp cls arglist = 
  let instance_object = (new Py_class.py_instance cls :> instance_t)
  in
    let maybeMethod =
      match cls # get_attr (PyString "__init__") with
      | Some f -> Some (PyBoundMethod (f,PyInstance instance_object ))
      | None -> None
    in 
      begin match maybeMethod with
      | Some theMethod ->
        begin try ignore (py_call interp theMethod arglist)
        with 
        | Returner _ -> ()
        | AttributeError (s,e) ->
          print_endline (
            "AttributeError '" ^ s ^
            "' calling __init__ method of class " ^ cls#get_name ^ 
            ": attribute " ^ (Py_functions.repr e)
          );
          raise (AttributeError (s,e))
        | x -> 
          print_endline ("Exception in __init__ method of class " ^ cls#get_name);
          print_endline (Printexc.to_string x);
          raise x
        end
      | None -> ()
      end;
      instance_object

and eval_arglist interp env a = 
  List.map
  begin fun x -> match x with
    | Argument1 v -> Argument1 (py_eval interp env v)
    | Argument2 (n,v) -> Argument2 (n, py_eval interp env v)
  end
  a

