(* special environments *)
open Py_types
open Py_exceptions
open Py_mtypes

(* NOTE: the exception label is currently not indexable,
   should be, like a function argument *)

class py_handler_environment (varname:string) (exc:expr_t) (env':environment_t) =
  object(self)
    val dict = new Py_dict.py_dictionary
    val super = env'

    initializer
      ignore (dict # set_item (PyString varname) exc)

    (* get an attribute *)
    method get_attr k =
      if dict # has_item k
      then dict # get_item k
      else super#get_attr k

    (* set an attribute *)
    method set_attr k v = super # set_attr k v
    method del_attr k = super # del_attr k
    method has_attr k = super # has_attr k

    (* test which methods will work *)
    method can_get_attr k = self#has_attr k
    method can_set_attr (k:expr_t) (v:expr_t) = true
    method can_set_any_attr (k:expr_t) = true
    method can_del_attr (k:expr_t) = true (* a lie for the moment *)
  
    (* Do indexed addressing *)
    method get_indexed level i = 
      super#get_indexed (level-1) i

    method set_indexed level i (v:expr_t) = 
      super#set_indexed (level-1) i v

    method del_indexed level i =
      super#del_indexed level i

    (* retrieve locals and globals dictionaries *)
    method get_locals = dict (* really, we should return function locals, if in a function *)
    method get_globals = super #get_globals

    method keys = dict#keys

    method get_module = super#get_module
  end

class py_null_environment =
  object
    method get_attr (k:expr_t): expr_t option = None
    method set_attr (k:expr_t) (v:expr_t): bool = false
    method del_attr (k:expr_t) :  bool = false
    method has_attr (k:expr_t) :  bool = false

    method can_get_attr (k:expr_t): bool = false
    method can_set_any_attr (k:expr_t): bool = false
    method can_set_attr (k:expr_t) (v:expr_t): bool = false
    method can_del_attr (k:expr_t): bool = false
    method keys : expr_t list = []
    method get_locals : dictionary_t = raise (ViperError "null environment has no locals")
    method get_globals : dictionary_t = raise (ViperError "null environment has no globals")
    method get_indexed (a:int) (b:int) : expr_t = PyInitial
    method set_indexed (a:int) (b:int) (c: expr_t) = ()
    method del_indexed (a:int) (b:int) = ()
    method get_module : module_t = raise (ViperError "null environment has no module")
  end

class py_python_environment builtins' globals' locals' =
  object(self)
    val builtins: dictionary_t = builtins'
    val globals: dictionary_t = globals'
    val locals: dictionary_t = locals'

    (* get an attribute *)
    method get_attr k =
      if locals # has_item k
      then locals # get_item k
      else if globals # has_item k
      then globals # get_item k
      else builtins # get_item k

    (* set an attribute *)
    method set_attr k v = locals # set_item k v
    method del_attr k = locals # del_item k
    method has_attr k = match self#get_attr k with 
      | Some x -> true 
      | None -> false 

    (* test which methods will work *)
    method can_get_attr k = self#has_attr k
    method can_set_attr (k:expr_t) (v:expr_t) = true
    method can_set_any_attr (k:expr_t) = true
    method can_del_attr (k:expr_t) = locals # has_item k 
  
    (* Do indexed addressing *)
    method get_indexed (i:int) (j:int) : expr_t = 
      raise (ViperError "get_indexed: python_environment does not support indexed addressing")
    method set_indexed (i:int) (j:int) (v:expr_t) : unit = 
      raise (ViperError "set_indexed: python_environment does not support indexed addressing")
    method del_indexed (i:int) (j:int) : unit =
      raise (ViperError "del_indexed: python_environment does not support indexed addressing")

    (* retrieve locals and globals dictionaries *)
    method get_locals = locals
    method get_globals = globals

    method keys = locals#keys

    method get_module : module_t = raise (ViperError "python_environment has no module")
  end


