(* Module console_fish,
  written by John Skaller,
  mailto:skaller@maxtal.com.au 
  June 1999 *)

(*
  This module provides the fish console type, which is responsible for
  sourcing data to the client lexer, and reporting errors. 
  
  It provides control over echoing and prompting, and maintains
  a history of lines read to facilitate error reporting.

  The principal distinction encapsulated by the abstraction
  is that interactive terminal input automatically echoes
  source lines, while reading a file need not. 

  Hence, errors during interactive input can be reported
  directly, referring to the previously entered line,
  whereas errors while reading a file without echoing
  require identification of the file and lines causing
  the problem. In addition, a history of lines read
  is maintained to provide some visual context.

  The principal machine interface method is lexer_func, 
  which has the interface
  required for the argument to Lexing.from_function, which
  constructs a lexbuf compatible with the ocamllex lexer.

  Three factory functions are provided to create consoles:
    interactive_console ()
      creates an interactive console
    console_from_channel chan 
      creates a console for an open input channel
      note the client is responsible for closing the channel if necessary
    stdin_console () 
      creates an console for standard input, it will be
      the interactive console if stdin is a terminal,
      else console_from_channel stdin
*)

type 'a parent = Parent of 'a | NoParent ;;

class type console =
  object
    method reset_line_number : unit (* 
      reset the line number to 1 *)

    method reset_prompt : unit (* 
      set the prompt to ps1, 
      the prompt is set to ps2 after ps1 is issued,
      this function also clears the history *)

    method set_echo_mode : bool -> unit (* 
      say whether or not to echo each line read
      to standard output [Note:  this doesn't apply to the
      interactive console] *)

    method set_number_switch : bool -> unit (* 
      determine whether or not line numbers should be output *)

    method set_prompt_mode : bool -> unit (*
      determine whether or not to issue prompts *)

    method set_ps1 : string -> unit (* 
      set ps1, the prompt for initial input *)

    method set_ps2 : string -> unit (*
      set ps2, the prompt for continued input *)

    method set_name : string -> unit (*
      set a convenient string name for the object *)

    method get_name : unit -> string (*
      get the name set with the set name function *)

    method lexer_func : string -> int -> int (*
      lexer_func s n fills in the string s with text from the input source,
      up to n characters, and returns the number of characters actually
      filled in; s must be at least n in length.
      0 is returned on end of input, otherwise the last character
      filled in will always be a newline *)

    method readline : unit -> string (*
      get one line *)

    method report_syntax_error : int -> unit (*
      print a syntax error message where the argument
      is the column number in the last line read *)

    method report_error : (unit->unit)-> unit (*
      print an error message with context, where the argument
      is a callback printing the actual error message *)

    method get_line_number : unit -> int (*
      return the line number of the last line read *)

    method set_parent: console parent -> unit (*
      set the parent console, to be used by %use actions
      to allow identifying the %use stack *)

    method show_inner_context: unit (*
      display the current location and the
      use stack used to get here: 
      used internally, NOT to be called by clients *)
  end
;;


val interactive_console : unit -> console (*
  a specialised console connected to the users terminal *)
;;

val console_from_channel : in_channel -> console (*
  a console connected to given input channel: the channel
  must be open at the time each call to lexer func is issued *)
;;

val stdin_console : unit -> console (*
  a console corresponding to standard input which will be 
  the interactive console if it is a terminal *)
;;


