(* publically known type *)
type 'a parent = Parent of 'a | NoParent ;;

class type console =
  object
    method reset_line_number: unit
    method reset_prompt : unit
    method set_echo_mode : bool -> unit
    method set_number_switch : bool -> unit
    method set_prompt_mode : bool -> unit
    method set_ps1 : string -> unit
    method set_ps2 : string -> unit
    method lexer_func : string -> int -> int 
    method set_name : string -> unit
    method get_name : unit -> string 
    method report_syntax_error : int -> unit
    method report_error : (unit->unit)-> unit
    method get_line_number: unit -> int
    method set_parent: console parent -> unit
    method show_inner_context: unit
    method readline: unit -> string
  end
;;


(* IMPLEMENTATION DETAILS *)
let history_limit = 200;;

class virtual console_base =
  object (self)
    val mutable ps1 = ">>>" 
    val mutable ps2 = "..>"
    val mutable prompt_switch = false
    val mutable echo_switch = false
    val mutable number_switch = false
    val mutable name = "Anonymous"
    val mutable line_number = 0
    val mutable prompt_state = 1
    val mutable last_prompt_len = 0
    val mutable history = Array.create history_limit ""
    val mutable history_index = 0
    val mutable history_count = 0
    val mutable my_parent: console parent = NoParent 
    method reset_line_number : unit = line_number <- 0
    method set_ps1 (ps1':string) = ps1 <- ps1'
    method set_ps2 (ps2':string)  = ps2 <- ps2'
    method set_prompt_mode (flag:bool) = prompt_switch <- flag
    method set_echo_mode (flag:bool) = echo_switch <- flag
    method set_number_switch (flag:bool) = number_switch <- flag
    method set_name (name':string) = name <- name'
    method get_name () = name
    method get_line_number () = line_number
    method set_parent p = my_parent <- p
    method private reset_history =
      history_index <- 0;
      history_count <- 0

    method reset_prompt : unit = 
      prompt_state <- 1;
      self#reset_history

    method private get_prompt = 
      let tmp = if prompt_state == 1 then ps1 else ps2 in
      prompt_state <- 2;
      last_prompt_len <- String.length tmp;
      tmp

    method private show_history =
      let j = ref (
        (history_index - history_count + history_limit) mod history_limit) 
      in
      for i=0 to history_count-1 do
        print_string history.(!j);
        j := (!j+1) mod history_limit 
      done;
      self#reset_history

    method private show_history_location = 
      print_string ("file \"" ^ name ^ "\", line " ^ (string_of_int line_number))

    (* this method unfortunately has to be public, because it invokes
       itself on its parent, which is only known in the abstract *)
    method show_inner_context = 
       self#show_history_location;
       print_endline "";
       self#show_parent_context

    method private show_parent_context =  
       begin match my_parent with
         NoParent -> ()
         | Parent con ->
             print_string "  .. used by "; 
             con#show_inner_context 
       end

    method report_syntax_error (column:int) = 
      begin if not echo_switch then
        print_endline ">>>>>>>>>>>>>>>>>>>>";
        self#show_history
      end;
      for i=0 to last_prompt_len + column - 1 do print_char ' ' done;
      print_endline "^ Syntax error";
      print_endline (
        "  in         file \"" ^ name ^ "\", " ^
        "line " ^ (string_of_int line_number) ^ ", " ^
        "column " ^ (string_of_int (column+1)));
      self#show_parent_context;
      begin if not echo_switch then
        print_endline "<<<<<<<<<<<<<<<<<<<<"
      end;
      flush stdout

    method report_error (callback:unit->unit) = 
      begin if not echo_switch then
        print_endline ">>>>>>>>>>>>>>>>>>>>";
        self#show_history
      end;
      callback ();
      print_string "  in         ";
      self#show_inner_context;
      begin if not echo_switch then
        print_endline "<<<<<<<<<<<<<<<<<<<<"
      end;
      flush stdout
  end
;;


(* this class takes a function which returns a string
   as an argument to construct a standard console object 
   The function must not fail: return a line including
   terminal newline, or an empty string at end of file.
*)

class console_from_function (callback:string->string) = 
  object (self) 
    inherit console_base
    val mutable buffer = ""
    val get_line = callback 


    method private transfer_data (request_buffer:string) (maxfill:int): int  = 
      let buffer_length = String.length buffer in
      let toCopy = min maxfill buffer_length in
        String.blit buffer 0 request_buffer 0 toCopy; 
          (* blit the head across *) 
        buffer <- String.sub buffer toCopy (buffer_length - toCopy); 
          (* cut it out of the buffer *)
        toCopy 
          (* return number of characters transfered *) 
    
    method private ensure_buffer_primed = 
      if buffer = "" 
      then self#prime_buffer

    method private prime_buffer =
      begin 
        let prompt = ref "" in
        begin (* compute the prompt *)
          if prompt_switch then begin
            if number_switch 
            then prompt := ((string_of_int (line_number + 1)) ^ ": ");
            prompt := !prompt ^ (self#get_prompt) 
          end;

          begin (* read input with prompt *)
              buffer <- (get_line !prompt) ; 
              line_number <- line_number + 1;

              if not echo_switch then begin
                (* keep track of the last three lines of input *)
                history.(history_index) <- buffer;
                history_index <- ((history_index + 1)  mod history_limit);
                history_count <- min (history_count+1)  history_limit
              end
          end
        end;
        
        (* output control *)
        if number_switch & not prompt_switch 
        then print_string ((string_of_int line_number) ^ ": ");
        
        if echo_switch then begin
          print_string buffer; 
          flush stdout
        end;
      end

    method lexer_func (request_buffer:string) (maxfill:int): int  = 
      self#ensure_buffer_primed;
      self#transfer_data request_buffer maxfill

    method readline () = 
      self#ensure_buffer_primed;
      let s =  buffer  in
        buffer <- "";
        s
 end
;; 

class terminal_console (callback:string->string) = 
  object (self) inherit console_from_function callback
    method set_echo_mode (flag:bool) = ()
(*
    method report_syntax_error (column:int) = 
      for i=0 to last_prompt_len + column - 1 do print_char ' ' done;
      print_endline "^ Syntax error";
      flush stdout

    method report_error (callback:unit->unit) = 
      callback ();
      flush stdout
*)
  end
;;

(* here is the implementation of the public API *)
let interactive_console () : console = 
  let con = new terminal_console Gnu_readline.readline
  in 
    con # set_name "Terminal";
    con # set_ps1 ">>>";
    con # set_ps2 "..>";
    con # set_prompt_mode true;
    con # set_number_switch false;
    (con : console)
;;


let console_from_channel (chan:in_channel) : console = 
  let line_from_channel (chan:in_channel) prompt =
    print_string prompt; flush stdout;
    try ((input_line chan) ^ "\n")
    with _ -> ""
  in
    let con = (new console_from_function (line_from_channel chan)) in
    con # set_prompt_mode false;
    con # set_echo_mode false;
    con # set_number_switch false;
    (con : console)
;;
  

let stdin_console () : console = 
  let isatty () = Gnu_readline.isatty 0 
  in
    if isatty () 
    then interactive_console ()
    else 
      let con = console_from_channel stdin in
      con # set_name "Standard Input";
      (con : console)
;;

