open Py_types
open Py_mtypes
open Py_exceptions

class py_class name' bases' dictionary' i2n n2i env' =
  object(self)
    val name : string = name'
    val bases : class_t list = bases'
    val super : environment_t =  env'
    val attribs = dictionary'
    val methods = new Py_dict.py_dictionary

    val name_to_index: int VarMap.t = n2i
    val index_to_name: string array  = i2n
    val vars = Array.create (Array.length i2n) PyInitial


    method get_name = name
    method get_bases = bases
    method get_dictionary = attribs
    method get_methods = methods 
    method get_environment = super
    method get_vars = vars

    method get_attr k = 
      match k with
      | PyString "__name__" -> Some (PyString name)
      | PyString "__dict__" -> Some (PyDictionary attribs)
      | PyString "__methods__" -> Some (PyDictionary methods)
      | PyString "__bases__" -> Some (PyTuple (List.map (fun x -> PyClass x) bases))
      | _ ->
        match attribs#get_item k with
        | Some v as y -> y
        | None ->
          match methods#get_item k with
          | Some v as y -> y
          | None ->
            let rec base_get k base_list =
            match base_list with 
            | h :: t ->
              begin match h#get_attr k
              with 
              | Some v as y -> y
              | None -> base_get k t 
              end
            | [] -> None 
            in base_get k bases

    method set_attr k v = attribs#set_item k v
    method has_attr k = attribs#has_item k
    method del_attr k = attribs#del_item k
    method can_del_attr k = attribs#has_item k
    method can_get_attr k = 
      match self#get_attr k with Some _ -> true | None -> false
    method can_set_attr (k:expr_t) (v:expr_t) = true 
    method can_set_any_attr (k:expr_t) = true 

    method get_index_to_global_name = index_to_name
    method get_global_name_to_index = name_to_index

    method get_indexed level i = 
      if level = 0 
      then vars.(i)
      else super #get_indexed (level-1) i

    method set_indexed level i (v:expr_t) = 
      if level = 0
      then vars.(i) <- v
      else super#set_indexed (level-1) i v
      
    method del_indexed level i =
      if level = 0
      then vars.(i) <- PyInitial
      else super#del_indexed level i

end

class py_class_environment (clas':class_t) = 
  object (self)
    val clas = clas'
    val dict = clas'#get_dictionary
    val super = clas'#get_environment
    val vars = clas'#get_vars

    val name_to_index = clas'#get_global_name_to_index

    (* get underlying module *)
    method get_module = super#get_module

    (* get an attribute *)
    method get_attr k =
      match k with 
      | PyString name -> 
        if Py_mtypes.VarMap.mem name name_to_index
        then Some vars.(Py_mtypes.VarMap.find name name_to_index)
        else if dict#has_item k
        then  dict#get_item k
        else super#get_attr k
      | _ -> 
        if dict#has_item k
        then  dict#get_item k
        else super#get_attr k

    (* set an attribute *)
    (* NOTE: we don't bother to keep the dictionary synched with
       indexed access. This means 'locals' will fail to show the
       correct dictionary, as in Python. The reason we don't,
       is that changes to the dictionary will not be reflected
       in the fast load array anyhow. This could be fixed,
       by storing to the dictionary as well as the array,
       and also by returning a n object attached to the
       array, instead of a dictionary (but with the same interface
     *)
    method set_attr k v = 
      match k with 
      | PyString name ->
        if Py_mtypes.VarMap.mem name name_to_index
        then begin 
          vars.(Py_mtypes.VarMap.find name name_to_index) <- v; 
          true
        end
        else dict#set_item k v 
      | _ -> dict#set_item k v  

    method del_attr k = dict # del_item k

    (* Python compatible has_attr *)
    method has_attr k = 
      match self#get_attr k with 
      | Some _ -> true 
      | None -> false

    (* test which methods will work *)
    method can_get_attr k = self#has_attr k
    method can_set_attr (k:expr_t) (v:expr_t) = true
    method can_set_any_attr (k:expr_t) = true
    method can_del_attr (k:expr_t) = true (* a lie for the moment *)
  
   (* Do indexed addressing *)
    method get_indexed level i = 
      if level = 0 
      then vars.(i)
      else super #get_indexed (level-1) i

    method set_indexed level i (v:expr_t) = 
      if level = 0
      then vars.(i) <- v
      else super#set_indexed (level-1) i v

    method del_indexed level i =
      if level = 0
      then vars.(i) <- PyInitial
      else super#del_indexed level i

    (* retrieve locals and globals dictionaries *)
    method get_locals = dict
    method get_globals = dict

    method keys = dict # keys
  end

class py_instance cls' = 
  object (self)
    val cls : class_t = cls'
    val dict = new Py_dict.py_dictionary
    method get_class = cls
    method get_dictionary = dict 
    method get_attr k = 
      match k with 
      | PyString "__class__" -> Some (PyClass cls)
      | PyString "__dict__" -> Some (PyDictionary dict)
      | _ ->
        match dict#get_item k
        with Some v as y -> y
        | None ->
          match cls#get_attr k with
          | Some cls_attr -> 
            if Py_functions.callable cls_attr 
            then Some (PyBoundMethod (cls_attr, PyInstance (self :> instance_t)) )
            else Some cls_attr
          | None -> None
          
    method set_attr k v = 
      match k with 
      | PyString "__dict__" 
      | PyString "__class__" -> false
      | _ -> dict#set_item k v
    method has_attr k = dict#has_item k
    method del_attr k = dict#del_item k
    method can_del_attr k = dict#has_item k
    method can_get_attr k = 
      match self#get_attr k with Some _ -> true | None -> false
    method can_set_attr k v = true 
    method can_set_any_attr k = true 

  end

