open Py_types
open Py_mtypes
open Py_exceptions
open Util
open Big_int 
open Num
open Py_builtins_util

(* depends on Py_exec *)

(* __import__ *)

let py_abs 
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with 
  | PyInt i -> PyInt (abs i)
  | PyLong l -> PyLong (abs_big_int l)
  | PyRational r -> PyRational (abs_num r)
  | PyFloat f -> PyFloat (abs_float f)
  | PyComplex (x,y) -> PyFloat (sqrt ((x *. x) +. (y *. y)))
  | _ -> raise (TypeError "abs of non number")
;;

let py_apply
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atleast e 2;
  atmost e 3;
  let f = arg e 0
  and args = arg e 1
  and keywords = optional e 2
  in let d = 
    match keywords with
    | None -> empty_dictionary
    | Some (PyDictionary d') -> d'
    | _ -> 
      raise (TypeError "Third argument to apply must be dictionary or omitted")
  in let args' = py_tuple_of_sequence args 
  in let args'' = 
    match args' with 
    | PyTuple ls -> ls 
    | _ -> raise (Failure "tuple_of_seqquence bombed in py_apply")
  in Py_exec.py_call interp f (make_argument_list args'' d)
;;

let py_buffer
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let s = arg e 0 in
  match s with
  | PyString _ -> s
  | _ -> raise (TypeError "buffer requires string argument")
;;

let py_callable
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let f = arg e 0
  in Py_functions.py_callable f
;;

(* WARNING: this function creates a UTF8 encoding of
   the given integer considered as an ISO-10646 code point.
   It is therefore not compatible with python's chr function,
   on codes 128-255. It may be better to preserved compatibility,
   and add a new builtin.
*)

let py_uchr        
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let a1 = arg e 0
  in match a1 with 
  | PyInt i -> PyString (Py_string.utf8_of_int i)
  | _ -> raise (TypeError "Argument to uchr must be integer")
;;

let py_chr        
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let a1 = arg e 0
  in match a1 with 
  | PyInt i -> 
    if i < 0 or i > 255 
    then raise (ValueError "Argument to chr must be integer 0..255")
    else PyString (String.make 1 (char_of_int i))
  | _ -> raise (TypeError "Argument to chr must be integer")
;;

(* WARNING: this code is wrong: it uses the ocaml builtin compare
   function, which doesn't work right for objects
*)

let py_cmp
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let a1 = arg e 0
  and a2 = arg e 1
  in try PyInt (Py_datum.py_compare a1 a2)
  with Invalid_argument _ -> raise (Failure "Unexpected failure of py_cmp")
;;


let py_coerce
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  PyTuple (coerce e)
;;

(* Note: we hack the string argument to compile with the exec option, 
   by adding a newline to the end; this handles a common client error
   without too many bad side effects.
*)
let py_compile
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 3;
  let s = arg e 0 
  and filename = arg e 1 
  and kind = arg e 2
  in let filename' = match filename with
  | PyString f -> f
  | _ -> raise (TypeError "Filename argument(2) to compile must be string")
  in match kind with
  | PyString "eval" ->
    begin match s with
    | PyString s' ->  
        Py_exec_module.parse_string_as_expr s' filename'
    | _ -> raise (TypeError "Argument1 to compile must be string")
    end
  | PyString "exec" ->
    begin match s with
    | PyString s' -> 
      PyStatement (Py_exec_module.parse_string (s'^ "\n") filename')
    | _ -> raise (TypeError "Argument1 to compile must be string")
    end
  | _ -> raise (TypeError "Argument 3 to compile must be 'exec' or 'eval'")
;;

(* NotImplemented: support for long and rational conversions *)
let py_complex
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atleast e 1;
  atmost e 2;
  let r = arg e 0 in
  let i = optional e 1 in
  let i' = match i with Some imag -> imag | None -> PyFloat 0.0
  in let r' = match r with 
  | PyInt i -> PyComplex (float_of_int i, 0.0)
  | PyFloat f -> PyComplex (f, 0.0)
  | PyComplex _ -> r 
  | _ -> raise (TypeError "Arg1 of complex must be number")
  and i'' = match i' with 
  | PyInt i -> PyComplex (float_of_int i, 0.0)
  | PyFloat f -> PyComplex (f, 0.0)
  | PyComplex _ -> i'
  | _ -> raise (TypeError "Arg2 of complex must be number")
  in Py_datum.py_add r' (Py_datum.py_mul i'' (PyComplex (0.0,1.0)))
;;

let py_ellipsis = PyNone;; (* temporary hack *)

let py_delattr    
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let attr = arg e 1 in
  ignore begin match arg e 0 with
  | PyInstance i -> i#get_dictionary#del_item attr
  | PyClass c -> c#get_dictionary#del_item attr 
  | PyModule c -> c#get_dictionary#del_item attr 
  | _ -> raise (TypeError "delattr requires module, instance or class argument")
  end;
  PyNone
;;

(* NotImplemented: support for long and rational types *)
let py_divmod     
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let (dividend, divisor) = Py_datum.py_coerce (arg e 0, arg e 1) in
  match (dividend, divisor) with
  | (PyInt dividend, PyInt divisor) -> 
    PyTuple [
      PyInt (dividend / divisor); 
      PyInt (dividend mod divisor)
    ]
  | (PyFloat dividend, PyFloat divisor) -> 
    let quot = floor (dividend /. divisor)  in
    PyTuple [PyFloat quot; PyFloat (dividend -. (divisor *. quot)) ]
  | (PyComplex _, PyComplex _) -> 
     raise (NotImplemented "divmod for complex")
  | _ -> raise (TypeError "divmod requires (non-complex) numeric arguments")

let py_exit
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  begin match arg e 0 with
  | PyInt n -> raise (SystemExit n)
  | _ -> raise (TypeError "exit function requires integer exit code")
  end
  ;
  PyNone
;;

let py_filter     
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let f = arg e 0 
  and x = arg e 1
  in let f' = match f with 
  |  PyNone -> PyNativeFunction ("identity", (fun interp args kwds -> arg args 0))
  | _ when Py_functions.callable f -> f
  | _ -> raise (TypeError "filter requires None or callable object as first argument")
  in let func = 
    begin fun x -> Py_datum.py_istrue 
      (Py_exec.py_call interp f' [Argument1 x])
    end
  in match x with
  | PyTuple ls -> PyTuple (List.filter func ls)
  | PyMutableList ls -> PyMutableList (Varray.filter func ls)
  | PyString s -> PyString (
      Py_util.string_of_list (List.filter func (Py_util.list_of_string s))
    )
  | _ -> raise (TypeError "filter requires tuple, list or string as second argument")
;;

(* NotImplemented: support for Long and Rational types *)
let py_float      
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let a = arg e 0 in
  match a with
  | PyString s -> 
    PyFloat (float_of_string s) 
  | PyInt i -> PyFloat (float_of_int i)
  | PyFloat _ -> a
  | _ -> raise (TypeError "float argument must be string, int or float")
;;

let py_getattr    
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atmost e 3;
  atleast e 2;
  let default = optional e 2
  and a = arg e 0 
  and b = arg e 1 
  in 
  let have_default =  
    match default with 
    | Some _ -> true 
    | None -> false 
  in 
    try Py_exec.get_attribute interp a b
    with 
    | AttributeError (x,y) -> 
      begin
        match default with
        | Some deflt -> deflt
        | None -> raise (AttributeError (x,y))
      end
    | x -> 
      raise x
;;

let py_hasattr    
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  PyInt (
    try ignore (py_getattr interp e d); 1
    with _ -> 0
  )
;;

let py_hash       
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  PyInt (Hashtbl.hash (arg e 0))
;;

(* NotImplemented: support for Long type *)
let py_hex        
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | PyInt x -> PyString (Printf.sprintf "0x%x" x)
  | _ -> raise (TypeError "hex requires integer argument")
;;

(* this requires a C function: it will be tricky, since the ocaml
   collector does storage compaction, and so the address of
   an object can change; we need to return the box address,
   not the physical address of the object. Of course, this will
   not work for unboxed types like int and float, since these
   are handled by value -- in Viper too.

   We cheat and just use the hash value. Technically, this isn't
   going to be unique for objects. Viper isn't 'object oriented'
   to the same extent as Python, immutable objects are treated
   as values, so it is actually appropriate to treat their id
   values as equal if the values are equal, and the hash function
   will do that.
*)

let py_id         
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  PyInt (Hashtbl.hash (arg e 0))
;;

let py_input      
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
raise (NotImplemented "input")
;;

(* Viper doesn't currently do interning of strings, so we just return
   the argument
*)
let py_intern     
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | PyString s as x -> x
  | _ -> raise (TypeError "Attempt to intern non string")
;;

let py_int        
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let a = arg e 0 in 
  match a with
  | PyInt _ -> a
  | PyFloat f -> PyInt (int_of_float f)
  | PyLong x -> PyInt (Big_int.int_of_big_int x)
  | PyRational x -> PyInt (Num.int_of_num x)
  | PyString s -> 
    begin try PyInt (int_of_string s)
    with Failure _ -> 
      raise (TypeError ("int: cannot convert string '"^s^"'to int"))
     end
  | _ -> raise (TypeError "Argument to int must be string, int, long, rational or float")
;;

let py_isinstance 
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let obj = arg e 0 and cls = arg e 1 in
  match cls with 
  | PyClass c ->
    begin match obj with 
    | PyInstance i -> PyInt (int_of_bool (Py_util.isbaseof c i#get_class))
    | _ ->
      raise (TypeError 
        "First argument to isinstance must be instance when second is a class")
    end
  | _ -> 
    raise (TypeError 
      "First argument of isinstance must be instance (Type for 2nd arg not implemented yet)")
;;

let py_issubclass 
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 2;
  let derived = arg e 0 and base = arg e 0 
  in let derived' = 
    match derived with 
    | PyClass d' -> d'
    | _ -> raise (TypeError "Arg1 of issubclass must be a class")
  in let base' = 
    match base with 
    | PyClass d' -> d'
    | _ -> raise (TypeError "Arg2 of issubclass must be a class")
  in PyInt (int_of_bool (Py_util.isbaseof base' derived'))
;;

let py_len        
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let a = arg e 0 in
  match a with
  | PyTuple _
  | PyMutableList _
  | PyString _ 
  | IntRange _ 
  | PyInstance _ -> Py_exec.py_seq_len interp a 
  | PyDictionary d -> PyInt (d#len)
  | _ -> raise (TypeError "len requires sequence or dictionary")
;;

let py_list       
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let a = arg e 0 in
  py_list_of_sequence a
;;

(* NotImplemented: support for Float, Complex and Rational types *)
let py_long       
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | PyInt i -> PyLong (big_int_of_int i)
  | PyLong l -> PyLong l
  | _ -> raise (TypeError "Argument to long must be int of long")
;;

let py_map        
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atleast e 2;
  let f = arg e 0 
  and x = arg e 1
  in let mapNone = 
    if (List.length e) = 2 
    then fun interp args kwds -> arg args 0
    else fun interp args kwds ->
      PyTuple (List.fold_right (fun x y -> x::y) [] args)
  in let f' = match f with 
  |  PyNone -> PyNativeFunction ("mapNone", mapNone)
  | _ when Py_functions.callable f -> f
  | _ -> raise (TypeError "map requires None or callable object as first argument")
  in 
    let n = (List.length e) - 1 in
    let a: expr_t list array = Array.create n [] in
    for i=0 to n-1 do
      a.(i) <- Py_util.list_of_sequence (List.nth e (i+1))
    done;
    let m = List.length a.(0) in
    for i=1 to n-1 do
      if (List.length a.(0)) <> m 
      then raise (ValueError "map requires list arguments of same length")
    done;

    let mklist n : argument_t list list=
      let rec mklist' n lst = if n = 0 then lst else [] :: mklist' (n-1) lst
      in mklist' n []
    in let binary_zip (a:expr_t list) (b:argument_t list list) : argument_t list list = 
      List.map2 (fun x y -> Argument1 x :: y) a b
    in let arguments_list = Array.fold_right binary_zip a (mklist m) 
    in let result = List.map (fun x -> Py_exec.py_call interp f' x) arguments_list
    in PyMutableList (Varray.of_list result)
;;

let py_max        
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | IntRange (start, stop, step) -> 
    if step>0 && stop > start then PyInt stop
    else if step <0 && stop < start then PyInt start
    else raise (TypeError "Max of empty Xrange")
  | x ->
  let l = Py_util.list_of_sequence x in
    let seq = coerce l in
    let result = 
      List.fold_left
      (fun x y -> if Py_datum.py_less x y then y else x)
      PyInitial
      seq
    in match result with 
    | PyInitial -> raise (TypeError "Max of empty sequence")
    | x -> x
;;

let py_min        
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | IntRange (start, stop, step) ->
    if step>0 && stop > start then PyInt start
    else if step <0 && stop > start then PyInt stop
    else raise (TypeError "Min of empty Xrange")
  | x ->
  let l = Py_util.list_of_sequence x in
    let seq = coerce l in
    List.fold_left
    (fun x y -> if Py_datum.py_less x y then x else y)
    PyTerminal
    seq
;;

(* NotImplemented: support for Long type *)
let py_oct        
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  match arg e 0 with
  | PyInt i ->
    if i = 0 then PyString "0"
    else let result = String.create 20 (* enough for our integers *)
    and count = ref 0 
    and a = ref i in
    while !count < 20 && !a <> 0 do
      incr count;
      result.[20 - !count] <- char_of_int (!a land 0x7  + (int_of_char '0'));
      a := !a asr 3
    done;
    PyString ("0" ^ (String.sub result (20 - !count) !count))
  | _ -> raise (TypeError "oct requires integer argument")
;;

(* the uord function requires a string argumnent which is a valid UTF-8 encoding
   of a single ISO-10646 encoding 
*)

let py_uord        
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let a = arg e 0 in
  match a with 
  | PyString s -> PyInt (fst (Py_string.parse_utf8 s 0))
  | _ -> raise (TypeError "uord requires utf-8 encoded string argument")
;;


(* this is the 8 bit ord function *)
let py_ord        
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let a = arg e 0 in
  match a with 
  | PyString s -> 
    if (String.length s) = 1
    then PyInt (int_of_char s.[0])
    else raise (ValueError "String argument of ord must have length 1")
  | _ -> raise (TypeError "ord requires string argument")
;;

let py_xrange     
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  match List.length e with
  | 1 -> IntRange (0,int_of_expr e 0,1)
  | 2 -> IntRange (int_of_expr e 0,int_of_expr e 1, 1)
  | 3 -> 
    let step = int_of_expr e 2 in
    if step = 0 then raise (TypeError "Step of xrange is zero")
    else let start = int_of_expr e 0 
    and stop = int_of_expr e 1 in
    let count = max ((stop - start) / step) 0 in
      IntRange (start, start + count * step, step)
  | _ -> raise (TypeError "Wrong number of arguments to (x)range")
;;


(* this function returns an actual tuple: xrange is usually more efficient *)
let py_range      
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  match py_xrange interp e d with
  | IntRange (start, stop, step) ->
    let li = ref [] and
    i = ref start in
    if step > 0 then while !i < stop do li := !li @ [PyInt !i]; i:= !i + step done
    else if step < 0 then while !i > stop do li := !li @ [PyInt !i]; i := !i + step done
    else if start <> step then raise (ValueError "Step in range cannot be 0 unless start=stop");
    PyMutableList (Varray.of_list !li)
  | _ -> raise (Failure "Unexpected case in py_range")
;;

let py_raw_input  
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
raise (NotImplemented "raw_input")
;;

let py_reduce     
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atleast e 2;
  atmost e 3;
  let init = optional e 2
  and func = arg e 0
  and seq = arg e 1
  in let n =
    match Py_exec.py_seq_len interp seq with 
    | PyInt i -> i
    | _ -> raise (ViperError "py_seq_len should return int")
  in 
    match init with
    | Some x ->
      let v = ref x in
      for i=0 to n-1 do
        v := Py_exec.py_call interp func 
          [Argument1 !v; Argument1 (Py_exec.py_get_seq_elem interp seq (PyInt i))]
      done;
      !v

    | None ->
      if n = 0 
      then raise (ValueError "Reduce empty list without initialiser")
      else if n = 1 
      then Py_exec.py_get_seq_elem interp seq (PyInt 0)
      else 
        let v = ref (Py_exec.py_get_seq_elem interp seq (PyInt 0)) in
        for i=1 to n-1 do
          v := Py_exec.py_call interp func 
            [Argument1 !v; Argument1 (Py_exec.py_get_seq_elem interp seq (PyInt i))]
        done;
        !v
;;

let py_reload     
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
raise (NotImplemented "reload")
;;

let py_repr       
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let a = arg e 0 in 
  try 
    let str_func = Py_exec.get_attribute interp a (PyString "__repr__")
    (* note that this is a bound method, so the call gets no arguments *)
    in Py_exec.py_call interp str_func []
  with _ -> PyString (Py_functions.repr a)
;;

let round x =
  if x >= 0.0
  then float_of_int (truncate (x +. 0.5))
  else float_of_int (truncate (x -. 0.5))
;;

(* NotImplemented: support for Rational type *)
let py_round      
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atleast e 1;
  atmost e 2;
  let a = arg e 0 in
  let b = match optional e 1 with 
    | Some v -> v 
    | _ -> PyInt 0 
  in match b with 
  | PyInt n ->
    let v = match a with 
    | PyFloat v -> v 
    | _ -> raise (TypeError "round requires float argument 1")
    in if n = 0 
      then PyFloat (round v)
      else let n' = float_of_int n in
        PyFloat (  (round (v *. 10.0 ** n')) *. 10.0 ** (-. n') )
  | _ -> raise (TypeError "round requires int argument 2")
;;

let py_setattr    
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 3;
  let k = arg e 1 and v = arg e 2 in
  ignore begin match arg e 0 with
  | PyClass c -> c#get_dictionary#set_item k v
  | PyInstance i -> i#set_attr k v
  | PyModule m -> m#get_dictionary#set_item k v
  | _ -> raise (TypeError "object of setattr must be class, instance or module")
  end;
  PyNone
;;

let py_slice      
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  py_xrange interp e d (* temporary hack *)
;;

(* this function hooks the __str__ methods of classes *)
(* NOTE: if a __str__ method is found and fails, we fall back 
   on the primitive str: NO EXCEPTION IS GENERATED.
   NOTE: the __str__ method need not return a string!
*)

let py_str        
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let a = arg e 0 in 
  try 
    let str_func = Py_exec.get_attribute interp a (PyString "__str__")
    (* note that this is a bound method, so the call gets no arguments *)
    in Py_exec.py_call interp str_func []
  with _ -> PyString (Py_functions.str a)
;;

let py_tuple      
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let a = arg e 0 in
  py_tuple_of_sequence a
;;

let py_type       
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  raise (SystemError "builtin 'type' not replaced by 'py_types.type'")
;;

(* extensions and methods *)

let py_erepr       
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let a = arg e 0 in 
  try 
    let str_func = Py_exec.get_attribute interp a (PyString "__erepr__")
    (* note that this is a bound method, so the call gets no arguments *)
    in Py_exec.py_call interp str_func []
  with _ -> PyString (Py_functions.erepr a)
;;


let py_trace_on
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  interp#incr_tracecount;
  PyNone
;;

let py_trace_off
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  interp#decr_tracecount;
  PyNone
;;

let to_rational x = 
match x with
| PyInt x' -> Num.num_of_int x'
| PyLong x' -> Num.num_of_big_int x'
(* we need to hack some conversion from float *)
| PyRational x' -> x'
| _ -> raise (TypeError "Rational requires int, long, rational")

let py_rational
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atleast e 1;
  atmost e 2;
  let x = to_rational (arg e 0)
  in let y = match optional e 1 with
    | Some y' -> to_rational y'
    | None -> Num.Int 1
  in PyRational (Num.div_num x y)
;;

let py_bind_function
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let f' = arg e 0 in
  match f' with 
  | PyFunction f -> Py_bind.bind_function f VarMap.empty
  | _ -> raise (TypeError "bind_function requires function argument")
;;

let py_bind_module
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 1;
  let f' = arg e 0 in
  match f' with 
  | PyModule m -> Py_bind.bind_module m
  | _ -> raise (TypeError "bind_module requires module argument")
;;

(* MACROS *)
let py_globals    
  (env:environment_t) 
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atmost e 1;
  PyDictionary begin
    match optional e 0 with
    | Some (PyEnv env) -> env#get_globals
    | None -> env#get_globals
    | _ -> raise (TypeError "globals requires optional arg1 be environment")
  end

let py_locals     
  (env:environment_t) 
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atmost e 1;
  PyDictionary begin
    match optional e 0 with
    | Some (PyEnv env) -> env#get_locals
    | None -> env#get_locals
    | _ -> raise (TypeError "locals requires optional arg1 be environment")
  end

let py_env
  (env:environment_t) 
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  exactly e 0;
  PyEnv env

let py_dir        
  (env:environment_t) 
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atmost e 1;
  let a = optional e 0 in
  PyMutableList begin
    Varray.of_list begin match a with 
    | Some v ->
      begin match v with
      | PyDictionary d -> d#keys
      | PyModule m -> m#get_dictionary#keys
      | PyClass c -> c#get_dictionary#keys
      | PyInstance i -> i#get_dictionary#keys
      | _ -> raise (TypeError "Argument to 'dir' must be dictionary, module, class or instance")
      end
    | None ->
      env#keys
  end end

let py_vars       
  (env:environment_t) 
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atmost e 1;
  match optional e 0 with
  | Some obj ->
    begin match obj with
    | PyDictionary _  -> obj
    | PyInstance i -> PyDictionary i#get_dictionary
    | PyClass c -> PyDictionary c#get_dictionary
    | PyModule m -> PyDictionary m#get_dictionary
    | _ -> raise (TypeError "vars requires dictionary, instance, class, module or no argument")
    end
  | None ->
    PyDictionary env#get_locals
;;


let py_evaluate   
  (env:environment_t) 
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atleast e 1;
  atmost e 3;
  let s' = arg e 0
  and env' = 
    match optional e 1 with
    | Some (PyEnv x) -> 
      atmost e 2;
      x
    | Some (PyDictionary globals) ->
      begin match optional e 2 with
      | Some (PyDictionary locals) ->
        new Py_env.py_python_environment 
        interp#get_builtins_dictionary globals locals
      | None ->
        new Py_env.py_python_environment 
        interp#get_builtins_dictionary globals globals 
      | _ -> raise (TypeError "eval requires optional arg 3 be dictionary")
      end
    | None -> env
    | _ -> raise (TypeError "eval requires optional arg 2 be dictionary or environment")
   in match s' with 
   | PyString s -> 
     let expr = Py_exec_module.parse_string_as_expr s "<string>" in
     Py_exec.py_eval interp env' expr
   | PyClosure (env,e) -> Py_exec.py_eval interp env e
   | _ -> Py_exec.py_eval interp env' s'

let py_execfile   
  (env:environment_t) 
  (interp:interpreter_t) 
  (e:expr_t list) 
  (d:dictionary_t): expr_t  = 
  empty_dict d;
  atleast e 1;
  atmost e 3;
  let filename = arg e 0 in
  let filename' = 
    match filename with 
    |  PyString f -> f 
    | _ -> raise (TypeError "String required for filename in execfile")
  in let env' = 
    match optional e 1 with
    | Some (PyEnv x) -> 
      atmost e 2;
      x
    | Some (PyDictionary globals) ->
      begin match optional e 2 with
      | Some (PyDictionary locals) ->
        new Py_env.py_python_environment 
        interp#get_builtins_dictionary globals locals 
      | None ->
        new Py_env.py_python_environment 
        interp#get_builtins_dictionary globals globals 
      | _ -> raise (TypeError "execfile requires optional arg 3 be dictionary")
      end
    | None -> env
    | _ -> raise (TypeError "execfile requires optional arg 2 be dictionary or environment")
  in let exec = Py_exec.py_exec interp env'
  in 
    Py_exec_module.exec_file interp exec filename'
    ;
    PyNone


let create_builtins_dictionary () =
  let d = new Py_dict.py_dictionary in

  (* MACROS *)
  List.iter
  begin fun x -> 
    ignore (d#set_item
      (PyString (fst x))
      (PyNativeMacro(fst x, snd x)) 
    )
  end
  [
    ("globals",   py_globals   );
    ("locals",    py_locals    ); 
    ("env",       py_env);
    ("eval",      py_evaluate  ); 
    ("execfile",  py_execfile  ); 
    ("dir",       py_dir       ); 
    ("vars",      py_vars      )
  ]
  ;

  (* FUNCTIONS *)
  List.iter
  begin fun x -> 
    ignore (d#set_item
      (PyString (fst x))
      (PyNativeFunction (fst x, snd x)) 
    )
  end
  [
    ("abs",       py_abs       );
    ("apply",     py_apply     );  
    ("buffer",    py_buffer    );  
    ("callable",  py_callable  );   
    ("chr",       py_chr       ); 
    ("cmp",       py_cmp       ); 
    ("coerce",    py_coerce    ); 
    ("compile",   py_compile   ); 
    ("complex",   py_complex   ); 
    ("delattr",   py_delattr   ); 
    ("divmod",    py_divmod    ); 
    ("exit",      py_exit      ); 
    ("filter",    py_filter    ); 
    ("float",     py_float     ); 
    ("getattr",   py_getattr   ); 
    ("hasattr",   py_hasattr   ); 
    ("hash",      py_hash      ); 
    ("hex",       py_hex       ); 
    ("id",        py_id        ); 
    ("input",     py_input     ); 
    ("intern",    py_intern    ); 
    ("int",       py_int       ); 
    ("isinstance",py_isinstance); 
    ("issubclass",py_issubclass); 
    ("len",       py_len       ); 
    ("list",      py_list      ); 
    ("long",      py_long      ); 
    ("map",       py_map       ); 
    ("max",       py_max       ); 
    ("min",       py_min       ); 
    ("oct",       py_oct       ); 
    ("ord",       py_ord       ); 
    ("range",     py_range     ); 
    ("raw_input", py_raw_input ); 
    ("reduce",    py_reduce    ); 
    ("reload",    py_reload    ); 
    ("repr",      py_repr      ); 
    ("round",     py_round     ); 
    ("setattr",   py_setattr   ); 
    ("slice",     py_slice     ); 
    ("str",       py_str       ); 
    ("tuple",     py_tuple     ); 
    ("type",      py_type      ); 
    ("xrange",    py_xrange    );


    (* files *)
    ("file_open",          Py_file.py_file_open ); 
    ("file_get_std_files", Py_file.py_file_get_std_files); 
    ("file_flush",         Py_file.py_file_flush); 
    ("file_close",         Py_file.py_file_close); 
    ("file_read",          Py_file.py_file_read); 
    ("file_write",         Py_file.py_file_write); 

    (* other posix stuff *)


    ("posix_chdir",   Py_posix.py_posix_chdir);
    ("posix_chmod",   Py_posix.py_posix_chmod);
    ("posix_chown",   Py_posix.py_posix_chown);
    ("posix_dup",   Py_posix.py_posix_dup);
    ("posix_dup2",   Py_posix.py_posix_dup2);
    ("posix_execv",   Py_posix.py_posix_execv);
    ("posix_execve",   Py_posix.py_posix_execve);
    ("posix__exit",   Py_posix.py_posix_exit);
    ("posix_fdopen",   Py_posix.py_posix_fdopen);
    ("posix_fork",   Py_posix.py_posix_fork);
    ("posix_fstat",   Py_posix.py_posix_fstat);
    ("posix_ftruncate",   Py_posix.py_posix_ftruncate);
    ("posix_getcwd",   Py_posix.py_posix_getcwd);
    ("posix_getegid",   Py_posix.py_posix_getegid);
    ("posix_geteuid",   Py_posix.py_posix_geteuid);
    ("posix_getgid",   Py_posix.py_posix_getgid);
    ("posix_getpgrp",   Py_posix.py_posix_getpgrp);
    ("posix_getpid",   Py_posix.py_posix_getpid);
    ("posix_getppid",   Py_posix.py_posix_getppid);
    ("posix_getuid",   Py_posix.py_posix_getuid);
    ("posix_kill",   Py_posix.py_posix_kill);
    ("posix_link",   Py_posix.py_posix_link);
    ("posix_listdir",   Py_posix.py_posix_listdir);
    ("posix_lseek",   Py_posix.py_posix_lseek);
    ("posix_lstat",   Py_posix.py_posix_lstat);
    ("posix_mkfifo",   Py_posix.py_posix_mkfifo);
    ("posix_mkdir",   Py_posix.py_posix_mkdir);
    ("posix_nice",   Py_posix.py_posix_nice);
    ("posix_pipe",   Py_posix.py_posix_pipe);
    ("posix_plock",   Py_posix.py_posix_plock);
    ("posix_popen",   Py_posix.py_posix_popen);
    ("posix_putenv",   Py_posix.py_posix_putenv);
    ("posix_strerror",   Py_posix.py_posix_strerror);
    ("posix_readlink",   Py_posix.py_posix_readlink);
    ("posix_rmdir",   Py_posix.py_posix_rmdir);
    ("posix_setgid",   Py_posix.py_posix_setgid);
    ("posix_setpgrp",   Py_posix.py_posix_setpgrp);
    ("posix_setpgid",   Py_posix.py_posix_setpgid);
    ("posix_setsid",   Py_posix.py_posix_setsid);
    ("posix_setuid",   Py_posix.py_posix_setuid);
    ("posix_stat",   Py_posix.py_posix_stat);
    ("posix_symlink",   Py_posix.py_posix_symlink);
    ("posix_system",   Py_posix.py_posix_system);
    ("posix_tcgetpgrp",   Py_posix.py_posix_tcgetpgrp);
    ("posix_tcsetpgrp",   Py_posix.py_posix_tcsetpgrp);
    ("posix_times",   Py_posix.py_posix_times);
    ("posix_umask",   Py_posix.py_posix_umask);
    ("posix_uname",   Py_posix.py_posix_uname);
    ("posix_unlink",   Py_posix.py_posix_unlink);
    ("posix_utime",   Py_posix.py_posix_utime);
    ("posix_wait",   Py_posix.py_posix_wait);
    ("posix_waitpid",   Py_posix.py_posix_waitpid);

    (* time *)
    ("time_time",         Py_time.py_time_time);
    ("time_clock",        Py_time.py_time_clock);
    ("time_gmtime",       Py_time.py_time_gmtime);
    ("time_localtime",    Py_time.py_time_localtime);
    ("time_asctime",      Py_time.py_time_asctime);
    ("time_sleep",        Py_time.py_time_sleep);

    (* list methods *)
    ("list_append",        Py_list_methods.py_list_append); 
    ("list_extend",        Py_list_methods.py_list_extend); 
    ("list_count",        Py_list_methods.py_list_count); 
    ("list_index",        Py_list_methods.py_list_index); 
    ("list_insert",        Py_list_methods.py_list_insert); 
    ("list_sort",        Py_list_methods.py_list_sort); 

    (* string methods *)
    ("string_strip",   Py_string_methods.py_string_strip); 
    ("string_rstrip",   Py_string_methods.py_string_rstrip); 
    ("string_lstrip",   Py_string_methods.py_string_lstrip); 
    ("string_split",   Py_string_methods.py_string_split); 
    ("string_join",   Py_string_methods.py_string_join); 
    ("string_find",   Py_string_methods.py_string_find); 
    
    (* dictionary methods *)
    ("dictionary_clear",   Py_dict_methods.py_dictionary_clear); 
    ("dictionary_copy",    Py_dict_methods.py_dictionary_copy); 
    ("dictionary_has_key", Py_dict_methods.py_dictionary_has_key); 
    ("dictionary_items",   Py_dict_methods.py_dictionary_items); 
    ("dictionary_keys",    Py_dict_methods.py_dictionary_keys); 
    ("dictionary_update",  Py_dict_methods.py_dictionary_update); 
    ("dictionary_values",  Py_dict_methods.py_dictionary_values); 
    ("dictionary_get",     Py_dict_methods.py_dictionary_get); 

    (* python compatible marshal [excluding code objects!] *)
    ("pmarshal_dump",     Py_marshal.py_pmarshal_dump);
    ("pmarshal_load",     Py_marshal.py_pmarshal_load);
    
    (* python compatible struct *)
    ("struct_pack",       Py_struct.py_struct_pack);
    ("struct_unpack",     Py_struct.py_struct_unpack);
    ("struct_calcsize",   Py_struct.py_struct_calcsize);

    (* regular expressions *)
    ("re_compile",         Py_pcre.py_re_compile);
    ("re_match",           Py_pcre.py_re_match);

    (* Viper extensions *)
    ("erepr",         py_erepr     ); 
    ("uord",          py_uord      ); 
    ("uchr",          py_uchr      );
    ("rational",      py_rational  );
    ("trace_on",      py_trace_on  );
    ("trace_off",     py_trace_off );
    ("bind_function", py_bind_function);
    ("bind_module",   py_bind_module);

    (* Graphics *)
    ("font_load", Py_gdk.py_font_load);
    ("string_extents", Py_gdk.py_string_extents);
    ("rgb_color", Py_gdk.py_create_color_rgb);

    (* GUI *)
    ("gui_mainloop", Py_gtk.py_gui_mainloop);
    ("gui_quit", Py_gtk.py_gui_quit);

    (* widgets *)
    ("widget_create", Py_gtk.py_widget_create);
    ("widget_destroy", Py_gtk.py_widget_destroy);
    ("widget_show", Py_gtk.py_widget_show);
    ("widget_hide", Py_gtk.py_widget_hide);
    ("widget_realize", Py_gtk.py_widget_realize);
    ("widget_unrealize", Py_gtk.py_widget_unrealize);
    ("widget_reparent", Py_gtk.py_widget_reparent);
    ("widget_set_usize", Py_gtk.py_widget_set_usize);
    ("signal_connect", Py_gtk.py_signal_connect);

    (* ranges *)
    ("range_get_adjustment", Py_gtk.py_range_get_adjustment);
    ("range_set_adjustment", Py_gtk.py_range_set_adjustment);

    (* containers *)
    ("container_add", Py_gtk.py_container_add);
    ("container_remove", Py_gtk.py_container_remove);
    ("pane_add", Py_gtk.py_pane_add);

    (* file selection *)
    ("file_selection_getattr", Py_gtk.py_file_selection_getattr);
    ("file_selection_setattr", Py_gtk.py_file_selection_setattr);

    (* text *)
    ("text_get", Py_gtk.py_text_get_text);
    ("text_replace", Py_gtk.py_text_replace);
    ("text_length", Py_gtk.py_text_length);
    ("text_getattr", Py_gtk.py_text_getattr);
    ("text_setattr", Py_gtk.py_text_setattr);
    
    (* notebook *)
    ("notebook_setattr", Py_gtk.py_notebook_setattr);
    ("notebook_insert_page", Py_gtk.py_notebook_insert_page);
    ("notebook_remove_page", Py_gtk.py_notebook_remove_page);
    ("notebook_set_page", Py_gtk.py_notebook_set_page);

    (* clist *)
    ("clist_insert", Py_gtk.py_clist_insert);

    (* scrolled window *)
    ("scrolled_window_add_with_viewport", Py_gtk.py_scrolled_window_add_with_viewport)

   ];

   ignore (d#set_item (PyString "Ellipsis") py_ellipsis);

   (* return *) d;
;;


