/* 
* This is an interface to the miniSQL database for Python.
* Based on a prior work by Anthony Baxter
* Updated, fixed and extended by David Gibson working for
* Thawte Consulting cc, South Africa.
*
* Copyright 1995 Thawte Consulting cc
* Portions copyright (C) 1994 Anthony Baxter.
*
* Permission is hereby granted, free of charge, to any person obtaining
* a copy of this source file to use, copy, modify, merge, or publish it
* subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or in any new file that contains a substantial portion of
* this file.
*
* THE AUTHOR MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF
* THE SOFTWARE FOR ANY PURPOSE. IT IS PROVIDED "AS IS" WITHOUT
* EXPRESS OR IMPLIED WARRANTY. THE AUTHOR DISCLAIMS ALL WARRANTIES
* WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
* OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NON-INFRINGEMENT OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE
* AUTHOR BE LIABLE TO YOU OR ANY OTHER PARTY FOR ANY SPECIAL,
* INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
* WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE, STRICT LIABILITY OR
* ANY OTHER ACTION ARISING OUT OF OR IN CONNECTION WITH THE USE OR
* PERFORMANCE OF THIS SOFTWARE.
*
* $Id: mSQL.c,v 1.1.1.1 1998/04/06 15:57:18 mdt Exp $
******************************************************

Modified by David Gibson December 1995

- listdbs and listtables now return a list of strings
- new Python naming conventions introduced
- queries now return data types in native Python format (String,Float,Int)
- solved spurious 'function requires at least one argument' error: old
 getargs would not handle optional arguments. ParseTuple is being used now.
 (so method table has got 1's in now)
 (old Parse routine still needed for subscript handling)
- msqlFreeResult now called after query!
- assignment to subscript trapped correctly. Ditto len()
- added DbType to the module dictionary
- mSQL.error object introduced

Modified by Jeffrey C. Ollie January 1997 to work with mSQL 2.0

- Queries can no longer be executed using the subscript syntax. I'm willing to put it back if someone needs it desperately.
- Connection objects have five read-only data members: 
	1.hostname - the connection string passed when the connection object is created. 
	2.serverinfo - the serverinfo string returned by the server when the connection is made. 
	3.hostinfo - string describing how the connection to the server was made. 
	4.protoinfo - the protocol version that is returned by the server when the connection is made. 
	5.dbname - the name of the database that is currently selected. 
	6.lastquery - a copy of the last query successfully executed. 
- Connection objects have several new methods: 
	1.createdb - create a database. 
	2.dropdb - drop a database. 
	3.shutdown - shut down the server. 
	4.reloadacls - reload the access control lists. 
	5.listindex - list the structure of an index. 
	6.close - close the connection to the server. 
- Memory leaks fixed. 
- Improved error detection and reporting. 
- see http://www.ollie.clive.ia.us/python/msql/

Modified by Michael Dietrich, Spring 1998

- fixed include files.
- PyMethodDef msql_methods contains now all four fields with
  documentation. doc is the readme.
- provide a makefile.
- added select/fetch and dictionary
- added error-strings
- added DECREF
- removed close(): use db=None instead
*/

/*#define DEBUG_MSQLMOD*/

#include <python/Python.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>

#include <msql.h>

#ifdef IS_UNIQUE
#define USING_MSQL2
#endif /* IS_UNIQUE */

static char *msqlmod_version = "v2.1.3";

static PyObject *pythonify_res();
static PyObject *pythonify_single_res();
static PyObject *pythonify_lf_res();

typedef struct
{
	PyObject_HEAD
	int handle;
	int valid;
	PyObject *hostname;
	PyObject *serverinfo;
	PyObject *hostinfo;
	PyObject *protoinfo;
	PyObject *dbname;
	PyObject *lastquery;
	m_result *res;
	int n;
	char** names;
	int* type;
} msqlobject;

staticforward PyTypeObject MsqlType;
static PyObject * mSQLError;

#define is_msqlobject(v) ((v)->ob_type == &MsqlType)

static void 
msqlobj_closecursor(self)
	msqlobject *self;
{
	if (self->res != NULL)
	{
		int i;
		msqlFreeResult(self->res);
		self->res = NULL;
		for (i=0;i<self->n;i++)
			free(self->names[i]);
		free(self->names);
		self->names = NULL;
		free(self->type);
		self->type = NULL;
		self->n = 0;
	}
}

static PyObject *
msqlmod_connect(self, args)
	PyObject *self;
	PyObject *args;
{
	char *hostname = NULL;
	msqlobject *n;
	int newhandle;
	
	if (!PyArg_ParseTuple(args, "|s", &hostname))
	{
		PyErr_SetString(mSQLError, "connect has one optional argument, a database host name");
		return NULL;
	}
	if ((newhandle = msqlConnect(hostname)) == -1)
	{
		PyErr_SetString(mSQLError, msqlErrMsg); 
		return NULL;
	}
	if ((n = PyObject_NEW(msqlobject, &MsqlType)) == NULL)
	{
		PyErr_SetString(mSQLError, "no new py-object");
		return NULL;
	}
	n->valid = 1;
	n->handle = newhandle;
	n->hostname = NULL;
	n->serverinfo = NULL;
	n->hostinfo = NULL;
	n->protoinfo = NULL;
	n->dbname = NULL;
	n->lastquery = NULL;
	n->res = NULL;
	n->n = 0;
	if (hostname == NULL)
	{
		Py_INCREF(Py_None);
		n->hostname = Py_None;
	}
	else
	{
		n->hostname = PyString_FromString(hostname);
		if (n->hostname == NULL)
		{
			Py_DECREF(n);
			PyErr_SetString(mSQLError, "python error");
			return NULL;
		}
	}
	n->serverinfo = PyString_FromString(msqlGetServerInfo());
	if (n->serverinfo == NULL)
	{
		Py_DECREF(n->hostname);
		n->hostname = NULL;
		Py_DECREF(n);
		PyErr_SetString(mSQLError, "python error");
		return NULL;
	}
	n->hostinfo = PyString_FromString(msqlGetHostInfo());
	if (n->hostinfo == NULL)
	{
		Py_DECREF(n->hostname);
		n->hostname = NULL;
		Py_DECREF(n->serverinfo);
		n->serverinfo = NULL;
		Py_DECREF(n);
		PyErr_SetString(mSQLError, "python error");
		return NULL;
	}
	n->protoinfo = PyInt_FromLong(msqlGetProtoInfo());
	if (n->protoinfo == NULL)
	{
		Py_DECREF(n->hostname);
		n->hostname = NULL;
		Py_DECREF(n->serverinfo);
		n->serverinfo = NULL;
		Py_DECREF(n->hostinfo);
		n->hostinfo = NULL;
		Py_DECREF(n);
		PyErr_SetString(mSQLError, "python error");
		return NULL;
	}
	Py_INCREF(Py_None);
	n->dbname = Py_None;
	Py_INCREF(Py_None);
	n->lastquery = Py_None;
	return (PyObject *)n;
}

static PyObject *
msqlobj_selectdb(self,args)
	msqlobject *self;
	PyObject *args;
{
	char *dbname;
	PyObject *newdbname;
	PyObject *olddbname;

	if (!PyArg_ParseTuple(args, "s", &dbname))
	{
		PyErr_SetString(mSQLError, "selectdb has one argument, a database name"); 
		return NULL;
	}
	if (!self->valid)
	{
		PyErr_SetString(mSQLError, "connection has been closed or lost");
		return NULL;
	}
	olddbname = self->dbname;
	newdbname = PyString_FromString(dbname);
	if (newdbname == NULL)
	{
		PyErr_SetString(mSQLError, "python error");
		return NULL;
	}
	if (msqlSelectDB(self->handle, dbname) == -1)
	{
		Py_DECREF(newdbname);
		PyErr_SetString(mSQLError,msqlErrMsg);
		return NULL;
	}
	self->dbname = newdbname;
	Py_DECREF(olddbname);
	Py_INCREF(Py_None);
	return Py_None;
} 

/*
	* Take an mSQL m_result, turn it into a list of strings.
	*/

static PyObject *
pythonify_single_res(res)
	m_result *res;
{
	PyObject *reslist;
	PyObject *rowtuple;
	PyObject *str;
	m_row thisrow;
	int i;
	int n;

#ifdef DEBUG_MSQLMOD
	printf("data ready, %d rows of %d fields\n",
				msqlNumRows(res),
				msqlNumFields(res));
#endif /* DEBUG_MSQLMOD */
	reslist = PyList_New(0);
	n = msqlNumFields(res);
	if (n != 1)
	{
		PyErr_SetString(mSQLError, "expected mSQL to return singletons"); 
		return NULL;
	}
	while(thisrow = msqlFetchRow(res))
	{
		str = PyString_FromString(thisrow[0]);
		if (str == NULL)
		{
			Py_DECREF(reslist);
			return NULL;
		}
		PyList_Append(reslist, str);
		Py_DECREF(str);
	}
	return reslist;
}

static PyObject * 
msqlobj_listdbs(self, args)
	msqlobject *self;
	PyObject *args;
{
	m_result *res;
	PyObject *resobj;
	
	if (!PyArg_ParseTuple(args, ""))
	{
		PyErr_SetString(mSQLError, "listdbs takes no arguments");
		return NULL;
	}
	if (!self->valid)
	{
		PyErr_SetString(mSQLError, "connection has been closed or lost");
		return NULL;
	}
	if ((res = msqlListDBs(self->handle)) == NULL)
	{
		/******** Need Better Error Reporting??? ********/
		Py_INCREF(Py_None);
		return Py_None;
	}
	resobj = pythonify_single_res(res);
	msqlFreeResult(res);
	return resobj;
}

static PyObject * 
msqlobj_listtables(self,args)
	msqlobject *self;
	PyObject *args;
{
	m_result *res;
	PyObject *resobj;

	if (!PyArg_ParseTuple(args, ""))
	{
		PyErr_SetString(mSQLError, "listtables takes no arguments");
		return NULL;
	}
	if (!self->valid)
	{
		PyErr_SetString(mSQLError, "connection has been closed or lost");
		return NULL;
	}
	if (self->dbname == Py_None)
	{
		PyErr_SetString(mSQLError, "must select a database first");
		return NULL;
	}
	if ((res = msqlListTables(self->handle)) == NULL)
	{
		/********* Need Better Error Reporting *****/
		Py_INCREF(Py_None);
		return Py_None;
	}
	resobj = pythonify_single_res(res);
	msqlFreeResult(res);
	return resobj;
}

/* Take an mSQL m_result, return a list of tuples of the FetchField data.  */
static PyObject *
pythonify_lf_res(res)
	m_result *res;
{
	PyObject *reslist;
	PyObject *thistuple;
	PyObject *str;
	int i;
	int n;
	char *type;
	char flags[14];
	m_field *tf;
		
#ifdef DEBUG_MSQLMOD
	printf("data ready, %d fields\n", msqlNumFields(res));
#endif /* DEBUG_MSQLMOD */
	reslist = PyList_New(0);
	if (reslist == NULL)
		return NULL;
	n = msqlNumFields(res);
	for(i = 0; i < n; i++)
	{
		tf = msqlFetchField(res);
		switch(tf->type)
		{
			case INT_TYPE:
			{
				type="int";
				break;
			}
			case CHAR_TYPE:
			{
				type="char";
				break;
			}
			case REAL_TYPE:
			{
				type="real";
				break;
			}
#ifdef USING_MSQL2
			case TEXT_TYPE:
			{
				type="text";
				break;
			}
			case UINT_TYPE:
			{
				type="uint";
				break;
			}
			case IDENT_TYPE:
			{
				type="ident";
				break;
			}
			case NULL_TYPE:
			{
				type="null";
				break;
			}
			case IDX_TYPE:
			{
				type="idx";
				break;
			}
			case SYSVAR_TYPE:
			{
				type="sysvar";
				break;
			}
#endif /* USING_MSQL2 */
			default:
			{
				type="????";
				break;
			}
		}
#ifdef USING_MSQL2
		if (IS_UNIQUE(tf->flags))
			strcpy(flags, "unique");
#else
		if (IS_PRI_KEY(tf->flags)) 
			strcpy(flags,"pri");
#endif /* USING_MSQL2 */
		else 
			flags[0]=0;
		if (IS_NOT_NULL(tf->flags)) 
			if (flags[0])
				strcat(flags," notnull");
			else
				strcpy(flags,"notnull");
		else 
			flags[0]=0;
		thistuple = Py_BuildValue("(sssis)", tf->name, tf->table, type, tf->length, flags);
		if (thistuple == NULL)
		{
			Py_DECREF(reslist);
			return NULL;
		}
		PyList_Append(reslist, thistuple);
		Py_DECREF(thistuple);
	}
	return reslist;
}

static PyObject *
msqlobj_listfields(self,args)
	msqlobject *self;
	PyObject *args;
{
	char *tname;
	m_result *res;
	PyObject *resobj;
	
	if (!PyArg_ParseTuple(args, "s", &tname))
	{
		PyErr_SetString(mSQLError, "listfields takes one arg, a table name");
		return NULL;
	}
	if (!self->valid)
	{
		PyErr_SetString(mSQLError, "connection has been closed or lost");
		return NULL;
	}
	if (self->dbname == Py_None)
	{
		PyErr_SetString(mSQLError, "must select a database first");
		return NULL;
	}
	if ((res = msqlListFields(self->handle, tname)) == NULL)
	{
		PyErr_SetString(mSQLError, "mSQL error");
		return NULL;
	}
	resobj = pythonify_lf_res(res);
	msqlFreeResult(res);
	return resobj;
}

#ifdef USING_MSQL2
static PyObject *
msqlobj_listindex(self, args)
	msqlobject *self;
	PyObject *args;
{
	char *tname;
	char *iname;
	m_result *res;
	PyObject *resobj;

	if (!PyArg_ParseTuple(args, "ss", &tname, &iname))
	{
		PyErr_SetString(mSQLError, "listindex takes two arguments, a table name and an index name");
		return NULL;
	}
	if (!self->valid)
	{
		PyErr_SetString(mSQLError, "connection has been closed or lost");
		return NULL;
	}
	if (self->dbname == Py_None)
	{
		PyErr_SetString(mSQLError, "must select a database first");
		return NULL;
	}
	if ((res = msqlListIndex(self->handle, tname, iname)) == NULL)
	{
		/************* Need Better Error Reporting ****/
		Py_INCREF(Py_None);
		return Py_None;
	}
	resobj = pythonify_single_res(res);
	msqlFreeResult(res);
	return resobj;
}
#endif /* USING_MSQL2 */

/* Take an mSQL m_result, turn it into a list of tuples.  */
static PyObject *
pythonify_res(res)
	m_result *res;
{
	PyObject *reslist;
	PyObject *rowtuple;
	PyObject *fieldobj;
	m_row thisrow;
	m_field *tf;
	int i;
	int n;

#ifdef DEBUG_MSQLMOD
	printf("data ready, %d rows of %d fields\n",
				msqlNumRows(res),
				msqlNumFields(res));
#endif /* DEBUG_MSQLMOD */
	reslist = PyList_New(0);
	n = msqlNumFields(res);
	while(thisrow = msqlFetchRow(res))
	{
		rowtuple = PyTuple_New(n);
		if (rowtuple == NULL)
		{
			Py_DECREF(reslist);
			return NULL;
		}
		msqlFieldSeek(res, 0);
		for(i=0; i<n; i++)
		{
			tf = msqlFetchField(res);
			if (thisrow[i])
				switch(tf->type)
				{
#ifdef USING_MSQL2
					case UINT_TYPE:
#endif /* USING_MSQL2 */
					case INT_TYPE:
					{
						fieldobj = PyInt_FromLong(atol(thisrow[i]));
						break;
					}
#ifdef USING_MSQL2
					case TEXT_TYPE:
#endif /* USING_MSQL2 */
					case CHAR_TYPE:
					{
						fieldobj = PyString_FromString(thisrow[i]);
						break;
					}
					case REAL_TYPE:
					{
						fieldobj = PyFloat_FromDouble(atof(thisrow[i]));
						break;
					}
					default: 
					{
						if ((fieldobj = PyTuple_New(2)) == NULL)
							break;
						PyTuple_SET_ITEM(fieldobj, 0, PyInt_FromLong(tf->type));
						if (PyTuple_GET_ITEM(fieldobj, 0) == NULL)
						{
							Py_DECREF(fieldobj);
							fieldobj = NULL;
							break;
						}
						PyTuple_SET_ITEM(fieldobj, 1, PyString_FromString(thisrow[i]));
						if (PyTuple_GET_ITEM(fieldobj, 1) == NULL)
						{
							Py_DECREF(fieldobj);
							fieldobj = NULL;
							break;
						}
/* fieldobj = NULL; */
/* PyErr_SetString(mSQLError, "mSQL database returned bad value"); */
						break;
					}
				}
			else
			{
				Py_INCREF(Py_None);
				fieldobj = Py_None;
			}
			if (fieldobj == NULL)
			{
				Py_DECREF(rowtuple);
				Py_DECREF(reslist);
				return NULL;
			}
			PyTuple_SetItem(rowtuple, i, fieldobj);
			Py_DECREF(fieldobj);
		}
		PyList_Append(reslist, rowtuple);
		Py_DECREF(rowtuple);
	}
	return reslist;
}

static PyObject * 
msqlobj_queryselect(self, args, prefetch)
	msqlobject *self;
	PyObject *args;
	int prefetch;
{
	m_result *res;
	PyObject *resobj;
	PyObject *oldquery;
	PyObject *newquery;
	char *query;

	if (self->res)
		msqlobj_closecursor(self);
	if (!PyArg_ParseTuple(args, "s", &query))
	{
		PyErr_SetString(mSQLError, "query has one arg, a query string");
		return NULL;
	}
	if (!self->valid)
	{
		PyErr_SetString(mSQLError, "connection has been closed or lost");
		return NULL;
	}
	if (self->dbname == Py_None)
	{
		PyErr_SetString(mSQLError, "must select a database first");
		return NULL;
	}
	oldquery = self->lastquery;
	newquery = PyString_FromString(query);
	if (newquery == NULL)
		return NULL;
	if (msqlQuery(self->handle, query) == -1)
	{
		PyErr_SetString(mSQLError, msqlErrMsg); 
		return NULL;
	}
	res = msqlStoreResult();
	if (!res)
	{
		/***** better error reporting *****/
		Py_DECREF(newquery);
		Py_INCREF(Py_None);
		return Py_None;
	}
	if (prefetch)
	{
		resobj = pythonify_res(res);
		msqlFreeResult(res);
	}
	else
	{
		int i;
		int n;
		n =
		self->n = msqlNumFields(res);
		self->res = res;
		if ((self->type = malloc(sizeof(*self->type) * n)) == NULL
		 || (self->names = malloc(sizeof(*self->names) * n)) == NULL)
		{
			msqlFreeResult(res);
			PyErr_SetString(mSQLError, "out of memory");
			return NULL;
		}
		msqlFieldSeek(res, 0);
		for(i=0; i<n; i++)
		{
		m_field *tf;
			tf = msqlFetchField(res);
			if ((self->names[i] = strdup(tf->name)) == NULL)
			{
				msqlFreeResult(res);
				PyErr_SetString(mSQLError, "out of memory");
				return NULL;
			}
			self->type[i] = tf->type;
		}
		Py_INCREF(Py_None);
		resobj = Py_None;
	}
	self->lastquery = newquery;
	Py_DECREF(oldquery);
	return resobj;
}

static PyObject * 
msqlobj_query(self, args)
	msqlobject *self;
	PyObject *args;
{
	return msqlobj_queryselect(self, args, 0);
}

static PyObject * 
msqlobj_fetch(self, args)
	msqlobject *self;
	PyObject *args;
{
	PyObject *rowdict;
	PyObject *value;
	PyObject *key;
	m_row thisrow;
	int i;
#ifdef DEBUG_MSQLMOD
	printf("data ready, %d rows of %d fields\n",
				msqlNumRows(res),
				msqlNumFields(res));
#endif /* DEBUG_MSQLMOD */
	if (!PyArg_ParseTuple(args, ""))
	{
		PyErr_SetString(mSQLError, "fetch takes no arguments");
		return NULL;
	}
	if (!self->valid)
	{
		PyErr_SetString(mSQLError, "connection has been closed or lost");
		return NULL;
	}
	if (self->res == NULL)
	{
		PyErr_SetString(mSQLError, "no cursor");
		return NULL;
	}
	if ((thisrow = msqlFetchRow(self->res)) == NULL)
	{
		Py_INCREF(Py_None);
		return Py_None;
	}
	if ((rowdict = PyDict_New()) == NULL)
	{
		PyErr_SetString(mSQLError, "python error");
		return NULL;
	}
	for(i=0; i<self->n; i++)
	{
		if (thisrow[i])
			switch(self->type[i])
			{
#ifdef USING_MSQL2
				case UINT_TYPE:
#endif /* USING_MSQL2 */
				case INT_TYPE:
				{
					value = PyInt_FromLong(atol(thisrow[i]));
					break;
				}
#ifdef USING_MSQL2
				case TEXT_TYPE:
#endif /* USING_MSQL2 */
				case CHAR_TYPE:
				{
					value = PyString_FromString(thisrow[i]);
					break;
				}
				case REAL_TYPE:
				{
					value = PyFloat_FromDouble(atof(thisrow[i]));
					break;
				}
				default: 
				{
					Py_DECREF(rowdict);
					PyErr_SetString(mSQLError, "mSQL database returned unknown data type");
					return NULL;
				}
			}
		else
		{
			Py_INCREF(Py_None);
			value = Py_None;
		}
		if (value == NULL)
		{
			Py_DECREF(rowdict);
			PyErr_SetString(mSQLError, "python error");
			return NULL;
		}
		if ((key = PyString_FromString(self->names[i])) == NULL)
		{
			Py_DECREF(rowdict);
			PyErr_SetString(mSQLError, "python error");
			return NULL;
		}
		PyDict_SetItem(rowdict, key, value);
		Py_DECREF(value);
		Py_DECREF(key);
	}
	return rowdict;
}

static PyObject * 
msqlobj_execute(self, args)
	msqlobject *self;
	PyObject *args;
{
	return msqlobj_queryselect(self, args, 1);
}

static PyObject *
msqlobj_createdb(self, args)
	msqlobject *self;
	PyObject *args;
{
	char *dbname;

	if (!PyArg_ParseTuple(args, "s", &dbname))
	{
		PyErr_SetString(mSQLError, "createdb needs one argument, the name of the database to be created");
		return NULL;
	}
	if (!self->valid)
	{
		PyErr_SetString(mSQLError, "connection has been closed or lost");
		return NULL;
	}
	if (msqlCreateDB(self->handle, dbname) == -1)
	{
		PyErr_SetString(mSQLError, msqlErrMsg);
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
msqlobj_dropdb(self, args)
	msqlobject *self;
	PyObject *args;
{
	char *dbname;

	if (!PyArg_ParseTuple(args, "s", &dbname))
	{
		PyErr_SetString(mSQLError, "dropdb needs one argument, the name of the database to be created");
		return NULL;
	}
	if (!self->valid)
	{
		PyErr_SetString(mSQLError, "connection has been closed or lost");
		return NULL;
	}
	if (msqlDropDB(self->handle, dbname) == -1)
	{
		PyErr_SetString(mSQLError, msqlErrMsg);
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
msqlobj_close(self, args)
	msqlobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
	{
		PyErr_SetString(mSQLError, "close takes no arguments");
		return NULL;
	}
	if (!self->valid)
	{
		PyErr_SetString(mSQLError, "connection has been closed or lost");
		return NULL;
	}
	msqlobj_closecursor(self);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
msqlobj_shutdown(self, args)
	msqlobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
	{
		PyErr_SetString(mSQLError, "shutdown takes no arguments");
		return NULL;
	}
	if (!self->valid)
	{
		PyErr_SetString(mSQLError, "connection has been closed or lost");
		return NULL;
	}
	if (msqlShutdown(self->handle) == -1)
	{
		PyErr_SetString(mSQLError, msqlErrMsg);
		return NULL;
	}
	self->valid = 0;
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
msqlobj_reloadacls(self, args)
	msqlobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
	{
		PyErr_SetString(mSQLError, "reloadacls takes no arguments");
		return NULL;
	}
	if (!self->valid)
	{
		PyErr_SetString(mSQLError, "connection has been closed or lost");
		return NULL;
	}
	if (msqlReloadAcls(self->handle) == -1)
	{
		PyErr_SetString(mSQLError, msqlErrMsg);
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static struct PyMethodDef msqlobj_methods[] =
{
	{"selectdb", (PyCFunction) msqlobj_selectdb, 1, "db.selectdb(string)
	attaches this object to a particular database. Queries executed
	will be directed to that database until another selectdb method
	call is made."},
	{"shutdown", (PyCFunction) msqlobj_shutdown, 1, ""},
	{"reloadacls", (PyCFunction) msqlobj_reloadacls, 1, ""},
	{"createdb", (PyCFunction) msqlobj_createdb, 1, ""},
	{"listdbs", (PyCFunction) msqlobj_listdbs, 1, "db.listdbs()
	returns a list of strings giving the names of the databases on the
	mSQL host to which one has connected with mSQL.connect()"},
	{"dropdb", (PyCFunction) msqlobj_dropdb, 1, ""},
	{"listtables", (PyCFunction) msqlobj_listtables, 1, "db.listtables()
	return a list of strings giving the table names in the selected
	database. Only valid after a selectdb call has been made."},
#ifdef USING_MSQL2
	{"listindex", (PyCFunction) msqlobj_listindex, 1, ""},
#endif /* USING_MSQL2 */
	{"query", (PyCFunction) msqlobj_execute, 1, "db.query(string)
	execute the given SQL query string against the selected database,
	returning a list of rows, where each row is a tuple of native
	Python data objects (string, int or real), or raising an exception
	if a parserror or type error occurred."},
	{"listfields", (PyCFunction) msqlobj_listfields, 1, "db.listfields(string)
	return a description of the fields in the given table, as a list
	of tuples, where each tuple contains the same information as is
	given by the relshow command in the mSQL admin suite."},
	{"select", (PyCFunction) msqlobj_query, 1, ""},
	{"fetch", (PyCFunction) msqlobj_fetch, 1, ""},
	{"closeselect", (PyCFunction) msqlobj_close, 1, ""},
	{NULL, NULL, 0, NULL} /* sentinel */
};

static PyObject *
msqlobj_getattr(self, name)
	msqlobject *self;
	char *name;
{
	if (strcmp(name, "hostname") == 0)
	{
		Py_INCREF(self->hostname);
		return self->hostname;
	}
	if (strcmp(name, "serverinfo") == 0)
	{
		Py_INCREF(self->serverinfo);
		return self->serverinfo;
	}
	if (strcmp(name, "hostinfo") == 0)
	{
		Py_INCREF(self->hostinfo);
		return self->hostinfo;
	}
	if (strcmp(name, "protoinfo") == 0)
	{
		Py_INCREF(self->protoinfo);
		return self->protoinfo;
	}
	if (strcmp(name, "dbname") == 0)
	{
		Py_INCREF(self->dbname);
		return self->dbname;
	}
	if (strcmp(name, "lastquery") == 0)
	{
		Py_INCREF(self->lastquery);
		return self->lastquery;
	}
	return Py_FindMethod(msqlobj_methods, (PyObject *)self, name);
}

static void 
msqlobj_del(self)
	register msqlobject *self;
{
	if (self->valid)
	{
		if (self->res)
			msqlobj_closecursor(self);
		msqlClose(self->handle);
		self->valid = 0;
	}
	Py_XDECREF(self->hostname);
	Py_XDECREF(self->serverinfo);
	Py_XDECREF(self->hostinfo);
	Py_XDECREF(self->protoinfo);
	Py_XDECREF(self->dbname);
	Py_XDECREF(self->lastquery);
	PyMem_DEL(self);
} 

int
msqlobj_print(sp, fp, flags)
	msqlobject *sp;
	FILE *fp;
	int flags;
{
	fprintf(fp, "mSQL handle");
	return 0;
}

static PyTypeObject MsqlType =
{
	PyObject_HEAD_INIT(&PyType_Type)
	0,
	"msqlobject",
	sizeof(msqlobject),
	0,
	(destructor)msqlobj_del, /*del*/
	0, /*print*/
	(getattrfunc)msqlobj_getattr, /*getattr*/
	0, /*setattr*/
	0, /*compare*/
	0, /*repr*/
	0, /*as_number*/
	0, /*as_sequence*/
	0, /*as_mapping*/
}; 

static struct PyMethodDef msql_methods[] =
{
	{"connect",msqlmod_connect, 1,"
USING THE MSQL MODULE

Instruction by example:

import mSQL
database = mSQL.connect('phoenix.thawte.co.za')
print database.listdbs()
database.selectdb('testdb')
database.listfields('alltypes')
result = database.query('SELECT * FROM alltypes WHERE id=24')
print result
print database.hostname
print database.serverinfo
print database.hostinfo
print database.protoinfo
print database.dbname
print database.lastquery"},
	{"mSQL", msqlmod_connect, 1,"
USING THE MSQL MODULE

Instruction by example:

import mSQL
database = mSQL.connect('localhost')
print aDataBase.listdbs()
database.selectdb('test')
database.listfields('alltypes')
result = database.query('SELECT * FROM alltypes WHERE id=24')
print result
print database.hostname
print database.serverinfo
print database.hostinfo
print database.protoinfo
print database.dbname
print database.lastquery"},
	{NULL, NULL, 0, NULL}
};

void
initmSQL()
{
	PyObject *module;
	PyObject *dict;
	PyObject *o;
	char *err;
	
	module = Py_InitModule("mSQL", msql_methods);
	dict = PyModule_GetDict(module);
	
	if (PyDict_SetItemString(dict, "DbType", (PyObject*)&MsqlType) != 0)
		Py_FatalError("Cannot add to mSQL dictionary");
	
	mSQLError = PyString_FromString("mSQL.error");
	if (PyDict_SetItemString(dict, "error", mSQLError) != 0)
		Py_FatalError("Cannot add to mSQL dictionary");

	o = PyString_FromString(msqlmod_version);
	if (PyDict_SetItemString(dict, "version", o) != 0)
		Py_FatalError("Cannot add to mSQL dictionary");
}
