/**
 *
 * OpenGL Module for Python
 * 
 * Version: 0.7
 *
 * Date:    96/09/12 (September)
 *
 * Authors: Tom Schwaller     <tom.schwaller@linux-magazin.de>
 *          Jim Hugunin       <hugunin@mit.edu>
 *          David Ascher      <david_ascher@brown.edu>
 * 
 * Notes:
 *
 *   - the file glconst.py is also needed, as it defines all kinds of
 *     useful constants
 *     
 *   - If you do not have Numeric python installed, undefine NUMERIC
 *   - We have included the glut shapes for convenience. If you do not 
 *     want to use them undefine the GLUT constant, otherwise add 
 *     glut_shapes.c in the setup file like:
 *     opengl openglmodule.c glut_shapes.c -lGL -lGLU
 *
 *   - thanks to Mike Hartshorn <mjh@yorvic.york.ac.uk> for quite 
 *     some bug reports.
 *
 *   - Some new stuff: Hope to have more from users of this module
 *
 *     96/08/30: gl_TranslateScene, gl_RotateScene
 *     96/09/01: gl_DistFromLine (from Mike)
 *     
***/

#define NUMERIC
#define GLUT

#include "Python.h"
#ifdef MS_WIN32
#include <windows.h>
#endif
#include <GL/gl.h>
#include <math.h>

static int typecode2gltype[] = {-1, GL_UNSIGNED_BYTE, GL_BYTE, GL_SHORT, GL_INT, -1, GL_FLOAT, -1, -1, -1, -1, -1};

static PyObject *gl_Error;

static PyObject *ErrorReturn(char *message)
{
  PyErr_SetString(gl_Error,message);
  return NULL;
}
#define ASSERT(E,M) if(!(E)) return ErrorReturn(M)

#define TRY(E) if(! (E)) return NULL

#ifdef NUMERIC

#include "arrayobject.h"

#define PyArray_ClearMemory(op, pitems) Py_DECREF((op))

static int PyArray_AsDoubleArray(PyObject **op, GLdouble **pitems, int *pn)
{
    PyArrayObject *mp;
    mp = (PyArrayObject *) PyArray_ContiguousFromObject (*op, PyArray_DOUBLE, 0, 1);
    if (!mp) return 0;
    *pitems = (GLdouble*) (mp->data);
    *pn = mp->dimensions[0];
    *op = (PyObject*) mp;
    return 1;
}

static int PyArray_AsFloatArray(PyObject **op, GLfloat **pitems, int *pn)
{
    PyArrayObject *mp;
    mp = (PyArrayObject *) PyArray_ContiguousFromObject (*op, PyArray_FLOAT, 0, 1);
    if (!mp) return 0;
    *pitems = (GLfloat*) (mp->data);
    *pn = mp->dimensions[0];
    *op = (PyObject*) mp;
    return 1;
}

static int PyArray_AsIntArray(PyObject **op, GLint **pitems, int *pn)
{
    PyArrayObject *mp;
    mp = (PyArrayObject *) PyArray_ContiguousFromObject (*op, PyArray_INT, 0, 1);
    if (!mp) return 0;
    *pitems =(GLint*) (mp->data);
    *pn = mp->dimensions[0];
    *op = (PyObject*) mp;
    return 1;
}

static int PyArray_AsShortArray(PyObject **op, GLshort **pitems, int *pn)
{
    PyArrayObject *mp;
    mp = (PyArrayObject *) PyArray_ContiguousFromObject (*op, PyArray_SHORT, 0, 1);
    if (!mp) return 0;
    *pitems =(GLshort*) (mp->data);
    *pn = mp->dimensions[0];
    *op = (PyObject*) mp;
    return 1;
}

#else

#include "abstract.h"

#define PyArray_ClearMemory(op, pitems) PyMem_DEL(pitems)

static int PyArray_AsDoubleArray(PyObject **op, GLdouble **pitems, int *pn)
{
    GLdouble *items;
    PyObject *item;
    int n, i;
    if (!PySequence_Check(*op)) return 0;
    n = PySequence_Length(*op);
    items = PyMem_NEW(GLdouble, n);
    if (items == NULL) {
	PyErr_NoMemory();
	return 0;
    }
    for (i=0; i<n; i++) {
	if (item = PySequence_GetItem(*op, i)) {
	   items[i] = PyFloat_AsDouble(item);
	   Py_DECREF(item);
        }
        if(PyErr_Occurred()) return 0;   
    }
    *pitems = items;
    *pn = n;
    return 1;
}

static int PyArray_AsFloatArray(PyObject **op, GLfloat **pitems, int *pn)
{
    GLfloat *items;
    PyObject *item;
    int n, i;
    if (!PySequence_Check(*op)) return 0;
    n = PySequence_Length(*op);
    items = PyMem_NEW(GLfloat, n);
    if (items == NULL) {
	PyErr_NoMemory();
	return 0;
    }
    for (i=0; i<n; i++) {
	if(item = PySequence_GetItem(*op, i)) {
	   items[i] = PyFloat_AsDouble(item);
	   Py_DECREF(item);
        }
        if(PyErr_Occurred()) return 0;
    }
    *pitems = items;
    *pn = n;
    return 1;
}

static int PyArray_AsIntArray(PyObject **op, GLint **pitems, int *pn)
{
    GLint *items;
    PyObject *item;
    int n, i;
    if (!PySequence_Check(*op)) return 0;
    n = PySequence_Length(*op);
    items = PyMem_NEW(GLint, n);
    if (items == NULL) {
	PyErr_NoMemory();
	return 0;
    }
    for (i=0; i<n; i++) {
	if (item = PySequence_GetItem(*op, i)) {
	   items[i] = PyInt_AsLong(item);
	   Py_DECREF(item);
	}
        if(PyErr_Occurred()) return 0;    
    }
    *pitems = items;
    *pn = n;
    return 1;
}

static int PyArray_AsShortArray(PyObject **op, GLshort **pitems, int *pn)
{
    GLshort *items;
    PyObject *item;
    int n, i;
    if (!PySequence_Check(*op)) return 0;
    n = PySequence_Length(*op);
    items = PyMem_NEW(GLshort, n);
    if (items == NULL) {
	PyErr_NoMemory();
	return 0;
    }
    for (i = 0; i < n; i++) {
	if (item = PySequence_GetItem(*op, i)) {
	   items[i] = PyInt_AsLong(item);
	   Py_DECREF(item);
	}
        if(PyErr_Occurred()) return 0;
    }
    *pitems = items;
    *pn = n;
    return 1;

}

#endif /* Not NUMERIC */

/* 
########################################################################
*/

static PyObject* gl_Accum(PyObject* self, PyObject* args)
{
  	GLenum arg1;
  	GLfloat arg2;
	TRY (PyArg_ParseTuple(args, "if", &arg1, &arg2));
  	glAccum(arg1, arg2);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_AlphaFunc(PyObject* self, PyObject* args)
{
	GLenum arg1 ;
  	GLclampf arg2 ;
	TRY (PyArg_ParseTuple(args, "if", &arg1, &arg2));
  	glAlphaFunc(arg1, arg2);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Begin(PyObject* self, PyObject* args)
{
  	GLenum arg1 ;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glBegin(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_BlendFunc(PyObject* self, PyObject* args)
{
	GLenum arg1, arg2;
	TRY (PyArg_ParseTuple(args, "ii", &arg1, &arg2));
  	glBlendFunc(arg1 , arg2);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_CallList(PyObject* self, PyObject* args)
{
	GLuint	arg1;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glCallList(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Clear(PyObject* self, PyObject* args)
{
	GLbitfield arg1;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glClear(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_ClearAccum(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, arg3, arg4;
	TRY (PyArg_ParseTuple(args, "ffff", &arg1, &arg2, &arg3, &arg4));
  	glClearAccum(arg1 , arg2 , arg3 , arg4);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_ClearColor(PyObject* self, PyObject* args)
{
	GLclampf arg1, arg2, arg3, arg4;
	TRY (PyArg_ParseTuple(args, "ffff", &arg1, &arg2, &arg3, &arg4));
  	glClearColor(arg1 , arg2 , arg3 , arg4);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_ClearDepth(PyObject* self, PyObject* args)
{
	GLclampd arg1;
	TRY (PyArg_ParseTuple(args, "d", &arg1));
  	glClearDepth(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_ClearIndex(PyObject* self, PyObject* args)
{
	GLfloat arg1;
	TRY (PyArg_ParseTuple(args, "f", &arg1));
  	glClearIndex(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_ClearStencil(PyObject* self, PyObject* args)
{
	GLint arg1;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glClearStencil(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Color3d(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ddd", &arg1, &arg2, &arg3)) 	
  		glColor3d(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsDoubleArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glColor3dv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Color3f(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "fff", &arg1, &arg2, &arg3)) 	
  		glColor3f(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsFloatArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glColor3fv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Color3i(PyObject* self, PyObject* args)
{
	GLint arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iii", &arg1, &arg2, &arg3)) 	
  		glColor3i(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsIntArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glColor3iv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Color3s(PyObject* self, PyObject* args)
{
	GLshort arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "hhh", &arg1, &arg2, &arg3)) 	
  		glColor3s(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsShortArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glColor3sv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Color4d(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, arg3, arg4, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "dddd", &arg1, &arg2, &arg3, &arg4)) 	
  		glColor4d(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsDoubleArray(&op, &vert, &n));
		if (n < 4) {
			PyErr_SetString(gl_Error,  "need element with at least 4 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glColor4dv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Color4f(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, arg3, arg4, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ffff", &arg1, &arg2, &arg3, &arg4)) 	
  		glColor4f(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsFloatArray(&op, &vert, &n));
		if (n < 4) {
			PyErr_SetString(gl_Error,  "need element with at least 4 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glColor4fv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Color4i(PyObject* self, PyObject* args)
{
	GLint arg1, arg2, arg3, arg4, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iiii", &arg1, &arg2, &arg3, &arg4)) 	
  		glColor4i(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsIntArray(&op, &vert, &n));
		if (n < 4) {
			PyErr_SetString(gl_Error,  "need element with at least 4 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glColor4iv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Color4s(PyObject* self, PyObject* args)
{
	GLshort arg1, arg2, arg3, arg4, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "hh", &arg1, &arg2, &arg3, &arg4)) 	
  		glColor4s(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsShortArray(&op, &vert, &n));
		if (n < 4) {
			PyErr_SetString(gl_Error,  "need element with at least 4 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glColor4sv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_ColorMask(PyObject* self, PyObject* args)
{
	GLboolean arg1, arg2, arg3, arg4;
	TRY (PyArg_ParseTuple(args, "bbbb", &arg1, &arg2, &arg3, &arg4));
  	glColorMask(arg1 , arg2 , arg3 , arg4);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_ColorMaterial(PyObject* self, PyObject* args)
{
	GLenum arg1, arg2;
	TRY (PyArg_ParseTuple(args, "ii", &arg1, &arg2));
  	glColorMaterial(arg1 , arg2);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_CopyPixels(PyObject* self, PyObject* args)
{
	GLint arg1, arg2;
 	GLsizei arg3, arg4;
	GLenum arg5 ;
	TRY (PyArg_ParseTuple(args, "iiiii", &arg1, &arg2, &arg3, &arg4, &arg5));
  	glCopyPixels(arg1 , arg2 , arg3 , arg4 , arg5);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_CullFace(PyObject* self, PyObject* args)
{
	GLenum arg1 ;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glCullFace(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_DeleteLists(PyObject* self, PyObject* args)
{
	GLuint arg1 ;
	GLsizei arg2 ;
	TRY (PyArg_ParseTuple(args, "ii", &arg1, &arg2));
  	glDeleteLists(arg1, arg2);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_DepthFunc(PyObject* self, PyObject* args)
{
	GLenum arg1 ;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glDepthFunc(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_DepthMask(PyObject* self, PyObject* args)
{
	GLboolean arg1;
	TRY (PyArg_ParseTuple(args, "b", &arg1));
  	glDepthMask(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_DepthRange(PyObject* self, PyObject* args)
{
	GLclampd arg1, arg2;
	TRY (PyArg_ParseTuple(args, "dd", &arg1, &arg2));
  	glDepthRange(arg1 , arg2);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Disable(PyObject* self, PyObject* args)
{
	GLenum arg1 ;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glDisable(arg1);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_DrawBuffer(PyObject* self, PyObject* args)
{
	GLenum arg1 ;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glDrawBuffer(arg1);
  	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_EdgeFlag(PyObject* self, PyObject* args)
{
	GLboolean arg1 ;
	TRY (PyArg_ParseTuple(args, "b", &arg1));
  	glEdgeFlag(arg1);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Enable(PyObject* self, PyObject* args)
{
	GLenum arg1 ;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glEnable(arg1);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_End(PyObject* self, PyObject* args)
{
	TRY (PyArg_ParseTuple(args, ""));
  	glEnd();
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_EndList(PyObject* self, PyObject* args)
{
	TRY (PyArg_ParseTuple(args, ""));
  	glEndList();
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_EvalCoord1d(PyObject* self, PyObject* args)
{
	GLdouble arg1 ;
	TRY (PyArg_ParseTuple(args, "d", &arg1));
  	glEvalCoord1d(arg1);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_EvalCoord1f(PyObject* self, PyObject* args)
{
	GLfloat arg1 ;
	TRY (PyArg_ParseTuple(args, "f", &arg1));
  	glEvalCoord1f(arg1);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_EvalCoord2d(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, *coords;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "dd", &arg1, &arg2)) 	
  		glEvalCoord2d(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsDoubleArray(&op, &coords, &n));
		if (n < 2) {
			PyErr_SetString(gl_Error,  "need element with at least 2 items");
			PyArray_ClearMemory(op, coords);
			return NULL;
		}
		glEvalCoord2dv(coords);
		PyArray_ClearMemory(op, coords);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_EvalCoord2f(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, *coords;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ff", &arg1, &arg2)) 	
  		glEvalCoord2f(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsFloatArray(&op, &coords, &n));
		if (n < 2) {
			PyErr_SetString(gl_Error,  "need element with at least 2 items");
			PyArray_ClearMemory(op, coords);
			return NULL;
		}
		glEvalCoord2fv(coords);
		PyArray_ClearMemory(op, coords);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_EvalMesh1(PyObject* self, PyObject* args)
{
	GLenum arg1 ;
	GLint arg2, arg3 ;
	TRY (PyArg_ParseTuple(args, "iii", &arg1, &arg2, &arg3));
  	glEvalMesh1(arg1, arg2, arg3);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_EvalMesh2(PyObject* self, PyObject* args)
{
	GLenum arg1;
	GLint arg2, arg3, arg4, arg5;
	TRY (PyArg_ParseTuple(args, "iiiii", &arg1, &arg2, &arg3, &arg4, &arg5));
  	glEvalMesh2(arg1, arg2, arg3, arg4, arg5);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_EvalPoint1(PyObject* self, PyObject* args)
{
	GLint arg1;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
	glEvalPoint1(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_EvalPoint2(PyObject* self, PyObject* args)
{
	GLint arg1, arg2;
	TRY (PyArg_ParseTuple(args, "ii", &arg1, &arg2));
	glEvalPoint2(arg1, arg2);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Finish(PyObject* self, PyObject* args)
{
	TRY (PyArg_ParseTuple(args, ""));
  	glFinish();
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Flush(PyObject* self, PyObject* args)
{
	TRY (PyArg_ParseTuple(args, ""));
  	glFlush();
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Fogf(PyObject* self, PyObject* args)
{
	GLenum arg1;
	GLfloat arg2, *rest;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "if", &arg1, &arg2))
  		glFogf(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "iO", &arg1, &op));
		TRY (PyArray_AsFloatArray(&op, &rest, &n));
		if ((n == 1) || (n > 3)) {
			glFogfv(arg1, rest);
			PyArray_ClearMemory(op, rest);
		}
		else {
			PyErr_SetString(gl_Error,  "second argument needs 1 or at least 4 items!");
			PyArray_ClearMemory(op, rest);
			return NULL;
		}
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Fogi(PyObject* self, PyObject* args)
{
	GLenum arg1;
	GLint arg2, *rest;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ii", &arg1, &arg2))
  		glFogi(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "iO", &arg1, &op));
		TRY (PyArray_AsIntArray(&op, &rest, &n));
		if ((n == 1) || (n>3)) {
			glFogiv(arg1, rest);
			PyArray_ClearMemory(op, rest);
		}
		else {
			PyErr_SetString(gl_Error,  "second argument needs 1 or at least 4 items!");
			PyArray_ClearMemory(op, rest);
			return NULL;
		}
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_FrontFace(PyObject* self, PyObject* args)
{
	GLenum arg1;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glFrontFace(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Frustum(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, arg3, arg4, arg5, arg6;
	TRY (PyArg_ParseTuple(args, "dddddd", &arg1, &arg2, &arg3, &arg4, &arg5, &arg6));
  	glFrustum(arg1, arg2, arg3, arg4, arg5, arg6);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Hint(PyObject* self, PyObject* args)
{ 
	GLenum arg1, arg2;
	TRY (PyArg_ParseTuple(args, "ii", &arg1, &arg2));
  	glHint(arg1 , arg2);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_IndexMask(PyObject* self, PyObject* args)
{
	GLuint arg1;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glIndexMask(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Indexd(PyObject* self, PyObject* args)
{
	GLdouble arg1;
	TRY (PyArg_ParseTuple(args, "d", &arg1));
  	glIndexd(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Indexf(PyObject* self, PyObject* args)
{
	GLfloat arg1;
	TRY (PyArg_ParseTuple(args, "f", &arg1));
  	glIndexf(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Indexi(PyObject* self, PyObject* args)
{
	GLint arg1;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glIndexi(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Indexs(PyObject* self, PyObject* args)
{
	GLshort arg1;
	TRY (PyArg_ParseTuple(args, "h", &arg1));
  	glIndexs(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_InitNames(PyObject* self, PyObject* args)
{
	TRY (PyArg_ParseTuple(args, ""));
  	glInitNames();
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_LightModelf(PyObject* self, PyObject* args)
{	
	GLenum arg1;
	GLfloat arg2, *rest;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "if", &arg1, &arg2))
  		glLightModelf(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "iO", &arg1, &op));
		TRY (PyArray_AsFloatArray(&op, &rest, &n));
		if ((n == 1) || (n>3)) {
			glLightModelfv(arg1, rest);
			PyArray_ClearMemory(op, rest);
		}
		else {
			PyErr_SetString(gl_Error,  "second argument needs 1 or at least 4 items!");
			PyArray_ClearMemory(op, rest);
			return NULL;
		}
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_LightModeli(PyObject* self, PyObject* args)
{
	GLenum arg1;
	GLint arg2, *rest;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ii", &arg1, &arg2))
  		glLightModeli(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "iO", &arg1, &op));
		TRY (PyArray_AsIntArray(&op, &rest, &n));
		if ((n == 1) || (n>3)) {
			glLightModeliv(arg1, rest);
			PyArray_ClearMemory(op, rest);
		}
		else {
			PyErr_SetString(gl_Error,  "second argument needs 1 or at least 4 items!");
			PyArray_ClearMemory(op, rest);
			return NULL;
		}
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Lightf(PyObject* self, PyObject* args)
{
	GLenum arg1, arg2;
	GLfloat arg3, *rest;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iif", &arg1, &arg2, &arg3))
  		glLightf(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "iiO", &arg1, &arg2, &op));
		TRY (PyArray_AsFloatArray(&op, &rest, &n));
		if ((n == 1) || (n > 2)) {
			glLightfv(arg1, arg2, rest);
			PyArray_ClearMemory(op, rest);
		}
		else {
			PyErr_SetString(gl_Error,  "second argument needs 1 or at least 3 items!");
			PyArray_ClearMemory(op, rest);
			return NULL;
		}
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Lighti(PyObject* self, PyObject* args)
{
	GLenum arg1, arg2;
	GLint arg3, *rest;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iii", &arg1, &arg2, &arg3))
  		glLighti(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "iiO", &arg1, &arg2, &op));
		TRY (PyArray_AsIntArray(&op, &rest, &n));
		if ((n == 1) || (n > 2)) {
			glLightiv(arg1, arg2, rest);
			PyArray_ClearMemory(op, rest);
		}
		else {
			PyErr_SetString(gl_Error,  "second argument needs 1 or at least 3 items!");
			PyArray_ClearMemory(op, rest);
			return NULL;
		}
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_LineStipple(PyObject* self, PyObject* args)
{
	GLint arg1;
	GLushort arg2;
	TRY (PyArg_ParseTuple(args, "ih", &arg1, &arg2));
  	glLineStipple(arg1, arg2);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_LineWidth(PyObject* self, PyObject* args)
{
	GLfloat arg1;
 	TRY (PyArg_ParseTuple(args, "f", &arg1));
  	glLineWidth(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_ListBase(PyObject* self, PyObject* args)
{
	GLuint arg1 ;
 	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glListBase(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_LoadIdentity(PyObject* self, PyObject* args)
{
	TRY (PyArg_ParseTuple(args, ""));
  	glLoadIdentity();
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_LoadName(PyObject* self, PyObject* args)
{
	GLuint arg1;
 	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glLoadName(arg1);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_LogicOp(PyObject* self, PyObject* args)
{
	GLenum arg1;
 	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glLogicOp(arg1);
 	Py_INCREF(Py_None);
	return Py_None;
}


static PyObject* gl_Map1d(PyObject* self, PyObject* args)
{

  GLenum arg1;
  GLdouble arg2;
  GLdouble arg3;
  GLint arg4;
  GLint arg5;
  PyObject *arg6;
  GLdouble *points;
  int n;

  TRY (PyArg_ParseTuple(args, "iddiiO", &arg1, &arg2, &arg3, &arg4, &arg5, &arg6));
  TRY (PyArray_AsDoubleArray(&arg6, &points, &n));
  glMap1d(arg1, arg2, arg3, arg4, arg5, points);
  Py_DECREF(arg6);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject* gl_Map1f(PyObject* self, PyObject* args)
{
  GLenum arg1;
  GLfloat arg2;
  GLfloat arg3;
  GLint arg4;
  GLint arg5;
  PyObject *arg6;
  GLfloat *points;
  int n;

  TRY (PyArg_ParseTuple(args, "iffiiO", &arg1, &arg2, &arg3, &arg4, &arg5, &arg6));
  TRY (PyArray_AsFloatArray(&arg6, &points, &n));
  glMap1f(arg1, arg2, arg3, arg4, arg5, points);
  Py_DECREF(arg6);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject* gl_Map2d(PyObject* self, PyObject* args)
{
  GLenum arg1;
  GLdouble arg2;
  GLdouble arg3;
  GLint arg4;
  GLint arg5;
  GLdouble arg6;
  GLdouble arg7;
  GLint arg8;
  GLint arg9;
  PyObject *arg10;
  GLdouble *points;
  int n;

  TRY (PyArg_ParseTuple(args, "iddiiddiiO",
			&arg1, &arg2, &arg3, &arg4, &arg5,
			&arg6, &arg7, &arg8, &arg9, &arg10));

  TRY (PyArray_AsDoubleArray(&arg10, &points, &n));
  glMap2d(arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, points);

  Py_DECREF(arg10);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject* gl_Map2f(PyObject* self, PyObject* args)
{
  GLenum arg1;
  GLfloat arg2;
  GLfloat arg3;
  GLint arg4;
  GLint arg5;
  GLfloat arg6;
  GLfloat arg7;
  GLint arg8;
  GLint arg9;
  PyObject *arg10;
  GLfloat *points;
  int n;

  TRY (PyArg_ParseTuple(args, "iffiiffiiO",
			&arg1, &arg2, &arg3, &arg4, &arg5,
			&arg6, &arg7, &arg8, &arg9, &arg10));

  TRY (PyArray_AsFloatArray(&arg10, &points, &n));
  glMap2f(arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, points);

  Py_DECREF(arg10);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject* gl_MapGrid1d(PyObject* self, PyObject* args)
{
	GLint arg1;
	GLdouble arg2, arg3;
	TRY (PyArg_ParseTuple(args, "idd", &arg1, &arg2, &arg3));
  	glMapGrid1d(arg1, arg2 , arg3);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_MapGrid1f(PyObject* self, PyObject* args)
{
	GLint arg1;
	GLfloat arg2, arg3;
	TRY (PyArg_ParseTuple(args, "iff", &arg1, &arg2, &arg3));
  	glMapGrid1f(arg1, arg2 , arg3);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_MapGrid2d(PyObject* self, PyObject* args)
{
	GLint arg1, arg4;
	GLdouble arg2, arg3, arg5, arg6;
	TRY (PyArg_ParseTuple(args, "iddidd", &arg1, &arg2, &arg3, &arg4, &arg5, &arg6));
  	glMapGrid2d(arg1, arg2, arg3, arg4, arg5, arg6);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_MapGrid2f(PyObject* self, PyObject* args)
{
	GLint arg1, arg4;
	GLfloat arg2, arg3, arg5, arg6;
	TRY (PyArg_ParseTuple(args, "iffiff", &arg1, &arg2, &arg3, &arg4, &arg5, &arg6));
  	glMapGrid2f(arg1, arg2, arg3, arg4, arg5, arg6);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Materialf(PyObject* self, PyObject* args)
{
	GLenum arg1, arg2;
	GLfloat arg3, *rest;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iif", &arg1, &arg2, &arg3))
  		glMaterialf(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "iiO", &arg1, &arg2, &op));
		TRY (PyArray_AsFloatArray(&op, &rest, &n));
		if ((n == 1) || (n>3)) {
			glMaterialfv(arg1, arg2, rest);
			PyArray_ClearMemory(op, rest);
		}
		else {
			PyErr_SetString(gl_Error,  "second argument needs 1 or at least 4 items!");
			PyArray_ClearMemory(op, rest);
			return NULL;
		}
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Materiali(PyObject* self, PyObject* args)
{
	GLenum arg1, arg2;
	GLint arg3, *rest;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iii", &arg1, &arg2, &arg3))
  		glMateriali(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "iiO", &arg1, &arg2, &op));
		TRY (PyArray_AsIntArray(&op, &rest, &n));
		if ((n == 1) || (n > 3)) {
			glMaterialiv(arg1, arg2, rest);
			PyArray_ClearMemory(op, rest);
		}
		else {
			PyErr_SetString(gl_Error,  "second argument needs 1 or at least 4 items!");
			PyArray_ClearMemory(op, rest);
			return NULL;
		}
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_MatrixMode(PyObject* self, PyObject* args)
{
	GLenum arg1 ;
 	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glMatrixMode(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_MultMatrixd(PyObject* self, PyObject* args)
{
	GLdouble *mat;
	PyObject *op;
	int n;
 	TRY (PyArg_ParseTuple(args, "O", &op));
	TRY (PyArray_AsDoubleArray(&op, &mat, &n));
	if (n < 16) {
		PyErr_SetString(gl_Error, "need element with at least 16 items");
		PyArray_ClearMemory(op, mat);
		return NULL;
	}
	glMultMatrixd(mat);
	PyArray_ClearMemory(op, mat);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_MultMatrixf(PyObject* self, PyObject* args)
{
	GLfloat *mat;
    PyObject *op;
	int n;
 	TRY (PyArg_ParseTuple(args, "O", &op));
	TRY (PyArray_AsFloatArray(&op, &mat, &n));
    if (n < 16) {
		PyErr_SetString(gl_Error, "need element with at least 16 items");
		PyArray_ClearMemory(op, mat);
		return NULL;
	}
	glMultMatrixf(mat);
	PyArray_ClearMemory(op, mat);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_NewList(PyObject* self, PyObject* args)
{
	GLuint arg1;
	GLenum arg2;
 	TRY (PyArg_ParseTuple(args, "ii", &arg1, &arg2));
  	glNewList(arg1, arg2);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Normal3d(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ddd", &arg1, &arg2, &arg3)) 	
  		glNormal3d(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsDoubleArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glNormal3dv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Normal3f(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "fff", &arg1, &arg2, &arg3)) 	
  		glNormal3f(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsFloatArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glNormal3fv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Normal3i(PyObject* self, PyObject* args)
{
	GLint arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iii", &arg1, &arg2, &arg3)) 	
  		glNormal3i(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsIntArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glNormal3iv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Normal3s(PyObject* self, PyObject* args)
{
	GLshort arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "hhh", &arg1, &arg2, &arg3)) 	
  		glNormal3s(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsShortArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glNormal3sv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Ortho(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, arg3, arg4, arg5, arg6;
	TRY (PyArg_ParseTuple(args, "dddddd", &arg1, &arg2, &arg3, &arg4, &arg5, &arg6));
  	glOrtho(arg1, arg2, arg3, arg4, arg5, arg6);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_PassThrough(PyObject* self, PyObject* args)
{
	GLfloat arg1;
	TRY (PyArg_ParseTuple(args, "f", &arg1));
	glPassThrough(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_PixelStoref(PyObject* self, PyObject* args)
{
	GLenum arg1;
  	GLfloat arg2;
	TRY (PyArg_ParseTuple(args, "if", &arg1, &arg2));
  	glPixelStoref(arg1, arg2);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_PixelStorei(PyObject* self, PyObject* args)
{
	GLenum arg1;
  	GLint arg2;
	TRY (PyArg_ParseTuple(args, "ii", &arg1, &arg2));
  	glPixelStorei(arg1, arg2);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_PixelTransferf(PyObject* self, PyObject* args)
{
	GLenum arg1;
  	GLfloat arg2;
	TRY (PyArg_ParseTuple(args, "if", &arg1, &arg2));
  	glPixelTransferf(arg1, arg2);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_PixelTransferi(PyObject* self, PyObject* args)
{
	GLenum arg1;
  	GLint arg2;
	TRY (PyArg_ParseTuple(args, "ii", &arg1, &arg2));
  	glPixelTransferi(arg1, arg2);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_PixelZoom(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2;
	TRY (PyArg_ParseTuple(args, "ff", &arg1, &arg2));
  	glPixelZoom(arg1, arg2);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_PointSize(PyObject* self, PyObject* args)
{
	GLfloat arg1;
	TRY (PyArg_ParseTuple(args, "f", &arg1));
  	glPointSize(arg1);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_PolygonMode(PyObject* self, PyObject* args)
{
	GLenum arg1, arg2;
	TRY (PyArg_ParseTuple(args, "ii", &arg1, &arg2));
  	glPolygonMode(arg1, arg2);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_PopAttrib(PyObject* self, PyObject* args)
{
	TRY (PyArg_ParseTuple(args, ""));
  	glPopAttrib();
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_PopMatrix(PyObject* self, PyObject* args)
{
	TRY (PyArg_ParseTuple(args, ""));
  	glPopMatrix();
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_PopName(PyObject* self, PyObject* args)
{
	TRY (PyArg_ParseTuple(args, ""));
  	glPopName();
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_PushAttrib(PyObject* self, PyObject* args)
{
	GLbitfield arg1 ;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glPushAttrib(arg1);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_PushMatrix(PyObject* self, PyObject* args)
{
	TRY (PyArg_ParseTuple(args, ""));
  	glPushMatrix();
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_PushName(PyObject* self, PyObject* args)
{
	GLuint arg1 ;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glPushName(arg1);
 	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_RasterPos2d(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "dd", &arg1, &arg2)) 	
  		glRasterPos2d(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsDoubleArray(&op, &vert, &n));
		if (n < 2) {
			PyErr_SetString(gl_Error,  "need element with at least 2 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glRasterPos2dv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_RasterPos2f(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ff", &arg1, &arg2)) 	
  		glRasterPos2f(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsFloatArray(&op, &vert, &n));
		if (n < 2) {
			PyErr_SetString(gl_Error,  "need element with at least 2 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glRasterPos2fv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_RasterPos2i(PyObject* self, PyObject* args)
{
	GLint arg1, arg2, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ii", &arg1, &arg2)) 	
  		glRasterPos2i(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsIntArray(&op, &vert, &n));
		if (n < 2) {
			PyErr_SetString(gl_Error,  "need element with at least 2 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glRasterPos2iv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_RasterPos2s(PyObject* self, PyObject* args)
{
	GLshort arg1, arg2, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "hh", &arg1, &arg2)) 	
  		glRasterPos2s(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsShortArray(&op, &vert, &n));
		if (n < 2) {
			PyErr_SetString(gl_Error,  "need element with at least 2 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glRasterPos2sv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_RasterPos3d(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ddd", &arg1, &arg2, &arg3)) 	
  		glRasterPos3d(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsDoubleArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glRasterPos3dv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_RasterPos3f(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "fff", &arg1, &arg2, &arg3)) 	
  		glRasterPos3f(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsFloatArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glRasterPos3fv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_RasterPos3i(PyObject* self, PyObject* args)
{
	GLint arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iii", &arg1, &arg2, &arg3)) 	
  		glRasterPos3i(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsIntArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glRasterPos3iv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_RasterPos3s(PyObject* self, PyObject* args)
{
	GLshort arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "hhh", &arg1, &arg2, &arg3)) 	
  		glRasterPos3s(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsShortArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glRasterPos3sv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_RasterPos4d(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, arg3, arg4, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "dddd", &arg1, &arg2, &arg3, &arg4)) 	
  		glRasterPos4d(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsDoubleArray(&op, &vert, &n));
		if (n < 4) {
			PyErr_SetString(gl_Error,  "need element with at least 4 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glRasterPos4dv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_RasterPos4f(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, arg3, arg4, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ffff", &arg1, &arg2, &arg3, &arg4)) 	
  		glRasterPos4f(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsFloatArray(&op, &vert, &n));
		if (n < 4) {
			PyErr_SetString(gl_Error,  "need element with at least 4 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glRasterPos4fv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_RasterPos4i(PyObject* self, PyObject* args)
{
	GLint arg1, arg2, arg3, arg4, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iiii", &arg1, &arg2, &arg3, &arg4)) 	
  		glRasterPos4i(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsIntArray(&op, &vert, &n));
		if (n < 4) {
			PyErr_SetString(gl_Error,  "need element with at least 4 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glRasterPos4iv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_RasterPos4s(PyObject* self, PyObject* args)
{
	GLshort arg1, arg2, arg3, arg4, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "hhhh", &arg1, &arg2, &arg3, &arg4)) 	
  		glRasterPos4s(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsShortArray(&op, &vert, &n));
		if (n < 4) {
			PyErr_SetString(gl_Error,  "need element with at least 4 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glRasterPos4sv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_ReadBuffer(PyObject* self, PyObject* args)
{
	GLenum arg1 ;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glReadBuffer(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Rectd(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, arg3, arg4, *vert1, *vert2;
	PyObject *op1, *op2;
	int n1, n2;	
	if (PyArg_ParseTuple(args, "dddd", &arg1, &arg2, &arg3, &arg4)) 	
  		glRectd(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "OO", &op1, &op2));
		TRY (PyArray_AsDoubleArray(&op1, &vert1, &n1));
		TRY (PyArray_AsDoubleArray(&op2, &vert2, &n2));
		if ((n1 < 2) || (n2 < 2)) {
			PyErr_SetString(gl_Error,  "need elements with at least 2 items");
			Py_DECREF(op1);
			Py_DECREF(op2);
			return NULL;
		}
		glRectdv(vert1, vert2);
		Py_DECREF(op1);
		Py_DECREF(op2);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Rectf(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, arg3, arg4, *vert1, *vert2;
	PyObject *op1, *op2;
	int n1, n2;	
	if (PyArg_ParseTuple(args, "ffff", &arg1, &arg2, &arg3, &arg4)) 	
  		glRectf(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "OO", &op1, &op2));
		TRY (PyArray_AsFloatArray(&op1, &vert1, &n1));
		TRY (PyArray_AsFloatArray(&op2, &vert2, &n2));
		if ((n1 < 2) || (n2 < 2)) {
			PyErr_SetString(gl_Error,  "need elements with at least 2 items");
			Py_DECREF(op1);
			Py_DECREF(op2);
			return NULL;
		}
		glRectfv(vert1, vert2);
		Py_DECREF(op1);
		Py_DECREF(op2);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Recti(PyObject* self, PyObject* args)
{
	GLint arg1, arg2, arg3, arg4, *vert1, *vert2;
	PyObject *op1, *op2;
	int n1, n2;	
	if (PyArg_ParseTuple(args, "iiii", &arg1, &arg2, &arg3, &arg4)) 	
  		glRecti(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "OO", &op1, &op2));
		TRY (PyArray_AsIntArray(&op1, &vert1, &n1));
		TRY (PyArray_AsIntArray(&op2, &vert2, &n2));
		if ((n1 < 2) || (n2 < 2)) {
			PyErr_SetString(gl_Error,  "need elements with at least 2 items");
			Py_DECREF(op1);
			Py_DECREF(op2);
			return NULL;
		}
		glRectiv(vert1, vert2);
		Py_DECREF(op1);
		Py_DECREF(op2);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Rects(PyObject* self, PyObject* args)
{
	GLshort arg1, arg2, arg3, arg4, *vert1, *vert2;
	PyObject *op1, *op2;
	int n1, n2;	
	if (PyArg_ParseTuple(args, "hhhh", &arg1, &arg2, &arg3, &arg4)) 	
  		glRects(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "OO", &op1, &op2));
		TRY (PyArray_AsShortArray(&op1, &vert1, &n1));
		TRY (PyArray_AsShortArray(&op2, &vert2, &n2));
		if ((n1 < 2) || (n2 < 2)) {
			PyErr_SetString(gl_Error,  "need elements with at least 2 items");
			Py_DECREF(op1);
			Py_DECREF(op2);
			return NULL;
		}
		glRectsv(vert1, vert2);
		Py_DECREF(op1);
		Py_DECREF(op2);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Rotated(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, arg3, arg4;
	TRY (PyArg_ParseTuple(args, "dddd", &arg1, &arg2, &arg3, &arg4));
  	glRotated(arg1, arg2, arg3, arg4);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Rotatef(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, arg3, arg4;
	TRY (PyArg_ParseTuple(args, "ffff", &arg1, &arg2, &arg3, &arg4));
  	glRotatef(arg1, arg2, arg3, arg4);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Scaled(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, arg3;
	TRY (PyArg_ParseTuple(args, "ddd", &arg1, &arg2, &arg3));
  	glScaled(arg1, arg2, arg3);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Scalef(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, arg3;
	TRY (PyArg_ParseTuple(args, "fff", &arg1, &arg2, &arg3));
  	glScalef(arg1, arg2, arg3);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Scissor(PyObject* self, PyObject* args)
{
	GLint arg1, arg2;
	GLsizei arg3, arg4;
	TRY (PyArg_ParseTuple(args, "iiii", &arg1, &arg2, &arg3, &arg4));
  	glScissor(arg1, arg2, arg3, arg4);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_ShadeModel(PyObject* self, PyObject* args)
{
	GLenum arg1;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glShadeModel(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_StencilFunc(PyObject* self, PyObject* args)
{
	GLenum arg1;
	GLint arg2;
	GLuint arg3;
	TRY (PyArg_ParseTuple(args, "iii", &arg1, &arg2, &arg3));
  	glStencilFunc(arg1, arg2, arg3);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_StencilMask(PyObject* self, PyObject* args)
{
	GLuint arg1;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glStencilMask(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_StencilOp(PyObject* self, PyObject* args)
{
	GLenum arg1, arg2, arg3;
	TRY (PyArg_ParseTuple(args, "iii", &arg1, &arg2, &arg3));
  	glStencilOp(arg1, arg2, arg3);
	Py_INCREF(Py_None);
	return Py_None;
}


static PyObject* gl_TexCoord1d(PyObject* self, PyObject* args)
{
	GLdouble arg1;
	TRY (PyArg_ParseTuple(args, "d", &arg1));
  	glTexCoord1d(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexCoord1f(PyObject* self, PyObject* args)
{
	GLfloat arg1;
	TRY (PyArg_ParseTuple(args, "f", &arg1));
  	glTexCoord1f(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexCoord1i(PyObject* self, PyObject* args)
{
	GLint arg1;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glTexCoord1i(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexCoord1s(PyObject* self, PyObject* args)
{
	GLshort arg1;
	TRY (PyArg_ParseTuple(args, "h", &arg1));
  	glTexCoord1s(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexCoord2d(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "dd", &arg1, &arg2)) 	
  		glTexCoord2d(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsDoubleArray(&op, &vert, &n));
		if (n < 2) {
			PyErr_SetString(gl_Error,  "need element with at least 2 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glTexCoord2dv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexCoord2f(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ff", &arg1, &arg2)) 	
  		glTexCoord2f(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsFloatArray(&op, &vert, &n));
		if (n < 2) {
			PyErr_SetString(gl_Error,  "need element with at least 2 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glTexCoord2fv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexCoord2i(PyObject* self, PyObject* args)
{
	GLint arg1, arg2, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ii", &arg1, &arg2)) 	
  		glTexCoord2i(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsIntArray(&op, &vert, &n));
		if (n < 2) {
			PyErr_SetString(gl_Error,  "need element with at least 2 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glTexCoord2iv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexCoord2s(PyObject* self, PyObject* args)
{
	GLshort arg1, arg2, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "hh", &arg1, &arg2)) 	
  		glTexCoord2s(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsShortArray(&op, &vert, &n));
		if (n < 2) {
			PyErr_SetString(gl_Error,  "need element with at least 2 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glTexCoord2sv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexCoord3d(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ddd", &arg1, &arg2, &arg3)) 	
  		glTexCoord3d(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsDoubleArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glTexCoord3dv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexCoord3f(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "fff", &arg1, &arg2, &arg3)) 	
  		glTexCoord3f(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsFloatArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glTexCoord3fv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexCoord3i(PyObject* self, PyObject* args)
{
	GLint arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iii", &arg1, &arg2, &arg3)) 	
  		glTexCoord3i(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsIntArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glTexCoord3iv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexCoord3s(PyObject* self, PyObject* args)
{
	GLshort arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "hhh", &arg1, &arg2, &arg3)) 	
  		glTexCoord3s(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsShortArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glTexCoord3sv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexCoord4d(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, arg3, arg4, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "dddd", &arg1, &arg2, &arg3, &arg4)) 	
  		glTexCoord4d(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsDoubleArray(&op, &vert, &n));
		if (n < 4) {
			PyErr_SetString(gl_Error,  "need element with at least 4 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glTexCoord4dv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexCoord4f(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, arg3, arg4, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ffff", &arg1, &arg2, &arg3, &arg4)) 	
  		glTexCoord4f(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsFloatArray(&op, &vert, &n));
		if (n < 4) {
			PyErr_SetString(gl_Error,  "need element with at least 4 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glTexCoord4fv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexCoord4i(PyObject* self, PyObject* args)
{
	GLint arg1, arg2, arg3, arg4, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iiii", &arg1, &arg2, &arg3, &arg4)) 	
  		glTexCoord4i(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsIntArray(&op, &vert, &n));
		if (n < 4) {
			PyErr_SetString(gl_Error,  "need element with at least 4 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glTexCoord4iv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexCoord4s(PyObject* self, PyObject* args)
{
	GLshort arg1, arg2, arg3, arg4, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "hh", &arg1, &arg2, &arg3, &arg4)) 	
  		glTexCoord4s(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsShortArray(&op, &vert, &n));
		if (n < 4) {
			PyErr_SetString(gl_Error,  "need element with at least 4 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glTexCoord4sv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexEnvf(PyObject* self, PyObject* args)
{
	GLenum arg1, arg2;
	GLfloat arg3, *rest;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iif", &arg1, &arg2, &arg3))
  		glTexEnvf(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "iiO", &arg1, &arg2, &op));
		TRY (PyArray_AsFloatArray(&op, &rest, &n));
		if ((n == 1) || (n > 3)) {
			glTexEnvfv(arg1, arg2, rest);
			PyArray_ClearMemory(op, rest);
		}
		else {
			PyErr_SetString(gl_Error,  "3. argument needs 1 or at least 4 items!");
			PyArray_ClearMemory(op, rest);
			return NULL;
		}
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexEnvi(PyObject* self, PyObject* args)
{
	GLenum arg1, arg2;
	GLint arg3, *rest;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iii", &arg1, &arg2, &arg3))
  		glTexEnvi(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "iiO", &arg1, &arg2, &op));
		TRY (PyArray_AsIntArray(&op, &rest, &n));
		if ((n == 1) || (n > 3)) {
			glTexEnviv(arg1, arg2, rest);
			PyArray_ClearMemory(op, rest);
		}
		else {
			PyErr_SetString(gl_Error,  "3. argument needs 1 or at least 4 items!");
			PyArray_ClearMemory(op, rest);
			return NULL;
		}
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexGend(PyObject* self, PyObject* args)
{
	GLenum arg1, arg2;
	GLdouble arg3, *rest;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iid", &arg1, &arg2, &arg3))
  		glTexGend(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "iiO", &arg1, &arg2, &op));
		TRY (PyArray_AsDoubleArray(&op, &rest, &n));
		if ((n == 1) || (n>3)) {
			glTexGendv(arg1, arg2, rest);
			PyArray_ClearMemory(op, rest);
		}
		else {
			PyErr_SetString(gl_Error,  "3. argument needs 1 or at least 4 items!");
			PyArray_ClearMemory(op, rest);
			return NULL;
		}
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexGenf(PyObject* self, PyObject* args)
{
	GLenum arg1, arg2;
	GLfloat arg3, *rest;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iif", &arg1, &arg2, &arg3))
  		glTexGenf(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "iiO", &arg1, &arg2, &op));
		TRY (PyArray_AsFloatArray(&op, &rest, &n));
		if ((n == 1) || (n > 3)) {
			glTexGenfv(arg1, arg2, rest);
			PyArray_ClearMemory(op, rest);
		}
		else {
			PyErr_SetString(gl_Error,  "3. argument needs 1 or at least 4 items!");
			PyArray_ClearMemory(op, rest);
			return NULL;
		}
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexGeni(PyObject* self, PyObject* args)
{
	GLenum arg1, arg2;
	GLint arg3, *rest;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iii", &arg1, &arg2, &arg3))
  		glTexGeni(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "iiO", &arg1, &arg2, &op));
		TRY (PyArray_AsIntArray(&op, &rest, &n));
		if ((n == 1) || (n>3)) {
			glTexGeniv(arg1, arg2, rest);
			PyArray_ClearMemory(op, rest);
		}
		else {
			PyErr_SetString(gl_Error,  "3. argument needs 1 or at least 4 items!");
			PyArray_ClearMemory(op, rest);
			return NULL;
		}
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexParameterf(PyObject* self, PyObject* args)
{
	GLenum arg1, arg2;
	GLfloat arg3, *rest;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iif", &arg1, &arg2, &arg3))
  		glTexParameterf(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "iiO", &arg1, &arg2, &op));
		TRY (PyArray_AsFloatArray(&op, &rest, &n));
		if ((n == 1) || (n > 3)) {
			glTexParameterfv(arg1, arg2, rest);
			PyArray_ClearMemory(op, rest);
		}
		else {
			PyErr_SetString(gl_Error,  "3. argument needs 1 or at least 4 items!");
			PyArray_ClearMemory(op, rest);
			return NULL;
		}
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_TexParameteri(PyObject* self, PyObject* args)
{
	GLenum arg1, arg2;
	GLint arg3, *rest;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iii", &arg1, &arg2, &arg3))
  		glTexParameteri(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "iiO", &arg1, &arg2, &op));
		TRY (PyArray_AsIntArray(&op, &rest, &n));
		if ((n == 1) || (n > 3)) {
			glTexParameteriv(arg1, arg2, rest);
			PyArray_ClearMemory(op, rest);
		}
		else {
			PyErr_SetString(gl_Error,  "3. argument needs 1 or at least 4 items!");
			PyArray_ClearMemory(op, rest);
			return NULL;
		}
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Translated(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, arg3;
	TRY (PyArg_ParseTuple(args, "ddd", &arg1, &arg2, &arg3));
  	glTranslated(arg1, arg2, arg3);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Translatef(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, arg3;
	TRY (PyArg_ParseTuple(args, "fff", &arg1, &arg2, &arg3));
  	glTranslatef(arg1, arg2, arg3);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Vertex2d(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "dd", &arg1, &arg2)) 	
  		glVertex2d(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsDoubleArray(&op, &vert, &n));
		if (n < 2) {
			PyErr_SetString(gl_Error,  "need element with at least 2 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glVertex2dv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

/* experimental */

static PyObject* gl_ColorVertex2d(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, *vert, *x, *y, *c, *max_cp;
	PyObject *op1, *op2, *colors=NULL;
	PyArrayObject *cp;
	int m, n, i, ok1, ok2, nc;	
	if (PyArg_ParseTuple(args, "dd", &arg1, &arg2)) 	
  		glVertex2d(arg1, arg2);
	else {
		PyErr_Clear();
		if (PyArg_ParseTuple(args, "O", &op1)) {
			TRY (PyArray_AsDoubleArray(&op1, &vert, &n));
			if (n < 2) {
				PyErr_SetString(gl_Error, "need element with at least 2 items");
				PyArray_ClearMemory(op1, vert);
				return NULL;
			}
			glVertex2dv(vert);
			PyArray_ClearMemory(op1, vert);
		}
#ifdef NUMERIC
		else {
			PyErr_Clear();
			TRY (PyArg_ParseTuple(args, "OO|O", &op1, &op2, &colors)); 
			ok1 = PyArray_AsDoubleArray(&op1, &x, &m);
			ok2 = PyArray_AsDoubleArray(&op2, &y, &n);
			if ((m != n) || (!ok1) || (!ok2)) {
					PyErr_SetString(gl_Error, "coordinate arrays must be of same length 
                                          or not enough memory");
					PyArray_ClearMemory(op1, x);
					PyArray_ClearMemory(op2, y);
					return NULL;
			}
			if (colors) {
				if (!(cp =(PyArrayObject *)PyArray_ContiguousFromObject(colors, 
																		PyArray_DOUBLE, 1, 2))) {
					PyArray_ClearMemory(op1, x);
					PyArray_ClearMemory(op2, y);
					return NULL;
				}
   			c = (GLdouble *)cp->data;
				nc = PyArray_Size((PyObject *)cp);
				if ( ((nc%3) != 0) || (n != nc/3)) {
					PyErr_SetString(gl_Error, "wrong color matrix size");
					PyArray_ClearMemory(op1, x);
					PyArray_ClearMemory(op2, y);
					PyArray_ClearMemory(cp, c);
					return NULL;
				}

				for (i=0; i<n; i++) {
					glColor3dv(c); c += 3;
					glVertex2d(*x++, *y++);
				}
			}
			else {
				for (i=0; i<n; i++) {
					glVertex2d(*x++, *y++);
				}
			}
			PyArray_ClearMemory(op1, x);
			PyArray_ClearMemory(op2, y);	
			PyArray_ClearMemory(cp, c);
		}
	}
#else
	}
#endif
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Vertex2f(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ff", &arg1, &arg2)) 	
  		glVertex2f(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsFloatArray(&op, &vert, &n));
		if (n < 2) {
			PyErr_SetString(gl_Error,  "need element with at least 2 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glVertex2fv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Vertex2i(PyObject* self, PyObject* args)
{
	GLint arg1, arg2, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ii", &arg1, &arg2)) 	
  		glVertex2i(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsIntArray(&op, &vert, &n));
		if (n < 2) {
			PyErr_SetString(gl_Error,  "need element with at least 2 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glVertex2iv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Vertex2s(PyObject* self, PyObject* args)
{
	GLshort arg1, arg2, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "hh", &arg1, &arg2)) 	
  		glVertex2s(arg1, arg2);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsShortArray(&op, &vert, &n));
		if (n < 2) {
			PyErr_SetString(gl_Error,  "need element with at least 2 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glVertex2sv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Vertex3d(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ddd", &arg1, &arg2, &arg3)) 	
  		glVertex3d(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsDoubleArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glVertex3dv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Vertex3f(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "fff", &arg1, &arg2, &arg3)) 	
  		glVertex3f(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsFloatArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glVertex3fv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Vertex3i(PyObject* self, PyObject* args)
{
	GLint arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iii", &arg1, &arg2, &arg3)) 	
  		glVertex3i(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsIntArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glVertex3iv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Vertex3s(PyObject* self, PyObject* args)
{
	GLshort arg1, arg2, arg3, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "hhh", &arg1, &arg2, &arg3)) 	
  		glVertex3s(arg1, arg2, arg3);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsShortArray(&op, &vert, &n));
		if (n < 3) {
			PyErr_SetString(gl_Error,  "need element with at least 3 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glVertex3sv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Vertex4d(PyObject* self, PyObject* args)
{
	GLdouble arg1, arg2, arg3, arg4, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "dddd", &arg1, &arg2, &arg3, &arg4)) 	
  		glVertex4d(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsDoubleArray(&op, &vert, &n));
		if (n < 4) {
			PyErr_SetString(gl_Error,  "need element with at least 4 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glVertex4dv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Vertex4f(PyObject* self, PyObject* args)
{
	GLfloat arg1, arg2, arg3, arg4, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "ffff", &arg1, &arg2, &arg3, &arg4)) 	
  		glVertex4f(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsFloatArray(&op, &vert, &n));
		if (n < 4) {
			PyErr_SetString(gl_Error,  "need element with at least 4 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glVertex4fv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Vertex4i(PyObject* self, PyObject* args)
{
	GLint arg1, arg2, arg3, arg4, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "iiii", &arg1, &arg2, &arg3, &arg4)) 	
  		glVertex4i(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsIntArray(&op, &vert, &n));
		if (n < 4) {
			PyErr_SetString(gl_Error,  "need element with at least 4 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glVertex4iv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Vertex4s(PyObject* self, PyObject* args)
{
	GLshort arg1, arg2, arg3, arg4, *vert;
	PyObject *op;
	int n;	
	if (PyArg_ParseTuple(args, "hhhh", &arg1, &arg2, &arg3, &arg4)) 	
  		glVertex4s(arg1, arg2, arg3, arg4);
	else {
		PyErr_Clear();
		TRY (PyArg_ParseTuple(args, "O", &op));
		TRY (PyArray_AsShortArray(&op, &vert, &n));
		if (n < 4) {
			PyErr_SetString(gl_Error,  "need element with at least 4 items");
			PyArray_ClearMemory(op, vert);
			return NULL;
		}
		glVertex4sv(vert);
		PyArray_ClearMemory(op, vert);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_Viewport(PyObject* self, PyObject* args)
{
	GLint arg1, arg2;
	GLsizei arg3, arg4;
	TRY (PyArg_ParseTuple(args, "iiii", &arg1, &arg2, &arg3, &arg4));
  	glViewport(arg1 , arg2 , arg3 , arg4);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_ClipPlane(PyObject* self, PyObject* args)
{
  	GLenum arg1;
  	PyObject *eop;
	GLdouble *equation;
	int n;
	TRY (PyArg_ParseTuple(args, "iO", &arg1, &eop));
	TRY (PyArray_AsDoubleArray(&eop, &equation, &n));
	if (n < 4) {
		PyErr_SetString(gl_Error,  "second argument needs at least 4 items");
		Py_DECREF(eop);
		return NULL;
		}
  	glClipPlane(arg1, equation);
	Py_DECREF(eop);      
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_GetClipPlane(PyObject* self, PyObject* args)
{
  	GLenum arg1;
#ifdef NUMERIC
  	PyArrayObject *values;
#else /* Not NUMERIC */
	PyObject *values;
	int i;
#endif /* Not NUMERIC */
	GLdouble equation[4];
	int dims[1];
	TRY (PyArg_ParseTuple(args, "i", &arg1));
	glGetClipPlane(arg1, equation);
	dims[0] = 4;
#ifdef NUMERIC
	values = (PyArrayObject *) PyArray_FromDims (1, dims, PyArray_DOUBLE);
	memcpy(values->data, equation, 4*sizeof(double));
#else /* Not NUMERIC */
	values = PyTuple_New(4);
	for (i = 0; i < 4; i++) {
	  PyTuple_SET_ITEM(values, i, PyFloat_FromDouble(equation[i]));
	}
#endif /* Not NUMERIC */
	return (PyObject*) values;
}


static PyObject* gl_IsEnabled(PyObject* self, PyObject* args)
{
  	GLenum arg1;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	return PyInt_FromLong(glIsEnabled(arg1));
}

static PyObject* gl_IsList(PyObject* self, PyObject* args)
{
  	GLuint arg1;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	return PyInt_FromLong(glIsList(arg1));
}

static PyObject* gl_RenderMode(PyObject* self, PyObject* args)
{
  	GLenum arg1;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	return PyInt_FromLong(glRenderMode(arg1));
}

static PyObject* gl_GenLists(PyObject* self, PyObject* args)
{
  	GLsizei arg1;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	return PyInt_FromLong(glGenLists(arg1));
}

static PyObject* gl_GetDoublev(PyObject* self, PyObject* args)
{
  GLenum pname;
  GLdouble oneparam;
  int nd=1, i, dims[3];
#ifdef NUMERIC
  PyArrayObject* params;
#else /* NUMERIC */
  int nitems;
  GLdouble *items;
  PyObject *params;
#endif /* NUMERIC */
  if (!PyArg_ParseTuple(args, "i", &pname))
    return NULL;
  switch(pname) {
    case GL_ACCUM_CLEAR_VALUE:
#ifdef GL_BLEND_COLOR_EXT
    case GL_BLEND_COLOR_EXT:
#endif
    case GL_COLOR_CLEAR_VALUE:
    case GL_COLOR_WRITEMASK:
    case GL_CURRENT_COLOR:
    case GL_CURRENT_RASTER_COLOR:
    case GL_CURRENT_RASTER_POSITION:
    case GL_CURRENT_RASTER_TEXTURE_COORDS:
    case GL_CURRENT_TEXTURE_COORDS:
    case GL_FOG_COLOR:
    case GL_LIGHT_MODEL_AMBIENT:
    case GL_MAP2_GRID_DOMAIN:
    case GL_SCISSOR_BOX:
    case GL_TEXTURE_ENV_COLOR:
    case GL_VIEWPORT:
	dims[0] = 4;
	break;
    case GL_CURRENT_NORMAL:
	dims[0] = 3;
	break;
    case GL_DEPTH_RANGE:	
    case GL_LINE_WIDTH_RANGE:
    case GL_MAP1_GRID_DOMAIN:
    case GL_MAP2_GRID_SEGMENTS:
    case GL_MAX_VIEWPORT_DIMS:
    case GL_POINT_SIZE_RANGE:
    case GL_POLYGON_MODE:
	dims[0] = 2;
	break;
    case GL_MODELVIEW_MATRIX:
    case GL_PROJECTION_MATRIX:
    case GL_TEXTURE_MATRIX:
	nd = 2;
	dims[0] = 4; dims[1] = 4;
	break;
    case GL_POLYGON_STIPPLE:
	dims[0] = 32;
	break;
    default:
        glGetDoublev(pname, &oneparam);
	return Py_BuildValue("d", oneparam);  
    }
#ifdef NUMERIC
    TRY(params = (PyArrayObject *)PyArray_FromDims(nd, dims, PyArray_DOUBLE));
    glGetDoublev(pname, (double *)params->data);
#else /* NUMERIC */
    nitems = 1;
    for(i = 0; i < nd; i++) {
      nitems *= dims[i];
    }
    TRY(items = PyMem_NEW(GLdouble, nitems));
    glGetDoublev(pname, items);
    TRY(params = PyTuple_New(nitems));
    for (i = 0; i < nitems; i++) {
      PyTuple_SET_ITEM(params, i, PyFloat_FromDouble(items[i]));
    }
    PyMem_DEL(items);
#endif /* NUMERIC */
    return (PyObject *)params;
}

/*OpenGL Extensions*/

static PyObject* gl_PolygonOffsetEXT(PyObject* self, PyObject* args)
{
#ifdef GL_EXT_polygon_offset
	GLfloat arg1, arg2;
	TRY (PyArg_ParseTuple(args, "ff", &arg1, &arg2));
  	glPolygonOffsetEXT(arg1, arg2);
 	Py_INCREF(Py_None);
	return Py_None;
#else
	PyErr_SetString(PyExc_ValueError, "PolygonOffsetEXT not implemented");
	return NULL;
#endif
}

static PyObject* gl_ArrayElementEXT(PyObject* self, PyObject* args)
{
#ifdef GL_EXT_array_element
  	GLint arg1;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glArrayElementEXT(arg1);
	Py_INCREF(Py_None);
	return Py_None;
#else
	PyErr_SetString(PyExc_ValueError, "ArrayElementEXT not implemented");
	return NULL;
#endif
}

/*These need to be fixed to match the above*/
#ifdef NOT_DEFINED
static PyObject* gl_DrawArraysEXT(PyObject* self, PyObject* args)
{
  	GLenum arg1;
  	GLint arg2;
	GLsizei arg3;
	TRY (PyArg_ParseTuple(args, "iii", &arg1, &arg2, &arg3));
  	glDrawArraysEXT(arg1, arg2, arg3);
	Py_INCREF(Py_None);
	return Py_None;
}
static PyObject* gl_BlendColorEXT(PyObject* self, PyObject* args)
{
  	GLclampf arg1, arg2, arg3, arg4;
	TRY (PyArg_ParseTuple(args, "ffff", &arg1, &arg2, &arg3, &arg4));
  	glBlendColorEXT(arg1, arg2, arg3, arg4);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_BlendEquationEXT(PyObject* self, PyObject* args)
{
  	GLenum arg1;
	TRY (PyArg_ParseTuple(args, "i", &arg1));
  	glBlendEquationEXT(arg1);
	Py_INCREF(Py_None);
	return Py_None;
}
#endif

/*Need to find proper size for glReadPixels array*/
static int glformat_size(GLenum format) {
	switch(format) {
	case GL_COLOR_INDEX:
	case GL_RED:
	case GL_GREEN:
	case GL_BLUE:
	case GL_ALPHA:
	case GL_STENCIL_INDEX:
	case GL_DEPTH_COMPONENT:
	case GL_LUMINANCE:
		return 1;

	case GL_LUMINANCE_ALPHA:
		return 2;

	case GL_RGB:
#ifdef GL_BGR_EXT
	case GL_BGR_EXT:
#endif
		return 3;

	case GL_RGBA:
#ifdef GL_BGRA_EXT
	case GL_BGRA_EXT:
#endif
#ifdef GL_ABGR_EXT
	case GL_ABGR_EXT:
#endif
		return 4;
	default:
		return -1;
	}
}

static int gltype_size(GLenum type) {
	switch(type) {
	case GL_BYTE:
	case GL_UNSIGNED_BYTE:
		return 8;

	case GL_SHORT:
	case GL_UNSIGNED_SHORT:
		return 16;

	case GL_INT:
	case GL_UNSIGNED_INT:
	case GL_FLOAT:
		return 32;

	case GL_BITMAP:
		return 1;
	
	default:
		return -1;
	}
}

static PyObject* gl_DrawPixels(PyObject* self, PyObject* args)
{
    char *data;  
    int format, type, width, height, n;
	int format_size, type_size, total_size;
	
    TRY (PyArg_ParseTuple(args, "iiiis#", &width, &height, &format, &type,
			&data, &n));
	/*This is a poor substitute for the RIGHT solution*/
	format_size = glformat_size(format);
	ASSERT(format_size != -1, "invalid format");
	type_size = gltype_size(type);
	ASSERT(format_size != -1, "invalid type");
	total_size = type_size*format_size*width*height/8;
	ASSERT(total_size >= n, "data area too small");

	glDrawPixels(width, height, format, type, data);

    Py_INCREF(Py_None);
    return Py_None; 
}

static PyObject* gl_TexImage2D(PyObject* self, PyObject* args)
{
    char *data; 
	int target, level, components, border;
    int format, type, width, height, n;
	int format_size, type_size, total_size;
	
    TRY (PyArg_ParseTuple(args, "iiiiiiiis#", &target, &level, &components, 
						&width, &height, &border, &format, &type,
						&data, &n));

	/*This is a poor substitute for the RIGHT solution*/
	format_size = glformat_size(format);
	ASSERT(format_size != -1, "invalid format");
	type_size = gltype_size(type);
	ASSERT(format_size != -1, "invalid type");
	total_size = type_size*format_size*width*height/8;
	ASSERT(total_size >= n, "data area too small");

	glTexImage2D(target, level, components, width, height, border,
					format, type, data);

    Py_INCREF(Py_None);
    return Py_None; 
}

static PyObject* gl_ReadPixels(PyObject* self, PyObject* args)
{
	PyObject *sp;
    char *data;  
	int x, y;
    int format, type, width, height, n=-1;
	int format_size, type_size, total_size;
	
    TRY (PyArg_ParseTuple(args, "iiiiii|i", &x, &y, &width, &height, &format, &type, &n));
	/*This is a poor substitute for the RIGHT solution*/
	format_size = glformat_size(format);
	ASSERT(format_size != -1, "invalid format");
	type_size = gltype_size(type);
	ASSERT(format_size != -1, "invalid type");
	total_size = type_size*format_size*width*height/8;
	if (n == -1) n = total_size;
	ASSERT(total_size >= n, "data area too small");

	data = (char *)malloc(n);
	glReadPixels(x, y, width, height, format, type, data);
	sp = PyString_FromStringAndSize(data, n);
	free(data);
	return sp;
}


#ifdef NUMERIC
static PyObject* gl_TrianglesWithNormals(PyObject* self, PyObject* args)
{
    int i, size;
    double d, d1[3], d2[3], norm[3], *v;	
    PyObject *vop;
    PyArrayObject *mp;
    char *data;
    TRY (PyArg_ParseTuple(args, "O", &vop));   	     
    TRY(mp = 
	(PyArrayObject *)PyArray_ContiguousFromObject(vop, PyArray_DOUBLE, 1, 0));
    size = PyArray_Size((PyObject *)mp);
    if ((size%9) != 0){
	PyErr_SetString(PyExc_ValueError, "matrix length sould be divisible by 9");
	return NULL;
    }   
    glBegin(GL_TRIANGLES);
    for (data = mp->data, i=0; i<size; i+=9, data+=9*mp->descr->elsize) {
	v=(double*)data;
	d1[0]=v[0]-v[3]; d2[0]=v[3]-v[6];     
	d1[1]=v[1]-v[4]; d2[1]=v[4]-v[7];       
	d1[2]=v[2]-v[5]; d2[2]=v[5]-v[8];         
	norm[0] = d1[1]*d2[2]-d1[2]*d2[1];
	norm[1] = d1[2]*d2[0]-d1[0]*d2[2];
	norm[2] = d1[0]*d2[1]-d1[1]*d2[0];
	d = sqrt(norm[0]*norm[0]+norm[1]*norm[1]+norm[2]*norm[2]);
	norm[0] /= d;
	norm[1] /= d;
	norm[2] /= d;
	glNormal3dv(norm);
	glVertex3dv(v);
	glVertex3dv(v+3);
	glVertex3dv(v+6);
    }
    glEnd();
    Py_DECREF(mp); 
    Py_INCREF(Py_None);   
    return Py_None; 
}

static PyObject* gl_Triangles(PyObject* self, PyObject* args)
{
    int i, size;
    PyObject *vop;
    PyArrayObject *mp;
    char *data;
    double *v;
    TRY (PyArg_ParseTuple(args, "O", &vop));   	     
    TRY(mp = 
      (PyArrayObject *)PyArray_ContiguousFromObject(vop, PyArray_DOUBLE, 1, 0));
    size = PyArray_Size((PyObject *)mp);
    if ((size%9) != 0){
	PyErr_SetString(PyExc_ValueError, "matrix length sould be divisible by 9");
	return NULL;
    }   
    glBegin(GL_TRIANGLES);
    for (data = mp->data, i=0; i<size; i+=9, data+=9*mp->descr->elsize) {
	v=(double*)data;
	glVertex3dv(v);
	glVertex3dv(v+3);
	glVertex3dv(v+6);
    }
    glEnd();
    Py_DECREF(mp); 
    Py_INCREF(Py_None);   
    return Py_None; 
}

static PyObject* gl_Lines(PyObject* self, PyObject* args)
{
    int i, size;
    PyObject *vop;
    PyArrayObject *mp;  
    char *data;
    TRY (PyArg_ParseTuple(args, "O", &vop));   	     
    TRY(mp = (PyArrayObject *)PyArray_ContiguousFromObject(vop, PyArray_DOUBLE, 1, 0));
    size = PyArray_Size((PyObject *)mp);
    if ((size%3) != 0){
	PyErr_SetString(PyExc_ValueError, "matrix length sould be divisible by 3");
	return NULL;
    }   
    glBegin(GL_LINES);
    for (data = mp->data, i=0; i<size; i+=3, data+=3*mp->descr->elsize) 
	glVertex3dv((double*)data);	
    glEnd(); 
    Py_INCREF(Py_None);
    return Py_None; 
}

static PyObject* gl_Points(PyObject* self, PyObject* args)
{
    int i, size;
    PyObject *vop;
    PyArrayObject *mp;  
    char *data;
    TRY (PyArg_ParseTuple(args, "O", &vop));   	     
    TRY(mp = (PyArrayObject *)PyArray_ContiguousFromObject(vop, PyArray_DOUBLE, 1, 0));
    size = PyArray_Size((PyObject *)mp);
    if ((size%3) != 0){
	PyErr_SetString(PyExc_ValueError, "matrix length sould be divisible by 3");
	return NULL;
    }   
    glBegin(GL_POINTS);
    for (data = mp->data, i=0; i<size; i+=3, data+=3*mp->descr->elsize) 
	glVertex3dv((double*)data);	
    glEnd(); 
    Py_INCREF(Py_None);
    return Py_None; 
}

static PyObject* gl_Tetrahedra(PyObject* self, PyObject* args)
{
    int i, size;
    PyObject *vop;
    PyArrayObject *mp;  
    char *data;
    double *v;
    TRY (PyArg_ParseTuple(args, "O", &vop));   	     
    TRY(mp = (PyArrayObject *)PyArray_ContiguousFromObject(vop, PyArray_DOUBLE, 1, 0));
    size = PyArray_Size((PyObject *)mp);
    if ((size%12) != 0){
	PyErr_SetString(PyExc_ValueError, "matrix length sould be divisible by 12");
	return NULL;
    }   
    glBegin(GL_LINE_LOOP);
    for (data = mp->data, i=0; i<size; i+=12, data+=12*mp->descr->elsize) {
	v=(double*)data;
	glVertex3dv(v);
	glVertex3dv(v+3);
	glVertex3dv(v+6);
	glVertex3dv(v);
	glVertex3dv(v+9);
	glVertex3dv(v+3);
	glVertex3dv(v+6);
	glVertex3dv(v+9);
    }	
    glEnd(); 
    Py_INCREF(Py_None);
    return Py_None; 
}

static PyObject* gl_Vertex(PyObject* self, PyObject* args)
{
    PyObject *vop;
    PyArrayObject *ap;  
    double *dp, *max_dp;
    
    TRY (PyArg_ParseTuple(args, "O", &vop));
    
    TRY(ap = (PyArrayObject *)PyArray_ContiguousFromObject(vop, PyArray_DOUBLE, 1, 2));
    
    max_dp = dp = (double *)ap->data;
    max_dp += PyArray_Size((PyObject *)ap);

    switch (ap->dimensions[1]) {
    case 2:
      while (dp<max_dp) { glVertex2dv(dp); dp+=2; }
      break;
    case 3:
      while (dp<max_dp) { glVertex3dv(dp); dp+=3; }
      break;
    case 4:
      while (dp<max_dp) { glVertex4dv(dp); dp+=4; }
      break;
    default:
      ASSERT(0, "1-4d vertices required");
    }

    Py_INCREF(Py_None);
    return Py_None; 
}

static PyObject* gl_CallLists(PyObject* self, PyObject* args)
{
	PyObject *op;
	PyArrayObject *ap;
	int i, type;
    
    TRY (PyArg_ParseTuple(args, "O", &op));
    if (PyArray_Check(op)) {
      ap = (PyArrayObject *)op;
	  if (ap->nd != 1) {
		  PyErr_SetString(gl_Error, "calllists: array must be 1d");
		  return NULL;
	  }
	  Py_INCREF(ap);
    } else {
      TRY(ap = (PyArrayObject *)PyArray_ContiguousFromObject(op, PyArray_INT, 1, 1));
    }

    type = typecode2gltype[ap->descr->type_num];
	if (type == -1) {
		PyErr_SetString(gl_Error, "can't callists on this type of array");
		Py_DECREF(ap);
		return NULL;
	}

	glCallLists(ap->dimensions[0], type, ap->data);

	Py_DECREF(ap);
	Py_INCREF(Py_None);
	return Py_None;
}

#endif /* Not NUMERIC */

/*######################## GLUT STUFF ##################################*/

#ifdef GLUT 

#include <GL/glu.h>

extern void glutWireSphere(GLdouble, GLint, GLint);
extern void glutSolidSphere(GLdouble, GLint, GLint);
extern void glutWireCone(GLdouble, GLdouble, GLint, GLint);
extern void glutSolidCone(GLdouble, GLdouble, GLint, GLint);
extern void glutWireCube(GLdouble);
extern void glutSolidCube(GLdouble);
extern void glutWireTorus(GLdouble, GLdouble, GLint, GLint);
extern void glutSolidTorus(GLdouble, GLdouble, GLint, GLint);
extern void glutWireDodecahedron(void);
extern void glutSolidDodecahedron(void);
extern void glutWireTeapot(GLdouble);
extern void glutSolidTeapot(GLdouble);
extern void glutWireOctahedron();
extern void glutSolidOctahedron();
extern void glutWireTetrahedron();
extern void glutSolidTetrahedron();
extern void glutWireIcosahedron();
extern void glutSolidIcosahedron();

static PyObject *gl_WireSphere(PyObject *self, PyObject *args)
{
	PyObject *res = NULL;
	double radius=1.0;
	int slices=30, stacks=30;
	if (!PyArg_ParseTuple(args, "|dii", &radius, &slices, &stacks))
		return NULL;
	glutWireSphere(radius, slices, stacks);
	Py_INCREF(Py_None);
	res = Py_None;
	return res;
}

static PyObject *gl_SolidSphere(PyObject *self, PyObject *args)
{
	double radius=1.0;
	int slices=30, stacks=30;
   GLUquadricObj *quadObj;
	if (!PyArg_ParseTuple(args, "|dii", &radius, &slices, &stacks))
		return NULL;
   glutSolidSphere(radius, slices, stacks);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *gl_WireCone(PyObject *self, PyObject *args)
{
	PyObject *res = NULL;
	double base, height;
	int slices, stacks;
	if (!PyArg_ParseTuple(args, "ddii", &base, &height, &slices, &stacks))
		return NULL;
	glutWireCone(base, height, slices, stacks);
	Py_INCREF(Py_None);
	res = Py_None;
	return res;
}

static PyObject *gl_SolidCone(PyObject *self,PyObject * args)
{
	PyObject *res = NULL;
	double base, height;
	int slices, stacks;
	if (!PyArg_ParseTuple(args, "ddii", &base, &height, &slices, &stacks))
		return NULL;
	glutSolidCone(base, height, slices, stacks);
	Py_INCREF(Py_None);
	res = Py_None;
	return res;
}

static PyObject *gl_WireCube(PyObject *self, PyObject *args)
{
	PyObject *res = NULL;
	double size;
	if (!PyArg_ParseTuple(args, "d", &size))
		return NULL;
	glutWireCube(size);
	Py_INCREF(Py_None);
	res = Py_None;
	return res;
}

static PyObject *gl_SolidCube(PyObject *self, PyObject *args)
{
	PyObject *res = NULL;
	double size;
	if (!PyArg_ParseTuple(args, "d", &size))
		return NULL;
	glutSolidCube(size);
	Py_INCREF(Py_None);
	res = Py_None;
	return res;
}

static PyObject *gl_WireTorus(PyObject *self, PyObject *args)
{
	PyObject *res = NULL;
	double innerRadius, outerRadius;
	int sides, rings;
	if (!PyArg_ParseTuple(args, "ddii", &innerRadius,&outerRadius,
	                      &sides, &rings))
		return NULL;
	glutWireTorus(innerRadius, outerRadius, sides, rings);
	Py_INCREF(Py_None);
	res = Py_None;
	return res;
}

static PyObject *gl_SolidTorus(PyObject *self, PyObject *args)
{
	PyObject *res = NULL;
	double innerRadius, outerRadius;
	int sides, rings;
	if (!PyArg_ParseTuple(args, "ddii", &innerRadius,&outerRadius,
	                      &sides, &rings))
		return NULL;
	glutSolidTorus(innerRadius, outerRadius, sides, rings);
	Py_INCREF(Py_None);
	res = Py_None;
	return res;
}

static PyObject *gl_WireDodecahedron(PyObject *self, PyObject *args)
{
	PyObject *res = NULL;
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	glutWireDodecahedron();
	Py_INCREF(Py_None);
	res = Py_None;
	return res;
}

static PyObject *gl_SolidDodecahedron(PyObject *self, PyObject *args)
{
	PyObject *res = NULL;
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	glutSolidDodecahedron();
	Py_INCREF(Py_None);
	res = Py_None;
	return res;
}

static PyObject *gl_WireTeapot(PyObject *self, PyObject *args)
{
	PyObject *res = NULL;
	double size;
	if (!PyArg_ParseTuple(args, "d", &size))
		return NULL;
	glutWireTeapot(size);
	Py_INCREF(Py_None);
	res = Py_None;
	return res;
}

static PyObject *gl_SolidTeapot(PyObject *self, PyObject *args)
{
	PyObject *res = NULL;
	double size;
	if (!PyArg_ParseTuple(args, "d", &size))
		return NULL;
	glutSolidTeapot(size);
	Py_INCREF(Py_None);
	res = Py_None;
	return res;
}

static PyObject *gl_WireOctahedron(PyObject *self, PyObject *args)
{
	PyObject *res = NULL;
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	glutWireOctahedron();
	Py_INCREF(Py_None);
	res = Py_None;
	return res;
}

static PyObject *gl_SolidOctahedron(PyObject *self, PyObject *args)
{
	PyObject *res = NULL;
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	glutSolidOctahedron();
	Py_INCREF(Py_None);
	res = Py_None;
	return res;
}

static PyObject *gl_WireTetrahedron(PyObject *self, PyObject *args)
{
	PyObject *res = NULL;
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	glutWireTetrahedron();
	Py_INCREF(Py_None);
	res = Py_None;
	return res;
}

static PyObject *gl_SolidTetrahedron(PyObject *self, PyObject *args)
{
	PyObject *res = NULL;
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	glutSolidTetrahedron();
	Py_INCREF(Py_None);
	res = Py_None;
	return res;
}

static PyObject *gl_WireIcosahedron(PyObject *self, PyObject *args)
{
	PyObject *res = NULL;
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	glutWireIcosahedron();
	Py_INCREF(Py_None);
	res = Py_None;
	return res;
}

static PyObject *gl_SolidIcosahedron(PyObject *self, PyObject *args)
{
	PyObject *res = NULL;
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	glutSolidIcosahedron();
	Py_INCREF(Py_None);
	res = Py_None;
	return res;
}

#endif /*GLUT*/

/* ###################### TOGL STUFF ######################################*/

static PyObject* gl_TranslateScene(PyObject *self, PyObject *args) 
{
	GLint x, y, mousex, mousey;
	double s;
	GLdouble mat[16];
	if(!PyArg_ParseTuple(args, "diiii", &s, &x, &y, &mousex, &mousey))
		return NULL;
	glMatrixMode(GL_MODELVIEW);
	glGetDoublev(GL_MODELVIEW_MATRIX, mat);
	glLoadIdentity();
	glTranslatef(s * (x - mousex), s * (mousey-y), 0.0);
	glMultMatrixd(mat);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject* gl_RotateScene(PyObject *self, PyObject *args) 
{
	GLint x, y, mousex, mousey;
	GLdouble s, xcenter, ycenter, zcenter, mat[16];
	if(!PyArg_ParseTuple(args, "ddddiiii", &s, &xcenter, &ycenter, &zcenter,
                                               &x, &y, &mousex, &mousey))
		return NULL;
	glMatrixMode(GL_MODELVIEW);
	glGetDoublev(GL_MODELVIEW_MATRIX, mat);
	glLoadIdentity();
	glTranslatef(xcenter, ycenter, zcenter);
	glRotatef(s * (y - mousey), 1., 0., 0.);
	glRotatef(s * (x - mousex), 0., 1., 0.);
	glTranslatef(-xcenter, -ycenter, -zcenter);
	glMultMatrixd(mat);
	Py_INCREF(Py_None);
	return Py_None;
}


#define v3op(a,op,b)      (a[0] op b[0], a[1] op b[1], a[2] op b[2])
#define v3eqn(a,aop,b,o,c)(a[0] aop b[0] o c[0], a[1] aop b[1] o c[1], \
                           a[2] aop b[2] o c[2])
#define v3dot(a,b)        ((a[0] * b[0]) + (a[1] * b[1]) + (a[2] * b[2]))
#define v3lengthsq(v)     v3dot(v, v)
#define VSQ(a)            ((a) * (a))
#define v3distsq(a,b)     (VSQ(a[0] - b[0]) + VSQ(a[1] - b[1]) + VSQ(a[2] - b[2]))

/*
 * Square of perpendicular distance of point from line.
 * gl.DistFromLine(x, p1, p2)
 * p1 and p2 are the end points of the line (for instance
 * returned from gl.UnProject.
 * Should transform this to Python documentation!
 */

static PyObject* gl_DistFromLine(PyObject *self, PyObject *args) 
{
   PyObject *ox, *op1, *op2, *ret;
   GLdouble *x, *p1, *p2;
   int nx, np1, np2;
   GLdouble p1top2[3], p2top1[3], p1tox[3], p2tox[3];
   GLdouble dsq, dot;

   if(!PyArg_ParseTuple(args, "OOO", &ox, &op1, &op2)) 
      return NULL;

   TRY (PyArray_AsDoubleArray(&ox, &x, &nx));
   TRY (PyArray_AsDoubleArray(&op1, &p1, &np1));
   TRY (PyArray_AsDoubleArray(&op2, &p2, &np2));

   if(nx != 3 || np1 != 3 || np2 != 3){
      PyErr_SetString(gl_Error, "All arguments should contain 3 items!");
		PyArray_ClearMemory(ox, x);
		PyArray_ClearMemory(op1, p1);
		PyArray_ClearMemory(op2, p2);
      return NULL;
   }

/* check if point lies off either end of line segment */

   v3eqn(p1top2, =, p2, -, p1);
   v3eqn(p1tox, =, x, -, p1);

   if(v3dot(p1tox, p1top2) < 0.0) {
      PyArray_ClearMemory(ox, x);
		PyArray_ClearMemory(op1, p1);
		PyArray_ClearMemory(op2, p2);
      return PyFloat_FromDouble(HUGE);
   }

   v3op(p2top1, =, - p1top2);
   v3eqn(p2tox, =, x, -, p2);

   if(v3dot(p2tox, p2top1) < 0.0) {
		PyArray_ClearMemory(ox, x);
		PyArray_ClearMemory(op1, p1);
		PyArray_ClearMemory(op2, p2);
      return PyFloat_FromDouble(HUGE);
	}

/* from dot product */

   dot = v3dot(p1top2, p1tox);

   dsq = dot * dot / v3lengthsq(p1top2);

/* from pythagoras */

   ret = PyFloat_FromDouble(v3distsq(p1, x) - dsq);
	PyArray_ClearMemory(ox, x);
	PyArray_ClearMemory(op1, p1);
	PyArray_ClearMemory(op2, p2);
	return ret;
}



/* ########################## END OF TOGL STUFF ###########################*/

static PyMethodDef gl_methods[] = {
  	{"Accum",				gl_Accum,				1},
  	{"AlphaFunc",			gl_AlphaFunc, 			1},
  	{"Begin",				gl_Begin, 				1},
  	{"BlendFunc",			gl_BlendFunc, 			1},
  	{"CallList",       	gl_CallList, 			1},
  	{"Clear",          	gl_Clear,				1},
  	{"ClearAccum",     	gl_ClearAccum, 		1},
  	{"ClearColor",     	gl_ClearColor, 		1},
  	{"ClearDepth",     	gl_ClearDepth, 		1},
  	{"ClearIndex",     	gl_ClearIndex, 		1},
  	{"ClearStencil",   	gl_ClearStencil, 		1},
  	{"Color3",        	gl_Color3d, 			1},
  	{"Color3b",        	gl_Color3s, 			1},
  	{"Color3bv",        	gl_Color3s, 			1},
  	{"Color3d",        	gl_Color3d, 			1},
  	{"Color3dv",        	gl_Color3d, 			1},
  	{"Color3f",        	gl_Color3f, 			1},
  	{"Color3fv",        	gl_Color3f, 			1},
  	{"Color3i",        	gl_Color3i, 			1},
  	{"Color3iv",        	gl_Color3i, 			1},
  	{"Color3s",        	gl_Color3s, 			1},
  	{"Color3sv",        	gl_Color3s, 			1},
  	{"Color3ub",       	gl_Color3s, 			1},
  	{"Color3ubv",       	gl_Color3s, 			1},
  	{"Color3ui",       	gl_Color3i, 			1},
  	{"Color3uiv",       	gl_Color3i, 			1},
  	{"Color3us",       	gl_Color3s, 			1},
  	{"Color3usv",       	gl_Color3s, 			1},
  	{"Color4",       	gl_Color4d, 			1},
  	{"Color4b",        	gl_Color4s, 			1},
  	{"Color4bv",       	gl_Color4s, 			1},
  	{"Color4d",       	gl_Color4d, 			1},
  	{"Color4dv",        	gl_Color4d, 			1},
  	{"Color4f",        	gl_Color4f, 			1},
  	{"Color4fv",        	gl_Color4f, 			1},
  	{"Color4i",        	gl_Color4i, 			1},
  	{"Color4iv",        	gl_Color4i, 			1},
  	{"Color4s",        	gl_Color4s, 			1},
  	{"Color4sv",        	gl_Color4s, 			1},
  	{"Color4ub",       	gl_Color4s, 			1},
  	{"Color4ubv",       	gl_Color4s, 			1},
  	{"Color4ui",       	gl_Color4i, 			1},
  	{"Color4uiv",       	gl_Color4i, 			1},
  	{"Color4us",       	gl_Color4s, 			1},
  	{"Color4usv",       	gl_Color4s, 			1},
  	{"ColorMask",      	gl_ColorMask, 			1},
	{"ColorMaterial",  	gl_ColorMaterial, 	1},
  	{"CopyPixels",     	gl_CopyPixels, 		1},
  	{"CullFace",       	gl_CullFace, 			1},
  	{"DeleteLists",    	gl_DeleteLists, 		1},
  	{"DepthFunc",      	gl_DepthFunc, 			1},
  	{"DepthMask",      	gl_DepthMask, 			1},
  	{"DepthRange",     	gl_DepthRange, 		1},
  	{"Disable",        	gl_Disable, 			1},
  	{"DrawBuffer",     	gl_DrawBuffer, 		1},
  	{"EdgeFlag",       	gl_EdgeFlag, 			1},
  	{"Enable",         	gl_Enable, 				1},
  	{"End",            	gl_End, 					1},
  	{"EndList",        	gl_EndList, 			1},
  	{"EvalCoord1",    	gl_EvalCoord1d, 		1},
  	{"EvalCoord1d",    	gl_EvalCoord1d, 		1},
 	{"EvalCoord1dv",    	gl_EvalCoord1d, 		1},
	{"EvalCoord1f",    	gl_EvalCoord1f, 		1},
	{"EvalCoord1fv",		gl_EvalCoord1f, 		1},
  	{"EvalCoord2",    	gl_EvalCoord2d, 		1},
  	{"EvalCoord2d",    	gl_EvalCoord2d, 		1},
 	{"EvalCoord2dv",    	gl_EvalCoord2d, 		1},
	{"EvalCoord2f",    	gl_EvalCoord2f, 		1},
	{"EvalCoord2fv",    	gl_EvalCoord2f, 		1},
  	{"EvalMesh1",      	gl_EvalMesh1, 			1},
  	{"EvalMesh2",      	gl_EvalMesh2, 			1},
  	{"EvalPoint1",     	gl_EvalPoint1, 		1},
  	{"EvalPoint2",     	gl_EvalPoint2, 		1},
  	{"Finish",         	gl_Finish, 				1},
  	{"Flush",          	gl_Flush, 				1},
  	{"Fog",           	gl_Fogf, 				1},
  	{"Fogf",           	gl_Fogf, 				1},
	{"Fogfv",           	gl_Fogf, 				1},
  	{"Fogi",           	gl_Fogi, 				1},
  	{"Fogiv",           	gl_Fogi, 				1},
  	{"FrontFace",      	gl_FrontFace, 			1},
  	{"Frustum",        	gl_Frustum, 			1},
  	{"Hint",           	gl_Hint, 				1},
  	{"IndexMask",      	gl_IndexMask, 			1},
  	{"Index",         	gl_Indexd, 				1},
  	{"Indexd",         	gl_Indexd, 				1},
 	{"Indexdv",         	gl_Indexd, 				1},
  	{"Indexf",         	gl_Indexf, 				1},
  	{"Indexfv",         	gl_Indexf, 				1},
  	{"Indexi",         	gl_Indexi, 				1},
 	{"Indexiv",         	gl_Indexi, 				1},
  	{"Indexs",         	gl_Indexs, 				1},
  	{"Indexsv",         	gl_Indexs, 				1},
  	{"InitNames",      	gl_InitNames, 			1},
  	{"LightModel",    	gl_LightModelf, 		1},
  	{"LightModelf",    	gl_LightModelf, 		1},
  	{"LightModelfv",    	gl_LightModelf, 		1},
  	{"LightModeli",    	gl_LightModeli, 		1},
  	{"LightModeliv",    	gl_LightModeli, 		1},
  	{"Light",         	gl_Lightf, 				1},
  	{"Lightf",         	gl_Lightf, 				1},
  	{"Lightfv",        	gl_Lightf, 				1},  
  	{"Lighti",         	gl_Lighti, 				1},
  	{"Lightiv",        	gl_Lighti, 				1},  
  	{"LineStipple",    	gl_LineStipple, 		1},
  	{"LineWidth",      	gl_LineWidth, 			1},
  	{"ListBase",       	gl_ListBase, 			1},
  	{"LoadIdentity",   	gl_LoadIdentity, 		1},
  	{"LoadName",       	gl_LoadName, 			1},
  	{"LogicOp",        	gl_LogicOp, 			1},
  	{"Map1d",      		gl_Map1d, 				1},
  	{"Map1f",      		gl_Map1f, 				1},
  	{"Map2d",      		gl_Map2d, 				1},
  	{"Map2f",      		gl_Map2f, 				1},
  	{"MapGrid1d",      	gl_MapGrid1d, 			1},
  	{"MapGrid1f",      	gl_MapGrid1f, 			1},
  	{"MapGrid2d",      	gl_MapGrid2d, 			1},
  	{"MapGrid2f",      	gl_MapGrid2f, 			1},
  	{"Material",      	gl_Materialf, 			1},
  	{"Materialf",      	gl_Materialf, 			1},
  	{"Materialfv",     	gl_Materialf, 			1},
  	{"Materiali",      	gl_Materiali, 			1},
  	{"Materialiv",     	gl_Materiali, 			1},  
  	{"MatrixMode",     	gl_MatrixMode, 		1},
  	{"MultMatrix",     	gl_MultMatrixd, 		1},
  	{"MultMatrixd",     	gl_MultMatrixd, 		1},
  	{"MultMatrixf",     	gl_MultMatrixd, 		1},
  	{"NewList",        	gl_NewList, 			1},
  	{"Normal3",       	gl_Normal3d, 			1},
  	{"Normal3b",       	gl_Normal3s, 			1},
  	{"Normal3bv",       	gl_Normal3s, 			1},
  	{"Normal3d",       	gl_Normal3d, 			1},
  	{"Normal3dv",       	gl_Normal3d, 			1},
  	{"Normal3f",       	gl_Normal3f, 			1},
  	{"Normal3fv",       	gl_Normal3f, 			1},
  	{"Normal3i",       	gl_Normal3i, 			1},
 	{"Normal3iv",       	gl_Normal3i, 			1},
 	{"Normal3s",       	gl_Normal3s, 			1},
 	{"Normal3sv",       	gl_Normal3s, 			1},
  	{"Ortho",          	gl_Ortho, 				1},
  	{"PassThrough",    	gl_PassThrough, 		1},
  	{"PixelStoref",    	gl_PixelStoref, 		1},
  	{"PixelStorei",    	gl_PixelStorei, 		1},
  	{"PixelTransferf", 	gl_PixelTransferf, 	1},
  	{"PixelTransferi", 	gl_PixelTransferi, 	1},
  	{"PixelZoom",      	gl_PixelZoom, 			1},
  	{"PointSize",      	gl_PointSize, 			1},
  	{"PolygonMode",    	gl_PolygonMode, 		1},
  	{"PopAttrib",      	gl_PopAttrib, 			1},
  	{"PopMatrix",      	gl_PopMatrix, 			1},
  	{"PopName",        	gl_PopName, 			1},
  	{"PushAttrib",     	gl_PushAttrib, 		1},
  	{"PushMatrix",     	gl_PushMatrix, 		1},
  	{"PushName",       	gl_PushName, 			1},
  	{"RasterPos2",    	gl_RasterPos2d, 		1},
  	{"RasterPos2d",    	gl_RasterPos2d, 		1},
  	{"RasterPos2dv",    	gl_RasterPos2d, 		1},
  	{"RasterPos2f",    	gl_RasterPos2f, 		1},
  	{"RasterPos2fv",    	gl_RasterPos2f, 		1},
  	{"RasterPos2i",    	gl_RasterPos2i, 		1},
  	{"RasterPos2iv",    	gl_RasterPos2i, 		1},
  	{"RasterPos2s",    	gl_RasterPos2s, 		1},
  	{"RasterPos2sv",    	gl_RasterPos2s, 		1},
  	{"RasterPos3",    	gl_RasterPos3d, 		1},
  	{"RasterPos3d",    	gl_RasterPos3d, 		1},
  	{"RasterPos3dv",    	gl_RasterPos3d, 		1},
  	{"RasterPos3f",    	gl_RasterPos3f, 		1},
  	{"RasterPos3fv",    	gl_RasterPos3f, 		1},
  	{"RasterPos3i",    	gl_RasterPos3i, 		1},
  	{"RasterPos3iv",    	gl_RasterPos3i, 		1},
  	{"RasterPos3s",    	gl_RasterPos3s, 		1},
  	{"RasterPos3sv",    	gl_RasterPos3s, 		1},
  	{"RasterPos4",    	gl_RasterPos4d, 		1},
  	{"RasterPos4d",    	gl_RasterPos4d, 		1},
  	{"RasterPos4dv",    	gl_RasterPos4d, 		1},
  	{"RasterPos4f",    	gl_RasterPos4f, 		1},
  	{"RasterPos4fv",    	gl_RasterPos4f, 		1},
  	{"RasterPos4i",    	gl_RasterPos4i, 		1},
  	{"RasterPos4iv",    	gl_RasterPos4i, 		1},
  	{"RasterPos4s",    	gl_RasterPos4s, 		1},
  	{"RasterPos4sv",    	gl_RasterPos4s, 		1},
  	{"ReadBuffer",     	gl_ReadBuffer, 		1},
  	{"Rect",          	gl_Rectd,				1},
  	{"Rectd",          	gl_Rectd,				1},
	{"Rectdv",          	gl_Rectd,				1},
  	{"Rectf",          	gl_Rectf,				1},
  	{"Rectfv",          	gl_Rectf,				1},
  	{"Recti",          	gl_Recti, 				1},
  	{"Rectiv",          	gl_Recti, 				1},
  	{"Rects",          	gl_Rects, 				1},
  	{"Rectsv",          	gl_Rects, 				1},
	{"Rotate",        	gl_Rotated, 			1},
	{"Rotated",        	gl_Rotated, 			1},
	{"Rotated",        	gl_Rotated, 			1},
  	{"Rotatef",        	gl_Rotatef, 			1},
  	{"Scale",         	gl_Scaled, 				1},
  	{"Scaled",         	gl_Scaled, 				1},
  	{"Scalef",         	gl_Scalef,				1},
  	{"Scissor",        	gl_Scissor,				1},
  	{"ShadeModel",     	gl_ShadeModel,			1},
  	{"StencilFunc",    	gl_StencilFunc,		1},
  	{"StencilMask",    	gl_StencilMask,		1},
  	{"StencilOp",      	gl_StencilOp,			1},
  	{"TexCoord1",     	gl_TexCoord1d, 		1},
  	{"TexCoord1d",     	gl_TexCoord1d, 		1},
 	{"TexCoord1dv",     	gl_TexCoord1d, 		1},
  	{"TexCoord1f",     	gl_TexCoord1f,			1},
  	{"TexCoord1fv",     	gl_TexCoord1f,			1},
  	{"TexCoord1i",     	gl_TexCoord1i,			1},
  	{"TexCoord1iv",     	gl_TexCoord1i,			1},
  	{"TexCoord1s",     	gl_TexCoord1s,			1},
  	{"TexCoord1sv",     	gl_TexCoord1s,			1},
 	{"TexCoord2",     	gl_TexCoord2d, 		1},
 	{"TexCoord2d",     	gl_TexCoord2d, 		1},
 	{"TexCoord2dv",     	gl_TexCoord2d, 		1},
  	{"TexCoord2f",     	gl_TexCoord2f,			1},
  	{"TexCoord2fv",     	gl_TexCoord2f,			1},
  	{"TexCoord2i",     	gl_TexCoord2i,			1},
  	{"TexCoord2iv",     	gl_TexCoord2i,			1},
  	{"TexCoord2s",     	gl_TexCoord2s,			1},
  	{"TexCoord2sv",     	gl_TexCoord2s,			1},
 	{"TexCoord3",     	gl_TexCoord3d, 		1},
 	{"TexCoord3d",     	gl_TexCoord3d, 		1},
 	{"TexCoord3dv",     	gl_TexCoord3d, 		1},
  	{"TexCoord3f",     	gl_TexCoord3f,			1},
  	{"TexCoord3fv",     	gl_TexCoord3f,			1},
  	{"TexCoord3i",     	gl_TexCoord3i,			1},
  	{"TexCoord3iv",     	gl_TexCoord3i,			1},
  	{"TexCoord3s",     	gl_TexCoord3s,			1},
  	{"TexCoord3sv",     	gl_TexCoord3s,			1},
 	{"TexCoord4",     	gl_TexCoord4d, 		1},
 	{"TexCoord4d",     	gl_TexCoord4d, 		1},
 	{"TexCoord4dv",     	gl_TexCoord4d, 		1},
  	{"TexCoord4f",     	gl_TexCoord4f,			1},
  	{"TexCoord4fv",     	gl_TexCoord4f,			1},
  	{"TexCoord4i",     	gl_TexCoord4i,			1},
  	{"TexCoord4iv",     	gl_TexCoord4i,			1},
  	{"TexCoord4s",     	gl_TexCoord4s,			1},
  	{"TexCoord4sv",     	gl_TexCoord4s,			1},
  	{"TexEnv",        	gl_TexEnvf, 			1},
  	{"TexEnvf",        	gl_TexEnvf, 			1},
  	{"TexEnvfv",        	gl_TexEnvf, 			1},
  	{"TexEnvi",        	gl_TexEnvi,				1},
 	{"TexEnviv",        	gl_TexEnvi,				1},
  	{"TexGen",        	gl_TexGend, 			1},
  	{"TexGend",        	gl_TexGend, 			1},
  	{"TexGendv",        	gl_TexGend, 			1},
  	{"TexGenf",        	gl_TexGenf,				1},
  	{"TexGenfv",        	gl_TexGenf, 			1},
  	{"TexGeni",        	gl_TexGeni,				1},
  	{"TexGeniv",        	gl_TexGeni, 			1},
  	{"TexParameter",  	gl_TexParameterf, 	1},
  	{"TexParameterf",  	gl_TexParameterf, 	1},
  	{"TexParameterfv",  	gl_TexParameterf, 	1},
  	{"TexParameteri",  	gl_TexParameteri,		1},
  	{"TexParameteriv",  	gl_TexParameteri, 	1},
  	{"Translate",     	gl_Translated, 		1},
  	{"Translated",     	gl_Translated, 		1},
  	{"Translatef",     	gl_Translatef,			1},
  	{"Vertex2",       	gl_Vertex2d, 			1},
  	{"Vertex2d",       	gl_Vertex2d, 			1},
  	{"Vertex2dv",      	gl_Vertex2d, 			1},
  	{"ColorVertex2",     gl_ColorVertex2d, 	1},
  	{"ColorVertex2d",    gl_ColorVertex2d, 	1},
  	{"Vertex2f",       	gl_Vertex2f,			1},
  	{"Vertex2fv",      	gl_Vertex2f, 			1},
  	{"Vertex2i",       	gl_Vertex2i,			1},
  	{"Vertex2iv",       	gl_Vertex2i, 			1},
  	{"Vertex2s",       	gl_Vertex2s, 			1},
  	{"Vertex2sv",       	gl_Vertex2s, 			1},
  	{"Vertex3",       	gl_Vertex3d, 			1},
  	{"Vertex3d",       	gl_Vertex3d, 			1},
  	{"Vertex3dv",      	gl_Vertex3d, 			1},
  	{"Vertex3f",       	gl_Vertex3f,			1},
  	{"Vertex3fv",      	gl_Vertex3f, 			1},
  	{"Vertex3i",       	gl_Vertex3i,			1},
  	{"Vertex3iv",       	gl_Vertex3i, 			1},
  	{"Vertex3s",       	gl_Vertex3s, 			1},
  	{"Vertex3sv",       	gl_Vertex3s, 			1},
  	{"Vertex4",       	gl_Vertex4d, 			1},
  	{"Vertex4d",       	gl_Vertex4d, 			1},
  	{"Vertex4dv",      	gl_Vertex4d, 			1},
  	{"Vertex4f",       	gl_Vertex4f,			1},
  	{"Vertex4fv",      	gl_Vertex4f, 			1},
  	{"Vertex4i",       	gl_Vertex4i,			1},
  	{"Vertex4iv",       	gl_Vertex4i, 			1},
  	{"Vertex4s",       	gl_Vertex4s, 			1},
  	{"Vertex4sv",       	gl_Vertex4s, 			1},
  	{"Viewport",       	gl_Viewport, 			1},
 
 	{"ClipPlane",			gl_ClipPlane,			1},   
 	{"GetClipPlane",		gl_GetClipPlane,		1},   
 	{"IsEnabled",			gl_IsEnabled,			1},   
 	{"IsList",				gl_IsList,				1},   
 	{"RenderMode",			gl_RenderMode,			1},
	{"GenLists",			gl_GenLists,			1}, 
  	{"GetDouble",			gl_GetDoublev, 		1},  
  	{"GetDoublev",			gl_GetDoublev, 		1},

/*Not terribly useful without NumPy, but they'll work*/
	{"DrawPixels",			gl_DrawPixels,			1},  
	{"TexImage2D",			gl_TexImage2D,			1},
	{"ReadPixels",			gl_ReadPixels,			1},
	
/*Various OpenGL extensions */
  	{"PolygonOffsetEXT", gl_PolygonOffsetEXT,	1},
  	{"ArrayElementEXT",	gl_ArrayElementEXT,	1}, 
#ifdef NOT_DEFINED
	{"DrawArraysEXT",		gl_DrawArraysEXT,		1},  
 	{"BlendColorEXT",		gl_BlendColorEXT,		1},   
 	{"BlendEquationEXT",	gl_BlendEquationEXT,	1},  
#endif

#ifdef NUMERIC
	{"TrianglesWithNormals",	gl_TrianglesWithNormals,	1}, 
	{"Triangles",			gl_Triangles,			1}, 
	{"Lines",				gl_Lines,				1}, 
	{"Points",				gl_Points,				1}, 
	{"Tetrahedra",			gl_Tetrahedra,			1},  
	{"Vertex",				gl_Vertex,				1},	
	{"CallLists",			gl_CallLists,			1},  
#endif /* Not NUMERIC */


#ifdef GLUT
	{"WireSphere", 		(PyCFunction)gl_WireSphere, 			1,
	 "(double radius, int slices, int stacks) -> None"},
	{"SolidSphere", 		(PyCFunction)gl_SolidSphere, 			1,
	 "(double radius, int slices, int stacks) -> None"},
	{"WireCone", 			(PyCFunction)gl_WireCone, 				1,
	 "(double base, double height, int slices, int stacks) -> None"},
	{"SolidCone", 			(PyCFunction)gl_SolidCone, 			1,
	 "(double base, double height, int slices, int stacks) -> None"},
	{"WireCube", 			(PyCFunction)gl_WireCube, 				1,
	 "(double size) -> None"},
	{"SolidCube", 			(PyCFunction)gl_SolidCube, 			1,
	 "(double size) -> None"},
	{"WireTorus", 			(PyCFunction)gl_WireTorus, 			1,
	 "(double innerRadius, double outerRadius, int sides, int rings) -> None"},
	{"SolidTorus", 		(PyCFunction)gl_SolidTorus, 			1,
	 "(double innerRadius, double outerRadius, int sides, int rings) -> None"},
	{"WireDodecahedron", 	(PyCFunction)gl_WireDodecahedron,1,
	 "() -> None"},
	{"SolidDodecahedron",(PyCFunction)gl_SolidDodecahedron, 	1,
	 "() -> None"},
	{"WireTeapot", 		(PyCFunction)gl_WireTeapot, 			1,
	 "(double size) -> None"},
	{"SolidTeapot", 		(PyCFunction)gl_SolidTeapot, 			1,
	 "(double size) -> None"},
	{"WireOctahedron", 	(PyCFunction)gl_WireOctahedron, 		1,
	 "() -> None"},
	{"SolidOctahedron", 	(PyCFunction)gl_SolidOctahedron, 	1,
	 "() -> None"},
	{"WireTetrahedron", 	(PyCFunction)gl_WireTetrahedron,	 	1,
	 "() -> None"},
	{"SolidTetrahedron", (PyCFunction)gl_SolidTetrahedron, 	1,
	 "() -> None"},
	{"WireIcosahedron", 	(PyCFunction)gl_WireIcosahedron, 	1,
	 "() -> None"},
	{"SolidIcosahedron", (PyCFunction)gl_SolidIcosahedron, 	1,
	 "() -> None"},
#endif /*GLUT*/

	{"TranslateScene",	gl_TranslateScene, 						1 },
	{"RotateScene", 		gl_RotateScene, 							1 },
	{"DistFromLine",		gl_DistFromLine,							1 },
  	{NULL,               NULL} 
};

void initopengl()
{
	PyObject *m, *d;
	m = Py_InitModule("opengl", gl_methods);
	d = PyModule_GetDict(m);
	gl_Error = Py_BuildValue("s", "opengl.error");
	PyDict_SetItemString(d, "error", gl_Error);
	if (PyErr_Occurred()) 
		Py_FatalError("can't initialize module opengl");
}
