/**
 *
 * GLU Module for Python (uncomplete)
 * 
 * Version: 0.7
 *
 * Date: 96/09/12 (September)
 *
 * Authors: Thomas Schwaller  <tom.schwaller@linux-magazin.de>
 *          Jim Hugunin       <jjh@Goldilocks.LCS.MIT.EDU>
 *          David Ascher      <david_ascher@brown.edu>
 * 
 * Notes:
 *
 *   - the file gluconst.py is also needed, as it defines all kinds of
 *     useful constants
 *     
***/

#include <GL/glu.h>
#include "Python.h"

typedef struct {
  PyObject_HEAD
  GLUquadricObj* quad;
} gluQuadricObject;

extern DL_IMPORT(PyTypeObject) GLUquadricType;

#define is_gluQuadricObject(op) ((op)->ob_type == &GLUquadricType)

GLUquadricObj* getgluquadricvalue(PyObject* op)
{
  if (!op || !is_gluQuadricObject(op)) {
    PyErr_BadInternalCall();
    return NULL;
  }
  else
    return ((gluQuadricObject*)op)->quad;    
}

PyObject* newgluQuadricObject(GLUquadricObj* Quad)
{
  gluQuadricObject* op;
  op = PyObject_NEW(gluQuadricObject, &GLUquadricType);
  if (op == NULL)
    return PyErr_NoMemory();
  op->ob_type = &GLUquadricType;
  op->quad = Quad;
  return (PyObject*)op;
}

static void gluquadric_delete(gluQuadricObject* op)
{
  if (op->quad) gluDeleteQuadric(op->quad);
  PyMem_DEL(op);
}

static PyObject* gluquadirc_getattr(gluQuadricObject* op, char* name)
{
  static struct PyMethodDef glu_methods[] = {  
    {NULL,                   NULL}
  };
  return Py_FindMethod(glu_methods, (PyObject *) op, name);
}

static PyObject* glu_NewQuadric(PyObject* self, PyObject* args)
{
  if (!PyArg_ParseTuple(args, ""))
    return NULL;
  return newgluQuadricObject(gluNewQuadric());
}

static PyObject* glu_Sphere(PyObject* self, PyObject* args)
{
  PyObject* quad;
  GLdouble radius;
  GLint slices, stacks;
  if (!PyArg_ParseTuple(args, "Odii", &quad, &radius, &slices, &stacks))
    return NULL;
  gluSphere(getgluquadricvalue(quad), radius, slices, stacks);
  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *glu_QuadricDrawStyle(PyObject* self, PyObject* args)
{
  PyObject* quad;
  GLenum drawStyle;
  if(!PyArg_ParseTuple(args, "Oi", &quad, &drawStyle))
    return NULL;
  gluQuadricDrawStyle(getgluquadricvalue(quad), drawStyle);
  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *glu_Perspective(PyObject* self, PyObject* args)
{
  GLdouble a, b, c, d;
  if(!PyArg_ParseTuple(args, "dddd", &a, &b, &c, &d))
    return NULL;
  gluPerspective(a, b, c, d);
  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject* glu_LookAt(PyObject* self, PyObject* args)
{
  GLdouble a, b, c, d, e, f, g, h, i;
  if(!PyArg_ParseTuple(args, "ddddddddd", &a, &b, &c, &d, &e, &f, &g, &h, &i))
    return NULL;
  gluLookAt(a, b, c, d, e, f, g, h, i);
  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject* glu_Ortho2D(PyObject* self, PyObject* args)
{
  GLdouble a, b, c, d;
  if(!PyArg_ParseTuple(args, "dddd", &a, &b, &c, &d))
    return NULL;
  gluOrtho2D(a, b, c, d);
  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject* glu_PickMatrix(PyObject* self, PyObject* args)
{
  int i;
  GLdouble x, y, width, height;
  GLint viewport[4];
  PyObject* viewportlist;
  if(!PyArg_ParseTuple(args, "ddddO!", &x, &y, &width, &height, 
                                       &PyList_Type, &viewportlist))
    return NULL;
  for (i=0; i<4; i++)
    viewport[i] = PyFloat_AsDouble(PyList_GetItem(viewportlist, i));
  gluPickMatrix(x, y, width, height, viewport);
  Py_INCREF(Py_None);
  return Py_None;  
}

static PyObject* glu_UnProject(PyObject* self, PyObject* args)
{
   GLdouble winx, winy, winz;
   GLdouble objx, objy, objz;
   GLdouble model[16], proj[16];
   GLint view[16];

   if(!PyArg_ParseTuple(args, "ddd", &winx, &winy, &winz))
      return NULL;

   glGetDoublev(GL_MODELVIEW_MATRIX, model);
   glGetDoublev(GL_PROJECTION_MATRIX, proj);
   glGetIntegerv(GL_VIEWPORT, view);
   gluUnProject(winx, winy, winz, model, proj, view, &objx, &objy, &objz);

   return Py_BuildValue("(ddd)", objx, objy, objz);
}

static PyTypeObject GLUquadricType = {
   PyObject_HEAD_INIT(&PyType_Type)  
  0,			           /* Number of items for varobject */
  "glu",                           /* Name of this type */
  sizeof(gluQuadricObject),        /* Basic object size */
  0,			           /* Item size for varobject */
  (destructor)gluquadric_delete,   /* tp_dealloc*/
  0,                               /* tp_print*/
  (getattrfunc)gluquadirc_getattr, /* tp_getattr*/
  0,			           /* tp_setattr*/
  0,                               /* tp_compare*/
  0,                               /* tp_repr*/
  0,                               /* tp_as_number*/
  0,	                           /* tp_as_sequence*/
  0,			           /* tp_as_mapping*/
};

static PyMethodDef glu_methods[] = {
  { "NewQuadric",        glu_NewQuadric , 1},
  { "Sphere",            glu_Sphere , 1},  
  { "QuadricDrawStyle",  glu_QuadricDrawStyle , 1},    
  { "Perspective",       glu_Perspective, 1}, 
  { "LookAt",            glu_LookAt, 1},   
  { "Ortho2D",           glu_Ortho2D, 1},    
  { "PickMatrix",        glu_PickMatrix, 1},  
  { "UnProject",         glu_UnProject, 1},
  { NULL,                NULL }
};

void initglu()
{
  Py_InitModule("glu", glu_methods);
}
