#!/usr/local/bin/python

# A class that creates an opengl widget.
# Mike Hartshorn
# Department of Chemistry
# University of York, UK
# 

import opengl
gl=opengl
from glconst import *
import glu
from Tkinter import *

class RawOpengl(Widget, Misc):
  """Widget without any sophisticated bindings\
     by Tom Schwaller"""

  def __init__(self, master=None, cnf={}, **kw):
    Widget.__init__(self, master, 'togl', cnf, kw)
    self.bind('<Map>', self.tkMap)
    self.bind('<Expose>', self.tkExpose)
    self.bind('<Configure>', self.tkExpose)

  def tkRedraw(self, *dummy):
    self.tk.call(self._w, 'makecurrent')
    gl.PushMatrix()
    self.update_idletasks()
    self.redraw(self)
    gl.Flush()
    gl.PopMatrix()
    self.tk.call(self._w, 'swapbuffers')

  def tkMap(self, *dummy):
    self.tkExpose()

  def tkExpose(self, *dummy):
    self.tkRedraw()

class Opengl(RawOpengl):
  """\
Tkinter bindings for an Opengl widget.
Mike Hartshorn
Department of Chemistry
University of York, UK
http://www.yorvic.york.ac.uk/~mjh/
"""

  def __init__(self, master=None, cnf={}, **kw):
    """\
    Create an opengl widget.
    Arrange for redraws when the window is exposed or when
    it changes size."""

    #Widget.__init__(self, master, 'togl', cnf, kw)
    apply(RawOpengl.__init__, (self, master, cnf), kw)
    self.initialised = 0

    # Current coordinates of the mouse.
    self.xmouse = 0
    self.ymouse = 0

    # Where we are centering.
    self.xcenter = 0.0
    self.ycenter = 0.0
    self.zcenter = 0.0

    # The _back color
    self.r_back = 1.
    self.g_back = 0.
    self.b_back = 1.

    # Where the eye is
    self.distance = 10.0

    # Field of view in y direction
    self.fovy = 30.0

    # Position of clipping planes.
    self.near = 0.1
    self.far = 1000.0

    # Is the widget allowed to autospin?
    self.autospin_allowed = 0

    # Is the widget currently autospinning?
    self.autospin = 0

    # Basic bindings for the virtual trackball
    self.bind('<Map>', self.tkMap)
    self.bind('<Expose>', self.tkExpose)
    self.bind('<Configure>', self.tkExpose)
    self.bind('<Shift-Button-1>', self.tkHandlePick)
    #self.bind('<Button-1><ButtonRelease-1>', self.tkHandlePick)
    self.bind('<Button-1>', self.tkRecordMouse)
    self.bind('<B1-Motion>', self.tkTranslate)
    self.bind('<Button-2>', self.StartRotate)
    self.bind('<B2-Motion>', self.tkRotate)
    self.bind('<ButtonRelease-2>', self.tkAutoSpin)
    self.bind('<Button-3>', self.tkRecordMouse)
    self.bind('<B3-Motion>', self.tkScale)

  def help(self):
    """Help for the widget."""

    import Dialog
    d = Dialog.Dialog(None, {'title': 'Viewer help',
			     'text':
			     'Button-1: Translate\n'
			     'Button-2: Rotate\n'
			     'Button-3: Zoom\n'
			     'Reset: Resets transformation to identity\n',
			     'bitmap': 'questhead',
			     'default': 0,
			     'strings': ('Done', 'Ok')})

  def activate(self):
    """Cause this Opengl widget to be the current destination for drawing."""

    self.tk.call(self._w, 'makecurrent')

  # This should almost certainly be part of some derived class.
  # But I have put it here for convenience.

  def basic_lighting(self):
    """\
    Set up some basic lighting (single infinite light source).

    Also switch on the depth buffer."""
   
    self.activate()
    light_position = (1, 1, 1, 0);
    gl.Lightf(GL_LIGHT0, GL_POSITION, light_position);
    gl.Enable(GL_LIGHTING);
    gl.Enable(GL_LIGHT0);
    gl.DepthFunc(GL_LESS);
    gl.Enable(GL_DEPTH_TEST);
    gl.MatrixMode(GL_MODELVIEW);
    gl.LoadIdentity()

  def report_opengl_errors(message = "OpenGL error:"):
    """Report any opengl errors that occured while drawing."""

    while 1:
      err_value = gl.GetError()
      if not err_value: break     
      print message, glu.ErrorString(err_value)

  def set_background(self, r, g, b):
    """Change the background colour of the widget."""

    self.r_back = r
    self.g_back = g
    self.b_back = b

    self.tkRedraw()

  def set_centerpoint(self, x, y, z):
    """Set the new center point for the model.
    This is where we are looking."""

    self.xcenter = x
    self.ycenter = y
    self.zcenter = z

    self.tkRedraw()

  def set_eyepoint(self, distance):
    """Set how far the eye is from the position we are looking."""

    self.distance = distance
    self.tkRedraw()

  def reset(self):
    """Reset rotation matrix for this widget."""

    gl.MatrixMode(GL_MODELVIEW);
    gl.LoadIdentity()
    self.tkRedraw()

  def tkHandlePick(self, event):
    """Handle a pick on the scene."""

    if hasattr(self, 'pick'):
      # here we need to use glu.UnProject

      # Tk and X have their origin top left, 
      # while Opengl has its origin bottom left.
      # So we need to subtract y from the window height to get
      # the proper pick position for Opengl

      realy = self.winfo_height() - event.y

      p1 = glu.UnProject(event.x, realy, 0.)
      p2 = glu.UnProject(event.x, realy, 1.)

      if self.pick(self, p1, p2):
	"""If the pick method returns true we redraw the scene."""

	self.tkRedraw()

  def tkRecordMouse(self, event):
    """Record the current mouse position."""

    self.xmouse = event.x
    self.ymouse = event.y

  def StartRotate(self, event):

    # Switch off any autospinning if it was happening

    self.autospin = 0

    self.tkRecordMouse(event)

  def tkScale(self, event):
    """Scale the scene.  Achieved by moving the eye position."""

    scale = 1 - 0.01 * (event.y - self.ymouse)
    self.distance = self.distance * scale
    self.tkRedraw()
    self.tkRecordMouse(event)

  def do_AutoSpin(self):
    s = 0.5
    self.activate()

    gl.RotateScene(0.5,
		   self.xcenter, self.ycenter, self.zcenter,
		   self.yspin, self.xspin, 0, 0)
    self.tkRedraw()

    if self.autospin:
      self.after(10, self.do_AutoSpin)

  def tkAutoSpin(self, event):
    """Perform autospin of scene."""

    self.after(4)
    self.update_idletasks()

    # This could be done with one call to pointerxy but I'm not sure
    # it would any quicker as we would have to split up the resulting
    # string and then conv

    x = self.tk.getint(self.tk.call('winfo', 'pointerx', self._w))
    y = self.tk.getint(self.tk.call('winfo', 'pointery', self._w))

    if self.autospin_allowed:
      if x != event.x_root and y != event.y_root:
	self.autospin = 1

	self.yspin = x - event.x_root
	self.xspin = y - event.y_root

	self.after(10, self.do_AutoSpin)

  def tkRotate(self, event):
    """Perform rotation of scene."""

    self.activate()
    gl.RotateScene(0.5,
		   self.xcenter, self.ycenter, self.zcenter,
		   event.x, event.y, self.xmouse, self.ymouse)
    self.tkRedraw()
    self.tkRecordMouse(event)

  def tkTranslate(self, event):
    """Perform translation of scene."""

    self.activate()
    gl.TranslateScene(0.05, event.x, event.y, self.xmouse, self.ymouse)
    self.tkRedraw()
    self.tkRecordMouse(event)

  def tkRedraw(self, *dummy):
    """Cause the opengl widget to redraw itself."""

    if not self.initialised: return
    self.activate()

    gl.PushMatrix()			# Protect our matrix
    self.update_idletasks()
    w = self.winfo_width()
    h = self.winfo_height()
    gl.Viewport(0, 0, w, h)

    # Clear the background and depth buffer.
    gl.ClearColor(self.r_back, self.g_back, self.b_back, 0.)
    gl.Clear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT)

    gl.MatrixMode(GL_PROJECTION);
    gl.LoadIdentity()
    glu.Perspective(self.fovy, float(w)/float(h), self.near, self.far)

    if 0:
      # Now translate the scene origin away from the world origin
      gl.MatrixMode(GL_MODELVIEW);
      mat = gl.GetDoublev(GL_MODELVIEW_MATRIX)
      gl.LoadIdentity()
      gl.Translatef(-self.xcenter, -self.ycenter,
		  -(self.zcenter+self.distance))
      gl.MultMatrixd(mat)
    else:
      glu.LookAt(self.xcenter, self.ycenter, self.zcenter + self.distance,
		 self.xcenter, self.ycenter, self.zcenter,
		 0., 1., 0.)
      gl.MatrixMode(GL_MODELVIEW);

    # Call objects redraw method.
    self.redraw(self)
    gl.Flush()				# Tidy up
    gl.PopMatrix()			# Restore the matrix

    self.tk.call(self._w, 'swapbuffers')

  def tkMap(self, *dummy):
    """Cause the opengl widget to redraw itself."""

    self.tkExpose()

  def tkExpose(self, *dummy):
    """Redraw the widget.
    Make it active, update tk events, call redraw procedure and
    swap the buffers.  Note: swapbuffers is clever enough to
    only swap double buffered visuals."""

    self.activate()
    if not self.initialised:
      self.basic_lighting()
      self.initialised = 1
    self.tkRedraw()


  def tkPrint(self, file):
    """Turn the current scene into PostScript via the feedback buffer."""

    self.activate()

    
