/* Copyleft (c) 1996 by Lele Gaifax.  All Rights Reserved
 *
 * This file is part of Nothing (yet).
 *
 * $RCSfile: mod_pyapache.c,v $
 * $Revision: 2.25 $
 * $Date: 1997/01/22 13:31:48 $
 *
 * Created Fri May  3 15:12:20 1996.
 *
 * Mon Jun 10 1996: This was formerly known as ``mod_python.c''. It
 * supercedes a previous work by Dave Mitchell <davem@magnet.com>, a
 * patch to Apache's ``mod_cgi.c'' to make it recognize scripts
 * written in Python and handling them consequently; it was
 * distribuited as tar file named ``pyapache.tar''. To avoid confusion
 * between the names, he kindly suggested to rename my module using
 * that name. I also find that ``mod_pyapache.c'' makes easier guess
 * its content, so I happily applied the suggestion. Thanx Dave.
 */

/* This module supports the Native Execution of Python Scripts within
   the Apache HTTPD Server. It does not add any new capability to the
   server, but lets it execute Python Scripts without the need of the
   external Python Interpreter.

   To use it, you must first rebuild the Apache Server linking in this
   module:

     a. Drop this source in the `src/' directory of the Apache distribution.
     b. Modify src/Configuration adding these lines (*NOTE* that the
        `EXTRA_LIBS' definition *must* be on a single line! Here I splitted
        it in several chunks for readability):

        PYTHONDIR= /usr/local/lib/python1.4/config
        EXTRA_CFLAGS= -I$(PYTHONDIR)
        EXTRA_LIBS= $(PYTHONDIR)/config.o $(PYTHONDIR)/getpath.o \
                    $(PYTHONDIR)/libModules.a $(PYTHONDIR)/libPython.a \
                    $(PYTHONDIR)/libObjects.a $(PYTHONDIR)/libParser.a
        Module pyapache_module mod_pyapache.o

        of course adapting PYTHONDIR value to your situation.

	If your compiler does not already look into /usr/local/include
	(where most probably resides the `python-1.4' include directory),
	you should add `-I/usr/local/include' to EXTRA_CFLAGS.
	
	On Fri, 17 Jan 1997, Kent Polk <kent@eaenki.nde.swri.edu> said
	that on Solaris 2.5.1 he had to explicitly add libdl.so and
	libm.so to EXTRA_LIBS.
	
	
        Please note that you should execute a `make libainstall' in
        the Python source tree to get `getpath.c' and `config.c'
        installed (of course, this must be done *after* a successful
        installation of the interpreter itself); moreover, you need to
        compile those sources by yourself, since the install procedure
        does not complete the operation (too bad, neither the Makefile
        installed in that directory will help you): look at the option
        used to build the interpreter, and reusing them compile
        `getpath.c' and `config.c' by hand. The following lines are
        from a real situation:

        $ cd /usr/local/lib/python1.4/config
        $ cc -O2 -I/usr/local/include/python1.4 -I. -DHAVE_CONFIG_H \
          -c config.c
        $ cc -O2 -I/usr/local/include/python1.4 -I. -DHAVE_CONFIG_H \
          -DPYTHONPATH='".:/usr/local/lib/python1.4:/usr/local/lib/python1.4/next3:/usr/local/lib/python1.4/sharedmodules"' -c getpath.c 

        that, adjusting the PYTHONPATH value (in particular the
        machine-dep directory) should work for you too.
        
        Alternatively, you may point PYTHONDIR directly where you keep
        the Python source tree: in this case you will have to correct
        the location of the libraries (`libParser.a' is in the
        `Parser/' directory, `libObjects.a' lives in the `Objects/'
        directory, and so on):

        PYTHONDIR= /LocalDeveloper/WiP/Python-1.4
        EXTRA_CFLAGS= -I$(PYTHONDIR)
        EXTRA_LIBS= $(PYTHONDIR)/Modules/config.o \
                    $(PYTHONDIR)/Modules/getpath.o \
                    $(PYTHONDIR)/Modules/libModules.a \
                    $(PYTHONDIR)/Python/libPython.a \
                    $(PYTHONDIR)/Objects/libObjects.a \
                    $(PYTHONDIR)/Parser/libParser.a
        Module pyapache_module mod_pyapache.o

     c. Run the `Configure' script to update the Makefile.
     d. Run `make' to rebuild the server.
        
   Once the server is built, to activate this module you need to put
   somewhere in its `conf/srm.conf' a line like

     AddHandler python/x-httpd-cgi .py

   meaning that whenever the server is asked for a resource whose name
   ends in `.py' it should handle it through this module
   (`python-cgi-script' is an alias for `python/x-httpd-cgi').
   

   You can configure further the Python Environment on a per directory
   basis, for example to set the Verbosity On in your testing
   directory, or set a particular module search path... For example,
   my `conf/access.conf' contains something like

     <Directory /usr/local/etc/apache/cgi-bin/lele>
     PythonPath /users/lele/Library/testing-python
     PythonVerbose On
     PythonDebug On
     </Directory>

   to test these features: the `PythonPath' command cause its argument
   to be prepended to the standard path (computed at compile time by
   Python); `PythonVerbose' is equivalent to the `-v' option of the
   interpreter, and `PythonDebug' to the `-d' option; in this case
   these settings are available only in the given directory. */

/**** IN ALPHA TEST --- BE WARNED --- MEMORY LEAKS WILL OCCUR ****
  
  You can avoid forking a new process for each script, creating a
  Python interpreter in the same process space of the server, by using
  the `PythonPersistent' directive, usable only in the srm.conf config
  file and in particular *not* in a directory config; if you set

    PythonPersistent On

  your script will fly as fast as possible ;-). I'm not convinced yet
  that this part doesn't leak memory: if this happens in a child
  process, it's a bug, yes, but inoffensive since it will exit soon,
  thus releasing its memory, but in persistent mode the problem will
  grow for a possibly long time.....

  Guido van Rossum, on Wed, 12 Jun 1996, said:

    """Alas, there really is no way to reset the interpreter to
       virginal state without the possibility of leaking memory. You
       either have to trust your CGI scripts not to mess with other
       modules and not to create circular links, or you have to fork
       it as a subprocess."""

  In other words, persistent mode *DOES* leak memory.
  
  So I added another directive, `PythonTrustedScripts', On by default:
  it can be used in persistent mode only, and when On the Python
  Environment gets initialized once, and never destroyed; imported
  module are never released; sys.stderr remains hooked to the server's
  error_log lifetime. On the other hand, a module gets imported almost
  once, so you cannot count on import side effects.

  When turned Off, each script execution is bracketed between a
  Py_Initialize() and PyImport_Cleanup(): since the latter does not
  completely release Python Environment, there will be memory leaks.

  Eventually these directives will be collapsed in a single one. */


/* BUGS:
********

   * PythonPersistent mode does not handle URL redirection very well.
   
   * PythonPersistent mode leaks memory. This depends on current Python
     implementation, and workarounds are difficult.

   * PythonTrustedScripts does alter interpreter's behaviour: a module
     gets imported almost once, so you cannot count on import side
     effects.
  
   * When handling a parsed HTML, the directive `<!--#exec cgi="file.py"-->'
     is performed as a normal CGI, not through this module. This is caused
     by the way Apache treats that directive. The easier workaround is to use
     `<!--#include virtual="file.py"-->', that is similar to the former but
     does not have this limitation.

   * Since the Apache Server executes itself under user `nobody', it is able
     to create the compiled version of the scripts it executes only when the
     directory containing them is writable by everybody, which may be an
     insane choice in some circumstances. The workaround is to create/update
     the compiled scripts whenever you install their sources (see the
     `compileall.py' library module in the Python Distribution).

   * This module requires Python 1.4 and Apache 1.2b4, and will
     require some work to adapt it to higher releases. If you are still
     using Apache 1.1[.1], then you need release 2.21 of this module.     

   * Suspicious code is marked with a 'XXX' sign: feel free to email me
     your mumblings on it. */


/* Apache includes */
#include "httpd.h"
#include "http_main.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_protocol.h"
#include "http_request.h"
#include "util_script.h"

/* Python includes */
#include <python1.4/Python.h>
#include <python1.4/import.h>
#include <python1.4/compile.h>
#include <python1.4/marshal.h>
#include <python1.4/graminit.h>
#include <python1.4/node.h>
#include <python1.4/structmember.h>

/* ...they miss this */
extern long PyOS_GetLastModificationTime (char *path);


/* Apache cries for autoconf!!. Till then, you should check the
   following manually. For example Solaris 2.5.1 needs this
   define enabled */
#if 0
#define HAVE_ALLOCA_H
#endif


/* AIX requires this to be the first thing in the file.  */
#ifdef __GNUC__
# ifndef NeXT
#   define alloca __builtin_alloca
# endif
#else
# if HAVE_ALLOCA_H
#  include <alloca.h>
# else
#  ifdef _AIX
#pragma alloca
#  else
#   ifndef alloca /* predefined by HP cc +Olibcalls */
      char *alloca();
#   endif
#  endif
# endif
#endif

/* This define is used to test whether the ApacheXXX instances get
   released at each cycle. Normally set it to 0 (or #undef it) */
/* #define RELEASE_TEST 1 */

/* XXX Is this right? Am I correctly understanding the form of the type? Or
   maybe "application/python-httpd-cgi" is a better choice? Any HTTP guru
   out there?*/
#define PYTHON_MAGIC_TYPE       "python/x-httpd-cgi"
#define ALT_PYTHON_MAGIC_TYPE   "python-cgi-script"


/**********************************************************************
                       CONFIGURATION DIRECTIVES
**********************************************************************/

module pyapache_module;

/* In PythonPersistent mode we execute the request in the same process
   as the server, thus saving a fork(). */
static int PythonPersistent = 0;

/* If PythonTrustedScripts is enabled, we do not create/destroy the
   entire Python environment each time, but only its `__main__' module.
   NB: as the name suggests, you better trust your scripts: they shouldn't
   modify other builtin modules, expecially `sys' and `__builtin__'. */
static int PythonTrustedScripts = 1;

static PyObject *ApacheRequest_new (request_rec *r);

/* This is the structure that holds the per-directory configuration
   of this module. */
typedef struct python_dir_config
{
  char *path;                   /* set with PythonPath new-path */
  int debug;                    /* set with PythonDebug On|Off */
  int verbose;                  /* set with PythonVerbose On|Off */
} python_dir_config;

/* Create and initialize the per-directory config-structure.
   This is called by the server at initialization time. */
static void *
create_python_dir_config (pool *p, char *dummy)
{
  python_dir_config *conf = (python_dir_config *) palloc (p, sizeof (*conf));

  conf->path = NULL;
  conf->debug = 0;
  conf->verbose = 0;
  
  return conf;
}

/* This function gets called at configuration time to handle `PythonPath'
   directives. */
static const char *
set_python_path (cmd_parms *parms, python_dir_config *dc, char *arg)
{
  dc->path = pstrdup (parms->pool, arg);
  
  return NULL;
}

/* Returns the path to be used by the Python Script. This is computed
   by joining the value of the `PythonPath' directives, if any, with
   the standard path, retrieved with Py_GetPath(). */
static const char *
get_python_path (request_rec *r)
{
  extern char *Py_GetPath();
  python_dir_config *conf = (python_dir_config *) get_module_config (r->per_dir_config, &pyapache_module);
  char *standard_path = Py_GetPath();
  
  if (conf->path)
    return pstrcat (r->pool, conf->path, ":", standard_path, NULL);
  else
    return standard_path;
}

/* This function gets called at configuration time when a PythonDebug
   directive is found. */
static const char *
set_python_debug (cmd_parms *parms, python_dir_config *dc, char *arg)
{
  if (!strcasecmp (arg, "off"))
    dc->debug = 0;
  else if (!strcasecmp (arg, "on"))
    dc->debug = 1;
  else
    return "PythonDebug must be set to `On' or `Off'";

  return NULL;
}

/* This function gets called at configuration time when a
   PythonVerbose directive is found. */
static const char *
set_python_verbose (cmd_parms *parms, python_dir_config *dc, char *arg)
{
  if (!strcasecmp (arg, "off"))
    dc->verbose = 0;
  else if (!strcasecmp (arg, "on"))
    dc->verbose = 1;
  else
    return "PythonVerbose must be set to `On' or `Off'";

  return NULL;
}

/* This function gets called at configuration time when a
   PythonPersistent directive is found. */
static const char *
set_python_persistent (cmd_parms *parms, python_dir_config *dc, char *arg)
{
  if (!strcasecmp (arg, "off"))
    PythonPersistent = 0;
  else if (!strcasecmp (arg, "on"))
    PythonPersistent = 1;
  else
    return "PythonPersistent must be set to `On' or `Off'";

  return NULL;
}

/* This function gets called at configuration time when a
   PythonTrustedScripts directive is found. */
static const char *
set_python_trusted_scripts (cmd_parms *parms, python_dir_config *dc, char *arg)
{
  if (!PythonPersistent)
    return "PythonTrustedScripts must be used with `PythonPersistent' enabled";
  else
    {
      if (!strcasecmp (arg, "off"))
        PythonTrustedScripts = 0;
      else if (!strcasecmp (arg, "on"))
        PythonTrustedScripts = 1;
      else
        return "PythonPersistent must be set to `On' or `Off'";
    }

  return NULL;
}
  
static command_rec python_cmds[] =
{
  { "PythonPath", set_python_path, NULL, ACCESS_CONF|RSRC_CONF|OR_ALL, TAKE1, "the path to be used for Python Scripts" },
  { "PythonVerbose", set_python_verbose, NULL, ACCESS_CONF|RSRC_CONF|OR_ALL, TAKE1, "On or Off" },
  { "PythonDebug", set_python_debug, NULL, ACCESS_CONF|RSRC_CONF|OR_ALL, TAKE1, "On or Off" },
  { "PythonPersistent", set_python_persistent, NULL, RSRC_CONF, TAKE1, "On or Off" },
  { "PythonTrustedScripts", set_python_trusted_scripts, NULL, RSRC_CONF, TAKE1, "On or Off" },
  { NULL }
};


/**********************************************************************
                               HANDLERS
**********************************************************************/

static PyObject *initialize_apache_module (request_rec *r);

typedef struct python_child_stuff
{
  request_rec *r;
  pid_t pid;
  char *argv0;                  /* This is r->filename with the path stripped off */
  FILE *script_in;              /* This gets attached to the script's stdin */
  FILE *script_out;             /* and this to its stdout, if not NPH. */
  FILE *script_err;             /* This gets attached to the script's stderr. */
  int nph;                      /* If != 0 this is an NPH script */
} python_child_stuff;

/* Validates the script, initializing the `argv0' and `nph' slots of
   PCSP. Returns OK or the error code. */
static int
check (python_child_stuff *pcsp)
{
  request_rec *r = pcsp->r;

  if ((pcsp->argv0 = strrchr (pcsp->r->filename, '/')) != NULL)
    pcsp->argv0++;
  else
    pcsp->argv0 = pcsp->r->filename;

  /* If the name of the script begins with "nph-" then we are not going
     to parse the headers coming from it, but we will pass the script output
     "as is" to the client. */
  pcsp->nph = !(strncmp (pcsp->argv0, "nph-", 4));
  
  if (!(allow_options (r) & OPT_EXECCGI) && !is_scriptaliased (r))
    {
      log_reason("Option ExecCGI is off in this directory", r->filename, r);
      return FORBIDDEN;
    }
  
  if (pcsp->nph && !strcmp (r->protocol, "INCLUDED"))
    {
      log_reason("attempt to include NPH CGI script", r->filename, r);
      return FORBIDDEN;
    }

  if (S_ISDIR(r->finfo.st_mode))
    {
      log_reason("attempt to invoke directory as script", r->filename, r);
      return FORBIDDEN;
    }
  
  if (r->finfo.st_mode == 0)
    {
      log_reason("script not found or unable to stat", r->filename, r);
      return NOT_FOUND;
    }
  
  if (! can_exec (&r->finfo))
    {
      log_reason("file permissions deny server execution", r->filename, r);
      return FORBIDDEN;
    }

  {
    int retval;

    if ((retval = setup_client_block (r, REQUEST_CHUNKED_ERROR)))
      return retval;
  }
    
  return OK;
}

/* Checks the existence of a valid compiled script. If it is there, returns
   an opened file on it, otherwise NULL. */
static FILE *
check_compiled_module (char *cpathname, long mtime)
{
  FILE *fp;
  long magic;
  long pyc_mtime;

  fp = fopen (cpathname, "rb");
  if (fp == NULL)
    return NULL;
  
  magic = PyMarshal_ReadLongFromFile (fp);
  if (magic != PyImport_GetMagicNumber())
    {
      fclose(fp);
      return NULL;
    }
  
  pyc_mtime = PyMarshal_ReadLongFromFile (fp);
  if (pyc_mtime != mtime)
    {
      fclose(fp);
      return NULL;
    }

  return fp;
}

#define LOG_REASON(msg,fn,r)    do {                                                            \
                                  if (PythonPersistent)                                         \
                                    log_reason ((msg), (fn), (r));                              \
                                  else                                                          \
                                    fprintf (stderr, "[%s] %s: %s\n", get_time(), (fn), (msg)); \
                                } while (0)
  
/* Executes the Python Script, maybe fetching the compiled version if it
   exists. Returns OK or SERVER_ERROR. */
static int
exec_the_script (python_child_stuff *pcsp)
{
  char *compiled;
  int argv0_len;
  PyObject *main, *maindict, *result, *PyEval_EvalCode(), *error;
  PyCodeObject *code;
  FILE *fp;
  long mtime;

  /* Running the script with a simpler PyRun_SimpleFile() has one major
     disadvantage: it does not look for the compiled script (.pyc). On
     the other hand, importing the script as a module and then executing
     it wouldn't perform exacty what we want: for example the common
     usage of doing something like 'if __name__ == "__main__": main()' to
     let the script be used both as an executable script as well as a
     module will fail, because in that way the script in not executed in
     the '__main__' namespace...

     So here I had to study the Python implementation and, cutting&pasting,
     write another way, merging their best qualities. */

  main = PyImport_AddModule ("__main__");
  if (!main)
    {
      LOG_REASON ("Cannot add __main__ module", pcsp->r->filename, pcsp->r);
      return SERVER_ERROR;
    }

  maindict = PyModule_GetDict (main);
  if (PyDict_SetItemString (maindict, "__builtins__", PyEval_GetBuiltins()))
    {
      LOG_REASON ("Cannot add __builtins__ to __main__ module", pcsp->r->filename, pcsp->r);
      return SERVER_ERROR;
    }
      
  mtime = PyOS_GetLastModificationTime (pcsp->argv0);

  argv0_len = strlen (pcsp->argv0);
  compiled = palloc (pcsp->r->pool, argv0_len+2);
  strcpy (compiled, pcsp->argv0);
  compiled[argv0_len] = 'c';
  compiled[argv0_len+1] = '\0';

  if ((fp = check_compiled_module (compiled, mtime)))
    {
      /* Ok, a valid compiled script exist. */
  
      code = (PyCodeObject *) PyMarshal_ReadObjectFromFile (fp);
      if (! code)
        {
          LOG_REASON ("Cannot load compiled codeobject", pcsp->r->filename, pcsp->r);
          return SERVER_ERROR;
        }
    }
  else
    {
      node *n, *PyParser_SimpleParseFile();
      FILE *compfile;
      
      /* Nop, parse it and execute */
      fp = fopen (pcsp->argv0, "r");

      if (! fp)
        {
          LOG_REASON ("Cannot open the Python Script", pcsp->r->filename, pcsp->r);
          return SERVER_ERROR;
        }
      
      n = PyParser_SimpleParseFile (fp, pcsp->argv0, file_input);
      if (! n)
        {
          LOG_REASON ("Cannot parse the Python Script", pcsp->r->filename, pcsp->r);
	  PyErr_Print();
          return SERVER_ERROR;
        }

      code = PyNode_Compile (n, pcsp->argv0);
      PyNode_Free (n);

      if (! code)
        {
          LOG_REASON ("Cannot compile the Python Script", pcsp->r->filename, pcsp->r);
	  PyErr_Print();
          return SERVER_ERROR;
        }

      /* Write out the compiled version */
      compfile = fopen (compiled, "wb");
      if (compfile)
        {
          PyMarshal_WriteLongToFile (PyImport_GetMagicNumber(), compfile);

          /* First write a 0 for mtime */
          PyMarshal_WriteLongToFile (0L, compfile);

          PyMarshal_WriteObjectToFile ((PyObject *) code, compfile);

          if (ferror (compfile))
            {
              /* Don't keep partial file */
              fclose (compfile);
              (void) unlink (compiled);
            }
          else
            {
              /* Now write the true mtime */
              fseek (compfile, 4L, 0);
              PyMarshal_WriteLongToFile (mtime, compfile);
              
              fflush (compfile);
              fclose (compfile);
            }
        }
    }

  PyErr_Clear();
  result = PyEval_EvalCode (code, maindict, maindict);
  Py_DECREF (code);

  Py_XDECREF (result);

  if ((error = PyErr_Occurred()))
    {
      if (error != PyExc_SystemExit)
        {
          LOG_REASON ("Errors evaluating Python Script", pcsp->r->filename, pcsp->r);
          PyErr_Print();
        }
    }

  if (! PythonPersistent)
    {
      /* We are the child, and our parent is not paying particular
         attention on us, so it's useless to send back an error condition
         when the script exits with errors: in any case, our parent is
         going to parse script's output. Eventually the script may
         print something like 'Status: 500' on its output... */

      return OK;
    }
  else
    {
      int ret = OK;
      
      if (error == PyExc_SystemExit)
        {
          PyObject *exc, *val, *tb;

          PyErr_Fetch (&exc, &val, &tb);

          if (val && PyInt_Check (val))
            {
              log_printf (pcsp->r->server, "`%s' exited with status %ld",
                          pcsp->r->filename, PyInt_AS_LONG ((PyIntObject *) val));
            }
          else
            {
              PyObject *str = PyObject_Str (val);
              
              log_printf (pcsp->r->server, "`%s' exited with strange status `%s'",
                          pcsp->r->filename, PyString_AS_STRING ((PyStringObject *) str));
              Py_DECREF(str);
            }

          Py_DECREF(exc);
          Py_XDECREF(val);
          Py_DECREF(tb);
        }
      else
        {
          if (error)
            {
              PyErr_Clear();

              /* XXX
                 The server will add its own error message at the end
                 of the script's output, if any. Is this desired? */
              ret = SERVER_ERROR;
            }
        }
      
      return ret;
    }
}

/* Initializes the Python Environment, the load path, the arguments list
   and its process environment.

   If PythonTrustedScripts is On, then do most of the work just once. */
static int
initialize_python_environment (python_child_stuff *pcsp)
{
  python_dir_config *conf = (python_dir_config *) get_module_config (pcsp->r->per_dir_config, &pyapache_module);
  extern int Py_VerboseFlag, Py_DebugFlag;
  PyObject *os_module;
  const char *ppath;
  
  Py_VerboseFlag = conf->verbose;
  Py_DebugFlag = conf->debug;

  /* The standard Py_Initialize() will not reexecute twice since it is
     protected against multiple calls... but we want it! so do here what
     it is used to do.

     Py_Initialize(); */

  error_log2stderr (pcsp->r->server);
  
  if (PythonPersistent && PythonTrustedScripts)
    {
      static int first_time = 1;

      if (first_time)
        {
          PyImport_Init();
          PyBuiltin_Init();
          PySys_Init();

          first_time = 0;
        }
    }
  else
    {
      PyImport_Init();
      PyBuiltin_Init();
      PySys_Init();
    }
    
  /* Set the default PYTHONPATH. */
  if ((ppath = get_python_path (pcsp->r)))
    PySys_SetPath ((char *) ppath);

  /* Set the command line arguments list */
  if (!pcsp->r->args || !pcsp->r->args[0] || (ind (pcsp->r->args, '=') >= 0))
    /* The aren't command line arguments: just set the first one to the
       script's name */
    PySys_SetArgv (1, &pcsp->argv0);
  else
    {
      int argc;
      char ** argv = create_argv (pcsp->r, pcsp->argv0, pcsp->r->args, NULL);

      for (argc=0; argv[argc]; argc++)
        /* do nothing */;
      
      PySys_SetArgv (argc, argv);
    }
  
  /* Reinitialize os.environ. */
  os_module = PyImport_ImportModule ("os");
  if (!os_module)
    {
      log_reason ("Cannot import ``os'' module", pcsp->r->filename, pcsp->r);
      PyErr_Print();
      return SERVER_ERROR;
    }
  else
    {
      int i;
      PyObject *module_dict;
      PyObject *environ_dict;
      array_header *env_arr = table_elts (pcsp->r->subprocess_env);
      table_entry *elts = (table_entry *) env_arr->elts;
      char *tz = getenv ("TZ");

      module_dict = PyModule_GetDict (os_module);
      
      /* extract the current environ dictionary from the OS module,
         make it empty and then fill it with our variables; in this
         way we keep `os.environ' and `posix.environ' in sync: since
         they actually are the same object, inserting a new dict in
         one of the two modules would break the link. */
      
      environ_dict = PyDict_GetItemString (module_dict, "environ");

      /* Python 1.4' os.py replaces, where possible, the environ
         dictionary member with a subclass of UserDict that calls
         putenv() in its __setitem__ method, to keep the real
         environment in sync. */
      if (! PyDict_Check (environ_dict))
        {
          PyObject *data = PyObject_GetAttrString (environ_dict, "data");

          if (data && PyDict_Check (data))
            PyDict_Clear (data);
          else
            {
              log_reason ("Unhandled environ type: %s",
                          environ_dict->ob_type->tp_name, pcsp->r);
              Py_DECREF (os_module);
              return SERVER_ERROR;
            }
        }
      else
        PyDict_Clear (environ_dict);
      
      if (tz != NULL)
        {
          PyObject *ptz = PyString_FromString (tz);

          PyMapping_SetItemString (environ_dict, "TZ", ptz);
          Py_DECREF (ptz);
        }

      i = env_arr->nelts;
      while (i--)
        {
          if (elts[i].key)
            {
              PyObject *value = PyString_FromString (elts[i].val);

              PyMapping_SetItemString (environ_dict, elts[i].key, value);
              Py_DECREF (value);
            }
        }
    }
  Py_DECREF (os_module);

  if (PythonPersistent && PythonTrustedScripts)
    {
      static PyObject *ApacheDict = NULL;

      /* Initialize the module once, then only update ``request'' member */
         
      if (! ApacheDict)
        {
          ApacheDict = initialize_apache_module (pcsp->r);

          if (!ApacheDict)
            {
              PyErr_Print();
              return SERVER_ERROR;
            }
        }
      else
        {
          PyObject *ar = ApacheRequest_new (pcsp->r);

          PyDict_SetItemString (ApacheDict, "request", ar);
          Py_DECREF (ar);
        }
    }
  else
    {
      if (!initialize_apache_module (pcsp->r))
        {
          PyErr_Print();
          return SERVER_ERROR;
        }
    }
      
  return OK;
}

/* This function gets executed in a child process. It initializes the
   Python environment, then executes the script. */
static void
python_child (void *child_stuff)
{
  python_child_stuff *pcsp = child_stuff;
  int status;

  if (pcsp->nph)
    client_to_stdout (pcsp->r->connection);
  
  if (initialize_python_environment (pcsp) != OK)
    Py_Exit (SERVER_ERROR);
  
  cleanup_for_exec();

  status = exec_the_script (pcsp);

  if (status == OK)
    Py_Exit (0);
  else
    Py_Exit (SERVER_ERROR);
}

/* Handles the script output, fetching the headers first, then forwarding it
   to the server. */
static int
copy_client_result (python_child_stuff *pcsp)
{
  int ret;

  if ((ret = scan_script_header_err (pcsp->r, pcsp->script_out, NULL)) == OK)
    {
      char *location = table_get (pcsp->r->headers_out, "Location");
      
      if (location && location[0] == '/' && pcsp->r->status == 200)
        {
          char buffer[HUGE_STRING_LEN];
          
          /* Soak up all the script output */
          hard_timeout ("read from script", pcsp->r);
          while (fgets (buffer, HUGE_STRING_LEN-1, pcsp->script_out) != NULL)
            /* do nothing */;
          while (fgets (buffer, HUGE_STRING_LEN-1, pcsp->script_out) != NULL)
            /* do nothing */;
          kill_timeout (pcsp->r);
                      
          /* This redirect needs to be a GET no matter what the original
           * method was.
           */
          pcsp->r->method = pstrdup (pcsp->r->pool, "GET");
          pcsp->r->method_number = M_GET;
              
          internal_redirect_handler (location, pcsp->r);
          ret = OK;
        }
      else if (location && pcsp->r->status == 200)
        {
          /* XX Note that if a script wants to produce its own Redirect
           * body, it now has to explicitly *say* "Status: 302"
           */
          ret = REDIRECT;
        }
      else
        {
          char buffer[HUGE_STRING_LEN];

          hard_timeout ("send script output", pcsp->r);
          
          send_http_header (pcsp->r);
          if (!pcsp->r->header_only)
            send_fd (pcsp->script_out, pcsp->r);

          /* Soak up stderr */
          while (fgets (buffer, HUGE_STRING_LEN-1, pcsp->script_err) != NULL)
            /* do nothing */;
          
          kill_timeout (pcsp->r);

          pfclose (pcsp->r->connection->pool, pcsp->script_out);
          pfclose (pcsp->r->connection->pool, pcsp->script_err);

          ret = OK;
        }
    }

  return ret;
}

/* Forks a new process and make it execute the python_child() function,
   then copies, if any, the arguments to the child's stdin, and at end
   read back script's output and copy it to the client. */
static int
child_execute_python_script (python_child_stuff *pcsp)
{
  /* Create a child process: if the script claims to be NPH, then do not
     use a timeout, just wait for it to finish. In this case do not even
     pass a pointer to script_out, since we will not be interested in it:
     the script's output is connected directly to the client. */
  
  if (! (pcsp->pid = spawn_child_err (pcsp->r->connection->pool,
                                      python_child,
                                      (void *) pcsp,
                                      pcsp->nph ? just_wait : kill_after_timeout,
                                      &pcsp->script_in,
                                      pcsp->nph ? NULL : &pcsp->script_out,
                                      &pcsp->script_err)))
    {
      log_reason ("couldn't spawn child process", pcsp->r->filename, pcsp->r);
      return SERVER_ERROR;
    }

  /* If there are args coming from the client, forward them to the
     script' stdin */
  if (should_client_block (pcsp->r))
    {
      void (*handler)();
      int len_read;
      char argsbuffer[HUGE_STRING_LEN];
      
      hard_timeout ("copy script args", pcsp->r);
      handler = signal (SIGPIPE, SIG_IGN);
    
      while (((len_read =
               get_client_block (pcsp->r, argsbuffer, HUGE_STRING_LEN)) > 0))
        {
          if (fwrite (argsbuffer, 1, len_read, pcsp->script_in) < (size_t) len_read)
            {
              /* silly script stopped reading, soak up remaining message */
              while (get_client_block (pcsp->r, argsbuffer, HUGE_STRING_LEN) > 0)
                /* do nothing */;
              break;
            }
        }
    
      fflush (pcsp->script_in);
      signal (SIGPIPE, handler);
      
      kill_timeout (pcsp->r);
    }

  pfclose (pcsp->r->connection->pool, pcsp->script_in);

  /* If the script is NPH our job finishes here; otherwise parse the
     incoming headers and copy the result to the client. */
  if (pcsp->nph)
    {
      waitpid (pcsp->pid, (int *) 0, 0);
      return OK;
    }
  else
    return copy_client_result (pcsp);
}

/* Execute the script in the same process space of the server.
   Hook an ApacheBuff instance to the script's stdin & stdout: in this
   way the script will read incoming data directly from the server, and
   will write its output directly to the client.
   Once done, cleanup the interpreter (well, almost...). */
static int
persistent_execute_python_script (python_child_stuff *pcsp)
{
  static PyObject *sysin_name = NULL, *sysout_name, *syserr_name;
  PyObject *sys_module, *sys_dict;
  PyObject *bltin_module;
  int status;

  if (initialize_python_environment (pcsp) != OK)
    return SERVER_ERROR;

  /* Setup Input/Output channels */
  
  if (!(sys_module = PyImport_AddModule ("sys")))
    {
      log_reason ("Cannot get ``sys'' module", pcsp->r->filename, pcsp->r);
      return SERVER_ERROR;
    }
  else
    {
      static PyObject *ApacheBuff_new (request_rec *r, int nph);
      PyObject *err, *buff = ApacheBuff_new (pcsp->r, pcsp->nph);

      /* Hook up the ApacheBuff to both stdin and stdout file objects of
         the script. In this way the script can comunicate directly with
         our client, without other intervention by us. */

      sys_dict = PyModule_GetDict (sys_module);

      if (! sysin_name)
        {
          sysin_name = PyString_FromString ("stdin");
          sysout_name = PyString_FromString ("stdout");
        }

      /* If PythonTrustedScripts is On, these sets will cause the release
         of the preceeding ApacheBuff object. */
      
      PyDict_SetItem (sys_dict, sysin_name, buff);
      PyDict_SetItem (sys_dict, sysout_name, buff);
      
      Py_DECREF (buff);

      /* Hook up stderr to server's error log */
      
      if (PythonTrustedScripts)
        {
          /* XXX Do this just once.
             This assumes that server->error_log does never change. */
          
          if (! syserr_name)
            {
              syserr_name = PyString_FromString ("stderr");

              err = PyFile_FromFile (pcsp->r->server->error_log, "<stderr>", "w", NULL);
              PyDict_SetItem (sys_dict, syserr_name, err);
      
              Py_DECREF (err);
            }
        }
      else
        {
          if (! syserr_name)
            syserr_name = PyString_FromString ("stderr");

          err = PyFile_FromFile (pcsp->r->server->error_log, "<stderr>", "w", fflush);
          PyDict_SetItem (sys_dict, syserr_name, err);
      
          Py_DECREF (err);
        }
    }

  status = exec_the_script (pcsp);

  /* Cleanup Python environment */
  
  if (PythonTrustedScripts)
    {
      PyObject *import_dict = PyImport_GetModuleDict();

      /* All we can do, without leaking memory, is remove __main__ module. */
      
      if (import_dict == NULL ||
          PyDict_DelItemString (import_dict, "__main__") != 0)
        {
          log_reason ("Cannot remove module ``__main__''", pcsp->r->filename, pcsp->r);

          /* XXX Is this right? */
          Py_Exit (SERVER_ERROR);
        }
      
      /* Flush stderr */

      if (PyObject_CallMethod (PyDict_GetItem (sys_dict, syserr_name), "flush", NULL) == NULL)
        log_reason ("Cannot flush script's stderr", pcsp->r->filename, pcsp->r);      
    }
  else
    {
      /* XXX __builtin__ and sys modules keep a static pointer to their
         dictionary, that is never freed. Those pointers gets initialized
         in the init function for the module, simply overriding their
         previous value. So it is safe, at least with the current
         implementation of Python, to get the modules's dictionary through
         PyImport_GetDict(), that returns a borrowed reference, and DECREF
         it: at cleanup time, these dictionaries will be released. */
      
      if (!(bltin_module = PyImport_AddModule ("__builtin__")))
        {
          log_reason ("Cannot get ``__builtin__'' module", pcsp->r->filename, pcsp->r);
          Py_Exit (SERVER_ERROR);
        }
      else
        {
          PyObject *bltin_dict = PyModule_GetDict (bltin_module);
          
          /* Force the __builtin__ dict to be released by PyImport_Cleanup()
             below. */
          Py_XDECREF (bltin_dict);
        }
      
      /* Force the sys dict to be released by PyImport_Cleanup() below. */
      Py_DECREF (sys_dict);
      
      PyImport_Cleanup();
    }
  
  if (status == OK)
    return OK;
  else
    return SERVER_ERROR;
}

/* This is the entry point: this function is called by the server to
   handle our preferite script type. */
static int
python_cgi_handler (request_rec *r)
{
  int ret;
  python_child_stuff pcs;

  pcs.r = r;
  /* These other members of pcs get initialized by check():
     pcs.argv0, pcs.nph */

  if ((ret = check (&pcs)) != OK)
    return ret;
  
  chdir_file (r->filename);
  
  add_common_vars (r);
  add_cgi_vars (r);

  if (PythonPersistent)
    ret = persistent_execute_python_script (&pcs);
  else
    ret = child_execute_python_script (&pcs);

  return ret;
}

static handler_rec python_handlers[] = {
  { PYTHON_MAGIC_TYPE, python_cgi_handler },
  { ALT_PYTHON_MAGIC_TYPE, python_cgi_handler },
  { NULL }
};

module pyapache_module = {
   STANDARD_MODULE_STUFF,
   NULL,                        /* initializer */
   create_python_dir_config,    /* dir config creater */
   NULL,                        /* dir merger --- default is to override */
   NULL,                        /* server config */
   NULL,                        /* merge server config */
   python_cmds,                 /* command table */
   python_handlers,             /* handlers */
   NULL,                        /* filename translation */
   NULL,                        /* check_user_id */
   NULL,                        /* check auth */
   NULL,                        /* check access */
   NULL,                        /* type_checker */
   NULL,                        /* fixups */
   NULL                         /* logger */
};

#if 0 /* This is not needed anymore. I use the pyrl module now!.
         Available on ftp.python.org, it encapsulate the GNU readline
         facility in an external module. Since somebody may be missing
         it, I left this code here. */

/* My Python Interpreter is configured to use GNU readline() for reading
   interactive inputs. Since I do not want to link in the entire readline
   library (and the termcap library too) in the Apache Server (where it is
   not used at all) I patched Python's Makefiles to not include `myreadline.o'
   in the `libParser.a' library. So now I have to furnish a simple-minded
   implementation of this function. */

char *
PyOS_Readline (char *prompt)
{
  int n = 1000;
  char *p = malloc (n);
  char *q;

  if (p == NULL)
    return NULL;

  if (prompt)
    fprintf (stderr, "%s", prompt);
  
  q = fgets (p, n, stdin);
  if (q == NULL)
    {
      *p = '\0';
      return realloc (p, 1);
    }
  n = strlen (p);
  if (n > 0 && p[n-1] != '\n')
    p[n-1] = '\n';
  return realloc(p, n+1);
}

#endif

/**********************************************************************
                    Interface with the Apache API
***********************************************************************/

/* This would be an useful extension of the structmember module: the notion
   of a computed member, ie the possibility of specifying a function returning
   the requested member as a Python object. */

typedef PyObject *(*ComputingFunction) Py_FPROTO((void *addr, char *name));

typedef struct computed_memberlist
{
  const char *name;
  ComputingFunction get;
} computed_memberlist;

static PyObject *
list_append_computed_members (PyObject *list, computed_memberlist *cml)
{
  while (cml && cml->name)
    {
      PyObject *s = PyString_FromString ((char *) cml->name);
      
      PyList_Append (list, s);
      Py_DECREF (s);
      cml++;
    }
  return list;
}

static PyObject *
computed_member_get (void *addr, computed_memberlist *cml, char *name)
{
  while (cml->name)
    {
      if (!strcmp (cml->name, name))
        return (*(cml->get)) (addr, name);
      cml++;
    }
  PyErr_SetString (PyExc_AttributeError, name);
  return NULL;
}

/* Our own exception object */
static PyObject *ApacheException;


/**********************************************************************
                         ApacheBuff Object

  This is probably the smarter (?) object in this module: it is an
  approximation of a Python fileobject, in the sense that it
  implements just the very basic methods of such an object: read,
  write and a do-nothing flush; they are enough to hook an instance of
  it in place of the standard fileobject sys.stdin and sys.stdout.

  The read method will return a string composed of bytes coming from
  the client, thus giving the script access to the arguments of a M_POST
  or a M_PUT request.

  The write method needs additional machinery: until it does not find
  the end-of-headers condition it will collect them, updating the
  internal tables of the server, like scan_script_headers() does. To
  complicate things, we have to parse headers `incrementally', since
  we aren't getting them from a stream, but directly as the script
  writes them on its output.  Once the headers come at end, the
  remaining stuff is sent to the client, as usual.
  This step does not happen for NPH scripts: the output of the script
  is sent directly to the client.

  The flush method is there for compatibility: it lets you use your `old'
  scripts, where you used to add a "sys.stdout.flush()" to sync the
  script output.
***********************************************************************/

typedef struct
{
  PyObject_HEAD

  /* The REQUEST_REC. We use it to access to Pool functions. */
  request_rec *r;

  /* FALSE until we parsed and validated all the headers coming from
     the script, TRUE after that.  TRUE all the time if NPH script. */
  int headers_sent;

  /* TRUE if we found a malformed header. */
  int malformed_header;

  /* If not NULL, contains yet-to-be-parsed headers, that is headers
     coming from previous calls to parse_headers() function that it
     couldn't parse because they weren't complete (eoln was missing). */
  char *headers;
} ApacheBuff;

staticforward PyTypeObject ApacheBuffType;

static char ApacheBuff_read_doc[] = "";

static PyObject *
ApacheBuff_read (ApacheBuff *self, PyObject *args)
{
  PyObject *v;
  int toread;
  int size;
  int got;
  
  if (args == NULL)
    toread = -1;                              /* until EOF */
  else
    {
      int s;

      if (!PyArg_ParseTuple (args, "i;SIZE", &s))
        return NULL;

      toread = s;
    }

  size = (toread >= 0 ? toread : BUFSIZ);
  v = PyString_FromStringAndSize (NULL, size);
  if (v == NULL)
    return NULL;

  got = 0;
  Py_BEGIN_ALLOW_THREADS
  while (1)
    {
      int gotnow = bread (self->r->connection->client,
                          PyString_AS_STRING((PyStringObject *) v)+got,
                          size-got);

      if (gotnow == 0)                        /* EOF */
        break;
      else if (gotnow == -1)                  /* ERROR */
        {
          PyErr_SetString (ApacheException, "Error reading from the client");
          Py_DECREF (v);
          return NULL;
        }

      got += gotnow;

      if (toread == -1)
        {
          size = got + BUFSIZ;
          Py_BLOCK_THREADS
            if (_PyString_Resize (&v, size) < 0)
              return NULL;
          Py_UNBLOCK_THREADS
        }
    }
  Py_END_ALLOW_THREADS
      
  if (got != toread)
    _PyString_Resize (&v, got);

  return v;
}

static void
log_malformed (const char *m, int lentoshow, ApacheBuff *self)
{
#define MALFORMED_MESSAGE "malformed header from script. Bad header="
#define MALFORMED_HEADER_LENGTH_TO_SHOW 30

  char malformed[sizeof (MALFORMED_MESSAGE)+1+MALFORMED_HEADER_LENGTH_TO_SHOW];

  if (lentoshow > MALFORMED_HEADER_LENGTH_TO_SHOW)
    lentoshow = MALFORMED_HEADER_LENGTH_TO_SHOW;
              
  strcpy (malformed, MALFORMED_MESSAGE);
  strncpy (malformed+sizeof (MALFORMED_MESSAGE)-1, m, MALFORMED_HEADER_LENGTH_TO_SHOW);
  malformed[sizeof (MALFORMED_MESSAGE)+1+MALFORMED_HEADER_LENGTH_TO_SHOW] = '\0';

  log_reason (malformed, self->r->filename, self->r);
}

/* This function gets called from the ApacheBuff_write method until
   the end-of-headers (two consecutive new line characters) condition is met.

   Splitting the input STR into '\n' separated lines, recognises the
   header and updates the various tables, like scan_script_headers() does.

   Then returns the number of characters consumed, or -1 on errors. */
static int
parse_headers (ApacheBuff *self, const char *str, int size)
{
  char *lbeg, *lend, *buff;
  int consumed;

  if (self->headers)
    {
      int len = strlen (self->headers);
      
      buff = alloca (len+size);
      strncpy (buff, self->headers, len);
      strncpy (buff+len, str, size);
      buff[len+size] = '\0';
    }
  else
    {
      buff = alloca (size);

      strncpy (buff, str, size);
      buff[size] = '\0';
    }
  
  lbeg = buff;
  lend = strchr (buff, '\n');

  while (lend)
    {
      int llen = lend - lbeg;

      /* Ok, we found a eoln, but isn't it a MessyDos CR-NL pair instead? */
      if (llen)
        {
          if (llen > 1 && *(lend-1) == '\015')
            {
              llen -= 2;
              *(lend-1) = '\0';
            }
          else
            {
              llen--;
              *lend = '\0';
            }
        }
      
      if (llen == 0)
        {
          /* Yeah, no more headers! Now, first of all send them
             back, then finish writing this string to the client. */

          send_http_header (self->r);
          self->headers_sent++;
          
          break;
        }
      else
        {
          /* Handle the header. Keep this up-to-date with Apache's
             scan_script_header(). */
          char *colon;
          
          for (colon = lbeg; colon < lend && *colon != ':'; colon++)
            /* do nothing */;
          
          if (*colon != ':')
            {
              log_malformed (lbeg, lend-lbeg, self);
              self->malformed_header++;
              
              return -1;
            }
          else
            {
              char *arg;
              
              *colon = '\0';
              arg = colon+1;
              while (arg < lend && *arg && isspace (*arg))
                arg++;
              
              if (! strcasecmp (lbeg, "Content-type"))
                {
                  /* Nuke trailing whitespace */
                  
                  char *argendp = lend - 1;
                  
                  while (argendp > arg && isspace (*argendp))
                    *argendp-- = '\0';
                  
                  self->r->content_type = pstrdup (self->r->pool, arg);
                }
              else if (! strcasecmp (lbeg, "Status"))
                {
                  sscanf (arg, "%d", &self->r->status);
                  self->r->status_line = pstrdup (self->r->pool, arg);
                }
              else if (! strcasecmp (lbeg, "Location"))
                {
                  table_set (self->r->headers_out, lbeg, arg);
                }   
              else
                /* The HTTP specification says that it is legal to
                 * merge duplicate headers into one.  Some browsers
                 * that support Cookies don't like merged headers and
                 * prefer that each Set-Cookie header is sent
                 * separately.  Lets humour those browsers.  */
                if (! strcasecmp(lbeg, "Set-Cookie"))
                  {
                    table_add (self->r->err_headers_out, lbeg, arg);
                  }
                else
                  {
                    table_merge (self->r->err_headers_out, lbeg, arg);
                  }
            }
        }
      
      lbeg = lend+1;
      lend = strchr (lbeg, '\n');
    }

  if (lend == NULL)
    {
      /* We can't complete the operation since the (last) header is not
         terminated by an eoln character. So record the tail of the string
         for the next call. */
      
      if (*lbeg)
        self->headers = pstrdup (self->r->pool, lbeg);
      else
        self->headers = NULL;

      return size;
    }
  else
    self->headers = NULL;
  
  consumed = lbeg - buff;
  if (self->malformed_header ||
      (consumed != size && !self->headers_sent))
    {
      log_malformed (buff, consumed, self);
      self->malformed_header++;

      return -1;
    }
  else
    return consumed + 1;        /* for the ending eoln */
}

static char ApacheBuff_write_doc[] = "";

static PyObject *
ApacheBuff_write (ApacheBuff *self, PyObject *args)
{
  char *str;
  int size;

  if (! self->malformed_header)
    {
      if (PyArg_ParseTuple (args, "s#;STRING", &str, &size))
        {
          int put;

          hard_timeout ("send script output", self->r);

          /* Did we parse and send the headers? Or maybe we belong to an
             NPH script? */
          if (!self->headers_sent)
            {
              int consumed = parse_headers (self, str, size);

              if (consumed < 0)
                {
                  kill_timeout (self->r);
                  
                  PyErr_SetString (ApacheException, "Script generated malformed headers");
                  return NULL;
                }

              size -= consumed;
              str += consumed;
            }

          if (size)
            {
              Py_BEGIN_ALLOW_THREADS
              put = bwrite (self->r->connection->client, str, size);
              Py_END_ALLOW_THREADS

              if (put != size)
                {
                  kill_timeout (self->r);
                  
                  PyErr_SetString (ApacheException, "Couldn't send output to client");
                  return NULL;
                }
            }

          kill_timeout (self->r);

          if (self->headers_sent)
            bgetopt (self->r->connection->client, BO_BYTECT, &self->r->bytes_sent);

          Py_INCREF(Py_None);
          return Py_None;
        }
      else
        return NULL;
    }
  else
    {
      PyErr_SetString (ApacheException, "Script generated malformed headers");
      return NULL;
    }
}

/* This was suggested by Lance Ellinghaus <lance@deserTelcom.com>: it's a
   no-op method, just to let you say ``sys.stdout.flush()'' in the scripts. */

static char ApacheBuff_flush_doc[] = "";

static PyObject *
ApacheBuff_flush (ApacheBuff *self, PyObject *args)
{
  if (! PyArg_NoArgs (args))
    {
      Py_INCREF(Py_None);
      return Py_None;
    }

  PyErr_SetString (PyExc_TypeError, "Too many arguments specified");
  return NULL;
}

static PyMethodDef ApacheBuff_methods[] =
{
  { "read",     (PyCFunction) ApacheBuff_read,  METH_VARARGS,   ApacheBuff_read_doc },
  { "write",    (PyCFunction) ApacheBuff_write, METH_VARARGS,   ApacheBuff_write_doc },
  { "flush",    (PyCFunction) ApacheBuff_flush, METH_VARARGS,   ApacheBuff_flush_doc },
  { 0, 0, 0, 0 }
};

static PyObject *
ApacheBuff_new (request_rec *r, int nph)
{
  ApacheBuff *self = PyObject_NEW (ApacheBuff, &ApacheBuffType);

  if (self == NULL)
    return NULL;

  self->r = r;

  self->malformed_header = 0;

  /* If we belong to an NPH script consider done the headers
     parse/validation step. */
  self->headers_sent = nph;

  self->headers = NULL;
  
  return (PyObject *) self;
}

static void
ApacheBuff_dealloc (ApacheBuff *self)
{
#if RELEASE_TEST
  fprintf (stderr, "%s: destroying instance at %lx\n", __FUNCTION__, self);
#endif
  PyMem_DEL (self);
}

static char ApacheBuff_doc[] = "\
ApacheBuff is a Python wrapper around Apache's `BUFF' object.\
";

static PyObject *
ApacheBuff_getattr (ApacheBuff *self, char *name)
{
  if (! strcmp (name, "__doc__"))
    return PyString_FromString (ApacheBuff_doc);
  
  return Py_FindMethod (ApacheBuff_methods, (PyObject *) self, name);
}

static PyTypeObject ApacheBuffType =
{
  PyObject_HEAD_INIT(&PyType_Type)
  0,                                          /*ob_size*/
  "ApacheBuff",                               /*tp_name*/
  sizeof(ApacheBuff),                         /*tp_basicsize*/
  0,                                          /*tp_itemsize*/
  
  /* methods */
  (destructor) ApacheBuff_dealloc,            /*tp_dealloc*/
  (printfunc) 0,                              /*tp_print*/
  (getattrfunc) ApacheBuff_getattr,           /*tp_getattr*/
  (setattrfunc) 0,                            /*tp_setattr*/
  (cmpfunc) 0,                                /*tp_compare*/
  (reprfunc) 0,                               /*tp_repr*/
  0,                                          /*tp_as_number*/
  0,                                          /*tp_as_sequence*/
  0,                                          /*tp_as_mapping*/
  (hashfunc) 0,                               /*tp_hash*/
  (ternaryfunc) 0,                            /*tp_call*/
  (reprfunc) 0,                               /*tp_str*/
  (getattrofunc) 0,                           /*tp_getattro*/
  (setattrofunc) 0,                           /*tp_setattro*/
 
  /* Space for future expansion */
  0L,0L,
  
  ApacheBuff_doc                              /* Documentation string */
};


/**********************************************************************
                         ApacheTable Object
***********************************************************************/

typedef struct
{
  PyObject_HEAD

  table *t;
} ApacheTable;

staticforward PyTypeObject ApacheTableType;

static char ApacheTable_has_key_doc[] = "\
Returns 1 if the specified KEY, that must be a string, is contained in\n\
the table, 0 otherwise.";

static PyObject *
ApacheTable_has_key (ApacheTable *self, PyObject *args)
{
  char *key;

  if (PyArg_ParseTuple (args, "s;KEY", &key))
    {
      char *value = table_get (self->t, key);

      if (value == NULL)
        {
          Py_INCREF(Py_False);
          return Py_False;
        }
      else
        {
          Py_INCREF(Py_True);
          return Py_True;
        }
    }

  return NULL;
}

static char ApacheTable_keys_doc[] = "\
Returns a list of the keys contained in the table.\
";

static PyObject *
ApacheTable_keys (ApacheTable *self, PyObject *args)
{
  int i = self->t->nelts;
  PyObject *list = PyList_New (i);
  table_entry *elts = (table_entry *) self->t->elts;
  
  while (i--)
    {
      PyObject *key = PyString_FromString (elts[i].key);

      PyList_SetItem (list, i, key);
    }

  return list;
}

static char ApacheTable_values_doc[] = "\
Returns a list of the values contained in the table.\
";

static PyObject *
ApacheTable_values (ApacheTable *self, PyObject *args)
{
  int i = self->t->nelts;
  PyObject *list = PyList_New (i);
  table_entry *elts = (table_entry *) self->t->elts;
  
  while (i--)
    {
      PyObject *val = PyString_FromString (elts[i].val);

      PyList_SetItem (list, i, val);
    }

  return list;
}

static char ApacheTable_items_doc[] = "\
Returns a list of tuples, each representing a (key,value) pair.\
";

static PyObject *
ApacheTable_items (ApacheTable *self, PyObject *args)
{
  int i = self->t->nelts;
  PyObject *list = PyList_New (i);
  table_entry *elts = (table_entry *) self->t->elts;
  
  while (i--)
    {
      PyObject *tuple = PyTuple_New (2);
      PyObject *key = PyString_FromString (elts[i].key);
      PyObject *val = PyString_FromString (elts[i].val);
      
      PyTuple_SetItem (tuple, 0, key);
      PyTuple_SetItem (tuple, 1, val);
      PyList_SetItem (list, i, tuple);
    }

  return list;
}

static PyMethodDef ApacheTable_methods[] =
{
  { "has_key",  (PyCFunction) ApacheTable_has_key,      METH_VARARGS,   ApacheTable_has_key_doc },
  { "keys",     (PyCFunction) ApacheTable_keys,         METH_VARARGS,   ApacheTable_keys_doc },
  { "values",   (PyCFunction) ApacheTable_values,       METH_VARARGS,   ApacheTable_values_doc },
  { "items",    (PyCFunction) ApacheTable_items,        METH_VARARGS,   ApacheTable_items_doc },
  { 0, 0, 0, 0 }
};

static PyObject *
ApacheTable_new (table *t)
{
  ApacheTable *self = PyObject_NEW (ApacheTable, &ApacheTableType);

  if (self == NULL)
    return NULL;

  self->t = t;
  return (PyObject *) self;
}

static void
ApacheTable_dealloc (ApacheTable *self)
{
#if RELEASE_TEST
  fprintf (stderr, "%s: destroying instance at %lx\n", __FUNCTION__, self);
#endif
  PyMem_DEL (self);
}

static char ApacheTable_doc[] = "\
ApacheTable is Python wrapper around Apache's `table' object.\n\
It behaves like a standard Python dictionary.\
";

static PyObject *
ApacheTable_getattr (ApacheTable *self, char *name)
{
  if (! strcmp (name, "__doc__"))
    return PyString_FromString (ApacheTable_doc);
  
  return Py_FindMethod (ApacheTable_methods, (PyObject *) self, name);
}

static int
ApacheTable_length (ApacheTable *self)
{
  return self->t->nelts;
}

static PyObject *
ApacheTable_subscript (ApacheTable *self, PyObject *key)
{
  if (PyString_Check (key))
    {
      char *str = PyString_AS_STRING((PyStringObject *) key);
      char *value = table_get (self->t, str);

      if (value == NULL)
        {
          PyErr_SetObject (PyExc_KeyError, key);
          return NULL;
        }
      else
        return PyString_FromString (value);
    }
  else
    {
      PyErr_SetObject (PyExc_TypeError, key);
      return NULL;
    }
}

static int
ApacheTable_ass_sub (ApacheTable *self, PyObject *key, PyObject *value)
{
  return 0;
}

static PyMappingMethods ApacheTable_as_mapping =
{
  (inquiry) ApacheTable_length,               /*mp_length*/
  (binaryfunc) ApacheTable_subscript,         /*mp_subscript*/
  (objobjargproc) ApacheTable_ass_sub         /*mp_ass_subscript*/
};

static PyTypeObject ApacheTableType =
{
  PyObject_HEAD_INIT(&PyType_Type)
  0,                                          /*ob_size*/
  "ApacheTable",                              /*tp_name*/
  sizeof(ApacheTable),                        /*tp_basicsize*/
  0,                                          /*tp_itemsize*/
  
  /* methods */
  (destructor) ApacheTable_dealloc,           /*tp_dealloc*/
  (printfunc) 0,                              /*tp_print*/
  (getattrfunc) ApacheTable_getattr,          /*tp_getattr*/
  (setattrfunc) 0,                            /*tp_setattr*/
  (cmpfunc) 0,                                /*tp_compare*/
  (reprfunc) 0,                               /*tp_repr*/
  0,                                          /*tp_as_number*/
  0,                                          /*tp_as_sequence*/
  &ApacheTable_as_mapping,                    /*tp_as_mapping*/
  (hashfunc) 0,                               /*tp_hash*/
  (ternaryfunc) 0,                            /*tp_call*/
  (reprfunc) 0,                               /*tp_str*/
  (getattrofunc) 0,                           /*tp_getattro*/
  (setattrofunc) 0,                           /*tp_setattro*/
 
  /* Space for future expansion */
  0L,0L,
  
  ApacheTable_doc                             /* Documentation string */
};


/**********************************************************************
                         ApacheServer Object
***********************************************************************/

typedef struct
{
  PyObject_HEAD

  server_rec *server;
} ApacheServer;

staticforward PyTypeObject ApacheServerType;

static PyMethodDef ApacheServer_methods[] =
{
  { 0, 0, 0, 0 }
};

#define AS_OFF(f) offsetof(server_rec, f)

static struct memberlist ApacheServer_memberlist[] =
{
  { "srm_confname",     T_STRING,       AS_OFF(srm_confname),   READONLY },
  { "access_confname",  T_STRING,       AS_OFF(access_confname),READONLY },
  { "server_admin",     T_STRING,       AS_OFF(server_admin),   READONLY },
  { "server_hostname",  T_STRING,       AS_OFF(server_hostname),READONLY },
  { "port",             T_SHORT,        AS_OFF(port),           READONLY },
  { 0, 0, 0, 0 }
};

static PyObject *
ApacheServer_new (server_rec *s)
{
  ApacheServer *self = PyObject_NEW (ApacheServer, &ApacheServerType);

  if (self == NULL)
    return NULL;

  self->server = s;
  return (PyObject *) self;
}

static void
ApacheServer_dealloc (ApacheServer *self)
{
#if RELEASE_TEST
  fprintf (stderr, "%s: destroying instance at %lx\n", __FUNCTION__, self);
#endif
  PyMem_DEL (self);
}

static char ApacheServer_doc[] = "\
ApacheServer is a Python wrapper around Apache's `server' object.\
";

static PyObject *
ApacheServer_getattr (ApacheServer *self, char *name)
{
  PyObject *rv;

  if (! strcmp (name, "__doc__"))
    return PyString_FromString (ApacheServer_doc);
  
  rv = PyMember_Get ((char *) self->server, ApacheServer_memberlist, name);
  if (rv)
    return rv;
  
  PyErr_Clear();
  return Py_FindMethod (ApacheServer_methods, (PyObject *) self, name);
}

static PyTypeObject ApacheServerType =
{
  PyObject_HEAD_INIT(&PyType_Type)
  0,                                          /*ob_size*/
  "ApacheServer",                             /*tp_name*/
  sizeof(ApacheServer),                       /*tp_basicsize*/
  0,                                          /*tp_itemsize*/
  
  /* methods */
  (destructor) ApacheServer_dealloc,          /*tp_dealloc*/
  (printfunc) 0,                              /*tp_print*/
  (getattrfunc) ApacheServer_getattr,         /*tp_getattr*/
  (setattrfunc) 0,                            /*tp_setattr*/
  (cmpfunc) 0,                                /*tp_compare*/
  (reprfunc) 0,                               /*tp_repr*/
  0,                                          /*tp_as_number*/
  0,                                          /*tp_as_sequence*/
  0,                                          /*tp_as_mapping*/
  (hashfunc) 0,                               /*tp_hash*/
  (ternaryfunc) 0,                            /*tp_call*/
  (reprfunc) 0,                               /*tp_str*/
  (getattrofunc) 0,                           /*tp_getattro*/
  (setattrofunc) 0,                           /*tp_setattro*/
 
  /* Space for future expansion */
  0L,0L,
  
  ApacheServer_doc                            /* Documentation string */
};

/**********************************************************************
                       ApacheConnection Object
***********************************************************************/

typedef struct
{
  PyObject_HEAD

  conn_rec *connection;
} ApacheConnection;

staticforward PyTypeObject ApacheConnectionType;

static PyMethodDef ApacheConnection_methods[] =
{
  { 0, 0, 0, 0 }
};

#define AC_OFF(f) offsetof(conn_rec, f)

static struct memberlist ApacheConnection_memberlist[] =
{
  /* Client's IP address */
  { "remote_ip",        T_STRING,       AC_OFF(remote_ip),      READONLY },

  /* If an authentication check was made, this gets set to the user
     name.  We assume that there's only one user per connection(!)  */
  { "user",             T_STRING,       AC_OFF(user),           READONLY },
  { "auth_type",        T_STRING,       AC_OFF(auth_type),      READONLY },

  /* Are we using HTTP Keep-Alive? */
  { "keepalive",        T_INT,          AC_OFF(keepalive),      READONLY },
  
  /* Did we use HTTP Keep-Alive? */
  { "keptalive",        T_INT,          AC_OFF(keptalive),      READONLY },
  
  /* How many times have we used it? */
  { "keepalives",       T_INT,          AC_OFF(keepalives),     READONLY },
  { 0, 0, 0, 0 }
};

static PyObject *
ApacheConnection_get_server (void *addr, char *name)
{
  ApacheConnection *self = addr;

  return ApacheServer_new (self->connection->server);
}
  
static computed_memberlist ApacheConnection_compmemberlist[] =
{
  { "server", ApacheConnection_get_server },
  { 0, 0 }
};

static PyObject *
ApacheConnection_new (conn_rec *c)
{
  ApacheConnection *self = PyObject_NEW (ApacheConnection, &ApacheConnectionType);

  if (self == NULL)
    return NULL;

  self->connection = c;
  return (PyObject *) self;
}

static void
ApacheConnection_dealloc (ApacheConnection *self)
{
#if RELEASE_TEST
  fprintf (stderr, "%s: destroying instance at %lx\n", __FUNCTION__, self);
#endif
  PyMem_DEL (self);
}

static char ApacheConnection_doc[] = "\
ApacheConnection is a Python wrapper around the Apache's `connection' object.\
";

static PyObject *
ApacheConnection_getattr (ApacheConnection *self, char *name)
{
  PyObject *rv;

  if (! strcmp (name, "__doc__"))
    return PyString_FromString (ApacheConnection_doc);
  
  rv = PyMember_Get ((char *) self->connection, ApacheConnection_memberlist, name);
  if (rv)
    {
      if (!strcmp (name, "__members__"))
        list_append_computed_members (rv, ApacheConnection_compmemberlist);
      
      return rv;
    }

  PyErr_Clear();
  rv = computed_member_get (self, ApacheConnection_compmemberlist, name);
  if (rv)
    return rv;
  else
    {
      PyErr_Clear();
      return Py_FindMethod (ApacheConnection_methods, (PyObject *) self, name);
    }
}

static PyTypeObject ApacheConnectionType =
{
  PyObject_HEAD_INIT(&PyType_Type)
  0,                                          /*ob_size*/
  "ApacheConnection",                         /*tp_name*/
  sizeof(ApacheConnection),                   /*tp_basicsize*/
  0,                                          /*tp_itemsize*/
  
  /* methods */
  (destructor) ApacheConnection_dealloc,      /*tp_dealloc*/
  (printfunc) 0,                              /*tp_print*/
  (getattrfunc) ApacheConnection_getattr,     /*tp_getattr*/
  (setattrfunc) 0,                            /*tp_setattr*/
  (cmpfunc) 0,                                /*tp_compare*/
  (reprfunc) 0,                               /*tp_repr*/
  0,                                          /*tp_as_number*/
  0,                                          /*tp_as_sequence*/
  0,                                          /*tp_as_mapping*/
  (hashfunc) 0,                               /*tp_hash*/
  (ternaryfunc) 0,                            /*tp_call*/
  (reprfunc) 0,                               /*tp_str*/
  (getattrofunc) 0,                           /*tp_getattro*/
  (setattrofunc) 0,                           /*tp_setattro*/
 
  /* Space for future expansion */
  0L,0L,
  
  ApacheConnection_doc                        /* Documentation string */
};


/**********************************************************************
                         ApacheRequest Object
***********************************************************************/

typedef struct
{
  PyObject_HEAD

  request_rec *request;
} ApacheRequest;

staticforward PyTypeObject ApacheRequestType;

static char ApacheRequest_get_remote_host_doc[] = "\
Returns the remote host of the request.\
";

static PyObject *
ApacheRequest_get_remote_host (PyObject *self, PyObject *args)
{
  ApacheRequest *rself = (ApacheRequest *) self;
  
  if (! PyArg_NoArgs (args))
    {
      const char *host = get_remote_host (rself->request->connection, rself->request->per_dir_config, REMOTE_NAME);

      if (host)
        {
          PyObject *pyhost = PyString_FromString ((char *) host);
          
          return pyhost;
        }
      else
        PyErr_SetString (ApacheException, "Cannot access `remote_host'");
    }
  else
    PyErr_SetString (PyExc_TypeError, "Too many arguments specified");
  return NULL;
}

static char ApacheRequest_get_remote_logname_doc[] = "\
Returns the remote logname of the request.\
";

static PyObject *
ApacheRequest_get_remote_logname (PyObject *self, PyObject *args)
{
  ApacheRequest *rself = (ApacheRequest *) self;
  
  if (! PyArg_NoArgs (args))
    {
      const char *logname = get_remote_logname (rself->request);

      if (logname)
        {
          PyObject *pylogname = PyString_FromString ((char *) logname);

          return pylogname;
        }
      else
        PyErr_SetString (ApacheException, "Cannot access `remote_logname'");
    }
  else
    PyErr_SetString (PyExc_TypeError, "Too many arguments specified");
  return NULL;
}

static PyMethodDef ApacheRequest_methods[] =
{
  { "get_remote_host",          (PyCFunction) ApacheRequest_get_remote_host,    METH_VARARGS,   ApacheRequest_get_remote_host_doc },
  { "get_remote_logname",       (PyCFunction) ApacheRequest_get_remote_logname, METH_VARARGS,   ApacheRequest_get_remote_logname_doc },
  { 0, 0, 0, 0 }
};

#define AR_OFF(f) offsetof(request_rec, f)

static struct memberlist ApacheRequest_memberlist[] =
{
  /* First line of request, so we can log it */
  { "the_request",      T_STRING,       AR_OFF(the_request),    READONLY },

  /* HTTP/0.9, "simple" request */
  { "assbackwards",     T_INT,          AR_OFF(assbackwards),   READONLY },

  /* A proxy request */
  { "proxyreq",         T_INT,          AR_OFF(proxyreq),       READONLY },

  /* HEAD request, as opposed to GET */
  { "header_only",      T_INT,          AR_OFF(header_only),    READONLY },

  /* Protocol, as given to us, or HTTP/0.9 */
  { "protocol",         T_STRING,       AR_OFF(protocol),       READONLY },

  /* Host, as set by full URI or Host: */
  { "hostname",         T_STRING,       AR_OFF(hostname),       READONLY },

  /* Status line, if set by script */
  { "status_line",      T_STRING,       AR_OFF(status_line),    READONLY },

  /* GET, HEAD, POST, etc. */
  { "method",           T_STRING,       AR_OFF(method),         READONLY },

  { "content_type",     T_STRING,       AR_OFF(content_type),   READONLY },
  { "handler",          T_STRING,       AR_OFF(handler),        READONLY },
  { "content_encoding", T_STRING,       AR_OFF(content_encoding),       READONLY },
  { "content_language", T_STRING,       AR_OFF(content_language),       READONLY },

  /* complete URI for a proxy req, or URL path for a non-proxy req */
  { "uri",              T_STRING,       AR_OFF(uri),            READONLY },
  { "filename",         T_STRING,       AR_OFF(filename),       READONLY },
  { "path_info",        T_STRING,       AR_OFF(path_info),      READONLY },

  /* QUERY_ARGS, if any */
  { "args",             T_STRING,       AR_OFF(args),           READONLY },
  
  { 0, 0, 0, 0 }
};

static PyObject *
ApacheRequest_get_server (void *addr, char *name)
{
  ApacheRequest *self = addr;

  return ApacheServer_new (self->request->server);
}

static PyObject *
ApacheRequest_get_connection (void *addr, char *name)
{
  ApacheRequest *self = addr;

  return ApacheConnection_new (self->request->connection);
}

static PyObject *
ApacheRequest_get_headers_in (void *addr, char *name)
{
  ApacheRequest *self = addr;

  return ApacheTable_new (self->request->headers_in);
}

static PyObject *
ApacheRequest_get_headers_out (void *addr, char *name)
{
  ApacheRequest *self = addr;

  return ApacheTable_new (self->request->headers_out);
}

static PyObject *
ApacheRequest_get_err_headers_out (void *addr, char *name)
{
  ApacheRequest *self = addr;

  return ApacheTable_new (self->request->err_headers_out);
}

static PyObject *
ApacheRequest_get_subprocess_env (void *addr, char *name)
{
  ApacheRequest *self = addr;

  return ApacheTable_new (self->request->subprocess_env);
}

static PyObject *
ApacheRequest_get_notes (void *addr, char *name)
{
  ApacheRequest *self = addr;

  return ApacheTable_new (self->request->notes);
}

static computed_memberlist ApacheRequest_compmemberlist[] =
{
  { "server", ApacheRequest_get_server },
  { "connection", ApacheRequest_get_connection },
  { "headers_in", ApacheRequest_get_headers_in },
  { "headers_out", ApacheRequest_get_headers_out },
  { "err_headers_out", ApacheRequest_get_err_headers_out },
  { "subprocess_env", ApacheRequest_get_subprocess_env },
  { "notes", ApacheRequest_get_notes },
  { 0, 0 }
};

static PyObject *
ApacheRequest_new (request_rec *r)
{
  ApacheRequest *self = PyObject_NEW (ApacheRequest, &ApacheRequestType);

  if (self == NULL)
    return NULL;

  self->request = r;
  return (PyObject *) self;
}

static void
ApacheRequest_dealloc (ApacheRequest *self)
{
#if RELEASE_TEST
  fprintf (stderr, "%s: destroying instance at %lx\n", __FUNCTION__, self);
#endif
  PyMem_DEL (self);
}

static char ApacheRequest_doc[] = "\
ApacheRequest is a Python wrapper around the Apache's `request_rec' object.\
";

static PyObject *
ApacheRequest_getattr (ApacheRequest *self, char *name)
{
  PyObject *rv;

  if (!strcmp (name, "__doc__"))
    return PyString_FromString (ApacheRequest_doc);
  
  rv = PyMember_Get ((char *) self->request, ApacheRequest_memberlist, name);
  
  if (rv)
    {
      if (!strcmp (name, "__members__"))
        list_append_computed_members (rv, ApacheRequest_compmemberlist);
      
      return rv;
    }
  
  PyErr_Clear();
  rv = computed_member_get (self, ApacheRequest_compmemberlist, name);
  if (rv)
    return rv;
  else
    {
      PyErr_Clear();
      return Py_FindMethod (ApacheRequest_methods, (PyObject *) self, name);
    }
}

static PyTypeObject ApacheRequestType =
{
  PyObject_HEAD_INIT(&PyType_Type)
  0,                                          /*ob_size*/
  "ApacheRequest",                            /*tp_name*/
  sizeof(ApacheRequest),                      /*tp_basicsize*/
  0,                                          /*tp_itemsize*/
  
  /* methods */
  (destructor) ApacheRequest_dealloc,         /*tp_dealloc*/
  (printfunc) 0,                              /*tp_print*/
  (getattrfunc) ApacheRequest_getattr,        /*tp_getattr*/
  (setattrfunc) 0,                            /*tp_setattr*/
  (cmpfunc) 0,                                /*tp_compare*/
  (reprfunc) 0,                               /*tp_repr*/
  0,                                          /*tp_as_number*/
  0,                                          /*tp_as_sequence*/
  0,                                          /*tp_as_mapping*/
  (hashfunc) 0,                               /*tp_hash*/
  (ternaryfunc) 0,                            /*tp_call*/
  (reprfunc) 0,                               /*tp_str*/
  (getattrofunc) 0,                           /*tp_getattro*/
  (setattrofunc) 0,                           /*tp_setattro*/
 
  /* Space for future expansion */
  0L,0L,
  
  ApacheRequest_doc                           /* Documentation string */
};

/**********************************************************************
                            Apache Module
***********************************************************************/

static PyMethodDef Apache_methods[] =
{
  { 0, 0, 0, 0 }
};

static PyObject *
initialize_apache_module (request_rec *r)
{
  PyObject *d, *ar;
  PyObject *m = Py_InitModule4 ("Apache", Apache_methods, NULL, NULL, PYTHON_API_VERSION);

  d = PyModule_GetDict (m);

  if (d)
    {
      ApacheException = PyString_FromString ("apache.error");
      PyDict_SetItemString (d, "error", ApacheException);
      
      /* This should be safe: after all none of you will do a
         del Apache.error
         right? So a reference to this object will remain in the
         dictionary until the entire module will be released. */
      
      Py_DECREF (ApacheException);
      
      ar = ApacheRequest_new (r);
      PyDict_SetItemString (d, "request", ar);
      Py_DECREF (ar);
    }
  
  /* Check for errors */
  if (PyErr_Occurred())
    {
      PyErr_Print();
      Py_FatalError ("Can't initialize module Apache");
      return NULL;
    }

  return d;
}

/* mod_pyapache.c ends here */

/*
** Local Variables:
** change-log-default-name:"../ChangeLog.PyApache"
** End:
*/
