
#include <sys/types.h>
#include <stdio.h>
#include "Python.h"
#include "ExtensionClass.h"
#include "ONEiiop.h"
/*
Copyright

       Copyright, 1997 Magnet Interactive Communications, L.L.C. All
rights reserved.  Copyright in this software is owned exclusively by
Magnet Interactive Communications, L.L.C.

       Permission to use, copy and distribute this software is hereby 
granted, provided that the above copyright notice appear in all 
copies and that both that copyright notice and this permission 
notice appear. Note that any product, process or technology 
described in this software may be the subject of other Intellectual 
Property rights reserved by Magnet and are not licensed hereunder.

No Warranty

       THE SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
EITHER EXPRESS OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
NON-INFRINGEMENT.

       This software could include technical inaccuracies or typographical 
errors. Magnet Interactive Communications, L.L.C. may periodically make 
improvements and/or changes to the software, which will be incorporated 
in new editions of the software. Magnet Interactive Communications, L.L.C. 
may make improvements and/or changes in this software at any time without 
notice and may or may not choose to make any such updates available.

*/

/* 
   Python Extension Class interface to Netscape Enterprise WAI API 
 
   v0.1 : Dave Mitchell (davem@magnet.com) 9/30/97

   This module implements Netscape's Java WAI interface as documented at 
http://developer.netscape.com/library/documentation/enterprise/wai/javaapi.htm

   To compile this:

   1) edit Setup.in to reflect the proper locations for the various enterprise
libraries, include files, and the ExtensionClass.h header file.

   2) edit Makefile.pre.in "NSDIR" variable to point to the directory
where enterprise-3.0 is installed

   3) you must have ExtensionClass.so in your pythonpath to run the
module.
  
   4) type "make -f Makefile.pre.in boot"
 
   5) "make all"

   6) If all goes well, you'll now have new copy of python in the current
directory, with a builtin "_wai" module. Due to linking errors I was unable
to build a working shared waimodule object file.

   7) Enable WAI applications in your netsite (it's in the Programs menu
of the admin server - you'll need to have "osagent" running when you
enable WAI or use WAI services)

   To create a new web application service in Python, you have to do a 
couple of things:

   1) import wai - this is a python layer that provides error tracebacks

Note!: you must have ExtensionClass.so somewhere in your pythonpath!
   
   2) derive your own class from wai.WebApplicationService, and provide
a Run() method which takes as an argument a HTTPServiceRequest object. This
HSR object contains methods with which you can query and set request and
response headers, as well as getting cgi environment data, and of course,
outputting the response to the client.

   3) call the WAS object's RegisterService() method, with the argument
being the "host:port" of the enterprise 3.0 server you are registering 
with.

   

   4) Access your wai application via the url "/iiop/<servicename>" where
<servicename> was the argument given to the WebApplicationService constructor.

   5) happy trails! try "./python waihello.py waihello <server>:<port>"
and go to the url "/iiop/waihello" for an example. 

*/

#define UNLESS(E) if(!(E))

static PyObject* error;

staticforward PyExtensionClass WAStype;
staticforward PyExtensionClass HSRtype;

staticforward long runfunc(ServerSession_t obj);

typedef struct {
	PyObject_HEAD
	IIOPWebAppService_t was_ptr;
} WASobject;

typedef struct {
	PyObject_HEAD
	ServerSession_t ss_ptr;
} HSRobject;

static WASobject* WAS;  /* the WAS instance containing the RUN method */


static PyObject *
WAS__init__(WASobject *self, PyObject *args) {

  char *name;
  PyObject *run_method = NULL;

  UNLESS(PyArg_ParseTuple(args, "s", &name)) return NULL;

  WAS = self;

  self->was_ptr = 
	WAIcreateWebAppService(name, runfunc, 0, NULL);
  /* ignoring argc, argv for now */

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *WAS_ActivateWAS(WASobject *self, PyObject *args) {

/* not sure what this one does or when it's called.. */

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *WAS_getServiceInfo(WASobject *self, PyObject *args) {

/* 
you're supposed to provide your own in a subclass, 
but I don't know when it gets called either
*/

  PyErr_SetString(error, "you must override the getServiceInfo method");
  return NULL;
}

static PyObject *WAS_RegisterService(WASobject *self, PyObject *args) {

  char *host;
  WAIBool rv;

  UNLESS(PyArg_ParseTuple(args, "s", &host)) {
    PyErr_SetString(error, "must give the server host:port");
    return NULL;
  }

  rv = WAIregisterService(self->was_ptr, host);

  UNLESS (rv == WAI_TRUE) {
    PyErr_SetString(error, "error registering service");
    return NULL;
  }

  WAIimplIsReady();

  /* tell the server it's ready to use */

  return PyInt_FromLong(1);	/* 1 means it worked */
}

static PyObject *WAS_Run(WASobject *self, PyObject *args) {

/* you're supposed to provide your own in a subclass */

  PyErr_SetString(error, "you must override the Run method");
  return NULL;
}

static PyObject *
HSR_addResponseHeader(HSRobject *self, PyObject *args) {

  char *header, *value;
  WAIReturnType_t rv;

  UNLESS(PyArg_ParseTuple(args, "ss", &header, &value)) {
    PyErr_SetString(error, "must provide header and value to be added");
    return NULL;
  }

  rv = WAIaddResponseHeader(self->ss_ptr, header, value);

  UNLESS(rv == WAISPISuccess) {
    PyErr_SetString(error, "error adding response header");
    return NULL;
  }

  return PyInt_FromLong(1);		/* 1 means it worked */
}

static PyObject *
HSR_delResponseHeader(HSRobject *self, PyObject *args) {

  char *header;
  WAIReturnType_t rv;

  UNLESS(PyArg_ParseTuple(args, "s", &header)) {
    PyErr_SetString(error, "must provide header to be deleted");
    return NULL;
  }

  rv = WAIdelResponseHeader(self->ss_ptr, header);

  UNLESS(rv == WAISPISuccess) {
    PyErr_SetString(error, "error deleting response header");
    return NULL;
  }
	       
  return PyInt_FromLong(1);             /* 1 means it worked */
}

static PyObject *
HSR_getConfigParameter(HSRobject *self, PyObject *args) {
  
  char *name, *value;
  PyObject *par;
  WAIReturnType_t rv;

  UNLESS(PyArg_ParseTuple(args, "s", &name)) {
    PyErr_SetString(error, "must provide config parameter to get");
    return NULL;
  }

  rv = WAIgetConfigParameter(self->ss_ptr, name, &value);

  UNLESS(rv == WAISPISuccess) {
    PyErr_SetString(error, "error retrieving config parameter");
    return NULL;
  }

  par = PyString_FromString(value);
  WAIstringFree(value);

  return par;
}

static PyObject *
HSR_getCookie(HSRobject *self, PyObject *args) {

  PyObject *cookies;
  char *c_cookies;
  WAIReturnType_t rv;

  UNLESS(PyArg_ParseTuple(args, "")) return NULL; 

  rv = WAIgetCookie(self->ss_ptr, &c_cookies);

  UNLESS(rv == WAISPISuccess) {
    PyErr_SetString(error, "error getting cookies");
    return NULL;
  }

  cookies = PyString_FromString(c_cookies);
  WAIstringFree(c_cookies);

  return cookies;
}

static PyObject *
HSR_getRequestHeader(HSRobject *self, PyObject *args) {
  
  char *header, *value;
  PyObject *py_value;
  WAIReturnType_t rv;

  UNLESS(PyArg_ParseTuple(args, "s", &header)) {
    PyErr_SetString(error, "must provide header to retrieve");
    return NULL;
  }

  rv = WAIgetRequestHeader(self->ss_ptr, header, &value);

  UNLESS(rv == WAISPISuccess) {
    PyErr_SetString(error, "error getting header");
    return NULL;
  }

  py_value = PyString_FromString(value);
  WAIstringFree(value);

  return py_value;
}

static PyObject *
HSR_getRequestInfo(HSRobject *self, PyObject *args) {

  char *header, *value;
  PyObject *py_value;
  WAIReturnType_t rv;
	  
  UNLESS(PyArg_ParseTuple(args, "s", &header)) {
    PyErr_SetString(error, "must provide cgi var to retrieve");
    return NULL;
  }
			
  rv = WAIgetRequestInfo(self->ss_ptr, header, &value);
			    
  UNLESS(rv == WAISPISuccess) {
    PyErr_SetString(error, "error getting cgi var");
    return NULL;
  } 
					  
  py_value = PyString_FromString(value);
  WAIstringFree(value);

  return py_value;
}

static PyObject *
HSR_getResponseContentLength(HSRobject *self, PyObject *args) {

  unsigned long length;
  PyObject *py_value;
  WAIReturnType_t rv;
	     
  UNLESS(PyArg_ParseTuple(args, "")) return NULL;
				
  rv = WAIgetResponseContentLength(self->ss_ptr, &length);

  UNLESS(rv == WAISPISuccess) {
    PyErr_SetString(error, "error getting content length");
    return NULL;
  }

  py_value = PyInt_FromLong(length);

  return py_value;
}

static PyObject *
HSR_getResponseHeader(HSRobject *self, PyObject *args) {

  char *header, *value;
  PyObject *py_value;
  WAIReturnType_t rv;

  UNLESS(PyArg_ParseTuple(args, "s", &header)) {
    PyErr_SetString(error, "must provide header to retrieve");
    return NULL;
  }

  rv = WAIgetResponseHeader(self->ss_ptr, header, &value);

  UNLESS(rv == WAISPISuccess) {
    PyErr_SetString(error, "error getting header");
    return NULL;
  }

  py_value = PyString_FromString(value);
  WAIstringFree(value);

  return py_value;
}

static PyObject *
HSR_LogError(HSRobject *self, PyObject *args) {

  int degree, clientinfo;
  char *fnname, *errmsg;
  WAIReturnType_t rv;

  clientinfo = 0;

  UNLESS(PyArg_ParseTuple(args, "iss|i", &degree, &fnname, 
			&errmsg, &clientinfo)) {
    PyErr_SetString(error, "must provide degree, fnname, errmsg, [clientinfo]");
    return NULL;
  }

  rv = WAILogError(self->ss_ptr, degree, fnname, errmsg, clientinfo);

  UNLESS(rv == WAISPISuccess) {
    PyErr_SetString(error, "error retrieving config parameter");
    return NULL;
  }

  return PyInt_FromLong(1);
}

static PyObject *
HSR_ReadClient(HSRobject *self, PyObject *args) {

  char *buffer;
  int buflen;
  PyObject *py_object;
  int rv;

  UNLESS(PyArg_ParseTuple(args, "i", &buflen)) {
    PyErr_SetString(error, "must provide length to data to read");
    return NULL;
  } 

  buffer = (char*)malloc(buflen*sizeof(char)+1);

  rv = WAIReadClient(self->ss_ptr, buffer, buflen);

  buffer[rv]  = '\0';

  py_object = PyString_FromStringAndSize(buffer, rv);

  free(buffer);

  return py_object;
}

static PyObject *
HSR_RespondRedirect(HSRobject *self, PyObject *args) {
  
  char *url;
  WAIReturnType_t rv;

  UNLESS(PyArg_ParseTuple(args, "s", &url)) {
    PyErr_SetString(error, "must provide url to redirect to");
    return NULL;
  }
  
  rv = WAIRespondRedirect(self->ss_ptr, url);

  UNLESS(rv == WAISPISuccess) {
    PyErr_SetString(error, "error redirecting client");
    return NULL;
  }

  return PyInt_FromLong(1);
}

static PyObject *
HSR_setCookie(HSRobject *self, PyObject *args) {

  char *name, *value, *expires, *domain, *path;
  int secure;
  WAIReturnType_t rv;

  secure = 0;

  UNLESS(PyArg_ParseTuple(args, "sssss|i", 
	&name, &value, &expires, &domain, &path, &secure)) {
    PyErr_SetString(error, "must provide cookie name, value, expiretime, domain, path, [secure]");
    return NULL;
  }

  rv = WAIsetCookie(self->ss_ptr, name, value, expires, domain, path, secure);

  UNLESS(rv == WAISPISuccess) {
    PyErr_SetString(error, "error setting cookie");
    return NULL;
  }

  return PyInt_FromLong(1);             /* 1 means it worked */
}

static PyObject *
HSR_setResponseContentLength(HSRobject *self, PyObject *args) {

  int contentlength;
  WAIReturnType_t rv;
  FILE*f;

  UNLESS(PyArg_ParseTuple(args, "i", &contentlength)) {
    PyErr_SetString(error, "must provide content length to set");
    return NULL;
  }

  rv = WAIsetResponseContentLength(self->ss_ptr, contentlength);

  UNLESS(rv == WAISPISuccess) {
    PyErr_SetString(error, "error setting content length");
    return NULL;
  }

  return PyInt_FromLong(contentlength);
}

static PyObject *
HSR_setResponseContentType(HSRobject *self, PyObject *args) {

  char *contenttype;
  WAIReturnType_t rv;

  UNLESS(PyArg_ParseTuple(args, "s", &contenttype)) {
    PyErr_SetString(error, "must provide content type to set");
    return NULL;
  }

  rv = WAIsetResponseContentType(self->ss_ptr, contenttype);

  UNLESS(rv == WAISPISuccess) {
    PyErr_SetString(error, "error setting content type");
    return NULL;
  }

  return PyInt_FromLong(1);
}

static PyObject *
HSR_setResponseStatus(HSRobject *self, PyObject *args) {

  int status;
  char *value;
  WAIReturnType_t rv;

  UNLESS(PyArg_ParseTuple(args, "is", &status, &value)) {
    PyErr_SetString(error, "must provide status and reason");
    return NULL;
  }

  rv = WAIsetResponseStatus(self->ss_ptr, status, value);

  UNLESS(rv == WAISPISuccess) {
    PyErr_SetString(error, "error setting response status");
    return NULL;
  }

  return PyInt_FromLong(1);             /* 1 means it worked */
}

static PyObject *
HSR_StartResponse(HSRobject *self, PyObject *args) {

  long rv;

  UNLESS(PyArg_ParseTuple(args, "")) return NULL;

  rv = WAIStartResponse(self->ss_ptr);

  return PyInt_FromLong(rv);	/* uhh.. not sure which is which */
/* 
   one value it returns means "no action required" because the 
   request was a HEAD and needs no content body. the other means,
  "do it".
*/
}

static PyObject *
HSR_WriteClient(HSRobject *self, PyObject *args) {

  char *buffer;
  int buflen;
  int rv;
  
  UNLESS(PyArg_ParseTuple(args, "s#", &buffer, &buflen)) {
    PyErr_SetString(error, "must provide data to write to client");
    return NULL;
  }

  rv = WAIWriteClient(self->ss_ptr, buffer, buflen);
 
  UNLESS(rv == 1) {
    PyErr_SetString(error, "error writing data to client");
    return NULL;
  }

  return PyInt_FromLong(rv);
}

static PyObject *
HSR_getHost(HSRobject *self, PyObject *args) {
  
  char *buffer;
  PyObject *p_obj;

  UNLESS(PyArg_ParseTuple(args, "")) return NULL; 

  buffer = WAIgetHost(self->ss_ptr);
 
  p_obj = PyString_FromString(buffer);
  WAIstringFree(buffer);

  return p_obj;
}

static PyObject *
HSR_getInfo(HSRobject *self, PyObject *args) {

  char *name, *value;
  PyObject *p_obj;
  WAIReturnType_t rv;

  UNLESS(PyArg_ParseTuple(args, "s", &name)) {
    PyErr_SetString(error, "must provide server var to get");
    return NULL;
  }
  
  rv = WAIgetInfo(self->ss_ptr, name, &value);

  UNLESS(rv == WAISPISuccess) {
    PyErr_SetString(error, "error getting server info");
    return NULL;
  }

  p_obj = PyString_FromString(value);

  WAIstringFree(value);

  return p_obj;
}

static PyObject *
HSR_getName(HSRobject *self, PyObject *args) {

  char *buffer;
  PyObject *p_obj;

  UNLESS(PyArg_ParseTuple(args, "")) return NULL;

  buffer = WAIgetName(self->ss_ptr);

  p_obj = PyString_FromString(buffer);
  WAIstringFree(buffer);

  return p_obj;
}

static PyObject *
HSR_getPort(HSRobject *self, PyObject *args) {

  long port;
  PyObject *p_obj;

  UNLESS(PyArg_ParseTuple(args, "")) return NULL;

  port = WAIgetPort(self->ss_ptr);

  return PyInt_FromLong(port); 
}

static PyObject *
HSR_getServerSoftware(HSRobject *self, PyObject *args) {

  char *buffer;
  PyObject *p_obj;

  UNLESS(PyArg_ParseTuple(args, "")) return NULL;

  buffer = WAIgetServerSoftware(self->ss_ptr);

  p_obj = PyString_FromString(buffer);
  WAIstringFree(buffer);

  return p_obj;
}

static PyObject *
HSR_isSecure(HSRobject *self, PyObject *args) {

  int secure;
  PyObject *p_obj;

  UNLESS(PyArg_ParseTuple(args, "")) return NULL;

  secure = WAIisSecure(self->ss_ptr);

  return PyInt_FromLong(secure);
}


static PyObject *
HSR__init__(HSRobject *self, PyObject *args) {

  UNLESS(PyArg_ParseTuple(args, "")) return NULL;

  ((HSRobject*)self)->ss_ptr = NULL;

  Py_INCREF(Py_None);
  return Py_None;
}

long runfunc(ServerSession_t obj) {
  PyObject *m, *n;
  HSRobject *req;
  char* msg = "hello\n";

  PyObject *a,*b,*c;


  req = (HSRobject*)PyObject_CallObject((PyObject*)&HSRtype, NULL);

  req->ss_ptr = obj;

  m = PyObject_CallMethod((PyObject*)WAS, "SafeRun", "O", req);

  return 0;
}

static PyObject * HSR_getattr(WASobject *self, char *name) {
  return Py_FindMethod(HSR_methods, (PyObject *)self, name);
}

static struct PyMethodDef WAS_methods[] = {
	{"__init__", (PyCFunction)WAS__init__, 1,
	"__init__(name) - Create a new empty Web Application Service and assign it an instance name."},
	{"ActivateWAS", (PyCFunction)WAS_ActivateWAS, 1,
	"ActivateWAS()"},
	{"getServiceInfo", (PyCFunction)WAS_getServiceInfo, 1,
	"getServiceInfo() - return the current service's name"},
	{"RegisterService", (PyCFunction)WAS_RegisterService, 1,
	"RegisterService(""host:port"") - Registers the WAI application with the web server"},
	{"Run", (PyCFunction)WAS_Run, 1,
	"Run(req) - called by the web server upon recieving a request"},
	{NULL, NULL}		/* sentinel */
};

static void WAS_dealloc(WASobject *self) {

/*
	WAIdeleteService(self->was_ptr);

 	it's documented, but not implemented!! doh!
*/
	Py_DECREF(WAS);
	WAS = NULL;	

	PyMem_DEL(self);
}

static PyObject * WAS_getattr(WASobject *self, char *name) {
  return Py_FindMethod(WAS_methods, (PyObject *)self, name);
}

static char WAStype__doc__[] = 
"Web Application Service - IIOP/WAI/Python";

static PyExtensionClass WAStype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,                            /*ob_size*/
	"_WebApplicationService",      /*tp_name*/
	sizeof(WASobject),	      /*tp_basicsize*/
	0,
	/* methods */
	(destructor)WAS_dealloc,      /*tp_dealloc*/
	(printfunc)0,                 /*tp_print*/
	(getattrfunc)WAS_getattr,      /*tp_getattr*/
	(setattrfunc)0,               /*tp_setattr*/
	(cmpfunc)0,                   /*tp_compare*/
	(reprfunc)0,                  /*tp_repr*/
        0,                            /*tp_as_number*/
	0,                            /*tp_as_sequence*/
	0,			      /*tp_as_mapping*/
	(hashfunc)0,                  /*tp_hash*/
	(ternaryfunc)0,               /*tp_call*/
	(reprfunc)0,                  /*tp_str*/
	/* Space for future expansion */
	0L,0L,0L,0L,
	WAStype__doc__, /* Documentation string */
        METHOD_CHAIN(WAS_methods)
};

static struct PyMethodDef HSR_methods[] = {
	{"addResponseHeader", (PyCFunction)HSR_addResponseHeader, 1,
	"addResponseHeader(name, value) - Adds a header to the HTTP response to be sent back to the client."},
	{"delResponseHeader", (PyCFunction)HSR_delResponseHeader, 1,
	"delResponseHeader(name) - Removes a header from the HTTP response to be sent back to the client."},
	{"getConfigParameter", (PyCFunction)HSR_getConfigParameter, 1,
	"getConfigParameter(name) - Gets the value of a parameter of the iiopexec function in the Service directive of the obj.conf file."},
	{"getCookie", (PyCFunction)HSR_getCookie, 1,
	"getCookie() - Retrieves any cookies sent by the client."},
	{"getRequestHeader", (PyCFunction)HSR_getRequestHeader, 1,
	"getRequestHeader(name) - Gets a header from the HTTP request sent by the client."},
	{"getRequestInfo", (PyCFunction)HSR_getRequestInfo, 1,
	"getRequestInfo(name) - Gets information about the client request (such as the value of a CGI 1.1 environment variable)."},
	{"getResponseContentLength", (PyCFunction)HSR_getResponseContentLength, 1,
	"getResponseContentLength() - Gets the content length (the value of the Content-length header) of the response."},
	{"getResponseHeader", (PyCFunction)HSR_getResponseHeader, 1,
	"getResponseHeader(name) - Gets a header from the HTTP response you plan to send to the client."},
	{"LogError", (PyCFunction)HSR_LogError, 1,
	"LogError(degree, fnname, reason) - Logs an entry to the server's error log file (<server_root>/<server_id>/logs/errors on UNIX and <server_root>\<server_id>\logs\errors on Windows NT)."},
	{"ReadClient", (PyCFunction)HSR_ReadClient, 1,
	"ReadClient(n) - Reads n bytes of data from the client (for example, for data sent through the HTTP POST method). "},
	{"RespondRedirect", (PyCFunction)HSR_RespondRedirect, 1,
	"RespondRedirect(url) - Redirects the client to a different URL"},
	{"setCookie", (PyCFunction)HSR_setCookie, 1,
	"setCookie(name, value, expires, domain, path, [secure]) - Creates a cookie and sends it to the client."},
	{"setResponseContentLength", (PyCFunction)HSR_setResponseContentLength, 1,
	"setResponseContentLength(n) - Sets the content length (the value of the Content-length header) of the response to be sent to the client."},
	{"setResponseContentType", (PyCFunction)HSR_setResponseContentType, 1,
	"setResponseContentType(mimetype) - Sets the content type (the value of the Content-type header) of the response to be sent to the client."},
	{"setResponseStatus", (PyCFunction)HSR_setResponseStatus, 1,
	"setResponseStatus(code, reason) - Sets the HTTP response code (for example, 404 for ""File Not Found"") of the response to be sent to the client."},
	{"StartResponse", (PyCFunction)HSR_StartResponse, 1,
	"StartResponse() - Starts sending the response back to the client."},
	{"WriteClient", (PyCFunction)HSR_WriteClient, 1,
	"WriteClient(data) - Writes data to the client."},
	{"getHost", (PyCFunction)HSR_getHost, 1,
	"getHost() - Gets the hostname of the machine where the web server is running. "},
	{"getInfo", (PyCFunction)HSR_getInfo, 1,
	"getInfo(name) - Retrieves information about the web server (such as the value of CGI 1.1 environment variables that describe the server)."},
	{"getPort", (PyCFunction)HSR_getPort, 1,
	"getPort() - Gets the port number that the web server listens to."},
	{"getServerSoftware", (PyCFunction)HSR_getServerSoftware, 1,
	"getServerSoftware() - Gets the type and version of the server software."},
	{"isSecure", (PyCFunction)HSR_isSecure, 1,
	"isSecure() - Specifies whether or not the server is run with SSL enabled."},
	{"__init__", (PyCFunction)HSR__init__, 1,
	"__init__() - Create a new empty request object."},
	{NULL, NULL}
};

static PyObject* HSR_dealloc(PyObject* self){
  PyMem_DEL(self);
}

static char HSRtype__doc__[] = 
"HTTP Server Request object";

static PyExtensionClass HSRtype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,                            /*ob_size*/
	"_HTTPServiceRequest",      /*tp_name*/
	sizeof(HSRobject),            /*tp_basicsize*/
	0,      
	/* methods */
	(destructor)HSR_dealloc,      /*tp_dealloc*/
	(printfunc)0,                 /*tp_print*/
	(getattrfunc)HSR_getattr,               /*tp_getattr*/
	(setattrfunc)0,               /*tp_setattr*/
	(cmpfunc)0,                   /*tp_compare*/
	(reprfunc)0,                  /*tp_repr*/
	0,                            /*tp_as_number*/
	0,                            /*tp_as_sequence*/
	0,                            /*tp_as_mapping*/ 
	(hashfunc)0,                  /*tp_hash*/
	(ternaryfunc)0,               /*tp_call*/
	(reprfunc)0,                  /*tp_str*/
	/* Space for future expansion */
	0L,0L,0L,0L,
	HSRtype__doc__, /* Documentation string */
	METHOD_CHAIN(HSR_methods)
};

static struct PyMethodDef wai_methods[] = {
	{NULL, NULL}	/* sentinel */
};

void init_wai() {
  PyObject *m, *d;

  m = Py_InitModule4(
	"_wai", wai_methods,
	"_wai -- netscape wai interface (C)",
	(PyObject*)NULL,PYTHON_API_VERSION);

  d = PyModule_GetDict(m);

  error = PyString_FromString("_wai.error");
  PyDict_SetItemString(d, "error", error);

  WAS = NULL;

PyExtensionClass_Export(d, "_WebApplicationService", WAStype);
PyExtensionClass_Export(d, "_HTTPServiceRequest", HSRtype);


  if (PyErr_Occurred()) 
    Py_FatalError("can't initialize module _wai");
}
	
