# $Id: newsfetch.py,v 1.2 1996/07/30 00:25:19 koert Exp $
#
# Python based news reader
#
# Copyright 1995 - Koert Zeilstra
#
# No warranty - use at your own risk
#

import string
import posixpath
import socket
import nntplib
import tempfile

#
# NewsFetcher
#   Base class for NNTP and local storage retrievers
#
class NewsFetcher:
  def __init__ (self, app_config):
    self.app_config = app_config
    self.current_group = None
    self.local_fetcher = LocalNewsFetcher(app_config)
    self.nntp_fetcher = NntpNewsFetcher(app_config)
    self.current_source = self.local_fetcher
    ARTICLE_HEAD = "head"
    ARTICLE_BODY = "body"

  def setSource (self, new_source):
    if new_source == "local":
      self.current_source = self.local_fetcher
    elif new_source == "nntp":
      self.current_source = self.nntp_fetcher
    if self.current_group != None:
      self.current_source.changeGroup(self.current_group)

  def changeGroup (self, group_name):
    self.current_source.changeGroup(group_name)
    self.current_group = group_name

  def groupStatus (self):
    return self.current_source.groupStatus()

  def getXhdr (self, start, end, field, func):
    self.current_source.getXhdr (start, end, field, func)

  def statusArticle (self, number):
    return self.current_source.statusArticle(number)

  def getArticle (self, number, part):
    # part is "head" or "body"
    return self.current_source.getArticle (number, part)

  def saveArticle (self, number, filename):
    return self.current_source.saveArticle (number, filename)

  def postArticle (self, article, author, newsgroup, subject, org, ref):
    return self.current_source.postArticle(article, author, newsgroup, subject, org, ref)

  def getGroupList (self):
    return self.current_source.getGroupList()

#
# NewsFetcherBase
#   Base class for NNTP and local storage retrievers
class BaseNewsFetcher:
  def __init__ (self, app_config):
    self.app_config = app_config
    self.post_error = "Posting Error"
    self.article_not_present = "Article not present"

  def splitLine (self, line):
    i = string.find(line, " ")
    if i != -1:
      field1 = line[:i]
      field2 = line[i+1:]
    else:
      field1 = ""
      field2 = ""
    return (field1, field2)


class LocalNewsFetcher(BaseNewsFetcher):
  def __init__ (self, app_config):
    BaseNewsFetcher.__init__(self, app_config)
    self.local_dir = posixpath.expandvars(posixpath.expanduser(self.app_config.get("storage")))

  def changeGroup (self, group_name):
    self.current_group = group_name

  def groupStatus (self):
    fp = open("%s/%s/index/status" % (self.local_dir, self.current_group), "r")
    line = fp.readline()
    fp.close()
    l = string.split (line)
    first = string.atoi(l[0])
    last = string.atoi(l[1])
    return (100, last-first, first, last)

  def getXhdr (self, start, end, field, func):
    fp = open("%s/%s/index/%s" % (self.local_dir, self.current_group,
		field), "r")
    found = 0
    line = fp.readline()
    while found == 0:
      i = string.find(line, " ")
      if i == -1:
        line = fp.readline()
      else:
        number = string.atoi(line[:i])
        if number >= start:
          found = 1
        else:
          line = fp.readline()
      if not line: found = 1
    while found == 1:
      i = string.find(line, " ")
      if i == -1:
        line = fp.readline()
      else:
        number = string.atoi(line[:i])
        if number > end:
          found = 0
        else:
          value = line[i+1:]
          func(number, value)
          line = fp.readline()
      if not line:
        found = 0
    fp.close()

  def statusArticle (self, number):
    filename = "%s/%s/%s" % (self.local_dir, self.current_group, number)
    if posixpath.exists(filename) and posixpath.isfile(filename):
      return "100"
    else:
      return "500"

  def getArticle (self, number, part):
    header = ""
    body = ""
    fp = open("%s/%s/%s" % (self.local_dir, self.current_group, number))
    article = fp.read()
    fp.close()
    lines = string.splitfields(article, "\n")
    header_done = 0
    for line in lines:
      if line == "":
        header_done = 1
      else:
        line = line + "\n"
        if header_done == 0:
          header = header + line
        else:
          body = body + line
    if part == "head":
      return header
    elif part == "body":
      return body
    return ""

  def postArticle (self, article, author, newsgroup, subject, org, ref):
    dummy = 1

class NntpNewsFetcherOld(BaseNewsFetcher):
  def __init__ (self, app_config):
    BaseNewsFetcher.__init__(self, app_config)
    self.nntp_socket = None
    self.group_status = 0

  def changeGroup (self, group_name):
    self.current_group = ""
    status, message = self.sendCommand("group %s" % group_name)
    print "changeGroup: %d - %s" % (status, message)
    self.group_status = message
    self.current_group = group_name

  def groupStatus (self):
    l = string.split(self.group_status)
    print l
    return string.atoi(l[0]), string.atoi(l[1]), string.atoi(l[2])

  def getXhdr (self, start, end, field, func):
    self.sendCommand("xhdr %s %d-%d" % (field, start, end))
    line = self.getDataLine()
    l = string.split(line)
    status = l[0]
    if status < 300:
      done = 0
      while done == 0:
        line = self.getDataLine()
        line = string.strip(line)
        if line == ".":
          done = 1
        else:
          i = string.find(line, " ")
          if i != -1:
            number = line[:i]
            value = line[i+1]
            func(number, value)

  def statusArticle (self, number):
    status, message = self.sendCommand ("stat %d", number)
    return (status, message)

  def getArticle (self, number, part):
    article = ""
    status = self.statusArticle(number)[0]
    if status >= 400:
      raise self.article_not_present
    else:
      status, msg = self.sendCommand(part)
      if status >= 400:
        raise self.article_not_present, msg
      else:
        done = 0
        while done == 0:
          line = self.getData()
          line = string.strip(line)
          if line == ".":
            done = 1
          else:
            article.append (line+"\n")
    return article

  def postArticle (self, article, author, newsgroup, subject, org, ref):
    status, msg = self.sendCommand("post")
    if status >= 400:
      raise self.post_error, msg
    else:
      self.sendData("From: %s\n" % author)
      self.sendData("Newsgroups: %s\n" % newsgroup)
      self.sendData("Subject: %s\n" % subject)
      self.sendData("Organization: %s\n" % org)
      self.sendData("References: %s\n" % ref)
      self.sendData("X-Newsreader: %s\n" % self.app_config("getAppVersion"))
      self.sendData("\n")
      lines = string.split(article, "\n")
      for line in lines:
        if line == ".":
          line = ".."
        self.sendData(line+"\n")
      self.sendData(".\n")
      line = self.nntp_socket.recv(1024)
      status, msg = self.splitLine (line)
      if string.atoi(status) >= 400:
        raise self.post_error, msg

  def getNntpSocket (self):
    if self.nntp_socket == None:
      host = socket.gethostbyname(self.app_config.get("nntp_server"))
      port = string.atoi(self.app_config.get("nntp_port"))
      self.nntp_socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
      self.nntp_socket.connect(host, port)
      line = self.nntp_socket.recv(1024)
      print "getNntpSocket: %s" % line

  def sendCommand (self, command):
    if self.nntp_socket == None:
      self.getNntpSocket()
    print "sendCommand: %s" % command
    self.nntp_socket.send(command+"\n")
    line = self.nntp_socket.recv(1024)
    print "sendCommand: %s" % line
    status, message = self.splitLine (line)
    status = string.atoi(status)
    print "sendCommand3: %d - %s - %s" % (status, message, line)
    # fp = self.getNntpSocket()
    #if [catch {dp_send $nntp_socket $command}] \
    #{ ::close $nntp_socket
    #  set nntp_socket ""
    #  getNntpSocket
    #  if {$group_name != ""} \
    #  { dp_send $nntp_socket "group $group_name"
    #    gets $nntp_socket line
    #    dp_send $nntp_socket $command
    #gets $nntp_socket line
    #return $line
    return status, message

  def sendData (self, data):
    if self.nntp_socket == None:
      getnntpSocket()
    self.nntp_socket.send(data)

  def getDataLine (self):
    return self.nntp_socket.recv(1024)

class NntpNewsFetcher(BaseNewsFetcher):
  def __init__ (self, app_config):
    BaseNewsFetcher.__init__(self, app_config)
    self.nntp_socket = None
    self.group_status = 0

  def changeGroup (self, group_name):
    self.getNntpSocket()
    self.current_group = ""
    status, count, first, last, name = self.nntp_socket.group(group_name)
    self.group_status = 100, string.atoi(count), string.atoi(first), string.atoi(last)
    self.current_group = group_name

  def groupStatus (self):
    return self.group_status

  def getXhdr (self, start, end, field, func):
    self.getNntpSocket()
    status, fields = self.nntp_socket.xhdr(field, "%d-%d" % (start, end))
    for number, value in fields:
      func(string.atoi(number), value)

  def statusArticle (self, number):
    status, message = self.sendCommand ("stat %d", number)
    return (status, message)

  def getArticle (self, number, part):
    if part == "head":
      status, nr, id, list = self.nntp_socket.head("%d" % number)
    if part == "body":
      status, nr, id, list = self.nntp_socket.body("%d" % number)
    return string.joinfields(list, "\n")

  def saveArticle (self, number, filename):
    status, nr, id, list = self.nntp_socket.article("%d" % number)
    file = open(filename, "w")
    file.write(string.joinfields(list, "\n"))

  def postArticle (self, article, author, newsgroup, subject, org, ref):
    filename = tempfile.mktemp()
    fp = open(filename, "w")
    fp.write("From: %s\n" % author)
    fp.write("Newsgroups: %s\n" % newsgroup)
    fp.write("Subject: %s\n" % subject)
    fp.write("Organization: %s\n" % org)
    fp.write("References: %s\n" % ref)
    fp.write("X-Newsreader: %s\n" % self.app_config.getAppVersion())
    fp.write("\n")
    fp.write(article)
    fp.close()
    fp = open(filename, "r")
    line = self.nntp_socket.post(fp)
    fp.close()
    status, msg = self.splitLine (line)
    if string.atoi(status) >= 400:
      raise self.post_error, msg
    return 0

  def getGroupList (self):
    self.getNntpSocket()
    result = self.nntp_socket.list()
    return result[1]

  def getNntpSocket (self):
    if self.nntp_socket == None:
      self.nntp_socket = nntplib.NNTP(self.app_config.get("nntp_server"))









