#
# FILE            $Id: RemoteCallServer.py,v 1.6 1996/01/11 15:49:15 omfadmin Exp $
#
# DESCRIPTION     Remote call handler for Python applications.
#
# AUTHOR          SEISY/LKSB Daniel Larsson
#
# All rights reserved. Reproduction, modification, use or disclosure 
# to third parties without express authority is forbidden.
# Copyright ABB Industrial Systems AB, Sweden, 1994.
# 
# HISTORY:
#
# $Log: RemoteCallServer.py,v $
# Revision 1.6  1996/01/11  15:49:15  omfadmin
# Since sock.accept doesn't seem to return the socket name when using UNIX
# sockets, I had to do some tweaking to make it work.
#
# Revision 1.5  1995/11/20  13:30:42  omfadmin
# Hmm, Replaced UDPServer with UnixStreamServer.
#
# Revision 1.4  1995/11/19  21:19:10  omfadmin
# Added support for servers using Unix sockets.
#
# Revision 1.3  1995/11/19  20:23:18  omfadmin
# Now uses SocketServer module from Python distribution. Does not close
# socket on every request for better performance. It should now also be
# easier to support per-client data (in the RequestHandler).
#
# Revision 1.2  1995/07/06  21:21:30  omfadmin
# Fixed doc heading for module.
#
# Revision 1.1  1995/07/06  18:25:04  omfadmin
# Initial revision
#
# Revision 1.15  1995/06/12  15:29:14  dlarsson
# *** empty log message ***
#
# Revision 1.14  1995/03/25  14:02:43  dlarsson
# from Agent import Agent -> import Agent
#
# Revision 1.13  1995/03/20  21:39:25  dlarsson
# VERBOSE -> _VERBOSE
#
# Revision 1.12  1995/03/13  16:03:50  dlarsson
# Removed support for recording (didn't work well with new Agent
# support). Other updates suggested by Guido.
#
# Revision 1.11  1995/03/13  12:45:39  dlarsson
# Now using pickle module.
#
# Revision 1.10  1995/02/27  20:11:32  dlarsson
# New documentation format (use __doc__ facility of Python 1.2)
#
# Revision 1.9  1995/02/06  20:08:22  dlarsson
# OMF_SERVER: _agent_refs_ is now a class variable.
#
# Revision 1.8  1995/01/13  09:04:42  dlarsson
# Added handling of tuples in MakeAgents.
#
# Revision 1.7  1995/01/12  16:39:27  dlarsson
# Automatic creation of agents from server side upon request added.
#
# Revision 1.6  1994/12/13  13:08:04  omfadmin
# At last manual generation works.
#
# Revision 1.5  1994/12/12  21:46:07  dlarsson
# Bug in docs (mangen failed again).
#
# Revision 1.4  1994/12/12  21:43:30  dlarsson
# Bug in docs (mangen failed).
#
# Revision 1.3  1994/12/12  13:53:23  dlarsson
# No change
#
# Revision 1.2  1994/12/06  20:43:56  dlarsson
# Added recording capabilities.
#
# Revision 1.1  1994/12/05  19:26:53  dlarsson
# Initial revision
#
# 

"""Handle remote calls from other python applications

DESCRIPTION
The RemoteCallServer is a class for handling remote calls from other Python
applications. Remote calls are one of

	*  arbitrary expression
	*  function call (i.e. anything you can 'apply' arguments on)
	*  attribute assignment (setattr(obj, attr, value))

Values transferred across the wire are marshalled using the 'pickle' module,
which means not only builtin data types (integers, lists, tuples, etc) can
be transferred, but also user defined classes.

After an incomming value is unpickled, but before it is further processed
(e.g. sent as arguments to a function) one can optionally process it with
a 'before' method. Likewise, before pickling the results and sending them back
an after method may massage it. The main reason for this facility is to
implement 'Agents'.

AGENTS
Agents are objects that acts as local representatives of remote objects.
Agents are also known in other systems as proxys, ambassadors, and other
names. Agents are useful for objects that use facilities local to the server
that might not be available at the client (e.g. they might interface to
an application with python embedded).

By installing agent support, and telling the RemoteCallServer object what
objects and/or classes not to send over the wire, all such objects appearing
in results from a RemoteCall will automatically be converted to agents.
An agent sent as an argument is likewise translated automatically to the
corresponding server object.

PROTOCOL
A remote request is a 2-tuple where the first element indicates whether
the second element is to be considered an expression, a call, an assignment
or a record control request.

If the request is an expression or a call, it is evaluated and if the evaluation
succeeds the tuple '(None, result)' is returned. In the case an exception is
raised the tuple '(exc_type, exc_value)' is returned instead.

If the request is an assignment it is executed and if successfull the tuple
'(None, None)' is returned. In the case an exception is raised the tuple
'(exc_type, exc_value)' is returned instead.

NOTES
When enabling agent support, currently all transferred tuples will be copied,
whether they contain agents or not.
"""

__version__ = '$Revision: 1.6 $'

from SocketServer import TCPServer, UnixStreamServer, BaseRequestHandler
from types import *


from RemoteCalldefs import *
import Agent


class RemCallHandlerMixin:
        """Handles a request from a client.

	This class should be mixed in with a SocketServer.RequestHandler.
	It handles a remote call request from a client."""

	def address(self):
		return self.request.getsockname()

	def fileno(self): return self.request.fileno()

	def handle(self, *args):
		"""Callback to call when a message is received on the socket.
		If 'AddInput' is used you don't have to bother with this
		function. If, however you use some other synchronization
		mechanism, this is the function to call to handle requests."""

		try:
			str = self.request.recv(10240)
		except:
			self.server.remove_client(self)
			return

		if self.server._VERBOSE:
			print 'Connected by ', self.client_address

		try:
			mode, request = eval(str)
		except:
			# Hmm, strange data. It seems we end up here
			# When a Unix socket is closed.
			self.server.remove_client(self)
			return
		else:
			if self.server._VERBOSE > 1:
				print 'Received :', request

			try:
				# Default return value
				result = (None, None)

				if mode == CALL:
					method, args = request
					result = None, self._call_(method, args)
				elif mode == EVAL:
					result = None, self._eval_expr_(request)
				elif mode == EXEC:
					result = None, self.Exec(request)
				elif mode == SETATTR:
					obj, attr, value = request
					result = None, self._setattr_(obj, attr, value)
				else:
					raise 'Mode error', mode

			except:
				import sys
				result = sys.exc_type, sys.exc_value
#				import tb
#				tb.printtb(sys.exc_traceback)

		if self.server._VERBOSE > 1:
			print 'Result :', result

		#self.wfile.write(`result`)
		self.request.send(`result`)

		if self.server._VERBOSE:
			print 'OK, wait for next...'


	def MakeAgents(self, result):
		"""Replaces objects that should be sent as agents with a corresponding
		agent.
		
		This method is normally used internally when agent support
		is enabled. You can use this method directly if you for some
		reason do not want general agent support but only for some
		selected cases."""

		resultType = type(result)
		if resultType == ListType:
		    result = map(self.MakeAgents, result)
		elif resultType == DictionaryType:
		    newresult = {}
		    for ix in result.keys():
			newresult[ix] = self.MakeAgents(result[ix])
		    result = newresult
		elif resultType == TupleType:
		    result = tuple(map(self.MakeAgents, result))
		elif result in self.server._makeagents_ \
		     or (hasattr(result, '__class__') and
			 result.__class__ in self.server._makeagents_):
		    self.server._agent_refs_[id(result)] = result
		    agent = Agent.Agent('RemoteCallServer.RemoteCallServerMixin._agent_refs_['+`id(result)`+']', self.address(), 0)
		    result = agent
		    if self.server._VERBOSE > 1:
			print 'Agent created :', result
		return result

	def ReplaceAgents(self, value):
		"""Replaces agents in 'value' with the actual objects they
		refer to.
		
		This method is normally used internally when agent support
		is enabled. You can use this method directly if you for some
		reason do not want general agent support but only for some
		selected cases."""
		valueType = type(value)
		if valueType == ListType:
			value = map(self.ReplaceAgents, value)
		elif valueType == DictionaryType:
			for ix in value.keys():
				value[ix] = self.ReplaceAgents(value[ix])
		elif valueType == TupleType:
			value = tuple(map(self.ReplaceAgents, value))
		elif hasattr(value, '__class__') and value.__class__ == Agent.Agent:
			value = self.Eval(value.remoteObject())
		return value		


	# ---  Evaluating and executing python code  ---
	
	def Eval(self, expr):
		"""This is the function which must evaluate the expression sent
		from a remote node. To be able to call application specific
		functions it is necessary to inherit from this class and
		override the eval method so that it executes in the right name
		space. By default they execute here, which means almost nothing
		is visible."""
		
		return eval(expr)

	def Exec(self, code):
		"""This is the function which executes the code sent
		from a remote node. To be able to call application specific
		functions it is necessary to inherit from this class and
		override the eval method so that it executes in the right name
		space. By default they execute here, which means almost nothing
		is visible."""
		
		exec code


	# ---  Private methods  ---

	def _call_(self, method, args):
		m = self.Eval(method)
		return picklify(
			self.server._after_(self,
				apply(m,
				      self.server._before_(self,
							   unpicklify(args)))))

	def _eval_expr_(self, expr):
		return picklify(
			self.server._after_(self,
				self.Eval(self.server._before_(self, expr))))

	def _setattr_(self, obj, attr, value):
		return picklify(
			setattr(self.Eval(obj),
				attr,
				self.server._before_(self,
						     unpicklify(self, value))))

	def _default_before_(self, value):
		return value

	_default_after_ = _default_before_





class RemCallHandler(RemCallHandlerMixin, BaseRequestHandler):
	def setup(self):
		"""Initialize Stream request and add myself to dispatcher"""
		BaseRequestHandler.setup(self)
		self.server.add_client(self)


class RemoteCallServerMixin:
	"""Handle remote calls from other python applications.
	Further documentation on module."""

	_agent_refs_ = {}

	# Verbose mode for printing trace and debug messages.
	_VERBOSE		= 0

	def __init__(self):
		# Initialize members
		self._makeagents_= []	# List of objects/classes which
					# shouldn't be transferred over
					# the wire, but send agents instead.
		self._after_	 = RemCallHandlerMixin._default_after_
		self._before_	 = RemCallHandlerMixin._default_before_
		
		self.clients     = {}

	# ---  Public methods  ---

	def add_input(self):
		"""Add myself to dispatcher."""
		# Call method defined by subclasses
		self.add_to_dispatcher(self, self._handle_request_)

	def SetBefore(self, before):
		"Set user defined 'before' method"
		self._before_ = before

	def SetAfter(self, after):
		"Set user defined 'after' method"
		self._after_ = after

	def EnableAgents(self):
		"""Sets up the server to handle agents. This function sets
		the 'before' and 'after' methods."""
		self.SetAfter(RemCallHandlerMixin.MakeAgents)
		self.SetBefore(RemCallHandlerMixin.ReplaceAgents)

	def SendAsAgent(self, obj):
		"""'obj' or instances of 'obj' should not be sent over the wire
		"as is", but rather as agents (i.e. the object stays at the
		server and the client gets a proxy object, here called agent)"""
		self._makeagents_.append(obj)


	# ---  Overrides SocketServer.TCPServer method  ---

	def process_request(self, request, client_address):
		# Due to problems getting the client address
		# of UNIX sockets, we replace them with the
		# server address
		if client_address == '':
			client_address = self.server_address
		self.finish_request(request, client_address)


	# ---  Private methods  ---

	def add_client(self, client):
		self.add_to_dispatcher(client, client.handle)
		self.clients[id(client)] = client

	def remove_client(self, client):
		self.remove_from_dispatcher(client)
		del self.clients[id(client)]

	def _handle_request_(self, *args):
		self.handle_request()


class TkDispatcherMixin:
	def __init__(self):
		import Tkinter
		self.tk = Tkinter.Tk()

	def add_to_dispatcher(self, object, fun):
		self.tk.tk.createfilehandler(object.fileno(), 1, fun)

	def remove_from_dispatcher(self, object):
		self.tk.tk.deletefilehandler(object.fileno())



class TkTCPRemoteCallServer(RemoteCallServerMixin, TCPServer, TkDispatcherMixin):
	def __init__(self, addr = ('', 0), RequestHandlerClass = RemCallHandler):
		RemoteCallServerMixin.__init__(self)
		TCPServer.__init__(self, addr, RequestHandlerClass)
		TkDispatcherMixin.__init__(self)


class TkUnixStreamRemoteCallServer(RemoteCallServerMixin, UnixStreamServer, TkDispatcherMixin):
	def __init__(self, addr, RequestHandlerClass = RemCallHandler):
		RemoteCallServerMixin.__init__(self)
		UnixStreamServer.__init__(self, addr, RequestHandlerClass)
		TkDispatcherMixin.__init__(self)


class XtDispatcherMixin:
	def add_to_dispatcher(self, object, fun):
		import Xt
		from Xtdefs import XtInputReadMask

		Xt.AddInput(object.fileno(), XtInputReadMask, fun, None)

	def remove_from_dispatcher(self, object):
		import Xt
		Xt.RemoveInput(object.fileno())


class XtTCPRemoteCallServer(RemoteCallServerMixin, TCPServer, XtDispatcherMixin):
	def __init__(self, addr = ('', 0), RequestHandlerClass = RemCallHandler):
		RemoteCallServerMixin.__init__(self)
		TCPServer.__init__(self, addr, RequestHandlerClass)
		XtDispatcherMixin.__init__(self)


class XtUnixStreamRemoteCallServer(RemoteCallServerMixin, UnixStreamServer, XtDispatcherMixin):
	def __init__(self, addr, RequestHandlerClass = RemCallHandler):
		RemoteCallServerMixin.__init__(self)
		UnixStreamServer.__init__(self, addr, RequestHandlerClass)
		XtDispatcherMixin.__init__(self)


