#
# FILE            $Id: RemoteCall.py,v 1.4 1996/01/11 15:51:41 omfadmin Exp $
#
# DESCRIPTION     RemoteCall is used to execute python statements in a remote
#		  python application.
#
# AUTHOR          SEISY/LKSB Daniel Larsson
#
# All rights reserved. Reproduction, modification, use or disclosure
# to third parties without express authority is forbidden.
# Copyright ABB Industrial Systems AB, Sweden, 1994.
#
# HISTORY:
#
# $Log: RemoteCall.py,v $
# Revision 1.4  1996/01/11  15:51:41  omfadmin
# Ooops, indentation error when raising exceptions from server.
#
# Revision 1.3  1995/11/20  13:33:39  omfadmin
# Indent with offset = 4.
#
# Revision 1.2  1995/11/19  20:20:24  omfadmin
# Made TCPRemoteCall and UnixStreamRemoteCall.
#
# Revision 1.1  1995/07/06  18:25:04  omfadmin
# Initial revision
#
#

"""Call remote python applications.

The RemoteCall is a class for sending calls to other Python applications.
Remote calls are one of

*  arbitrary expression
*  function call (i.e. anything you can 'apply' arguments on)
*  attribute assignment (setattr(obj, attr, value))

Values transferred across the wire are marshalled using the 'pickle' module,
which means not only builtin data types (integers, lists, tuples, etc) can
be transferred, but also user defined classes.

Server side exceptions are reraised on the client side with the original
value, unless the exception value contains unsupported values (see below).

SEE ALSO
Agent.py - A framework for building agent-based applications.
"""

__version__ = '$Revision: 1.4 $'


import socket
from RemoteCalldefs import *

error = 'RemoteCall.error'

class TCPRemoteCall:
	"See module documentation."

	address_family = socket.AF_INET
	socket_type    = socket.SOCK_STREAM

	def __init__(self, server_address = None):
	    # Arguments:
	    #   host	name of remote host
	    #   port	socket port of remote application
	    self.server_address = server_address
	    self._socket_ = None

	def __del__(self):
	    if self._socket_: self._socket_.shutdown(1)

	def __getstate__(self):
	    """'Magic' method used by the pickle module when packing RemoteCalls to
	    send them across the wire."""
	    return self.server_address

	def __setstate__(self, state):
	    """'Magic' method used by the pickle module when initializing a RemoteCall
	    from the wire."""
	    self.server_address = state

	def Exec(self, call):
	    """Executes 'call', which is a string of python commands, 
	    on a remote application. Exceptions in the remote application
	    due to the call are propagated to the caller.
	    """
	    return self._rpc_(`(EXEC, call)`)

	def Call(self, call, *args):
	    """Executes 'call', which is a string of python commands, 
	    on a remote application. Exceptions in the remote application
	    due to the call are propagated to the caller.
	    """
	    return unpicklify(self._rpc_(`(CALL, (call, picklify(args)))`))

	def Setattr(self, obj, attr, value):
	    """Sets the attribute 'attr', in object 'obj' (which is a string
	    referring to a remote object), to 'value'. Exceptions in the remote
	    application due to the assignment are propagated to the caller.
	    """
	    return self._rpc_(`(SETATTR, (obj, attr, picklify(value)))`)

	def Eval(self, expr):
	    """Executes 'expr', which is a python expression in string form,
	    in a remote application. The value of the expression is
	    returned to the caller. Exceptions in the remote application
	    due to the call are propagated to the caller.
	    """
	    return unpicklify(self._rpc_(`(EVAL, expr)`))


	# ---  Private methods:  ---

	def _connect_(self):
	    if not self._socket_:
		self._socket_ = socket.socket(self.address_family,
					      self.socket_type)
		self._socket_.connect(self.server_address)
	    return self._socket_

	def _rpc_(self, rpc):
	    sock = self._connect_()
	    sock.send(rpc)
	    try:
		# result = sock.makefile('r').read()
		result = sock.recv(10240)
		value = eval(result)
	    except:
		# Could not parse response
		raise error, rpc+': Invalid return value'

	    # The result is returned as a 2-tuple, where the first
	    # element is None if everything went ok. In case of an
	    # error the first element is the exception name and the
	    # second the exception value.
	    # 
	    # NOTE: Exceptions are identified by the exception object's
	    # id, not the contents of the string (of course). In Python
	    # releases prior to 1.2, exceptions are always strings, and
	    # I assume here that the contents of the string is equal to
	    # the name of the exception itself. If this assumption is
	    # false, there is no way of catching the exception :(.
	    #
	    # NOTE II: In the face of Python 1.2, this whole scheme will
	    # need to be readdressed, since exceptions can not only be
	    # string objects but instance objects also.
	    if value[0] != None:
		try:
		    exc = eval(value[0])
		except:
		    exc = value[0]
		raise exc, value[1]
	    else:
		return value[1]


class UnixStreamRemoteCall(TCPRemoteCall):
    "See module documentation."
    address_family = socket.AF_UNIX


# I assume TCP is the more commonly used transport medium
RemoteCall = TCPRemoteCall
