#
# FILE            $Id: Agent.py,v 1.5 1996/01/11 15:47:27 omfadmin Exp $
#
# DESCRIPTION     Agents are client side representatives of server objects.
#
# AUTHOR          SEISY/LKSB Daniel Larsson
#
# All rights reserved. Reproduction, modification, use or disclosure
# to third parties without express authority is forbidden.
# Copyright ABB Industrial Systems AB, Sweden, 1994.
#
# HISTORY:
#
# $Log: Agent.py,v $
# Revision 1.5  1996/01/11  15:47:27  omfadmin
# Had to add getinitargs, otherwise the server would call __getattr__ which
# would call the server through the socket...
#
# Revision 1.4  1995/11/20  13:36:41  omfadmin
# Support for Unix stream sockets.
#
# Revision 1.3  1995/11/19  20:25:12  omfadmin
# Reuse remote calls between agents to obtain better performance (no separate
# sockets for individual agents).
#
# Revision 1.2  1995/08/03  13:33:13  omfadmin
# __repr__ may be called before running __init__ when debugging.
#
# Revision 1.1  1995/07/06  18:25:04  omfadmin
# Initial revision
#
#

"""Agents provide a simple mechanism to implement distributed applications.
An agent is a client side representative of a server object. Through the
agent it is possible to access attributes and execute methods on the server.
Communication with servers are done via sockets.

The server object can be any kind of python object having attributes (e.g.
instance, class, module).

Server side exceptions are reraised on the client side with the original
value, unless the exception value contains unsupported values (see below).

Normally, agents are created by clients to represent a known server object.
Sometimes, however, server methods return objects that should not be moved
to the client process, but rather stay in the server. In such cases it is
possible to ask the client to create an agent for the returned object instead.
The server then keeps a reference to the object to avoid garbage collection.
This reference is removed when the client removes his agent. The value to
return from the server in this case is an agent identification string which
contains a reference to the object in string form (the reference is prefixed
by a "magic" string to make it distinguishable from normal return values).

EXAMPLE
To implement an agent for the server object

	class Server:
		def __init__(self):
			self.anAttr = 47

		def method(sel, arg1, arg2):
			....

	server = Server()

you must create an agent for it:

	# Import Agent module
	import Agent

	server = Agent.Agent('host.name.here', port)

	# get an attribute
	print server.anAttr
	
	# Call a method
	server.method('To be or not to be...', 'do be do be do')

NOTE
The types of values that can be transferred over the net is limited by
what the standard Python module 'pickle' can handle. Supported types
include all standard builtin types (ints, lists, tuples, etc) and
user defined classes, with some restrictions (see documentation in
pickle.py for a thorough description).
"""

import types

error = 'Agent.error'

class Agent:
	__doc__ = __doc__
	
	def __init__(self, remote=0, server_address=None, do_config=1, remote_call=None):
	    """Initialize the agent.

	    Arguments:

	    remote	name of the remote object
	    host		name of host where remote object resides
	    port		socket port of application holding remote object
	    byServer	Agent created by server (only used internally)
	    """

	    # Since this class has a '__setattr__' method, we must go
	    # via the '__dict__' attribute to fool python.
	    if remote:
		if not (server_address or remote_call):
		    raise error, 'You must give address to remote object, or a RemoteCall object'
		self.__dict__['_remote_']   = remote
		if remote_call:
		    self.__dict__['_remcall_'] = remote_call
		else:
		    from RemoteCall import TCPRemoteCall, UnixStreamRemoteCall
		    if type(server_address) == type(()):
			self.__dict__['_remcall_']  = TCPRemoteCall(server_address)
		    else:
			self.__dict__['_remcall_']  = UnixStreamRemoteCall(server_address)
		    self.__dict__['_byServer_'] = 0
		    if do_config:
			self._config_()
	    else:
		self.__dict__['_remote_']   = None
		self.__dict__['_remcall_']  = None

	def __repr__(self):
	    # Prohibit call to __getattr__
	    return '<Agent instance at %s>' % hex(id(self))[2:]

	__str__ = __repr__
			
	def __getinitargs__(self):
	    return ()

	def __getstate__(self):
	    """'Magic' method used by the pickle module when packing Agents to
	    send them across the wire."""
	    return self._remote_, self._remcall_

	def __setstate__(self, state):
	    """'Magic' method used by the pickle module when initializing an Agent
	    from the wire."""
	    self.__dict__['_remote_']   = state[0]
	    self.__dict__['_remcall_']  = state[1]
	    self.__dict__['_byServer_'] = 1

	def __getattr__(self, attr):
	    """Get an attribute from the remote object.

	    All variable read accesses (i.e. agent.attr) result in
	    a call to this function, unless 'attr' is found in
	    the agent object itself. This function will get 'attr'
	    from the remote server object."""

	    result = self._remcall_.Eval(self._remote_+'.'+attr)
	    self._config_agents_(result)
	    return result

	def __setattr__(self, attr, value):
	    """Set an attribute in the remote object.

	    All variable write accesses (i.e. agent.attr = 4) result in
	    a call to this function, unless 'attr' is found in
	    the agent object itself. This function will set 'attr'
	    in the remote server object."""

	    return self._remcall_.Setattr(self._remote_, attr, value)

	def remoteObject(self):
	    return self._remote_

	def remote_call(self):
	    """Return the remote call object"""
	    return self._remote_call_

	def Remove(self):
	    """Remove agent.

	    Remove agent (i.e. remove all method objects and if the
	    agent was created by server, also remove the book keeping
	    reference to the server object from the server).
	    The reason we have an explicit remove service is because all
	    method objects have references to the agent so no garbage
	    collection will occur if we simply delete the reference to
	    the agent."""

	    if self._byServer_:
		self._remcall_.Exec('del '+self._remote_)
	    for name in self.__dict__.keys():
		object = self.__dict__[name]
		if type(object) == types.InstanceType and \
		   object.__class__ == self.Method:
		    del self.__dict__[name]


	class Method:
	    """Represents a method on the remote object.

	    Overloads the function call method to look like a function."""

	    def __init__(self, agent, method):
		"""Arguments:

		agent	the agent instance the method shall use
		method	name of the method (string)"""

		self._agent_ = agent
		self._method_ = method

	    def __call__(self, *args):
		"""This 'magic' method enables calling instances as if
		they were functions. It will perform a remote procedure
		call on the server object the agent is connected to."""

		result = apply(self._agent_._remcall_.Call,
			       (self._agent_._remote_+'.'+self._method_,)+args)
		self._agent_._config_agents_(result)
		return result

	# Private methods:
	def _config_(self):
	    """Tries to create Method objects (see below) to match the
	    connected server object's interface. This method checks
	    the server's __class__ attribute (if it has any) and the
	    object itself."""

	    try:
		class_obj = self._remote_+'.__class__'
		# Pick up the object's class' methods, if any
		self._configobj_(class_obj)

		# Pick up the object's class' bases' methods
		self._config_bases_(class_obj)				
	    except: pass

	    self._configobj_(self._remote_)

	def _configobj_(self, object):
	    """Tries to create Method objects (see below) to match "object"'s
	    interface. This method assumes the object has a '__dict__'
	    attribute."""

	    # Find all functions in 'object'
	    f_type = 'type('+object+'.__dict__[f])'
	    is_callable = object+'.__dict__[f].__class__.__dict__.has_key("__call__")'
	    filter_fun = 'lambda f:'+f_type+' in [ClassType, FunctionType, BuiltinFunctionType, MethodType] or ('+f_type+' == InstanceType and '+is_callable+')'
	    funs_cmd = 'filter('+filter_fun+','+object+'.__dict__.keys())'

	    # Send the filter function to server.
	    funs = self._remcall_.Eval(funs_cmd)

	    # Create method objects for all attributes not starting with '_'
	    for fun in funs:
		if fun[0] != '_':
		    self.__dict__[fun] = Agent.Method(self, fun)


	def _config_bases_(self, class_obj):
	    """Traverse the inheritance hierarchy of the class object
	    and pick up inherited methods."""

	    bases = self._remcall_.Eval('len('+class_obj+'.__bases__)')
	    for base in range(bases):
		# Add methods from base
		base_obj = class_obj+'.__bases__['+`base`+']'
		self._configobj_(base_obj)
		# Add methods from base's bases
		self._config_bases_(base_obj)

	def _config_agents_(self, value):
	    "If 'value' contains agents, configure them."

	    valueType = type(value)
	    if valueType == types.ListType:
		for item in value:
		    self._config_agents_(item)
	    elif valueType == types.DictionaryType:
		for item in value.values():
		    self._config_agents_(item)
	    elif valueType == types.TupleType:
		for item in value:
		    self._config_agents_(item)
	    elif hasattr(value, '__class__') and value.__class__ == Agent:
		# If agent's remote call is connected to the same
		# server as I am, make it use my remote call
		# object.
		#if value._remcall_.server_address == self._remcall_.server_address:
		#    value.__dict__['_remcall_'] = self._remcall_
		value._config_()
