#!/usr/bin/env python

"""
Author: David Schere
Purpose: Allow for concurrent xml processing.
	
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.





Sets up a simple client/server for processing multiple xml documents
remotely. The user need only derive off of the netsax_default_serverside_handler
class and pass the class (not the instantiation of the class) to the
server as part of its initializations arguments. Each client will
process the xml document and send the processed entities down the
wire to server for final processing.

At the buttom of this module is a test harness. In order to run setup
two consoles one for the server and one for the client.

Please make sure that netsax.py is installed where you installed 
saxlib! 

# The server console
> python netsax.py server &

# The console 
> python netsax.py client test.xml


The client routine forks 10 child clients which process the same xml file
andombard the server. The server will instantiate 10 separate handlers for
each document and proces them concurrently. When all child processes
are done the client exits.


Have fun !

David



"""
from xml.sax import saxlib
import signal, cPickle, socket, select, sys, traceback, os, strop, string

"""
sys.path.append("xml/sax")
sys.path.append("xml/sax/drivers")
sys.path.append("xml/sax/parsers")
"""


SERVER_TIMEOUT = 30

false,true=0,1

version="0.1"


ACK  = 'a'
NACK = 'n'


ServerError = "netsax.ServerError"
ConnectionError = "netsax.ConnectionError"				

# Two functions lifted from Carlos Maltzahn's Paos object database 
# implememts a length oriented send and recv on top of TCP send and recv
def SEND(sock, message):
  return sock.send(`len(message)` + ' ' + message)

def RECV(sock, buf_len):
  recvd_data = sock.recv(20)
  if len(recvd_data) == 0:
    raise EOFError
  msg_len_header_len = string.index(recvd_data, ' ') + 1
  msg_len = strop.atoi(recvd_data[0:msg_len_header_len-1])
  while len(recvd_data) < msg_len + msg_len_header_len:
    act_buf_len = msg_len + msg_len_header_len - len(recvd_data)
    recvd_data = recvd_data + sock.recv(act_buf_len)
  return recvd_data[msg_len_header_len:]


""" Sends the parsed xml elements down the wire to a server for proccesing.
    All communication is lock step with the server, that is, we send a message
    and get an acknowledgement if succes or a no-acknowledgment followed
    by an error message. 
    
    We also setup a timeout if no response received from the server within
    30 (default value) seconds
"""
class netsax_client(saxlib.HandlerBase):
	def __init__(self, serveraddr, portnum, timeout):
		self.s = socket.socket(socket.AF_INET, socket.SOCK_STREAM )
		try:
			self.s.connect(serveraddr,portnum)
		except:
			msg = "Unable to connect to " + serveraddr + ", port = " + str(portnum)
			raise ConnectionError, msg
	
		# setup timeout interval and event handler
		self.timeout = timeout
		signal.signal(signal.SIGALRM, self.timeout_handler)
		
	# process an xml document and send off entity events to the server	
	def xmlproc(self, docName):
		self.docName = docName		
	

		#import saxexts, saxlib, saxutils
		from xml.sax import  saxexts, saxlib, saxutils
		#import drv_xmllib
		from xml.sax.drivers import drv_xmllib
		p = drv_xmllib.create_parser()
		p.setErrorHandler(saxutils.ErrorPrinter())
		
		try:
			p.setDocumentHandler( self )
			p.parse( docName )
		except IOError,e:
			print "Error: "+str(e)
		except saxlib.SAXException,e:
			print str(e)### Cleaning up.


	def timeout_handler(self, *args):
		raise ServerError, "sever timed out"	

	""" Main routine that communicates with server
	"""
	def comm(self, cmd, *args):
		
	
		# serialization
		data = cPickle.dumps( (cmd, args) )
		
		# send down the wire
		SEND( self.s, data )
		
		# enable alarm for timeout
		signal.alarm( self.timeout )
		
		# get the response
		try:
			reply = RECV( self.s, 20000 )
		except socket.error:
			raise ServerError, "Lost connection with server"
		
		# disable alarm
		signal.alarm( 0 )
		
		# de-serialize reply
		x = cPickle.loads( reply )
		if x[0] == NACK:
			print "Error message from server"
			raise ServerError, x[1]
			import sys; sys.exit(0)
			
		if len(x) == 1:
			return ()
		return x[1:]
	
	def startDocument(self):
		self.docId = self.comm("startDocument", self.docName)[0]
		
	def startElement(self, name, amap):
		self.comm( "startElement", self.docId, name, amap)

	def endElement(self, name, amap = {}):
		self.comm("endElement",  self.docId, name, amap)

	def processingInstruction(self, name, content):
		self.comm("processingInstruction", self.docId, name, content)

	def characters(self, v, s, e):
		self.comm("characters", self.docId, v, s, e)
	
	def endDocument(self):
		self.comm("endDocument", self.docId )
	

# This class is called within ther server namespace. You derive
# off of this to process the xml document. This is a skelton only
# that prints events to stdout
class netsax_default_serverside_handler:
	def __init__(self, server, docName):
		self.server = server
		self.docName = docName

	def startElement(self, name, amap):
		return (false, None)

	def endElement(self, name, amap = {}):
		return (false, None)

	def processingInstruction(self, name, content):
		return (false, None)

	def characters(self, v, s, e):
		return (false, None)
	
	def endDocument(self):
		return (false, None)



""" Base class for xml server.
"""		
class netsax_server_base:
	def __init__(self):
		self.table = {}
		self.docId = 0
		self.hproto = netsax_default_serverside_handler
		
	# run the server
	# portnum - 
	#	port number forTPC/IP socket connection
	# handler_prototype -
	#	This is a class object which gets created to process a 
	#	particular document.
	def run(self, portnum, handler_prototype = netsax_default_serverside_handler):
		self.hproto = handler_prototype
		
		s = socket.socket(socket.AF_INET, socket.SOCK_STREAM )
		s.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
		s.bind('',portnum)
		self.conn_table = {}

		s.listen(64)						
		
		self.mainloop( s )

	# Overloadthis method to perform background tasks		
	def bgtask(self):
		pass	
		
	# interprets clint side commands 	
	def server_cmd(self, ready_conn, args):
		cmd = args[0]
		if cmd == "startDocument":
			# we are directed to create a new document
			docName = args[1]
			# create a new document object
			d = self.hproto(self, docName)
			# assign it to a table by docId
			self.table[ str(self.docId)  ] = d
			# update docId
			self.docId = self.docId + 1
			
			# return the assigned docId 
			return self.docId - 1
			
	# process command from a client. Creates and removes
	# handler objects for xml documents.	
	def proc(self, ready_conn, args):
		# trap server commands
		msg = self.server_cmd( ready_conn, args )
		if msg != None:
			return (ACK, msg)
		else:
			# fetch the docId from the list of args
			attr  = args[0]
			docId, params = args[1][0], args[1][1:]
			result = None
			
			# try to get the document object
			if self.table.has_key( str(docId) ):
				# apply the arguments to a method
				d = self.table[str(docId)]
				if hasattr(d,attr):
					f = getattr(d,attr)
					self.handler_diag(attr, d, params)
					if len(params) > 0:
						(err, result) = apply(f,params)
					else:
						(err, result) = apply(f,())
				self.table[str(docId)] = d	

				if err == false:
					return (ACK, result)
				else:
					return (NACK, result)	
			else:
				msg = "No matching document for docId %d" % (docId)
				return (NACK, msg)	

	# Loop forever polling socket 's'		
	def mainloop(self, s):	
		self.conn_list = [s]
		while 1:
			(ready_list, x, y) = select.select(self.conn_list, [], [], 0)
			for ready_conn in ready_list:
				if ready_conn is s:
				      (conn, addr) = s.accept()
				      self.conn_table[conn] = addr
				      self.conn_list.append(conn)
				else:
					try:
						data = RECV(ready_conn, 20000)
						args = cPickle.loads(data)

						self.proc_diag( ready_conn, args )
						result = self.proc( ready_conn, args )
						
						data = cPickle.dumps(result)
						SEND(ready_conn, data)
					except:
						self.errHandler( ready_conn )		
			self.bgtask_diag()			
			self.bgtask()		
					
	# Called when an exception is thrown to send an error message back to
	# the client	
	def errHandler(self, ready_conn):			
		if sys.exc_type in [EOFError, socket.error]:
			ready_conn.close()
			self.conn_list.remove(ready_conn)
			del self.conn_table[ready_conn]
		else:
			msg = traceback.format_stack()
			data = cPickle.dumps((NACK, msg ))			
			#data = pickle.dumps((NACK, sys.exc_type, sys.exc_value, temp.read() ))			
			SEND(ready_conn, data)
			
			
	# --- diagnostic hooks -----
	# These are useful hooks so that monitors can be setup
	def proc_diag(self, ready_conn, args): 			
		pass
		
	def bgtask_diag(self):
		pass
		
	def handler_diag(self, method, handler, params):
		pass	




################################################################
#	Test Harness
#
# Tests operation of netsax.
#	   Verify that the server can handle multiple concurrent
#          clients parsing xml documents. The server willl Evoke 
#	   the TesHandler to perform processing. 
#
################################################################

# Dummy server side document handler
class TestHandler(netsax_default_serverside_handler):
	def __init__(self, server, docName):
		netsax_default_serverside_handler.__init__(self, server, docName)

	def startElement(self, name, amap):
		print "TestHandler.startElement", name, amap
		return (false, None)

	def endElement(self, name, amap = {}):
		print "TestHandler.endElement", name, amap
		return (false, None)

	def processingInstruction(self, name, content):
		print "TestHandler.processingInstruction", name, content
		return (false, None)

	def characters(self, v, s, e):
		print "TestHandler.characters", v, s, e
		return (false, None)
	
	def endDocument(self):
		print "TestHandler.endDocument"
		return (false, None)

# multiplexing socket port
NETSAX_PORT = 2112  	# Good alblum !



class TestServer(netsax_server_base):
	def __init__(self):
		netsax_server_base.__init__(self)
	
def server_test():
	s = TestServer() 
	# Note: TestHandler, not TestHandler()
	s.run( NETSAX_PORT, TestHandler)		
	
def client_test( xmlfile ):
	import os

	for i in range(0,10):
		if os.fork():
			# host = "", port = 2112, timeout if no response in 10 seconds
			c = netsax_client("", NETSAX_PORT, 10)	

			# parse the xml file and send it to the server
			c.xmlproc( xmlfile )
			
			sys.exit(0)

	# wait for all of our children to exit
	os.wait()
			
			
if __name__ == '__main__':
	if sys.argv[1] == "server":
		server_test()
	else:
		try:
			xmlfile = sys.argv[2]
		except:
			xmlfile = "test.xml"
		client_test( xmlfile )



		
			
		
		
		
		
		
		
		
		
		
		
