/* This module makes BSD editline available to Python.  It is based
 * on the GNU readline module by Guido van Rossum.
 *
 * David Leonard <d@openbsd.org>, 1999. Public Domain.
 */

/* Standard definitions */
#include "Python.h"
#include <setjmp.h>
#include <signal.h>
#include <errno.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h> /* For isatty() */
#endif

/* BSD editline definitions */
#include <histedit.h> 

/* BSD's editline contains a simple, undocumented, but helpful tokenizer. */
struct tokenizer;
typedef struct tokenizer Tokenizer;
Tokenizer *tok_init(const char *);
void tok_reset(Tokenizer *);
void tok_end(Tokenizer *);
int tok_line(Tokenizer *, const char *, int *, char ***);

/* Pointers needed from outside (but not declared in a header file). */
extern int (*PyOS_InputHook)();
extern char *(*PyOS_ReadlineFunctionPointer) Py_PROTO((char *));

/* Local state */

static EditLine *el;
static History *El_history;
static char *El_prompt;
static const char *El_word_brk = " \t\n`~!@#$%^&*()-=+[{]}\\|;:'\",<>/?";

/* Exported function to send one line to editline's init file parser */

static PyObject *
parse_and_bind(self, args)
	PyObject *self;
	PyObject *args;
{
	char *line, *s;
	int ret;
	char **argv;
	int argc;
	Tokenizer *tok;

	if (!PyArg_ParseTuple(args, "s", &s))
		return NULL;

	line = strdup(s);
	if (line == NULL)
		return PyErr_NoMemory();

	tok = tok_init(NULL);
	tok_line(tok, line, &argc, &argv);
	ret = el_parse(el, argc, argv);
	tok_end(tok);
	free(line);

	switch (ret) {
	case 0:
		Py_INCREF(Py_None);
		return Py_None;
	case -1:
		PyErr_SetString(PyExc_ValueError, "unknown command");
		return NULL;
	case 1:
		PyErr_SetString(PyExc_RuntimeError, "command returned error");
		return NULL;
	}
}

static char doc_parse_and_bind[] = "\
parse_and_bind(string) -> None\n\
Parse and execute single line of a editline init file. See editrc(5).\
";


/* Exported function to parse a editline init file */

static PyObject *
read_init_file(self, args)
	PyObject *self;
	PyObject *args;
{
	char *s = NULL;
	if (!PyArg_ParseTuple(args, "|z", &s))
		return NULL;

	if (el_source(el, s) == -1)
		return PyErr_SetFromErrno(PyExc_IOError);

	Py_INCREF(Py_None);
	return Py_None;
}

static char doc_read_init_file[] = "\
read_init_file([filename]) -> None\n\
Parse a editline initialization file.\n\
The default filename is the last filename used.\
";


/* Exported function to specify a word completer in Python */

static PyObject *completer = NULL;
static PyThreadState *tstate = NULL;

static PyObject *
set_completer(self, args)
	PyObject *self;
	PyObject *args;
{
	PyObject *function = Py_None;
	if (!PyArg_ParseTuple(args, "|O", &function))
		return NULL;

	if (function == Py_None) {
		Py_XDECREF(completer);
		completer = NULL;
		tstate = NULL;
	}
	else if (PyCallable_Check(function)) {
		PyObject *tmp = completer;
		Py_INCREF(function);
		completer = function;
		Py_XDECREF(tmp);
		tstate = PyThreadState_Get();
	}
	else {
		PyErr_SetString(PyExc_TypeError,
				"set_completer(func): argument not callable");
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static char doc_set_completer[] = "\
set_completer([function]) -> None\n\
Set or remove the completer function.\n\
The function is called as function(text, state),\n\
for i in [0, 1, 2, ...] until it returns a non-string.\n\
It should return the next possible completion starting with 'text'.\
";

/* Exported function to read the current line buffer */

static PyObject *
get_line_buffer(self, args)
	PyObject *self;
        PyObject *args;
{
	const LineInfo *li;

	if (PyArg_NoArgs(args))
		return NULL;
	li = el_line(el);
	return PyString_FromStringAndSize(li->buffer,
		li->lastchar - li->buffer);
}

static char doc_get_line_buffer[] = "\
get_line_buffer() -> string\n\
return the current contents of the line buffer.\
";

/* Exported function to insert text into the line buffer */

static PyObject *
insert_text(self, args)
	PyObject *self;
	PyObject *args;
{
	char *s;
	if (!PyArg_ParseTuple(args, "s", &s))
		return NULL;

	if (strlen(s) && el_insertstr(el, s) == -1) {
		PyErr_SetString(PyExc_ValueError, "won't fit");
		return NULL;
	}

	Py_INCREF(Py_None);
	return Py_None;
}


static char doc_insert_text[] = "\
insert_text(string) -> None\n\
Insert text into the command line at the cursor.\
";


/* Table of functions exported by the module */

static struct PyMethodDef Editline_methods[] =
{
	{"parse_and_bind", parse_and_bind, 1, doc_parse_and_bind},
	{"get_line_buffer", get_line_buffer, 1, doc_get_line_buffer},
	{"insert_text", insert_text, 1, doc_insert_text},
	{"read_init_file", read_init_file, 1, doc_read_init_file},
	{"set_completer", set_completer, 1, doc_set_completer},
	{0, 0}
};

/* C function to call the Python completer. */

unsigned char
on_completion(el, ch)
	EditLine *el;
	int ch;
{
	int state = 0;
	char *result = NULL;
	int ret;
	const char *text;
	int textlen;

	if (completer != NULL) {
		PyObject *r;
		PyThreadState *save_tstate;
		const LineInfo *li;

		/* walk backwards from the cursor to find the leading text */
		li = el_line(el);
		for (text = li->cursor; text > li->buffer; text--)
			if (strchr(El_word_brk, *(text-1)) != NULL)
				break;
		textlen = li->cursor - text;

		/* Note that editline is called with the interpreter
		   lock released! */
		save_tstate = PyThreadState_Swap(NULL);
		PyEval_RestoreThread(tstate);
		r = PyObject_CallFunction(completer, "s#i", text, textlen,
			state);
		if (r == NULL)
			goto error;
		if (r == Py_None)
			result = NULL;
		else {
			char *s = PyString_AsString(r);
			if (s == NULL)
				goto error;
			result = strdup(s);
			if (result == NULL)
				PyErr_NoMemory();
		}
		Py_DECREF(r);
		goto done;
	  error:
		PyErr_Clear();
		Py_XDECREF(r);
	  done:
		PyEval_SaveThread();
		PyThreadState_Swap(save_tstate);
	}

	if (result) {
		el_deletestr(el, textlen);
		el_insertstr(el, result);
		free(result);
		ret = CC_REDISPLAY;
	} else {
		ret = CC_ERROR;
	}

	return ret;
}

static char on_completion_text[] = "\
Call the completer set by editline.set_completer() to complete the word \
before the cursor.Complete the word before the cursor\
";

/* Helper to pass the prompt through the editline API. */

static char *
editline_prompt_fn(el)
	EditLine *el;
{
	return El_prompt;
}

/* Helper to initialize BSD editline properly. */

static void
setup_Editline()
{
	El_history = history_init();
	el = el_init("python", stdin, stdout);
	el_set(el, EL_PROMPT, editline_prompt_fn); /* use el_prompt */
	el_set(el, EL_SIGNAL, 1); /* editline to handle signals */
	el_set(el, EL_ADDFN, "complete", on_completion_text, on_completion);
	el_set(el, EL_HIST, history, El_history);
	history(El_history, H_EVENT, 1000);
}


/* Interrupt handler */

static jmp_buf jbuf;

/* ARGSUSED */
static RETSIGTYPE
onintr(sig)
	int sig;
{
	longjmp(jbuf, 1);
}


/* Wrapper around BSD editline that handles signals differently. */

static char *
call_Editline(prompt)
	char *prompt;
{
	int n;
	char *result;
	RETSIGTYPE (*old_inthandler)();

	old_inthandler = signal(SIGINT, onintr);
	if (setjmp(jbuf)) {
		result = NULL;
	} else {
		const char *p; 

		El_prompt = prompt;
		p = el_gets(el, &n);
		if (p == NULL || n == 0) 
			result = strdup("");
			if (result == NULL)
				PyErr_NoMemory();
		else {
			result = malloc(n+1);
			if (result == NULL) 
				PyErr_NoMemory();
			else {
				memcpy(result, p, n);
				result[n] = '\0';
				history(El_history, H_ENTER, result);
			}
		}
	}

	signal(SIGINT, old_inthandler);
	return result;
}


/* Initialize the module */

static char doc_module[] =
"Importing this module enables command line editing using BSD editline. \
See editrc(5).";

DL_EXPORT(void)
initeditline()
{
	PyObject *m;

	m = Py_InitModule4("editline", Editline_methods, doc_module,
			   (PyObject *)NULL, PYTHON_API_VERSION);
	if (isatty(fileno(stdin))) {
		PyOS_ReadlineFunctionPointer = call_Editline;
		setup_Editline();
	}
}
