# -*- Python -*-
#
# $Source: /home/bhowes/src/BoilerPlate/BoilerPlate.py,v $
# $Author: bhowes $
# $Date: 1997/09/10 23:12:29 $
# $State: Exp $
#
#---------------------------------------------------------------------------
#
# Copyright 1997 B-Ray Software
#
# Copyright in this software is owned by B-Ray Software, unless otherwise
# indicated. Permission to use, copy and distribute this software royalty-free
# is hereby granted, provided that the above copyright notice appears in all
# copies and that both the copyright notice and this permission notice appear.
#
# THERE IS ABSOLUTELY NO WARRANTY FOR THIS SOFTWARE.
#
# The software is provided "as is" without warranty of any kind, either express
# or implied, including, but not limited to, the implied warranties of
# merchantability, fitness for a particular purpose, or non-infringement. This
# software could include technical inaccuracies or typographical errors. B-Ray
# Software may make improvements and/or changes in this software at any time
# without notice.
#
# --------------------------------------------------------------------------
#
# BoilerPlate -- collection of classes that provide simple yet powerful
# macro-type processing of Python text. Blocks within text can be conditionally
# included or excluded based on runtime values. Also, text blocks can be
# iterated over by a sequence or a dictionary.
#
# Although BoilerPlate was based on ideas found in Jim Fulton's DocTemplate
# module, it is *not* a replacement. Most notably, only dictionaries are
# support in the __init__ and __call__ methods of the BoilerPlate classes,
# whereas Fulton's code supports an instance variable and name resolution for
# it. Also, the HTML server-side include format is not recognized, and the
# block tag structures are radically different.
#
# The following tags are recognized within text blocks:
#  o  @if <condition>: -- start of a conditional block. Condition is evaluated
#     at format time and if it returns a Python `true' value, the text block is
#     formatted.
#
#  o  @for <tag> in <data>: -- start of an iteration block. Data is evaluated
#     at format time and is iterated over. Unlike Python's `for' statement,
#     this one supports dictionaries the value for <data>. The following data
#     elements are available within a `for' block iteration over a sequence:
#
#       o <tag>.counter -- count of values; starts at 1
#       o <tag>.index -- index of value in sequence; starts at 0
#       o <tag>.value -- current value in iteration
#
#     These values are available for dictionary iterations:
#
#       o <tag>.key -- current key value in iteration
#       o <tag>.value -- current value in iteration
#
#     Finally, `for' blocks attempt to calculate statistics based on the values
#     in the iteration. See class `ForCooker' for more info about these.
#
#  o  @elif <condition>: -- (only available after an @if block). Condition is
#     evaluated at format time if the owning @if block or previous @elif block
#     did not format.
#
#  o  @else: -- (only after an @if or @elif block). Represents text to format
#     if all preceding @if and @elif blocks returned a Python `false'.
#
# Runtime text substitution is denoted by using Python's normal %() constructs.
# During formatting, we obtain the contents between the `(' and ')' characters,
# and evaluate it. Evaluation scope is limited to a set of formatting
# operations, Python builtin functions, and a provided dictionary.
#
# *** NOTE: this module REQUIRES a patch to Python's stringobject.c file to
# *** allow for embedded `(' and `)' characters within a %() construct. This
# *** change should be in Python release 1.5. A patch file should be included
# *** with this file.
#
# The Formatter class provides formatting routines that can be embedded inside
# of a %() construct to change the value before it is given to Python. These
# functions are supported:
#
#  o  Lower() -- convert all text to lowercase
#  o  Upper() -- convert all text to uppercase
#  o  Capitalize() -- convert first character of every word to uppercase and
#     all the rest to lowercase
#  o  Spacify() -- convert all '_' characters to spaces
#  o  Null() -- sets an alternative to return if the final value of the %()
#     construct matches a set of empty values.
#  o  Letter() -- returns letter corresponding to a number (`A' == 0)
#  o  Roman() -- returns conversion of value into roman numerals
#  o  HtmlEncode -- returns value with appropriate conversions for use within
#     an HTML tag.
#  o  UrlEncode -- returns value with appropriate conversions for use within an
#     HTML URL.
#
# Note that since Python's `eval' function is used, any built-in function can
# also be used in the %() construct.
#
# Users should create instances of either the String or the File BoilerPlate
# class. Here are some examples and expected results:
#
# EXAMPLES:
#
#   python% a={'foo': 'HeLlO WoRlD', 'bar': (1,2,3,4,5)} 
#   python% BoilerPlate.String( '%(foo)s %(Lower(foo))s %(Upper(foo))s', a )()
#   'HeLlO WoRlD hello world HELLO WORLD'
#
#   python% BoilerPlate.String( '%(map(Roman(bar))s', a )()
#   "['I', 'II', 'III', 'IV', 'V']"
#
#   python% print BoilerPlate.String( '''%for each in bar:
#   > index = %(each.index)d   value = %(each.value)d
#   > @end for
#   > avg = %(each.mean)d''', a )()
#   index = 0   value = 1
#   index = 1   value = 2
#   index = 2   value = 3
#   index = 3   value = 4
#   index = 4   value = 5
#   avg = 3
#
#---------------------------------------------------------------------------
# $Log: BoilerPlate.py,v $
# Revision 1.1  1997/09/10  23:12:29  bhowes
# Initial revision
#

version = "$Id: BoilerPlate.py,v 1.1 1997/09/10 23:12:29 bhowes Exp $"

import array, copy, regex, string

#
# Class Sink -- wrapper for Python `array' module. Used to collect formatted
# text from Block instances.
#
class Sink:

	#
	# __init__ -- create array to hold formatted text
	#
	def __init__( self ):
		self.Reset()
		return

	#
	# Reset -- dump previous result and start over.
	#
	def Reset( self ):
		self.array = array.array( 'c' )

	#
	# Append -- add text to end of current value
	#
	def Append( self, text ):
		self.array.fromstring( text )

	#
	# Value -- return accumulated text.
	#
	def Value( self ):
		return self.array.tostring()

	#
	# __str__ -- return accumulated text
	#
	__str__ = Value

#
# RawText -- instances describe a range within a text string.
#
class RawText:

	#
	# __init__ -- remember text block and positions within it that we own.
	#
	def __init__( self, text, startPos, endPos ):
		self.text = text
		self.startPos = startPos
		self.endPos = endPos

	#
	# Cook -- apply the given dictionary to a range of text we own.
	#
	def Cook( self, sink, dict ):
		if self.startPos != self.endPos:
			sink.Append( self.text[ self.startPos : self.endPos ] % dict )

#
# Class Block -- fundamental class used to record blocks of unformatted text.
# Blocks nest within other blocks. See the derived classes IfBlock and
# ForBlock.
#
class Block:

	#
	# Distinguish ourselves from other Block types
	#
	kBlockKind = None

	#
	# Define regular expression used to locate start and end of blocks. Does
	# not detect `elif' and `else' blocks. First grouping is the entire match.
	# The second grouping is the kind of block that was found. The third group
	# contains any additional text following the tag.
	#
	flowTag = regex.compile( '\([ \t]*@\(end\|for\|if\)[ \t]*\(.*\)$\)' )

	#
	# AddChild -- add the given child to our list of owned blocks, and let the
	# child eat text until it is done. Returns position of first unprocessed
	# character.
	#
	def AddChild( self, child, text, startPos ):
		self.blocks.append( child )
		return child.EatText( text, startPos )

	#
	# NewChild -- create a subblock based on the tag found in the text. Returns
	# position of first unprocessed character in text.
	#
	def NewChild( self, kind, text, startPos ):

		#
		# From a top-level, only `for' and `if' blocks are allowed
		#
		if kind == 'for':
			child = ForBlock( self.flowTag.group( 3 ) )

		elif kind == 'if':
			child = IfBlock( self.flowTag.group( 3 ) )

		else:
			raise RuntimeError, 'invalid tag found:', kind

		return self.AddChild( child, text, startPos )

	#
	# IsClosure -- returns true if the statement is a block closure. This is a
	# method so that derived classes may override it.
	#
	def IsClosure( self, kind ):
		return kind == 'end'

	#
	# CloseBlock -- finish working with a block of text. Calculate the position
	# of the next character in the text string to process.
	#
	def CloseBlock( self, text, startPos, endPos ):

		#
		# Verify that the `end' statement belongs to us.
		#
		check = string.strip( self.flowTag.group( 3 ) )
		if check != self.kBlockKind:
			raise RuntimeError, "expected 'end' to '%s' block -- got '%s'" % \
				  ( self.kBlockKind, check )

		#
		# Only grab something if there's text involved.
		#
		if startPos != endPos:
			self.blocks.append( RawText( text, startPos, endPos ) )

		#
		# Skip over `end' statement and trailing newline character
		#
		return endPos + len( self.flowTag.group( 0 ) ) + 1

	#
	# EatText -- process the given text string until we run out of characters
	# or we have encountered a proper block closure. Returns an index into the
	# given text string of the first unprocessed character.
	#
	def EatText( self, text, startPos ):

		#
		# Initialize list of blocks we control
		#
		self.blocks = []
		limit = len( text )
		index = -1
		while startPos < limit:

			#
			# Look for next block tag
			#
			index = self.flowTag.search( text, startPos )
			if index < 0:
				break

			#
			# Found a tag.
			#
			else:
				kind = self.flowTag.group( 2 )

				#
				# Are we done? If so, calculate the last character we processed
				# to return to the caller.
				#
				if self.IsClosure( kind ):
					endPos = self.CloseBlock( text, startPos, index )
					break

				#
				# Nope -- must have a embedded block. Grab any unprocessed text
				# as ours and create a new child block.
				#
				else:
					if index != startPos:
						self.blocks.append( RawText( text, startPos, index ) )

					startPos = index + len( self.flowTag.group( 0 ) ) + 1
					startPos = self.NewChild( kind, text, startPos )

			index = -1

		#
		# Finished processing -- check for premature end of text.
		#
		if index == -1:

			#
			# If we are not a named block, just take everything as a final text
			# block
			#
			if self.kBlockKind == None:
				endPos = limit
				if startPos < endPos:
					self.blocks.append( RawText( text, startPos, endPos ) )

			#
			# Should've seen an `end' block.
			#
			else:
				raise RuntimeError, "no 'end' to '%s' block" % self.kBlockKind

		return endPos

	#
	# Cook -- visit all of the blocks under our control and format each using
	# the given dictionary.
	#
	def Cook( self, sink, dict ):
		for each in self.blocks:
			each.Cook( sink, dict )

#
# Class IfBlock -- handles sequences of `if', `elif', and `else' text blocks.
# Each block except an `else' block must have a condition that will be
# evaluated during text formatting to see if the block's text is to be used.
#
class IfBlock( Block ):

	kBlockKind = 'if'

	#
	# Define regular expression to locate all block tags, including `elif' and
	# `else'.
	#
	flowTag = regex.compile( '\([ \t]*@\(elif\|else\|end\|for\|if\)[ \t]*' \
							 '\(.*\)$\)' )

	#
	# __init__ -- remember condition to test during formatting. Accept but do
	# not require trailing `:' character.
	#
	def __init__( self, condition ):
		condition = string.strip( condition )
		if condition[ -1 ] == ':':
			condition = string.strip( condition[ : -1 ] )
		if len( condition ) == 0:
			raise RuntimeError, "empty 'if' condition"
		self.condition = condition

	#
	# AddAlternative -- add a new alternative block to our list.
	#
	def AddAlternative( self, child, text, startPos ):
		self.alternatives.append( child )
		return child.EatText( text, startPos )

	#
	# NewChild -- override of Block method. We handle `elif' and `else' blocks
	# here.
	#
	def NewChild( self, kind, text, startPos ):
		if kind == 'elif':
			child = ElifBlock( self.flowTag.group( 3 ) )
			endPos = self.AddAlternative( child, text, startPos )

		#
		# There should not be any conditions following `else'
		#
		elif kind == 'else':
			extra = string.strip( self.flowTag.group( 3 ) )
			if len( extra ) > 0:
				if extra[ -1 ] == ':':
					extra = string.strip( extra[ : -1 ] )
					if len( extra ) > 0:
						raise RuntimeError, "text after 'else' tag: %s" % \
							  self.flowTag.group( 3 )
			child = ElseBlock()
			endPos = self.AddAlternative( child, text, startPos )

		#
		# Must be something in our main block -- have not seen an `elif' or
		# `else' block yet.
		#
		else:
			endPos = Block.NewChild( self, kind, text, startPos )

		return endPos

	#
	# EatText -- override of Block method. Initializes a list of alternative
	# blocks to hold any `elif' or `else' blocks after any text blocks.
	#
	def EatText( self, text, startPos ):
		self.alternatives = []
		return Block.EatText( self, text, startPos )

	#
	# Cook -- override of Block method. If our condition returns `true', we
	# format whatever text we have. Otherwise, we look for an alternative
	# clause (`elif' or `else' block) that has a true condition.
	#
	def Cook( self, sink, dict ):
		if dict[ self.condition ]:
			Block.Cook( self, sink, dict )
		else:

			#
			# Visit each alternative until a true value is returned.
			#
			for each in self.alternatives:
				value = each.Cook( sink, dict )
				if value:
					break

#
# Class ElseBlock -- handles text blocks starting with an `else' statement.
#
class ElseBlock( Block ):

	kBlockKind = 'if'

	#
	# CloseBlock -- override of Block method. We don't move past the `@'
	# statement that closed us so that our parent `if' block will see it.
	#
	def CloseBlock( self, text, startPos, endPos ):
		if startPos != endPos:
			self.blocks.append( RawText( text, startPos, endPos ) )
		return endPos

#
# Class ElifBlock -- handles text blocks starting with an `elif' statement.
#
class ElifBlock( IfBlock ):

	#
	# CloseBlock -- override of Block method. We don't move past the `@'
	# statement that closed us so that our parent `if' block will see it.
	#
	def CloseBlock( self, text, startPos, endPos ):
		if startPos != endPos:
			self.blocks.append( RawText( text, startPos, endPos ) )
		return endPos

	#
	# IsClosure -- override of Block method. Treats another `elif' block or an
	# `else' block as an end to our block.
	#
	def IsClosure( self, kind ):
		return kind == 'elif' or kind == 'else' or kind == 'end'

	#
	# Cook -- override of IfBlock Cook method. Only formats its text if its
	# condition returns `true'. Returns result of condition evaluation so that
	# our parent `if' block can detect if we formatted our text.
	#
	def Cook( self, sink, dict ):
		value = dict[ self.condition ]
		if value:
			Block.Cook( self, sink, dict )

		return value

#
# Class ForBlock -- handles text blocks starting with a `for' statement. We
# accept sequences and dictionaries for iteration values.
#
class ForBlock( Block ):

	kBlockKind = 'for'

	#
	# __init__ -- install name and values to use for sequence/map operations.
	# Accept but do not require trailing `:' character
	#
	def __init__( self, contents ):

		#
		# Expect data following the `for' tag formatted as <name> in <data>,
		# similar to Python's syntax.
		#
		contents = string.split( string.strip( contents ) )
		if len( contents ) < 3 or contents[ 1 ] != 'in':
			raise RuntimeError, "invalid 'for' block format: %s" % contents

		self.name = contents[ 0 ]
		contents = string.join( contents[ 2 : ] )
		if contents[ -1 ] == ':':
			contents = string.strip( contents[ : -1 ] )
		if len( contents ) == 0:
			raise RuntimeError, "invalid 'for' block format: %s" % contents

		#
		# Remember data we will use for iteration.
		#
		self.data = contents
		return

	#
	# IterCook -- method called by a cooker instance (see below) to format our
	# blocks
	#
	IterCook = Block.Cook

	#
	# Cook -- apply the given formatter to the contents of the `for' block.
	#
	def Cook( self, sink, formatter ):

		#
		# Obtain sequence/mapping to iterate over
		#
		data = formatter[ self.data ]

		#
		# Create appropriate cooker based on type of data
		#
		varType = type( data )
		if varType == type( [] ) or varType == type( () ):

			#
			# Have a sequence -- create a cooker to format references to
			# sequence values.
			#
			cooker = ForSeqCooker( formatter, self.name, data )

		elif varType == type( {} ):

			#
			# Have a mapping -- create a cooker to format references to
			# dictionary values.
			#
			cooker = ForMapCooker( formatter, self.name, data )

		else:
			raise RuntimeError, "invalid argument for 'for' block"

		#
		# Let the cooker iterate over the elements of the sequence or mapping
		# and repeatedly cook our blocks.
		#
		cooker.Cook( formatter, sink, self )
		return

#
# Formatter -- handles resolution of %() format constructs within text blocks.
# Name resolution is handled thru Python's `eval' built-in function. This class
# provides a set of formattting functions that can be used within a %() format
# construct to change the data before it is given to Python:
#
#   o Upper -- convert given value to all uppercase characters
#   o Lower -- convert given value to all lowercase characters
#   o Capitalize -- make the first character of each word in the value
#     uppercase, and all other characters lowercase.
#   o Spacify -- convert all underscore (`_') characters to spaces.
#   o Null -- define replacement to use if value is found in kNullSet (see
#     below)
#   o Letter -- returns the letter that corresponds to adding the given value
#     to the letter 'A'.
#   o Roman -- returns a conversion of a numeric value into roman numerals.
#   o HtmlEncode -- returns value with appropriate conversions for use within
#     an HTML tag.
#   o UrlEncode -- returns value with appropriate conversions for use within an
#     HTML URL.
#
# All functions except Null take just one argument -- the value to work on. The
# Null function takes an additional argument that is returned to Python if the
# value to be returned is None or the empty string.
#
class Formatter:

	#
	# Define the roman characters we support
	#
	kRomanSet = [ ( 'M', 1000 ), ( 'D', 500 ), ( 'C', 100 ), ( 'L', 50 ),
				  ( 'X', 10 ), ( 'V', 5 ), ( 'I', 1 ) ]

	#
	# Define values that can have an alternative expression when encountered.
	# Used by __getitem__ method below.
	#
	kNullSet = ( None, '', 0 )

	#
	# __init__ -- remember instance and dictionary to use for evaluation of
	# formatting names and expressions.
	#
	def __init__( self, env ):
		self.env = env
		dict = self.procs = {}

		#
		# Install functions available for formatting values
		#
		dict[ 'Upper' ] = self.Upper
		dict[ 'Lower' ] = self.Lower
		dict[ 'Capitalize' ] = self.Capitalize
		dict[ 'Spacify' ] = self.Spacify
		dict[ 'Null' ] = self.Null
		dict[ 'Letter' ] = self.Letter
		dict[ 'Roman' ] = self.Roman
		dict[ 'HtmlEncode' ] = self.HtmlEncode
		dict[ 'UrlEncode' ] = self.UrlEncode
		return

	#
	# Upper -- returns text in uppercase
	#
	def Upper( self, value ):
		return string.upper( value )

	#
	# Lower -- returns text in lowercase
	#
	def Lower( self, value ):
		return string.lower( value )

	#
	# Capitalize -- returns text with first letter of all words in uppercase
	# and the rest in lowercase.
	#
	def Capitalize( self, value ):
		return string.join( map( lambda a: string.upper( a[ 0 ] ) + \
								 string.lower( a[ 1 : ] ),
								 string.split( value ) ) )

	#
	# Spacify -- convert all `_' characters into spaces
	#
	def Spacify( self, value ):
		return string.join( string.splitfields( value, '_' ) )

	#
	# Null -- set alternative value to use if final value is found in kNullSet
	#
	def Null( self, value, noneValue  ):
		self.none = noneValue
		return value

	#
	# Letter -- returns letter that corresponds to the given number.
	#
	def Letter( self, value ):
		return chr( ord( 'A' ) + value )

	#
	# Roman -- converts the value into Roman numerals.
	#
	def Roman( self, value ):
		from regsub import sub

		acc = array.array( 'c' )
		for ( char, charVal ) in self.kRomanSet:
			while value >= charVal:
				value = value - charVal
				acc.append( char )

		#
		# Substitute acceptable contractions.
		#
		acc = sub( 'DCCCC', 'CM', acc.tostring() )
		acc = sub( 'CCCC', 'CD', acc )
		acc = sub( 'LXXXX', 'XC', acc )
		acc = sub( 'XXXX', 'XL', acc )
		acc = sub( 'VIIII', 'IX', acc )
		acc = sub( 'IIII', 'IV', acc )
		return acc

	#
	# HtmlEncode -- convert certain characters into HTML equivalents
	#
	def HtmlEncode( self, value ):
		jf = string.joinfields
		sf = string.splitfields
		value = jf( sf( value, '&' ), '&amp;' )
		value = jf( sf( value, '<' ), '&lt;' )
		value = jf( sf( value, '>' ), '&gt;' )
		value = jf( sf( value, '>' ), '&quot;' )
		return value

	#
	# UrlEncode -- convert certain characters into safe HTML variants.
	#
	def UrlEncode( self, value ):
		import urllib
		return urllib.quote( value )

	#
	# ProcDict -- returns our dictionary of formatting functions.
	#
	def ProcDict( self ):
		return self.procs

	#
	# Resolve -- attempts to evaluate given key using the contents of our
	# formatting dictionary and our active environment.
	#
	def Resolve( self, key ):
		return eval( key, self.procs, self.env )

	#
	# __getitem__ -- provides rvalue indexing. Called by Python kernel to
	# resolve name values for `%' formats.
	#
	def __getitem__( self, key ):
		self.none = None
		value = self.Resolve( key )

		#
		# See if there is a replacement value to use.
		#
		if self.none != None and value in self.kNullSet:
			value = self.none

		return value

#
# Class ForCooker -- performs the formatting of `for' blocks. Installs
# attributes in the formatting dictionary so that %() format constructs can
# reference the current index, counter, and value of the iterator, as well as
# statistical values calculated for numeric sequences.
#
class ForCooker:

	#
	# __init__ -- remembers name of iterator attribute and the data to iterate
	# over (must be a sequence or dictionary). Installs ourselves into the
	# given format dictionary under the name of the iterator.
	#
	def __init__( self, formatter, tagName, data ):
		self.tagName = tagName
		self.data = data

		#
		# Add to formatter a reference to ourselves for user access to iterator
		# values and statistics.
		#
		formatter.ProcDict()[ tagName ] = self
		return

	#
	# SetStats -- attempt to calculate various statistical values from the
	# sequence given to us. If a non-numeric value is encountered, the attempt
	# is aborted, and incomplete statistics will result. NOTE: sequence is
	# sorted in-place.
	#
	def SetStats( self, values ):
		import math

		#
		# Create/init statistic attributes. These can be reference within a
		# %() construct with a name of tagName + `.' + statName.
		#
		self.count = 0
		self.sum = 0
		self.sumsq = 0
		self.mean = 0.0
		self.variance = 0.0
		self.variance_n = 0.0
		self.sdev = 0.0
		self.sdev_n = 0.0
		self.min = None
		self.max = None

		count = self.count = len( values )
		if count > 0:

			#
			# Ignore any type/value conflicts.
			#
			try:
				sum = 0;
				sumsq = 0

				#
				# Calculate sum and sum of squares for use in other
				# statistics.
				#
				for each in values:

					#
					# If we have a string, try to make it a number.
					#
					if type( each ) == type( '' ):
						try:
							each = string.atoi( each )
						except ValueError:
							each = string.atof( each )

					#
					# Calculate values
					#
					sum = sum + each
					sumsq = sumsq + each * each

				self.sum = sum
				self.sumsq = sumsq

				mean = self.mean = float( sum ) / count
				mean = mean * mean
				tmp = self.variance_n = float( sumsq ) / count - mean
				self.sdev_n = math.sqrt( tmp )

				if count > 1:
					tmp = self.variance = tmp * count / ( count - 1 )
					self.sdev = math.sqrt( tmp )

			except ( TypeError, ValueError ):
				pass

			#
			# Sort the values so we can calculate the median value and min/max
			# values.
			#
			values.sort()
			self.min = values[ 0 ]
			self.max = values[ count - 1 ]

			#
			# Calculate the median element. Attempt to handle non-numeric
			# elements.
			#
			if count == 1:
				self.median = values[ 0 ]
			else:
				index = count / 2
				self.median = values[ index ]
				if index * 2 == count:
					try:
						self.median = ( values[ index - 1 ] +
										values[ index ] ) / 2
					except TypeError:
						pass

#
# ForSeqCooker -- provides resolution of sequencing variables available within
# a `for' block.
#
class ForSeqCooker( ForCooker ):

	#
	# __init__ -- install attributes for sequences.
	#
	def __init__( self, formatter, tagName, data ):
		ForCooker.__init__( self, formatter, tagName, data )

		#
		# Initialize iterator attributes available inside the blocks
		#
		self.counter = 0
		self.index = 0
		self.value = None

		#
		# Attempt to calculate statistics from the data. NOTE: we make a copy
		# of the sequence because SetStats sorts.
		#
		self.SetStats( list( copy.copy( data ) ) )
		return

	#
	# Cook -- visit the given blocks as many times as there are elements in the
	# sequence we are using. Updates iterator attributes.
	#
	def Cook( self, formatter, sink, blocks ):
		for each in self.data:
			self.value = each
			self.counter = self.counter + 1
			blocks.IterCook( sink, formatter )
			self.index = self.index + 1

#
# ForMapCooker -- provides resolution of sequencing variables available within
# a `for' block.
#
class ForMapCooker( ForCooker ):

	#
	# __init__ -- install attributes for dictionaries.
	#
	def __init__( self, formatter, tagName, data ):
		ForCooker.__init__( self, formatter, tagName, data )

		#
		# Initialize iterator attributes available inside the blocks
		#
		self.counter = 0
		self.key = None
		self.value = None

		#
		# Attempt to calculate statistics from the values found in the
		# dictionary. Unlike ForSeqDict, we don't need to make a copy of this.
		#
		self.SetStats( data.values() )
		return

	#
	# Cook -- visit the given blocks as many times as there are elements in the
	# mapping we are using. Updates iterator attributes.
	#
	def Cook( self, formatter, sink, blocks ):
		data = self.data
		for key in data.keys():
			self.key = key
			self.value = data[ key ]
			self.counter = self.counter + 1
			blocks.IterCook( sink, formatter )

#
# Class BoilerPlate -- abstract base class that combines text block parsing and
# dictionary management for formatting.
#
class BoilerPlate( Block ):

	#
	# __init__ -- start with an empty dictionary of formatting values.
	#
	def __init__( self ):
		self.dict = {}
		self.cooked = None

	#
	# ApplyDicts -- copy given dictionary values into our formatting
	# dictionary. *Internal method*
	#
	def ApplyDicts( self, dict, kw ):
		tmp = self.dict
		if dict:
			for key in dict.keys():
				tmp[ key ] = dict[ key ]

		if kw:
			for key in kw.keys():
				tmp[ key ] = kw[ key ]

	#
	# Add -- copy key/values found in the given dictionary and keywords to our
	# formatting dictionary.
	#
	def Apply( self, dict = None, **kw ):
		self.ApplyDicts( dict, kw )

	#
	# Reset -- forget any formatted (cooked) text
	#
	def Reset( self ):
		self.cooked = None

	#
	# Cook -- format our text by applying whatever dictionary entries we have
	# to our text blocks.
	#
	def Cook( self ):
		if self.cooked == None:

			#
			# Create new sink to hold formatted output
			#
			sink = Sink()

			#
			# Create formatter to do %() construct resolution, apply it to our
			# text blocks and cache the result.
			#
			Block.Cook( self, sink, Formatter( self.dict ) )
			self.cooked = sink.Value()

		return self.cooked

	#
	# Value -- apply contents of given dictionary and keywords, format text,
	# and return string result.
	#
	def Value( self, dict = None, **kw ):

		#
		# If we have new information, forget previous formatted text.
		#
		if dict != None or len( kw ) > 0:
			self.Reset()
			self.ApplyDicts( dict, kw )

		return self.Cook()

	#
	# __call__ -- provide call semantics for instance.
	#
	__call__ = Value

	#
	# __str__ -- potentially format text. Return string result.
	#
	def __str__( self ):
		return self.Cook()

#
# Class String -- class that uses Python string as source for text.
#
class String( BoilerPlate ):

	#
	# __init__ -- process the given text into blocks
	#
	def __init__( self, text, dict = None, **kw ):
		BoilerPlate.__init__( self )
		self.ApplyDicts( dict, kw )
		self.EatText( text, 0 )

#
# Class File -- class that uses a file as source for text.
#
class File( BoilerPlate ):

	#
	# __init__ -- remember name of file to use and load it.
	#
	def __init__( self, name, dict = None, **kw ):
		BoilerPlate.__init__( self )
		self.ApplyDicts( dict, kw )
		self.name = name
		self.Reload()

	#
	# Reload -- cause instance to reload template from file and reparse it.
	# Text is not reformatted but will be when asked for.
	#
	def Reload( self ):

		#
		# Forget any cached formatted text.
		#
		self.Reset()

		#
		# Read file contents
		#
		fd = open( self.name )
		text = fd.read()
		fd.close()

		#
		# Rebuild list of blocks from text.
		#
		self.EatText( text, 0 )

#
# *** TESTING CRUFT ***
#

class Foo:
	def __init__( self, a, b, c ):
		self.name = a
		self.description = b
		self.mode = c

	def CanChange( self ):
		return self.mode

	def Name( self ):
		return self.name

	def Description( self ):
		return self.description

#
# *Incomplete* -- should cause all code above to be visited
#
def RegressionTesting( create = None ):
	if create:
		fd = open( 'regress.cmp', 'w' )
	else:
		fd = open( 'regress.out', 'w' )

	#
	# Test some formatting functions
	#
	a = { 'foo':'HeLlO WoRlD', 'bar':( 1, 2, 3, 4, 5 ), 'z':{ 'a':1, 'b':2 } }
	b = String( '%(Capitalize(foo))s %(Upper(foo))s %(Lower(foo))s', a )()
	fd.write( '*** 1 ***\n%s\n' % b )

	#
	# Look mom! Builtin function in %() construct
	#
	b = String( '%(map(Roman,bar))s', a )()
	fd.write( '*** 2 ***\n%s\n' % b )

	#
	# Test HTML boilerplate
	#
	b = File( 'foo.html',
			  title = 'Hello',
			  header = 'My Header',
			  usingBorders = 1,
			  listing = ( Foo( 'barbie', 'just a doll', 0 ),
						  Foo( 'ken', 'another one', 1 ),
						  Foo( 'muffin', 'something to eat', 0 ),
						  Foo( 'banana', 'yellow', 1 ) ),
			  footer = 'So long<P>' )()

	fd.write( '*** 3 ***\n%s\n' % b )

	#
	# Test sequence iterator
	#
	b = String( '''@for each in bar:
index = %(each.index)d   value = %(each.value)d
@end for
avg = %(each.mean)d''', a )()
	fd.write( '*** 4 ***\n%s\n' % b )

	#
	# Test dictionary iterator
	#
	b = String( '''@for each in z:
key = %(each.key)s   value = %(each.value)d
@end for''', a )()
	fd.write( '*** 5 ***\n%s\n' % b )

	fd.close()

	if not create:
		old = open( 'regress.cmp', 'r' ).read()
		new = open( 'regress.out', 'r' ).read()
		if old != new:
			print 'FAILED'
		else:
			print 'OK'

if __name__ == '__main__':
	RegressionTesting()
