# -------------------------------------------------------------------------
# MODULE:      Window
#
# DESCRIPTION: 
#     Contains the Window class.
#     
# AUTHOR:
#     Per Spilling <per@cwi.nl>, CWI, Amsterdam.

import Xt, Xm, X, Xmd            # Xt & Motif interface

import vp                        # global data & constants for vpApp

from WidgetObject import WidgetObject
from Application  import Application

debug = vp.FALSE

# -------------------------------------------------------------------------
# CLASS:         Window
#
# INHERITS FROM: WidgetObject : TreeNode : Object
#
# DESCRIPTION: 
#     A concrete class for creating windows (top-level-shells). A window
#     can have the following components: 
#
#     - MenuBar     => <window>.AddMenu( <pulldown-menu> )
#     - WorkArea    => <window>.AddWorkArea( <work-area> )
#     - CommandArea => <window>.AddCommandArea( <command-area> )
#     - MessageArea => <window>.AddMessageArea() and/or 
#                      <window>.SetMessage( <string> )
#
#     The 'window_type' can be either a MAIN_WINDOW or a DIALOG_WINDOW. A
#     DIALOG_WINDOW must be explicitly poppped-up with the Show() method.
#
#     The layout of the components can be either CWM (CommandArea:WorkArea:-
#     MessageArea) or MWC (MessageArea:WorkArea:CommandArea).
#

class Window( WidgetObject ):

	widget_class = Xm.MainWindow

	# ------------------------------------------------------------------
	# Initialization & finalization methods

	def __init__( self, argdict = {} ):
		#
		# Args can be:
		# - 'name'           : <string> (<appname>=default)
		# - 'size'           : (w,h)
		# - 'show_separators': <bool> (vp.TRUE=default)
		# - 'window_type'    : [vp.MAIN_WINDOW=default | vp.DIALOG_WINDOW]
		# - 'layout'         : [vp.CWM=default | vp.MWC]
		# - 'focus_policy'   : [vp.EXPLICIT | vp.POINTER=default]
		#
		argdict = self.MergeDefaults( argdict, {
			'name'           : vp.theApplication.GetName(),
			'show_separators': vp.TRUE,
			'window_type'    : vp.MAIN_WINDOW,
			'layout'         : vp.CWM,
			'focus_policy'   : vp.POINTER,
			'size'           : (100, 100)
			})
		WidgetObject.__init__( self, argdict ) 

		self.work_area       = None
		self.message_area    = None
		self.command_area    = None
		self.menu_bar        = None

		# private attributes

		self.separator_height = 0
		self.did_set_size     = vp.FALSE
		self.did_init         = vp.FALSE

		self.Subscribe( vp.RESIZE, self.ResizeEH, None )
		self.SetDefaultXRes()

		vp.theApplication.AddChild( self )  # register with the application



	def SetDefaultXRes( self ):
		#
		# The Window class will create a window without scrollbars. If 
		# scrollbars are needed, then the ScrolledWindow class should be used.
		#
		if self.show_separators:
			self.SetXResources({ 
				  'visualPolicy'              : Xmd.VARIABLE,
				  'initialResourcesPersistent': X.FALSE,
				  'showSeparator'             : X.TRUE 
				  })
		else:
			self.SetXResources({ 
				  'visualPolicy'              : Xmd.VARIABLE,
				  'initialResourcesPersistent': X.FALSE
				  })
			
		
	def CreateWidget( self ):
		#
		# The Window widget is special; it will be implemented as a popup shell
		# off the Application's widget. An XmMainWindow widget will be used
		# to handle the window layout. 
		#
		if debug: print 'Window.CreateWidget called'

		xr = {'keyboardFocusPolicy': self.focus_policy}
		self.top_w = self.parent.GetWidget().CreatePopupShell( self.GetName(),
	                                                   Xt.ApplicationShell, xr)

		self.w = self.top_w.CreateManagedWidget( 'MainWindow', 
			                                     Xm.MainWindow, 
												 self.xresources )
		self.InitWidget()


	def InitWindow( self ):
		size_needed = self.GetNaturalSize()
			
		if not self.did_set_size:
			self.SetSize( size_needed[0], size_needed[1] )
			self.SetNaturalSize( size_needed[0], size_needed[1] )

		self._InstallComponents()
		self._SetComponentSizes()
		self.SetMinSize( self.natural_size[0], self.natural_size[1] )
		self.did_init = vp.TRUE


	def _InstallComponents( self ):
		self.scomp_height = 0  # the height of the static components

		if self.work_area != None:
			if debug:
				print 'Window._InstallComponents: nat. size of workarea =', \
					  self.work_area.GetNaturalSize()

			self.w.workWindow = self.work_area.GetWidget()
			self.work_area.AddCallback('size_changed', self.ChildSizeChangedCB)

		if self.command_area:
			self.command_area.AddCallback( 'size_changed', 
										   self.ChildSizeChangedCB )
			self.scomp_height = self.scomp_height + self._ca_nh

		if self.message_area:
			self.scomp_height = self.scomp_height + self.message_area.w.height

		if self.menu_bar:
			self.w.menuBar = self.menu_bar.GetWidget()
			self.scomp_height = self.scomp_height + self.menu_bar.w.height

		if self.layout == vp.CWM:
			if self.message_area:
				self.w.messageWindow = self.message_area.GetWidget()

			if self.command_area:
				self.w.commandWindow = self.command_area.GetWidget()
		else:
			if self.command_area:
				self.w.messageWindow = self.command_area.GetWidget()

			if self.message_area:
				self.w.commandWindow = self.message_area.GetWidget()


	def _SetComponentSizes( self ):
		(w,h) = self.GetSize() 
		if self.work_area: self.work_area.SetSize( w, h - self.scomp_height )
		if self.command_area: self.command_area.SetSize( w, self._ca_nh )

		if debug:
			print 'Window._SetComponentSizes: tot-size =', w,h, \
				  'scomp_height =', self.scomp_height, \
				  'wa size =', self.work_area.GetSize()


	# ------------------------------------------------------------------
	# Child management methods: 

	def InitAddedNode( self, child ):
		child.AddCallback( 'deleted', self.ChildDeletedCB )
		WidgetObject.InitAddedNode( self, child )

		
	def _AddChild( self, child ): WidgetObject.AddChild( self, child )

	def AddChild( self, child ):
		#
		# This method should not be used
		#
		print 'Window.AddChild: This method should not be used.'
		print 'Use one of the following instead: '
		print '    AddMenu( <menu> )'
		print '    AddMessageArea() or SetMessage( <message> )'
		print '    AddWorkArea( <work_area> )'
		print '    AddCommandArea( <command_area> )'


	def AddMenu( self, menu ):
		if not self.menu_bar:
			from Menu import MenuBar 

			if debug: print 'Window: adding menu_bar'

			self.menu_bar = MenuBar() 
			WidgetObject.AddChild( self, self.menu_bar )
			self.separator_height = self.separator_height + 2

		self.menu_bar.AddChild( menu )


	def AddMessageArea( self ):
		if not self.message_area:
			from MiscGraphic import Label

			if debug: print 'Window: adding message_area'

			self.message_area = Label()
			self.message_area.SetXResources({
				  'alignment'   : Xmd.ALIGNMENT_BEGINNING,
				  'marginWidth' : 8,
				  'marginHeight': 8
				  })
			WidgetObject.AddChild( self, self.message_area )
			self.separator_height = self.separator_height + 2
		else:
			print 'Window.AddMessageArea: message_area already exists!'


	def AddWorkArea( self, work_area ):
		if debug: print 'Window: adding work_area'

		if not self.did_init:
			self.work_area = work_area
			WidgetObject.AddChild( self, work_area )
		else:
			#
			# This can be the case if a work-area is replaced with a new one 
			#
			if self.work_area != None: self.work_area.Finalize()

			self.top_w.UnmanageChild()
			self.work_area = work_area
			WidgetObject.AddChild( self, work_area )
			self.InitWindow()
			self.top_w.ManageChild()


	def AddCommandArea( self, command_area ):
		if debug: print 'Window: adding command_area'

		if not self.command_area:  
			self.command_area   = command_area
			WidgetObject.AddChild( self, command_area )
			self.separator_height = self.separator_height + 2
		else:
			print 'Window.AddCommandArea: command_area already exists!' 
	

	def ChildSizeChangedCB( self, child ):
		ns = self.GetNaturalSize()
		self.SetMinSize( ns[0], ns[1] )

		self.w.UnmanageChild()

		if ns[0] > self.GetWidth() or ns[1] > self.GetHeight():
			self.SetSize( ns[0], ns[1] )

		if hasattr( child, 'DoLayout' ): child.DoLayout()

		if debug: 
			print 'Window.ChildSizeChangedCB:', \
				  'wa ns =', self.work_area.GetNaturalSize(),\
				  'win size =', self.GetSize()

		self.w.ManageChild()


	def ChildDeletedCB( self, child ):
		if debug: print 'Window.ChildDeletedCB: child =', child.GetName()

		if child == self.work_area: self.work_area = None
		elif child == self.command_area: self.command_area = None
		elif child == self.message_area: self.message_area = None


	# ------------------------------------------------------------------
	# Misc. public methods:

	def SetMessage( self, message ):
		if not self.message_area: self.AddMessageArea()
		self.message_area.SetName( message )


	# ------------------------------------------------------------------
	# Notify methods:

	def GetNaturalSize( self ):
		w = 0
		h = 0

		if self.work_area:
			ns = self._GetNaturalSizeWorkArea()
			self._wa_nh = ns[1]

			if debug: print 'Natural size of work_area =', ns

			w = max(w, ns[0])
			h = h + ns[1] + 2  # give the workarea a little extra height

		if self.menu_bar:
			ns = self.menu_bar.GetNaturalSize()

			if debug: print 'Natural size of menu_bar =', ns

			w = max(w, ns[0])
			h = h + ns[1]

		if self.command_area:
			ns = self.command_area.GetNaturalSize()
			self._ca_nh = ns[1]

			if debug: print 'Natural size of command_area =', ns

			w = max(w, ns[0])
			h = h + ns[1]
			
		if self.message_area:
			ns = self.message_area.GetNaturalSize()

			if debug: print 'Natural size of message_area =', ns

			w = max(w, ns[0])
			h = h + ns[1]
			
		h = h + self.separator_height
		(self_w,self_h) = WidgetObject.GetNaturalSize( self )

		if self.work_area != None and self_h > h:
			self._wa_nh = self._wa_nh + (self_h - h)

		self.natural_size = (max( w, self_w) , max( h, self_h ))
		return self.natural_size


	def _GetNaturalSizeWorkArea( self ):
		#
		# This method is used so that it can be overridden in subclasses.
		#
		return self.work_area.GetNaturalSize()


	# ------------------------------------------------------------------
	# Event handler method:

	def ResizeEH( self, target, void, xevent, void ):
		if self.w.width != self.GetWidth() or \
			  self.w.height != self.GetHeight():
			if debug:
				print 'Window.ResizeEH: old size =', self.GetSize(), \
					  'new size =', self.w.width, self.w.height
			WidgetObject.ResizeEH( self, target, void, xevent, void )

			if self.work_area != None:
				self.work_area.ResizeEH( target, void, xevent, void )

			if self.command_area != None:
				self.command_area.ResizeEH( target, void, xevent, void )


	# ------------------------------------------------------------------
	# "Draw" methods:

	def Show( self ):
		if debug: print 'Window.Show called'

		self.visible = vp.TRUE
		if not self.did_init: 
			self.InitWindow()
		self.top_w.Popup(0)


	def Hide( self ):
		if debug: print 'Window.Hide called'

		self.top_w.Popdown()
		self.visible = vp.FALSE


	# ------------------------------------------------------------------
	# Geometry methods:

	def SetSize( self, width, height ):
		#
		# Must resize the toplevel shell as well 
		#
		if self.IsRealized():
			self.top_w.width = width
			self.top_w.height = height

		WidgetObject.SetSize( self, width, height )

		if self.IsRealized() and self.work_area != None:
			self.work_area.SetSize( width, height - self.scomp_height )

			if self.command_area != None:
				self.command_area.SetSize( width, self._ca_nh )

		self.did_set_size = vp.TRUE


	def SetMinSize( self, width, height ):
		#
		# Set the 'minWidth' and 'minHeight' resources of the ApplicationShell
		# widget. This will make it impossible to resize the window to a 
		# smaller size than the natural size.
		#
		self.top_w.minWidth  = self.natural_size[0]
		self.top_w.minHeight = self.natural_size[1]



# -------------------------------------------------------------------------
# CLASS:         ScrolledWindow
#
# INHERITS FROM: Window : WidgetObject : Graphic : (TreeNode,DatumDict) : \
#                Object
#
# DESCRIPTION: 
#

class ScrolledWindow( Window ):

	# ------------------------------------------------------------------
	# Initialization & finalization methods

	def SetDefaultXRes( self ):
		#
		# The ScrolledWindow class will create a window with scrollbars.
		#
		self.SetXResources({
			  'scrollingPolicy'       : Xmd.AUTOMATIC,
			  'scrollBarDisplayPolicy': Xmd.AS_NEEDED,
			  'showSeparator'         : X.TRUE
			  })
		

	def _GetNaturalSizeWorkArea( self ): 
		#
		# Since the window has scrollbars we do not have to consider the 
		# size of the workarea.
		#
		return ( 10, 10 )


	def _SetComponentSizes( self ):
		if self.command_area:
			self.command_area.SetSize( self.width.value, self._ca_nh )


	# ------------------------------------------------------------------
	# Notify methods:

	def ResizeEH( self, target, void, xevent, void ):
		#
		# Do not resize the work_area as in the Window class
		#
		if self.w.width != self.GetWidth() or \
			  self.w.height != self.GetHeight():
			WidgetObject.ResizeEH( self, target, void, xevent, void )

			if self.command_area:
				self.command_area.ResizeEH( target, void, xevent, void )


# -------------------------------------------------------------------------
# CLASS:         PanedWindow
#
# INHERITS FROM: Window : WidgetObject : Graphic : (TreeNode,DatumDict) : \
#                Object
#
# DESCRIPTION: 
#

from MiscGraphic import PanedViewPort, ScrolledViewPort

class PanedWindow( Window ):
	
	# ------------------------------------------------------------------
	# Initialization & finalization methods

	def __init__( self, argdict = {} ):
		Window.__init__( self, argdict )

		# the work-area will be managed by a PanedViewPort

		self.AddWorkArea( PanedViewPort() )


	def AddWorkArea( self, work_area ):
		if self.work_area == None:
			Window.AddWorkArea( self, work_area )
		else:
			self.work_area.AddChild( work_area )

