# -------------------------------------------------------------------------
# MODULE:      WidgetObject
#
# DESCRIPTION: 
#     Contains the WidgetObject class.
#
# USAGE:      
#     > from WidgetObject import WidgetObject
# 
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, per@cwi.nl

import X
import vp, EventNotifier

from vp      import TRUE, FALSE
from Graphic import Graphic
from Datum   import *

# -------------------------------------------------------------------------
# CLASS:         WidgetObject
#
# INHERITS FROM: Graphic : TreeNode : Object
#
# DESCRIPTION: 
#     Base-class for all graphical objects which can be implemented with 
#     (X-windows) widgets.
#

class WidgetObject( Graphic ):

	# ------------------------------------------------------------------
	# Class attributes.

	widget_class = None
	_GetWcBBox   = Graphic.GetBBox
	_SetWcBBox   = Graphic.SetBBox

	# ------------------------------------------------------------------
	# Initialisation.

	def __init__( self, argdict = {} ):
		if hasattr( self, 'w' ):
			return

		self.w = None
		Graphic.__init__( self, argdict )

		self.Subscribe( vp.WIDGET_DESTROYED, self.WidgetDestroyed, None )


	def AddDatums( self ):
		#
		# Explanation of X-widget geometry resources:
		# -------------------------------------------
		# 'x'           = x-coordinate of the upper-left outside corner 
		#                 (outside border)
		# 'y'           = y-coordinate of the upper-left outside corner
		# 'width'       = inside width (excluding the border)
		# 'height'      = inside height
		# 'borderWidth' = the width of the border that surrounds the widget
		#                 on all four sides
		#
		# In this method it is assumed that 'borderWidth' and 'borderHeight' 
		# are both 0. 
		#
		if debug: print 'WidgetObject.AddDatums called for', self.GetName()

		self.AddDatum('wc_x',   Datum(0)) 
		self.AddDatum('wc_y',   Datum(0))
		self.AddDatum('x',      ExternalDatum( XResourceShell('x', self)))
		self.AddDatum('y',      ExternalDatum( XResourceShell('y', self)))
		self.AddDatum('_width', ExternalDatum( XResourceShell('width', self)))
		self.AddDatum('_height',ExternalDatum( XResourceShell('height', self)))
		self.width  = self._width
		self.height = self._height


	def _InstallExtraBBoxDatums( self, bw ):
		#
		# This method is used to install extra datums when the 'borderWidth'
		# is not equal to zero. The method assumes that the 'AddDatums' method
		# has already been called. (that should be done in Graphic.__init__).
		# 
		self.AddDatum('borderWidth', 
			  ExternalDatum( XResourceShell('borderWidth', self)))
		self.borderWidth.Set( bw )
		
		self.width  = Add( self._width, Multiply( 2, self.borderWidth ))
		self.height = Add( self._height, Multiply( 2, self.borderWidth ))


	def Realize( self ):
		if debug: print 'WidgetObject.Realize: realizing', self.GetName()

		self.CreateWidget()
		for child in self.children: 
			child.Realize()
		Graphic.Realize( self )   # actually Responder.Realize
		

	def CreateWidget( self ):
		if debug: 
			print 'WidgetObject.CreateWidget called for', self.GetName(), \
				  'parent =', self.parent

		self.w = self.parent.GetWidget().CreateWidget( self.GetName(), 
													   self.widget_class,
													   self.xresources )
		if self.IsVisible(): self.w.ManageChild()
		self.InitWidget()



	def InitWidget( self ):
		self.realized = TRUE
		self.InitXResShells( self.w )  

		if self.natural_size[0] == 0:
			self.natural_size = (self.width.value, self.height.value)
		else:
			self.SetSize( self.natural_size[0], self.natural_size[1] )

		if self.editmode == TRUE: self._SetEditMode( TRUE )
		if self.hilight == TRUE: self.SetHilight( TRUE )


	def GetAttrKeys( self ):
		if self.IsRealized():
			xres_keys = self.w.Class().GetResourceList()
		elif self.widget_class != None:
			xres_keys = self.widget_class.GetResourceList()
		else:
			xres_keys = []

		keys = []
		for item in xres_keys:
			if type(item) == type(()):
				keys.append( item[0] )
			else:
				keys.append( item )

		return Graphic.GetAttrKeys(self) + keys


	# ------------------------------------------------------------------
	# Destruction.

	def Finalize( self ):
		if debug_final:
			print 'WidgetObject.Finalize called for', self.GetClassName()

		Graphic.Finalize( self )

		if self.w != None:
			self.w.DestroyWidget()
			self.w = None


	# ------------------------------------------------------------------
	# 'Drawing' methods:

	def Show(self):               # make visible (manage, popup)
		Graphic.Show( self )
		if self.IsVisible() and self.w and not self.w.IsManaged():
			self.w.ManageChild()


	def Hide(self):               # make invisible (unmanage, popdown)
		Graphic.Hide( self )
		if not self.IsVisible() and self.w and self.w.IsManaged():
			self.w.UnmanageChild()


	def ShowControlPoints( self ):
		Graphic.ShowControlPoints( self )

		if hasattr( self.parent, 'AddToDisplayList' ):
			self.parent.AddToDisplayList( self )


	def HideControlPoints( self ):
		Graphic.HideControlPoints( self )

		if hasattr( self.parent, 'RemoveFromDisplayList' ):
			self.parent.RemoveFromDisplayList( self )
		

	# ------------------------------------------------------------------
	# Geometry methods: (overridden to reduce the number of redraws)

	def Translate( self, dx, dy ):
		self.parent.RedrawOff()
		Graphic.Translate( self, dx, dy )
		self.parent.RedrawOn()
		

	# ------------------------------------------------------------------
	# Datum callback methods:

	def RedrawCB( self, value ): self.Redraw()

	# ------------------------------------------------------------------
	# Event handler methods:

	def ResizeEH( self, target, void, xevent, void ):
		#
		# Must update the datums that are dependent on the size of the widget
		#
		if debug:
			print 'WidgetObject.Resize called for ', self.GetName(), \
				  'old size =', self.w.width, self.w.height

		self.width.UpdateInternal()
		self.height.UpdateInternal()


	def Mouse1DragEH( self, target, client_data, xevent, e ):
		if debug:
			print 'WidgetObject.Mouse1DragEH called for', self.GetName()

		if self.show_cntrl_pnts and self._cp != 0:        # resize bbox
			if e == None: e = xevent      # XButtonEvent

			if self._cp == 1:                             # upper-left
				bb = self.GetBBox()
				if e.x > bb[2]: e.x = bb[2]
				if e.y > bb[3]: e.y = bb[3]
				self.SetBBox( e.x, e.y, bb[2], bb[3] )
			elif self._cp == 2:                           # upper-right
				ll = (self.x.value, self.y.value+self.height.value)
				if e.x < ll[0]: e.x = ll[0]
				if e.y > ll[1]: e.y = ll[1]
				self.SetBBox( ll[0], e.y, e.x, ll[1] )
			elif self._cp == 3:                           # lower-right
				bb = self.GetBBox()
				if e.x < bb[0]: e.x = bb[0]
				if e.y < bb[1]: e.y = bb[1]
				self.SetBBox( bb[0], bb[1], e.x, e.y )
			elif self._cp == 4:                           # lower-left
				ur = (self.x.value+self.width.value, self.y.value)
				if e.x > ur[0]: e.x = ur[0]
				if e.y < ur[1]: e.y = ur[1]
				self.SetBBox( e.x, ur[1], ur[0], e.y )
				
			self.parent.Redraw()                          # redraw prims


	# ------------------------------------------------------------------
	# Modify methods

	def SetEditMode( self, bool ):
		if self.editmode != bool:
			self.editmode = bool
			if self.IsRealized(): self._SetEditMode( bool )


	def _SetEditMode( self, bool ):
		if bool == TRUE:
			try:
				#
				# The events for the control points will be sent to the 
				# parent. By calling 'parent.SubscribePrimitive' the
				# parent will know that this object's 'IsPicked' method
				# must be called to check if any of its control points has
				# been picked.
				#
				self.Subscribe( vp.MOUSE_DOWN, self.MouseDownEH, None )
				self.parent.SubscribePrimitive( self, vp.MOUSE_DOWN )
				self.Subscribe( vp.MOUSE1_DRAG, self.Mouse1DragEH, None )
				self.parent.SubscribePrimitive( self, vp.MOUSE1_DRAG )
				self.Subscribe( vp.MOUSE_UP, self.MouseUpEH, None )
				self.parent.SubscribePrimitive( self, vp.MOUSE_UP )
			except:
				print 'WidgetObject.SetEditMode: a WidgetObject can', \
					  'only be edited if its parent is a Composite!'
		else:
			if self.show_cntrl_pnts: self.HideControlPoints()
			try:
				self.UnSubscribe( vp.MOUSE_DOWN )
				self.parent.UnSubscribePrimitive( self, vp.MOUSE_DOWN )
				self.UnSubscribe( vp.MOUSE1_DRAG )
				self.parent.UnSubscribePrimitive( self, vp.MOUSE1_DRAG )
				self.UnSubscribe( vp.MOUSE_UP )
				self.parent.UnSubscribePrimitive( self, vp.MOUSE_UP )
			except:
				print 'WidgetObject.SetEditMode: a WidgetObject can', \
					  'only be edited if its parent is a Composite!'
		
		
	def SetXResources( self, xres_dict ):
		for res in xres_dict.keys():
			val = xres_dict[res]
			if res in ('foreground','background') and type(val) == type(''):
				#
				# Convert color
				#
				val = vp.theApplication.w.Convert(val,'Pixel')

			elif res == 'borderWidth':
				#
				# Must change the datums dealing with the bbox
				#
				if not hasattr( self, 'borderWidth' ):
					self._InstallExtraBBoxDatums( val )

			self.xresources[res] = xres_dict[res] = val
			
		if self.IsRealized():
			self.w.SetValues( xres_dict )
			self.UpdateXResShells()


		
	def SetHiliXRes( self, res, val ):
		if not hasattr( self, 'hili_xres'):
			self.hili_xres = {}

		if type(val) == type('') and \
			  (res == 'foreground' or res == 'background'):
			#
			# Convert color
			#
			self.hili_xres[res] = vp.theApplication.w.Convert(val,'Pixel')
		else:
			self.hili_xres[res] = val

		if self.IsRealized() and self.hilight == TRUE:
			if res not in self.xresources.keys():
				self.xresources[res] = getattr(self.w, res)

			setattr(self.w, res, val)

			# Notify the datums that depend on the Xresources

			self.UpdateXResShells()


	def SetHilight( self, bool ):
		self.hilight = bool
		
		if self.IsRealized():
			if bool == TRUE:
				if not hasattr( self, 'hili_xres'):
					self.hili_xres = {}
					self.hili_xres['foreground'] = self.w.background
					self.hili_xres['background'] = self.w.foreground

				for res in self.hili_xres.keys():
					if res not in self.xresources.keys():
						self.xresources[res] = getattr(self.w, res )
					self.w.SetValues( self.hili_xres )
			else:
				self.w.SetValues( self.xresources )


	# ------------------------------------------------------------------
	# Access methods

	def GetWidget( self ): return self.w

	
	# ------------------------------------------------------------------
	# Query methods:

	def IsPicked( self, x, y ):
		#
		# This method is only used to check if the point is on one of the 
		# control points of the WidgetObject.
		#
		if self.show_cntrl_pnts: 
			return self._IsControlPointPicked( x, y, self.GetBBox() )
		else:
			return FALSE
		
