# -------------------------------------------------------------------------
# MODULE:      TreeMV
#
# DESCRIPTION: 
#     Contains model and view classes for trees.
#
# AUTHOR:
#     Per Spilling <per@cwi.nl> and Dirk Soede <soede@cwi.nl>, CWI, Amsterdam.

import vp

from ModelView         import Model, View, EditorView 
from OrnatedTreeNode   import OrnatedTreeNode
from NestableComposite import NestableCanvas
from WidgetButton      import PushButton
from GraphicLink       import ArrowLink
from Menu              import PopupMenu

debug       = vp.FALSE
debug_mv    = vp.FALSE
debug_final = vp.FALSE

# -------------------------------------------------------------------------
# CLASS:      TreeEditorView
#
# DESCRIPTION:  
#     An editor-view class for TreeNodeViews.
#   

class TreeEditorView( EditorView ):

	# ------------------------------------------------------------------
	# Initialisation.

	def __init__( self, model, view_category ):
		EditorView.__init__( self, model, view_category )

		# attributes for doing the layout

		self.tree_size    = (0,0)
		self.tree_pos     = (0,0)
		self.border_width = 10
		self.vgap         = 10
		self.hgap         = 40
		self.outline      = None
		self.first_expose = vp.TRUE
		self.do_layout    = vp.TRUE
		
		# Install change handlers

		self.InstallChangeHandler( 'node', self.DoLayoutCH )
		self.InstallChangeHandler( 'label', self.DoLayoutCH )
		self.InstallChangeHandler( 'realize', self.DoLayoutCH )


	def CreateGraphic( self ):
		graphic = EditorView.CreateGraphic( self )
		graphic.Subscribe( vp.EXPOSE, self.ExposeEH, None )
		return graphic


	def Finalize( self ):
		if self.edit_object_view != None: self.edit_object_view.Finalize()
		EditorView.Finalize( self )

		
	# ------------------------------------------------------------------
	# Model-View change-handlers

	def DoLayoutCH( self, data ):
		if debug: print 'DoLayoutCH:', self.mv_view_category
		if self.IsRealized(): self.DoLayout()


	# ------------------------------------------------------------------
	# Event handlers: 
	
	def ExposeEH( self, target, client_data, xevent, e ):
		if self.first_expose:
			self.first_expose = vp.FALSE
			self.DoLayout()

		target.ExposeEH( target, client_data, xevent, e )


	# ------------------------------------------------------------------
	# Layout methods:

	def CalculateNaturalSize( self ):
		self.tree_size = (0, 0)
		self.tree_size = self.edit_object_view.CalculateSize()
		w = self.tree_size[0] + (2*self.border_width) + 1
		h = self.tree_size[1] + (2*self.border_width) + 1
		return (w,h)
		
	
	def DoLayout( self ):
		if debug: print 'TreeEditorView.DoLayout: Start'

		(w,h) = self.CalculateNaturalSize()

		# Increasing own size can lead to problems if parent does not
		# agree... This works however if the parent is a ScrollView.

		tree_pos = (self.border_width, self.border_width )
		self.edit_object_view.SetTreePosition( tree_pos[0], tree_pos[1] )

		if debug:
			print 'TreeEditorView.DoLayout: tree size =', w,h

		if w > self.mv_graphic.GetWidth() or h > self.mv_graphic.GetHeight():
			self.mv_graphic.SetNaturalSize( w, h )

		if debug:
			print 'TreeEditorView.DoLayout: End'


	# ------------------------------------------------------------------
	# Draw methods:

	def DrawTreeOutline( self ):
		if self.outline == None:
			from Rectangles import Rectangle

			self.outline = Rectangle( {
				  'place': ( self.tree_pos[0], self.tree_pos[1] ),
				  'size':  ( self.tree_size[0], self.tree_size[1] ),
				  } )
			self.AddChild( self.outline )
		else:
			self.outline.SetSize( self.tree_size[0], self.tree_size[1] )

	
	# ------------------------------------------------------------------
	# Access methods:

	def GetVGap( self ): return self.vgap


	def GetHGap( self ): return self.hgap



# -------------------------------------------------------------------------
# CLASS:      TreeNodeModel
#
# DESCRIPTION:
#       The data structure is a treenode supplemented with graph links
#       between arbitrary nodes.
#   

class TreeNodeModel( OrnatedTreeNode, Model ):

	# ------------------------------------------------------------------
	# Initialisation.

	def __init__( self, editor, name ):
		self.name       = name
		OrnatedTreeNode.__init__( self )
		Model.__init__( self, editor )

		# The OrnatedTreeNode maintains a list of inlinks and outlinks.
		# These are lists of other nodes.


	def Finalize( self ):
		if debug_final:
			print 'TreeNodeModel.Finalize called for', self.GetName()

		OrnatedTreeNode.Finalize( self )
		Model.Finalize( self )


	# ------------------------------------------------------------------
	# Model-View 'action' methods, i.e. methods which lead to change 
	# notification.

	def InitAddedNode( self, child ):
		OrnatedTreeNode.InitAddedNode( self, child )
		child.NotifyChange( 'parent', self )  
		child.SynchronizeViewRealizations()
		self.mv_editor.NotifyChange( 'node', None )


	def RemoveChild( self, child ):
		OrnatedTreeNode.RemoveChild( self, child )
		self.mv_editor.NotifyChange( 'node', None )


	def AddLink( self, other ):
		#
		# The OrnatedTreeNode maintains a list of inlinks and outlinks
		#
		OrnatedTreeNode.AddLink( self, other )
		self.NotifyChange( 'link_added', other )


	def SetName( self, string ):
		OrnatedTreeNode.SetName( self, string )
		self.NotifyChange( 'label', string )          # notify the views
		self.mv_editor.NotifyChange('label', None)    # will cause re-layout

			
	# ------------------------------------------------------------------
	# View->Model methods:

	def GetChildViews( self, view_category ):
		#
		# This method assumes that all children have the same view categories
		#
		child_views = []
		if self.children != None:
			for child in self.children:
				child_views.append( child.mv_views[view_category] )
		return child_views



# -------------------------------------------------------------------------
# CLASS:         AbstractTreeNodeView
#
# INHERITS FROM: View : Object
#
# DESCRIPTION:  
#     An abstract view class which defines the protocol (and some 
#     functionality) for all kinds of tree-node-views.
#

class AbstractTreeNodeView(  View ):

	# ------------------------------------------------------------------
	# Creation:

	def __init__( self, tree_node_model, view_category ):
		View.__init__( self, tree_node_model, view_category )

		# Install change handlers

		self.InstallChangeHandler( 'label', self.NameCH )


	def SynchronizeRealization( self ):
		#
		# Bring realization in accord with realization status of the
		# realization reference view.
		#
		a = self.GetSynchronizeAction()

		if a == -1: self.UnRealize()
		elif a == 1: self.Realize()
			

	def GetSynchronizeAction( self ):  
		# 
		#  0 means do nothing
		#  1 means realize
		# -1 means unrealize
		#
		r = 0

		if self.mv_model.IsRoot():
			ref_view = self.GetEditorView()
		else:
			ref_view = self.GetParentView()

		if ref_view.IsRealized() != self.IsRealized():
			if self.IsRealized():
				r = -1
			else:
				r = 1

		return r
			

	def Realize( self ):
		#
		# Default implementation
		#
		View.Realize( self )
		for child_view in self.GetChildViews(): 
			child_view.Realize()
		

	def UnRealize( self ):
		#
		# Default implementation
		#
		for child in self.GetChildViews():
			if child.IsRealized(): child.UnRealize()
		self.MVDeleteGraphic()


	# ------------------------------------------------------------------
	# Model-View change-handlers

	def NameCH( self, new_name ):
		if self.mv_graphic != None: self.mv_graphic.SetName( new_name )


	# ------------------------------------------------------------------
	# Misc public methods:

	def GetChildViews( self ):
		#
		# Return the child-views of the same category
		#
		return self.mv_model.GetChildViews( self.mv_view_category )


	def GetParentView( self ):
		#
		# Return the parent-view of the same category
		#
		parent_model = self.mv_model.GetParent()
		if parent_model != None:
			parent_view = parent_model.GetView( self.mv_view_category )
		else:
			parent_view = None
		return parent_view


	def GetTreeNodeModel( self, mv_graphic ): 
		return mv_graphic.mv_view.mv_model


	def GetTreeNodeView( self, mv_graphic ): return mv_graphic.mv_view


	# ------------------------------------------------------------------
	# Callbacks for user input. ('class methods')

	def EditNameCB( self, menu_item ):
		if not hasattr( vp, 'thePromptDialog'):
			  from MiscDialogs import PromptDialog
			  vp.thePromptDialog = PromptDialog()

		vp.thePromptDialog.Post( 'Enter new name:',
			                     self._EditNameCB, None, None)


	def _EditNameCB( self, prompt_dialog ):
		new_label = prompt_dialog.GetText()
		self.GetTreeNodeModel( vp.theCurrentPopupNode ).SetName( new_label )
		

	def AddChildCB( self, menu_item ):
		if not hasattr( vp, 'thePromptDialog'):
			  from MiscDialogs import PromptDialog
			  vp.thePromptDialog = PromptDialog()

		vp.thePromptDialog.Post( 'Enter name:',
			                     self._AddChildCB, None, None)


	def _AddChildCB( self, prompt_dialog ):
		#
		# This method must be overridden if the arguments to the 
		# constructor change in the subclass.
		#
		node = self.mv_model.__class__( self.mv_model.mv_editor, 
			                            prompt_dialog.GetText() )
		self.GetTreeNodeModel( vp.theCurrentPopupNode ).AddChild( node )


	def RemoveNodeCB( self, menu_item ):
		node = self.GetTreeNodeModel(vp.theCurrentPopupNode)
		node.Finalize()   # will remove itself from the tree
		node = None



# -------------------------------------------------------------------------
# CLASS:         NestableTreeNodeView
#
# INHERITS FROM: AbstractTreeNodeView : View : Object
#
# DESCRIPTION:  
#     NestableTreeNodeView presents a nested tree-view.
#

class NestableTreeNodeView( AbstractTreeNodeView ):

	# ------------------------------------------------------------------
	# Creation and finalization methods: 

	def __init__( self, tree_node_model, view_category ):
		AbstractTreeNodeView.__init__( self, tree_node_model, view_category )

		self.graphic_iconified = vp.FALSE


	def GetSynchronizeAction( self ):  
		# 
		#  0 means do nothing
		#  1 means realize
		# -1 means unrealize
		#
		r = 0
		if self.mv_model.IsRoot():
			ref_view = self.GetEditorView()
		else:
			ref_view = self.GetParentView()

		if ref_view.IsRealized() != self.IsRealized():
			if self.IsRealized():
				r = -1
			else:
				r = 1
		return r


	def CreateMainMenu( self ):
		#
		# Called by NestableRectangle (see NestableTreeNodeView.CreateGraphic)
		#
		return PopupMenu({
			'title':   'Actions:',
			'items':  [
				('Iconify',       self.mv_graphic.IconifyCB),
				('Shrink to fit', self.mv_graphic.ShrinkToFitCB),
				('Add child',     self.AddChildCB),
				('Remove node',   self.RemoveNodeCB)
				] 
			})


	def CreateIconMenu( self ):
		#
		# Called by NestableRectangle (see NestableTreeNodeView.CreateGraphic)
		#
		return PopupMenu({
			'title':   'Actions:',
			'items':  [
				('Open',         self.mv_graphic.RestoreCB),
				('Edit name',    self.EditNameCB),
				('Remove node',  self.RemoveNodeCB)
				] 
			})

		
	def CreateGraphic( self ):
		#
		# Called by View.MVCreateGraphic
		# 
		if debug: print 'NestableTreeNodeView.CreateGraphic called'

		graphic = NestableCanvas({ 'name': self.mv_model.GetName() })
		graphic.icon.mv_view = self   # to enable MV access  

		# let the this class define the menus

		graphic.CreateMainMenu = self.CreateMainMenu
		graphic.CreateIconMenu = self.CreateIconMenu

		graphic.AddCallback( 'iconified', self.GraphicIconifiedCB )
		graphic.AddCallback( 'restored', self.GraphicRestoredCB )

		if not self.visible:
			graphic.Hide()

		return graphic


	def GetGraphicParent( self ):
		#
		# The parent of the top graphic is the enclosing editor-view-graphic, 
		# while the others use the parent-view-graphic.
		#
		if self.mv_model.IsRoot():
			return self.GetEditorView().GetGraphic()
		else:
			return self.GetParentView().GetGraphic()
			
		
	# ------------------------------------------------------------------
	# Graphic->View callbacks

	def GraphicIconifiedCB( self, graphic ): self.graphic_iconified = vp.TRUE


	def GraphicRestoredCB( self, graphic ): self.graphic_iconified = vp.FALSE



# -------------------------------------------------------------------------
# CLASS:         TreeNodeView
#
# INHERITS FROM: AbstractTreeNodeView : View : Object
#
# DESCRIPTION:  
#     TreeNodeView presents a standard hierarchical tree-view.
#

class TreeNodeView( AbstractTreeNodeView ):

	# ------------------------------------------------------------------
	# Class attributes:

	node_menu = None # shared menu for all TreeNodeViews

	# ------------------------------------------------------------------
	# Creation and finalization methods: 
		
	def __init__( self, tree_node_model, view_category ):
		AbstractTreeNodeView.__init__( self, tree_node_model, view_category )

		# attributes for doing the layout

		self.depth         = 0
		self.tree_size     = (0,0)
		self.tree_pos      = (0,0)
		self.sec_outlinks  = []         # list of secondary link nodes
		self.graphic_links = []         # TreeLinks

		# Install change handlers

		self.InstallChangeHandler( 'label', self.NameCH )
		self.InstallChangeHandler( 'link_added', self.LinkAddedCH )


	def CreateGraphic( self ):
		#
		# Called by View.__init__. The mv_graphic will be added to its parent
		# in the ParentCH method.
		#
		if TreeNodeView.node_menu == None:
			TreeNodeView.node_menu = self.CreateNodeMenu()
			
		button = PushButton({ 'name': self.mv_model.GetName() })
		button.SetXResources({
			  'borderColor'    : 'black',
			  'borderWidth'    : 1,
			  'background'     : 'Azure',
			  'shadowThickness': 0
			  })
		button.AddPopup( self.node_menu ) # add a node-menu for user input

		if not self.visible:
			button.Hide()

		return button


	def CreateNodeMenu( self ):
		from Menu import PopupMenu
		return PopupMenu({
			'title':   'Node actions',
			'items':  [
				('Hide children', self.HideChildrenCB),
				('Show children', self.ShowChildrenCB),
				('Add child',     self.AddChildCB),
				('Remove node',   self.RemoveNodeCB),
				('Edit name',     self.EditNameCB)
				] 
			})


	def Finalize( self ):
		#
		# All links are contained in the 'links' list
		#
		if debug_final:
			print 'TreeNodeView.Finalize called for', self.GetClassName()

		for link in self.graphic_links[:]:   # make a tmp copy of the list
			link.Finalize()
			del link

		self.graphic_links = None
		View.Finalize( self )  # The View class should take care of the rest


	def Realize( self ):
		if debug_mv: 
			print 'TreeNodeView.Realize called for', self.mv_model.GetName()

		self.MVCreateGraphic()

		# save the editor-view as an attribute since it will be accessed a lot

		self.editor_view = self.GetEditorView()
		self.GetGraphicParent().AddChild( self.GetGraphic() )

		# create link to parent

		if not self.mv_model.IsRoot(): self._CreateParentLinkGraphic()

		# realize children views

		for child_view in self.GetChildViews(): child_view.Realize()

		# create secondary links and let the editor-view do the layout

		if self.mv_model.IsRoot():
			self._CreateSecondaryLinks()
			if hasattr( self.editor_view, 'DoLayout' ):
				self.editor_view.DoLayout()


	def _CreateParentLinkGraphic( self ):
		vc = self.mv_view_category
		link = TreeLink({
			'head_length'  : 15,
			'head_width'   : 8,
			'begin_graphic': self.GetParentView().GetGraphic(), 
			'end_graphic'  : self.GetGraphic()
			})
		self.GetGraphicParent().AddChild( link )


	def _CreateSecondaryLinkGraphic( self, other_node ):
		vc = self.mv_view_category
		link = TreeLink({
			'head_length'  : 15,
			'head_width'   : 8,
			'xres'         : {'foreground': 'azure' },
			'begin_graphic': self.GetGraphic(), 
			'end_graphic'  : other_node.GetView(vc).GetGraphic()
			})
		self.GetGraphicParent().AddChild( link )


	def _CreateSecondaryLinks( self ):
		for node in self.sec_outlinks:
			self._CreateSecondaryLinkGraphic( node )

		for child_view in self.GetChildViews():
			child_view._CreateSecondaryLinks()


	def UnRealize( self ):
		for link in self.graphic_links[:]:   # make a tmp copy of the list
			link.Finalize()

		AbstractTreeNodeView.UnRealize( self )


	# ------------------------------------------------------------------
	# Callbacks for user input.

	def HideChildrenCB( self, menu_item ):
		self.editor_view.GraphicUpdateOff()
		self.GetTreeNodeView( vp.theCurrentPopupNode ).HideChildren()
		self.editor_view.DoLayout()
		self.editor_view.GraphicUpdateOn()


	def ShowChildrenCB( self, menu_item ):
		self.editor_view.GraphicUpdateOff()
		self.GetTreeNodeView( vp.theCurrentPopupNode ).ShowChildren()
		self.editor_view.DoLayout()
		self.editor_view.GraphicUpdateOn()


	# ------------------------------------------------------------------
	# Model-View change-handlers

	def LinkAddedCH( self, other_node ):
		#
		# Adds 'secondary' links (f.ex. in case of multiple inheritance). The
		# TreeLink, which is a graphic, will add itself to the 'links' list in
		# both the begin-node and the end-node.
		#
		if debug_mv:
			print 'TreeNodeView.LinkAddedCH called for',self.mv_model.GetName()

		if self.IsRealized():
			self._CreateSecondaryLinkGraphic( other_node )
		else:
			self.sec_outlinks.append( other_node ) 

		
	# ------------------------------------------------------------------
	# 'Draw' methods:

	def HideChildren( self ):
		ch_views = self.GetChildViews()
		for child in ch_views:
			child.HideChildren()
			child.Hide()

		if len(ch_views) != 0:
			self.mv_graphic.SetXResources({
				  'background': 'Black',
				  'foreground': 'Azure'
				  })


	def ShowChildren( self ):
		ch_views = self.GetChildViews()
		for child in ch_views:
			child.ShowChildren()
			child.Show()

		if len(ch_views) != 0:
			self.mv_graphic.SetXResources({
				  'background': 'Azure',
				  'foreground': 'Black'
				  })


	# ------------------------------------------------------------------
	# Layout methods:

	def CalculateSize( self ):
		vgap           = self.editor_view.GetVGap()
		hgap           = self.editor_view.GetHGap()
		tree_w         = 0
		tree_h         = 0

		count = 0
		for child in self.GetChildViews():
			(w,h) = child.CalculateSize()
			tree_w = max( tree_w, w )
			tree_h = tree_h + h

			# In case of more than one child additional gaps are required.
			if h > 0:
				if count > 0:
					tree_h = tree_h + vgap
				count = count + 1

		if self.mv_graphic.IsVisible():
			tree_w = tree_w + self.mv_graphic.GetWidth()
			if not self.mv_model.IsLeaf():
				tree_w = tree_w + hgap
			tree_h = max( tree_h, self.mv_graphic.GetHeight() )

		self.tree_size = ( tree_w, tree_h )
		if debug:
			print 'CalculateSize of', self.mv_model.GetName(), ':',\
				  self.tree_size

		return self.tree_size

		
	def SetTreePosition( self, x, y ):
		vgap          = self.editor_view.GetVGap()
		hgap          = self.editor_view.GetHGap()
		self.tree_pos = (x,y)

		delta_x = delta_y = 0

		if self.mv_graphic.IsVisible():
			ypos = y + self.GetTreeHeight()/2 - self.mv_graphic.GetHeight()/2
			delta_x  = self.mv_graphic.GetWidth() + hgap
			delta_y  = 0
			self.mv_graphic.SetPosition( x, ypos )
		else:
			delta_x = 0
			delta_y = 0

		for child in self.GetChildViews():
			cpos = (self.tree_pos[0]+delta_x, self.tree_pos[1]+delta_y)

			if not child.mv_model.IsLeaf():
				child.SetTreePosition( cpos[0], cpos[1] )
				delta_y = delta_y + child.GetTreeHeight() + vgap

			elif child.mv_graphic.IsVisible():
				child.mv_graphic.SetPosition( cpos[0], cpos[1] )
				delta_y = delta_y + child.mv_graphic.GetHeight() + vgap
			else:
				pass


	# ------------------------------------------------------------------
	# Access methods:

	def GetTreeHeight( self ): return self.tree_size[1]
	

	def GetTreeWidth( self ): return self.tree_size[0]



# -------------------------------------------------------------------------
# CLASS:         TreeLink
#
# INHERITS FROM: ArrowLink
#
# DESCRIPTION:   
#     A graphic which connects TreeNodeView's by an arrow.  
#

class TreeLink( ArrowLink ):
	
	# ------------------------------------------------------------------
	# Initialisation.

	def __init__( self, argdict = {} ):
		ArrowLink.__init__( self, argdict )
			
		try:
			self.begin_graphic.mv_view.graphic_links.append(self)
			self.end_graphic.mv_view.graphic_links.append(self)
		except: 
			print 'TreeLink requires a -graphic_links- list to be present in',\
				  'the View classes to which it should link!'


	def Finalize( self ):
		if debug_final:
			print 'TreeLink.Finalize called for', self.GetName()

		self.begin_graphic.mv_view.graphic_links.remove(self)
		self.end_graphic.mv_view.graphic_links.remove(self)

		ArrowLink.Finalize( self )



