
# -------------------------------------------------------------------------
# MODULE:      StructuredText
#
# DESCRIPTION: 
#     Provides a Text in which a Phrases can be incorporated. A Phrase
#     contains either a text string or a number of other Phrases.
#
# USAGE:      
#

import vp

from Text    import Text
from Graphic import Graphic


# -------------------------------------------------------------------------
# CLASS:         StructuredText
#
# INHERITS FROM: Text
#
# DESCRIPTION: 
#

class StructuredText(Text):

	# ------------------------------------------------------------------
	# Init methods.

	def __init__( self, argdict = {} ):
		Text.__init__( self, argdict )


	def InitAddedNode(self, phrase):
		Text.InitAddedNode(self, phrase)
		#
		# Try to draw the text onto the widget.
		#
		phrase.SetTextView(self)



# -------------------------------------------------------------------------
# CLASS:         Phrase
#
# INHERITS FROM: Graphic
#
# DESCRIPTION: 
#

class Phrase(Graphic):

	# ------------------------------------------------------------------
	# Init methods.

	def __init__(self):
		Graphic.__init__(self)
		self.size        = 0
		self.text_view   = None
		self.shown       = vp.FALSE
		self.AddFunctionDatum('string', self._SetString)

	
	def InitAddedNode(self, phrase):
		Graphic.InitAddedNode(self, phrase)
		phrase.SetTextView(self.text_view)


	# ------------------------------------------------------------------
	# Misc public methods

	def SetString(self, string):
		self.string.Set(string)


	def GetString(self):
		return self.string.Get()


	def GetPosition(self):
		#
		# Get the position in the widget.
		# 
		parent = self.GetParent()
		if parent.IsA(Phrase):
			position = parent.GetPosition()
		else:
			position = 0
		prev = self.GetPrevSibling()
		while prev != None:
			position = position + prev.GetSize()
			prev = prev.GetPrevSibling()
		return position


	def GetSize(self):
		#
		# Get the size in the widget.
		#
		if self.shown:
			return self.size
		else:
			return 0


	def Write(self, file):
		string = self.GetString()
		if string != None:
			 file.Write(string)
		for child in self.GetChildren():
			child.Write(file)


	# ------------------------------------------------------------------
	# Private methods

	def _SetString(self, new_string):
		if self.text_view == None or (not self.text_view.IsRealized()):
			return

		if new_string == None:
			new_string = ''			

		old_size = self.GetSize()
		new_size = len(new_string)
		position = self.GetPosition()
		size     = self.GetSize()

		if old_size > 0:
			self.text_view.Replace(position, position + size, '')

		if new_size > 0:
			self.text_view.Replace(position, position, new_string)

		self.ChangeSize(new_size - old_size)
		self.shown = vp.TRUE
		

	def ChangeSize(self, diff):
		self.size = self.size + diff
		parent = self.GetParent()

		if parent.IsA(Phrase):
			parent.ChangeSize(diff)


	def SetTextView(self, text_view):
		#
		# Set text_view attribute and do an update.
		# 
		if self.text_view != text_view:
			self.text_view = text_view
			self._SetString(self.GetString())	 # Set the string.
			for node in self.GetChildren():
				node.SetTextView(text_view)



# -------------------------------------------------------------------------
# TEST FUNCTIONS
#


def test():
	from Application import Application
	from Window      import Window

	app    = Application('StructuredText')
	window = Window()

	text   = StructuredText()
	window.AddWorkArea(text)

	for i in range(3):
		comp = Phrase()
		text.AddChild(comp)
		for i in range(2):
			phrase = Phrase()
			comp.AddChild(phrase)
			phrase.SetString('Hello world\n')
	app.Run()

