# -------------------------------------------------------------------------
# MODULE:      Responder
#
# DESCRIPTION: 
#     Contains the Responder class.
# 
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, <per@cwi.nl>
#

import EventNotifier

from EventNotifier import debugswitch
from vp import ENTER, LEAVE, MOUSE_DOWN, MOUSE2_DRAG, MOUSE1_DRAG, MOUSE_UP, \
               KEYPRESS, RESIZE, EXPOSE, WIDGET_DESTROYED, TRUE, FALSE

from Mixin import Mixin

ds = debugswitch
debug = FALSE

# -------------------------------------------------------------------------
# CLASS:         Responder
#
# INHERITS FROM: Mixin
#
# DESCRIPTION: 
#     A mixin class which provides the functionality for event-handling. All
#     graphic classes inherit from Responder. Non-graphic classes can handle
#     events also by inheriting from the Responder class. They will not be 
#     able to subscribe to events directly however, but must do this
#     indirectly via a graphic object.
#

class Responder( Mixin ):

	# ------------------------------------------------------------------
	# Initializations and finalizations.

	def __init__( self, argdict = {} ):
		self.eh_dict = {}

		init_method = self.GetSuperAttr( Responder, '__init__' )
		if init_method != None:
			if argdict: init_method( self, argdict )
			else: init_method( self )


	def Realize( self ):
		#
		# Called by Primitive or WidgetObject as part of the Realization
		# procedure.
		#
		if hasattr( self, 'w' ):
			self.eh_dict[WIDGET_DESTROYED] = (self.WidgetDestroyed, None, 1)

		for etype in self.eh_dict.keys():
			(eh, client_data, count) = self.eh_dict[etype]
			if count > 0: self._Subscribe( etype )


	def SetEH( self, etype, event_handler, client_data ):
		#
		# Install an event-handler without subscribing to the event; used
		# by the Sketcher class.
		#
		self.eh_dict[etype] = (event_handler, client_data, 0)


	def Finalize( self ):
		#
		# Called by Primitive or WidgetObject as part of the Finalize
		# procedure.
		#
		for etype in self.eh_dict.keys():
			(eh, client_data, count) = self.eh_dict[etype]
			if count > 0: self._UnSubscribe( etype )
		self.eh_dict = None


	# ------------------------------------------------------------------
	# Private methods

	def _Subscribe( self, etype ):
		if hasattr( self, 'w' ):                            # WidgetObject
			EventNotifier.Subscribe( self, self.w, etype )
		elif hasattr( self, 'canvas' ):                     # Primitive
			self.canvas.SubscribePrimitive( self, etype )
			if debug: 
				print 'Responder._Subscribe:', ds[etype], \
					  'for', self.GetClassName(), 'canvas =', self.canvas

			
		
	def _UnSubscribe( self, etype ):
		if hasattr( self, 'w' ):                            # WidgetObject
			EventNotifier.UnSubscribe( self, self.w, etype )
		elif hasattr( self, 'canvas' ):                     # Primitive
			self.canvas.UnSubscribePrimitive( self, etype )
			

	# ------------------------------------------------------------------
	# Event subscription methods:
	#
	# These methods can be called by other objects than 'self' in order to
	# redirect event to the event-handlers of other objects.
	#
	# A count is used to determine if a 'real' Subscribe/UnSubscribe must be
	# done.

	def Subscribe( self, etype, event_handler, client_data ):
		if not self.eh_dict.has_key( etype ):
			self.eh_dict[etype] = (event_handler, client_data, 1)
			if self.IsRealized():
				self._Subscribe( etype )
		else:
			(eh,cd,count) = self.eh_dict[etype]
			if event_handler == eh and client_data == cd:
				count = count + 1
				self.eh_dict[etype] = (event_handler, client_data, count)
				if count == 1 and self.IsRealized():
					self._Subscribe( etype )
			else:
				self.eh_dict[etype] = (event_handler, client_data, 1)
				if self.IsRealized():
					self._Subscribe( etype )


	def UnSubscribe( self, etype ):
		if self.eh_dict.has_key( etype ):
			(eh,cd,count) = self.eh_dict[etype]
			count = count - 1
			self.eh_dict[etype] = (eh, cd, count)
			if count == 0 and self.IsRealized(): 
				self._UnSubscribe( etype )


	# ------------------------------------------------------------------
	# Event-handler protocol-methods: 

	def DummyEH( self, target, client_data, xevent, e ): pass

	def WidgetDestroyed( self ):
		#
		# Set the base widget reference to None so that it will be known that
		# this widget has been destroyed.
		#
		if debug: print 'WidgetObject.WidgetDestroyed called'
		if hasattr( self, 'w' ): self.w = None


	# ------------------------------------------------------------------
	# Query methods

	def DidSubscribe( self, etype ):
		if self.eh_dict.has_key(etype):
			(eh, cd, count) = self.eh_dict[etype]
			return count > 0
		else:
			return FALSE
