# -------------------------------------------------------------------------
# MODULE:      Primitives
#
# DESCRIPTION: 
#      Contains the Primitive, Text, Bitmap, Line, Arrow, Rectangle, 
#      FillRectangle and Circle classes.
#
# USAGE:
#      > from Primitives import <class>
#
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, per@cwi.nl

import X

import vp, EventNotifier

from vp      import TRUE, FALSE
from Graphic import Graphic
from Canvas  import Canvas
from Datum   import Add

debug       = FALSE
debug_final = FALSE

# -------------------------------------------------------------------------
# The 'valid_gc_attributes' are members of the XGCValues struct. (there are
# more as well..)

valid_gc_attributes = ['function', 'plane_mask', 'foreground', 'background',
	                   'line_width', 'line_style', 'cap_style', 'join_style',
					   'fill_style', 'fill_rule', 'arc_mode', 'tile',
					   'stipple', 'ts_x_origin', 'ts_y_origin', 'font']


# -------------------------------------------------------------------------
# CLASS:         Primitive
#
# INHERITS FROM: Graphic : (TreeNode, DatumDict) : Object
#
# COLLABORATORS: Canvas
#
# DESCRIPTION: 
#     Base-class for all classes which are not implemented via X-widgets.
#

class Primitive( Graphic ): 

	# ------------------------------------------------------------------
	# Init methods

	def __init__( self, argdict = {} ): 
		self.canvas     = None
		self.do_redraw  = TRUE
		self.GetGC      = self._GetGC
		self.gc         = None                     # Graphics context

		argdict = self.MergeDefaults(argdict, {
			  'pos'      : (0,0),
			  'size'     : (10,10),
			  'show_name': FALSE
			  })
		Graphic.__init__( self, argdict )


	def Realize( self ):
		if debug: print 'Primitive.Realize: realizing', self.GetName()

		self.CreatePrimitive()

		self.wc_x = Add( self.parent.wc_x, self.x )
		self.wc_y = Add( self.parent.wc_y, self.y )

		for child in self.children: 
			child.Realize()
		Graphic.Realize( self )     # actually Responder.Realize


	def CreatePrimitive( self ):
		#
		# set fpointer to the real redraw method
		#
		if hasattr( self, 'name' ) and self.show_name == TRUE:
			self.__Redraw = self._Redraw  # basic redraw method
			self._Redraw  = self._RedrawL # wrapper around basic redraw method

		self.Redraw = self._Redraw 

		# add self to the display-list in the first parent which is a canvas

		self.canvas = self.GetCanvas()
		self.canvas.AddToDisplayList( self )    

		self.realized = TRUE


	def Finalize( self ):
		if debug_final: print 'Primitive.Finalize called for', self.GetName()

		Graphic.Finalize( self )

		if self.canvas != None:
			self.canvas.RemoveFromDisplayList( self )
			self.canvas = None
		
		self.GetGC    = None
		self.Redraw   = None
		self.gc       = None
		self.realized = FALSE


	# ------------------------------------------------------------------
	# Datum methods:

	def AddDatums( self ):
		Graphic.AddDatums( self )  

		# add redraw-callback-datum

		self.AddFunctionDatum('_redraw', self.RedrawCB)
		self.x.LinkTo( self._redraw )
		self.y.LinkTo( self._redraw )
		self.width.LinkTo( self._redraw )
		self.height.LinkTo( self._redraw )

	# ------------------------------------------------------------------
	# Public methods for setting and accessing attributes (both datum, 
	# xresources and others).

	def GetAttrKeys( self ):
		return Graphic.GetAttrKeys( self ) + valid_gc_attributes

		
	# ------------------------------------------------------------------
	# Event handler methods:

	def Mouse1DragEH( self, target, client_data, xevent, e ):
		if debug: print 'Primitive.Mouse1DragEH called for', self.GetName()

		if self.show_cntrl_pnts and self._cp != 0:      # resize bbox
			if self._cp == 1:                           # upper-left
				bb = self._GetWcBBox()
				if e.x > bb[2]: e.x = bb[2]
				if e.y > bb[3]: e.y = bb[3]
				self._SetWcBBox( e.x, e.y, bb[2], bb[3] )
			elif self._cp == 2:                         # upper-right
				ll = (self.wc_x.value, self.wc_y.value+self.height.value)
				if e.x < ll[0]: e.x = ll[0]
				if e.y > ll[1]: e.y = ll[1]
				self._SetWcBBox( ll[0], e.y, e.x, ll[1] )
			elif self._cp == 3:                         # lower-right
				bb = self._GetWcBBox()
				if e.x < bb[0]: e.x = bb[0]
				if e.y < bb[1]: e.y = bb[1]
				self._SetWcBBox( bb[0], bb[1], e.x, e.y )
			elif self._cp == 4:                         # lower-left
				ur = (self.wc_x.value+self.width.value, self.wc_y.value)
				if e.x > ur[0]: e.x = ur[0]
				if e.y < ur[1]: e.y = ur[1]
				self._SetWcBBox( e.x, ur[1], ur[0], e.y )
				

	# ------------------------------------------------------------------
	# Datum callback methods:

	def RedrawCB( self, value ):
		#
		# Must update the whole canvas so that old drawings get wiped out.
		#
		if self.canvas != None and self.do_redraw:
			self.canvas.Redraw()


	# ------------------------------------------------------------------
	# 'Drawing' methods:

	def _RedrawL( self ):
		#
		# Redraw with label; 
		#
		self.__Redraw()

		xc = self.x.value + self.width.value/2
		yc = self.y.value + self.height.value/2

		self.GetBackgroundGC().FillRectangle( xc - (self.lsize[0]/2 + 2), 
			                                  yc - (self.lsize[1]/2 + 2),
											  self.lsize[0] + 4, 
											  self.lsize[1] + 4 )
									
		self.GetOLGC().DrawString( xc - self.lsize[0]/2, 
			                       yc + self.lsize[1]/2, self.name )


	def RedrawOff( self ):
		self.do_redraw = FALSE
		if self.canvas:
			self.canvas.RedrawOff()


	def RedrawOn( self ):
		self.do_redraw = TRUE
		if self.canvas:
			self.canvas.RedrawOn()


	def Show( self ):
		self.RedrawOff()
		if not self.visible:
			self.visible = TRUE
			self.ExecuteCallback( 'visible', self )
		self.RedrawOn()


	def Hide( self ):
		self.RedrawOff()
		if self.visible:
			self.visible = FALSE
			self.ExecuteCallback( 'invisible', self )
		self.RedrawOn()


	# ------------------------------------------------------------------
	# Geometry methods: 

	def Translate( self, dx, dy ):
		self.RedrawOff()
		Graphic.Translate( self, dx, dy )
		self.RedrawOn()
		

	# ------------------------------------------------------------------
	# Modifying methods:

	def SetName( self, name ):
		self.RedrawOff()
		self.name = name 
		self.lsize = vp.theDefaultFont.GetStringSize( name )
		self.RedrawOn()


	def SetXResources( self, xres_dict ):
		self.gc = None     # a new gc must be made for the next redraw

		for res in xres_dict.keys():
			if res in valid_gc_attributes:
				val = xres_dict[res]
				if type(val) == type('') and \
					  (res == 'foreground' or res == 'background'):
					#
					# Convert color
					#
					self.xresources[res] = \
						  vp.theApplication.w.Convert(val,'Pixel')
				else:
					self.xresources[res] = val
			

	def SetHiliXRes( self, res, val ):
		global valid_gc_attributes
		
		if res in valid_gc_attributes:
			if not hasattr( self, 'hili_xres'): self.hili_xres = {}

			self.hili_gc = None
			if type(val) == type('') and \
				  (res == 'foreground' or res == 'background'):
				#
				# Convert color
				#
				self.hili_xres[res] = vp.theApplication.w.Convert(val,'Pixel')
			else:
				self.hili_xres[res] = val


	def SetHilight( self, bool ):
		if debug: print 'SetHilight: bool =', bool

		self.hilight = bool
		if hasattr( self, 'ol_gc' ): del self.ol_gc

		if bool == TRUE:
			if not hasattr( self, 'hili_xres'):
				self.hili_xres = {}
			self.GetGC = self._GetHiliGC
		else:
			self.GetGC = self._GetGC
		
		if self.canvas != None and self.do_redraw:
			self.canvas.Redraw()


	# ------------------------------------------------------------------
	# Access methods:

	def GetCanvas( self ):
		if self.parent.IsA( Primitive ):
			return self.parent.GetCanvas()
		else:
			return self.parent


	def GetOLGC( self ):  
		#
		# Return GC for outline and label
		#
		if not hasattr( self, 'ol_gc' ):
			convert = vp.theApplication.w.Convert
			xres = {}
			xres['line_width'] = 1
			xres['font']       = vp.theDefaultFont.GetFont()

			if not self.hilight:
				xres['foreground'] = convert('black','Pixel')
			else:
				xres['foreground'] = convert('white','Pixel')
			
			self.ol_gc = self.GetWidget().GetGC( xres )
		return self.ol_gc


	def GetBackgroundGC( self ):
		#
		# Return GC for drawing in the background 
		#
		if not hasattr( self, 'bg_gc' ):
			self.bg_gc = self.GetWidget().CreateGC({ 
				  'foreground': self.GetWidget().background
				  })
		return self.bg_gc


	def _GetGC( self ):
		if self.gc == None:
			self.gc = self.GetWidget().GetGC( self.xresources )
		return self.gc
		

	def _GetHiliGC( self ):
		if not hasattr( self, 'hili_gc' ) or self.hili_gc == None:
			self.hili_gc = self.GetWidget().GetGC( self.hili_xres )
		return self.hili_gc


	# ------------------------------------------------------------------
	# Query methods:
	
	def IsPicked( self, x, y ):
		bb = self._GetWcBBox()   # bbox in widget (canvas) coordinates

		if self.show_cntrl_pnts == TRUE:
			return (self._IsControlPointPicked( x, y, bb ) or \
				  (bb[0] < x and bb[1] < y and bb[2] > x and bb[3] > y))
		else:
			return (bb[0] < x and bb[1] < y and bb[2] > x and bb[3] > y)


 
