# -------------------------------------------------------------------------
# MODULE:      Poly Primitives
#
# DESCRIPTION: 
#     Primitives based on a sequence of points (PolyPath) like PolyLine
#     and Polygon.
# 
# AUTHOR:
#     Dirk Soede, CWI, Amsterdam, <soede@cwi.nl>


import vp

from Primitive   import Primitive
from Datum       import *
from             math import *


# -------------------------------------------------------------------------
# CLASS:         PolyPath
#
# INHERITS FROM: Primitive : Graphic : (TreeNode,DatumDict) : Object
#
# DESCRIPTION:
#    Specifies a sequence of 2D points forming a path. The path can be
#    manipulated per point and as a whole (eg. rotation, scaling).
#    PolyLine and Polygon use it as base class.
#
#    NB. Some methods for point manipulation could be added.

class PolyPath( Primitive ):

	# ------------------------------------------------------------------
	# Init methods

	def __init__( self, argdict = {} ):
		#
		# Additional args can be:
		# - 'points'    : <list of 2D tuples>
		# - 'rotation'  : <angle in degrees>
		# Store data of the polygon.
		self.points    = None

		# The original points and their attributes are stored here.
		self._points   = None
		self._x        = None
		self._y        = None
		self._width    = None
		self._height   = None

		argdict = self.MergeDefaults(argdict, {
			  'points': [(0,0), (20,0), (20, 20), (0, 20)],
			  'x': 0,
			  'y': 0,
			  })

		Primitive.__init__( self, argdict )

		# Assign and evaluate the polygon points.
		self.InstallPoints(self.points)

		# Now link geometry attributes to the geometry callback.
		self.x.LinkTo( self.geometry_cb )
		self.y.LinkTo( self.geometry_cb )
		self.width.LinkTo( self.geometry_cb )
		self.height.LinkTo( self.geometry_cb )
		self.rotation.LinkTo( self.geometry_cb )

		self.GeometryCB(None)


	def AddDatums( self ):
		#
		# Initialisation of Datums.
		#
		Primitive.AddDatums( self )
		self.AddDatum('rotation', Datum(None))
		self.AddFunctionDatum('geometry_cb', self.GeometryCB)
		self.SetDatum('x', None)
		self.SetDatum('y', None)
		self.SetDatum('width', None)
		self.SetDatum('height', None)



	# ------------------------------------------------------------------
	# Geometry updates:

	def InstallPoints( self, points ):
		#
		# Set the geometry explicitly by a series of points. Attributes
		# like size, position and rotation are re-initialize
		#
		self.points = points
		
		if len(self.points) > 0:
			xmin = xmax = self.points[0][0]
			ymin = ymax = self.points[0][1]

			for point in self.points:
				xmin = min(point[0], xmin)
				xmax = max(point[0], xmax)
				ymin = min(point[1], ymin)
				ymax = max(point[1], ymax)
		else:
			xmin = xmax = 0
			ymin = ymax = 0			

		self._x      = xmin
		self._y      = ymin
		self._width  = xmax - xmin
		self._height = ymax - ymin

		# The bounding box Datums must be initialized.

		if self.x.value == None:
			self.x.Set( self._x )
		if self.y.value == None:
			self.y.Set( self._y )
		if self.width.value == None:
			self.width.Set( self._width )
		if self.height.value == None:
			self.height.Set( self._height )
		if self.rotation.value == None:
			self.rotation.Set(0)
		self.real_points = points[:]


	def RotateVector(self, vector, angle):
		#
		# Rotate a 2D vector over `angle' degrees. Should be a method of a
		# vector class
		#
		length = sqrt(pow(vector[0], 2) + pow(vector[1], 2) )
		if length > 0:
			angle  = angle * pi / 180.0
			angle  = atan2(vector[1], vector[0]) + angle
			vector = ( int(cos(angle) * length), int(sin(angle) * length) )
		else:
			vector = ( 0, 0 )
		return vector	



	# ------------------------------------------------------------------
	# Managing the polygons points.

	def GetPoints(self):
		return self.points


	def AppendPoint(self, point):
		self.InstallPoints(self.GetPoints() + [point])


	def ClosePath(self):
		if len(self.points) > 0:
			self.InstallPoints(self.points + [self.points[0]])


	# ------------------------------------------------------------------
	# Callbacks.

	def GeometryCB( self, void):
		#
		# A shared callback to be called when geometry Datums change.
		# 
		x       = self.x.value
		y       = self.y.value
		xfactor = self.width.value / self._width
		yfactor = self.height.value / self._height

		self.real_points = []

		for point in self.points:
			new_point     = point[0] - self._x, point[1] - self._y
			new_point = new_point[0] * xfactor,  new_point[1] * yfactor
			if self.rotation.value != 0:
				new_point =  self.RotateVector(new_point, self.rotation.value)
			new_point = x + new_point[0],  y + new_point[1]
			self.real_points.append(new_point)



# -------------------------------------------------------------------------
# CLASS:         Polygon
#
# INHERITS FROM: PolyPath : Primitive : Graphic : (TreeNode,DatumDict) : Object
#
# DESCRIPTION:
#    

class Polygon( PolyPath ):

	def _Redraw( self ):
		self.GetGC().FillPolygon(self.real_points, 0, 0)


# -------------------------------------------------------------------------
# CLASS:         PolyLine
#
# INHERITS FROM: PolyPath : Primitive : Graphic : (TreeNode,DatumDict) : Object
#
# DESCRIPTION:
#    

class PolyLine( PolyPath ):

	def _Redraw( self ):
		self.GetGC().DrawLines(self.real_points, 0)


