# -------------------------------------------------------------------------
# MODULE:      NestableComposite 
#
# DESCRIPTION: 
#     Contains the following *abstract* 'mix-in' classes:
#         - Nestable, and
#         - Icon
#
#     and the follwoing *concrete* classes:
#         - NestableRectangle, 
#         - NestableCircle, 
#         - NestableCanvas,
#         - RectangleIcon, 
#         - CircleIcon, and
#         - ButtonIcon
#
# AUTHOR:
#     Per Spilling <per@cwi.nl>, CWI, Amsterdam.

import vp, EventNotifier, X, Xm

from Mixin        import Mixin
from Graphic      import Graphic
from Composite    import Composite
from Primitive    import Primitive
from Rectangle    import Rectangle
from Circle       import Circle
from Canvas       import Canvas
from WidgetObject import WidgetObject
from WidgetButton import PushButton
from Menu         import PopupMenu

debug       = vp.FALSE
debug_final = vp.FALSE


# -------------------------------------------------------------------------
# CLASS:         Nestable
#
# DESCRIPTION: 
#     An abstract 'mix-in' class which contains the functionality for nestable 
#     graphical objects.
#
#     A nestable graphical object is a graphic which can be in an opened- or 
#     iconified state. The children can only be seen when the graphic is in 
#     its opened state, i.e. similar to 'folders' on a Mac. Opening a child 
#     will open it 'in place', expanding the size of the parent if necessary.
#
#     The two menus 'main_menu' and 'icon_menu' are class attributes, instead
#     of instance attributes since they are the same for all instances.
#

class Nestable( Mixin ):

	# ---------------------------------------------------------------------
	# Class attributes:

	main_menu  = None
	icon_menu  = None
	icon_class = None

	# ---------------------------------------------------------------------
	# Creation and finalization methods:

	def __init__( self, argdict = {} ):
		#
		# Set some default attribute values
		#
		argdict = self.MergeDefaults(argdict, {
			'iconified': vp.FALSE,
			'draggable': vp.TRUE,
			'editmode' : vp.TRUE
			})
		apply( self.GetSuperAttr( Nestable, '__init__' ), (self, argdict ))

		# create icon graphic

		self.icon = self.icon_class( self, self.GetClassName() )
		self.icon.Hide()


	def CreateMainMenu( self ):
		#
		# This method should be overridden in subclasses or by 'clients'
		#
		if debug:
			print 'Nestable.CreateMainMenu called for', self.GetClassName()

		return PopupMenu({
			  'title':   'Actions:',
			  'items':  [('Iconify',       self.IconifyCB),
				         ('Shrink to fit', self.ShrinkToFitCB),
						 ('Delete',        self.FinalizeCB)] 
			})


	def CreateIconMenu( self ):
		#
		# This method should be overridden in subclasses or by 'clients'
		#
		if debug:
			print 'Nestable.CreateIconMenu called for', self.GetClassName()

		return PopupMenu({
			  'title':   'Actions:',
			  'items':  [('Restore',   self.RestoreCB)] 
			})


	def Realize( self ):
		self.GetSuperAttr( Nestable, 'Realize' )( self )

		if len(self.children) != 0:
			self._CalculateNaturalSize()
			(w,h) = self.GetSize()
			if w < self.natural_size[0]: w = self.natural_size[0]
			if h < self.natural_size[1]: h = self.natural_size[1]
			self.SetSize( w, h )

		if Nestable.main_menu == None:
			Nestable.main_menu = self.CreateMainMenu()

		if Nestable.icon_menu == None:
			Nestable.icon_menu = self.CreateIconMenu()

		self.icon.SetName(self.GetName())
		
		self.x.DoubleLink( self.icon.x )
		self.y.DoubleLink( self.icon.y )

		if hasattr( self, 'title' ):
			self.width.LinkTo( self.title.width )

		# give the same callbacks to the icon as to self

		if hasattr( self, 'cb_dict' ):
			for key in self.cb_dict.keys():
				if type( self.cb_dict[key] ) == type([]):
					for cb in self.cb_dict[key]:
						self.icon.AddCallback( key, cb )
				else:
					self.icon.AddCallback( key, self.cb_dict[key] )

		self.parent.AddChild( self.icon )

		self.AddPopup( self.main_menu )
		self.icon.AddPopup( self.icon_menu )

		if self.iconified:
			self.Iconify()
		else:
			self.InitialRestore()

		self.SetXResources({ 'foreground': 'bisque' })


	def InitialRestore( self ):
		self.iconified = vp.FALSE
		self.SetPosition( self.icon.x.value, self.icon.y.value )
		self._Show()


	def Finalize( self ):
		if debug_final: print 'Nestable.Finalize called'

		self.icon.Finalize()
		del self.icon
		self.GetSuperAttr( Nestable, 'Finalize' )( self )


	# ------------------------------------------------------------------
	# Callback support:

	def AddCallback( self, cb_type, cb_handler ):
		self.GetSuperAttr( Nestable, 'AddCallback' )( self,cb_type,cb_handler )

		if self.IsRealized():
			self.icon.AddCallback( cb_type, cb_handler )


	# ------------------------------------------------------------------
	# Nestable 'action' methods:

	def Iconify( self ):
		self.iconified = vp.TRUE

		if self.IsRealized() and not self.icon.IsVisible():
			self._Hide()
			self.icon.SetPosition( self.x.value, self.y.value )
			self.icon.Show()
			self.ExecuteCallback( 'iconified', self )


	def Restore( self ):
		self.iconified = vp.FALSE

		if self.IsRealized() and self.icon.IsVisible():
			self.icon.Hide()
			self.SetPosition( self.icon.x.value, self.icon.y.value )
			self._Show()
			self.ExecuteCallback( 'restored', self )
			self.ExecuteCallback( 'size_changed', self )


	def ShrinkToFit( self ): 
		self.RedrawOff()
		bb = self._GetMinWcBBox()
		self._SetWcBBox( bb[0], bb[1], bb[2], bb[3] )
		self.RedrawOn()
		

	def SetName( self, name ):
		self.GetSuperAttr( Nestable, 'SetName' )( self, name )
		if hasattr( self, 'icon' ): self.icon.SetName( name )


	# ------------------------------------------------------------------
	# Callbacks for user input. ('class methods')

	def IconifyCB( self, menu_item ): vp.theCurrentPopupNode.Iconify()

	def RestoreCB( self, menu_item ): vp.theCurrentPopupNode.Restore()

	def FinalizeCB( self, menu_item ): vp.theCurrentPopupNode.Finalize()
		
	def ShrinkToFitCB( self, menu_item ): vp.theCurrentPopupNode.ShrinkToFit()
		

	# ------------------------------------------------------------------
	# Misc methods:

	def InitAddedNode( self, child ):
		self.GetSuperAttr( Nestable, 'InitAddedNode' )( self, child )

		child.Subscribe( vp.MOUSE2_DRAG, self.ChildMouse2DragEH, None )
		child.Subscribe( vp.MOUSE1_DRAG, self.ChildMouse1DragEH, None )
		child.Subscribe( vp.MOUSE_UP, self.ChildMouseUpEH, None )
		child.AddCallback( 'size_changed', self.ChildSizeChangedCB )

		if child.IsA( Nestable ): child.Iconify()
			
		if self.IsRealized():
			self._CalculateNaturalSize()  # new natural size
			(w,h) = self.GetSize()
			if w < self.natural_size[0]: w = self.natural_size[0]
			if h < self.natural_size[1]: h = self.natural_size[1]
			self.SetSize( w, h )

			self._CheckChildPosition( child )

		if self.iconified: child.Hide()

		

	def IsIconified( self ): return self.iconified
		
	# ------------------------------------------------------------------
	# 'Drawing' methods:

	def Hide( self ):
		if self.iconified and hasattr( self, 'icon' ):
			self.icon.Hide()
		else:
			self._Hide()


	def _Hide( self ):
		for child in self.children:
			if not child.IsA( Icon ):
				child.Hide()
		self.GetSuperAttr( Nestable, 'Hide' )( self )


	def Show( self ):
		if self.iconified and hasattr( self, 'icon' ):
			self.icon.Show()
		else:
			self._Show()


	def _Show( self ):
		self._CalculateNaturalSize()
		self.SetSize( max(self.natural_size[0],self.width.value), 
			          max(self.natural_size[1],self.height.value) )
		self.GetSuperAttr( Nestable, 'Show' )( self )
		for child in self.children:
			if not child.IsA( Icon ):
				child.Show()

	# ------------------------------------------------------------------
	# 'Layout' methods:

	def _CalculateNaturalSize( self ):
		(oxmin,oymin,oxmax,oymax) = (0,0,20,20)

		for child in self.children:
			(cxmin,cymin,cxmax,cymax) = child.GetBBox()
			if cxmax > (oxmax-10): oxmax = cxmax + 10
			if cymax > (oymax-10): oymax = cymax + 10

		if hasattr( self, 'title' ):
			self.natural_size = (oxmax-oxmin, 
								 oymax-oymin+self.title.height.value)
		else:
			self.natural_size = (oxmax-oxmin, oymax-oymin)


	def _GetMinWcBBox( self ): 
		if len(self.children) == 0:
			return (self.wc_x.value, 
				    self.wc_y.value, 
				    self.wc_x.value+20,
				    self.wc_y.value+20)
		else:
			first_child = vp.TRUE
			(xmin, ymin, xmax, ymax) = (0,0,0,0)

			for child in self.children:
				if child.IsVisible():
					if first_child:
						(xmin, ymin, xmax, ymax)  = child._GetWcBBox()
						first_child = vp.FALSE
					else:
						(cxmin,cymin,cxmax,cymax) = child._GetWcBBox()
						xmin = min(cxmin,xmin)
						ymin = min(cymin,ymin)
						xmax = max(cxmax,xmax)
						ymax = max(cymax,ymax)

			return (xmin, ymin, xmax, ymax)


	def _GetMinResizeWcBBox( self ):
		(oxmin,oymin,oxmax,oymax) = self._GetWcBBox()
		(xmin, ymin, xmax, ymax)  = self._GetMinWcBBox()
		dx = oxmax - xmax
		dy = oymax - ymax

		if self._cp == 1: 
			minbb = (oxmin+dx, oymin+dy, oxmax, oymax)
		elif self._cp == 2:
			minbb = (oxmin, oymin+dy, xmax, oymax)
		elif self._cp == 3:
			minbb = (oxmin, oymin, xmax, ymax)
		else:
			minbb = (oxmin+dx, oymin, oxmin, ymax)

		return minbb
		

	def ChildSizeChangedCB( self, child ):
		#
		# Called when a child was 'restored' (opened); must check own size. 
		# Need only check the max values.
		#
		(w,h) = self.GetSize()
		(cxmin,cymin,cxmax,cymax) = child.GetBBox()

		must_change_size = vp.FALSE

		if cxmax > w:
			w = cxmax
			must_change_size = vp.TRUE

		if cymax > h:
			h = cymax
			must_change_size = vp.TRUE
			
		if must_change_size:
			self.SetSize( w, h )
			self.ExecuteCallback( 'size_changed', self )

		
	def _CheckSize( self ):
		#
		# 'self' is not allowed to be resized smaller than its minimum bbox
		#
		if not hasattr( self, '_min_bb' ):
			return

		(oxmin,oymin,oxmax,oymax) = self._GetWcBBox()

		if oxmin >= self._min_bb[0]: oxmin = self._min_bb[0]
		if oymin >= self._min_bb[1]: oymin = self._min_bb[1]
		if oxmax <= self._min_bb[2]: oxmax = self._min_bb[2]
		if oymax <= self._min_bb[3]: oymax = self._min_bb[3]

		self._SetWcBBox( oxmin, oymin, oxmax, oymax )
		
		
	def _CheckChildSize( self, child  ):
		#
		# Children are not allowed to be resized larger than 'self'
		#
		(oxmin,oymin,oxmax,oymax) = self._GetWcBBox()
		(cxmin,cymin,cxmax,cymax) = child._GetWcBBox()

		if cxmin <= oxmin: cxmin = oxmin + 1
		if cymin <= oymin: cymin = oymin + 1
		if cxmax >= oxmax: cxmax = oxmax - 1
		if cymax >= oymax: cymax = oymax - 1

		child._SetWcBBox( cxmin, cymin, cxmax, cymax )


	def _CheckChildPosition( self, child ):
		#
		# Children are not allowed to move outside the bbox of 'self'
		#
		xdelta = 0
		ydelta = 0
		
		if hasattr( self, 'title' ):
			(oxmin,oymin,oxmax,oymax) = (0, self.title.height.value,
										 self.width.value, self.height.value)
		else:
			(oxmin,oymin,oxmax,oymax) = (0, 0,
										 self.width.value, self.height.value)
		(cxmin,cymin,cxmax,cymax) = child.GetBBox()

		if cxmin < oxmin: xdelta = oxmin - (cxmin - 1)
		if cymin < oymin: ydelta = oymin - (cymin - 1)
		if cxmax > oxmax: xdelta = oxmax - (cxmax + 1)
		if cymax > oymax: ydelta = oymax - (cymax + 1)

		if xdelta != 0 or ydelta != 0:
			child.SetBBox( cxmin+xdelta, cymin+ydelta, 
				           cxmax+xdelta, cymax+ydelta )
		

	# ------------------------------------------------------------------
	# Event handlers:

	def MouseDownEH( self, target, client_data, xevent, e ):
		if e == None: e = xevent
		if e.button == X.Button1 and hasattr( self, '_cp' ): 
			self._min_bb = self._GetMinResizeWcBBox()
		self.GetSuperAttr( Nestable, 'MouseDownEH' )( self, target, 
											client_data, xevent, e )
		

	def Mouse1DragEH( self, target, client_data, xevent, e ):
		self.GetSuperAttr( Nestable, 'Mouse1DragEH' )( self, target, 
											 client_data, xevent, e )
		if not hasattr( self, 'w' ):
			self._CheckSize()


	def ChildMouse1DragEH( self, child, client_data, xevent, e ):
		self.RedrawOff()
		child.Mouse1DragEH( child, client_data, xevent, e )
		if not hasattr( self, 'w' ):
			self._CheckChildSize( child )
		self.RedrawOn()


	def ChildMouse2DragEH( self, child, client_data, xevent, e ):
		self.RedrawOff()
		child.Mouse2DragEH( child, client_data, xevent, e )
		if not hasattr( self, 'w' ):
			self._CheckChildPosition( child )
		self.RedrawOn()


	def ChildMouseUpEH( self, child, client_data, xevent, e ):
		self.RedrawOff()
		child.MouseUpEH( child, client_data, xevent, e )
		if not hasattr( self, 'w' ):
			self._CheckChildPosition( child )
			self._CheckChildSize( child )
		self.RedrawOn()


# -------------------------------------------------------------------------
# CLASS:         Icon
#
# INHERITS FROM: None
#
# DESCRIPTION: 
#     A 'collaborator class' for the Nestable class. 
#

class Icon:

	def __init__( self, master_object ):
		self.master_object = master_object


	def Restore( self ): self.master_object.Restore()
		


# -------------------------------------------------------------------------
# CLASS:         RectangleIcon
#
# INHERITS FROM: (Rectangle,Icon) : Primitive : Graphic : \
#                (TreeNode, DatumDict) : Object
#
# DESCRIPTION: 
#

class RectangleIcon( Rectangle, Icon ):

	def __init__( self, master_object, name ):
		Icon.__init__( self, master_object )
		Rectangle.__init__( self, {
			'name'     : name,
			'size'     : (48,48),
			'fill'     : vp.TRUE,
			'outline'  : vp.TRUE,
			'show_name': vp.TRUE,
			'draggable': vp.TRUE
			})
		self.SetXResources({ 'foreground': 'turquoise3' })
			  


# -------------------------------------------------------------------------
# CLASS:         ButtonIcon
#
# INHERITS FROM: (PushButton,Icon) : WidgetObject : Graphic : \
#                (TreeNode, DatumDict) : Object
#
# DESCRIPTION: 
#

class ButtonIcon( PushButton, Icon ):

	def __init__( self, master_object, name ):
		Icon.__init__( self, master_object )
		PushButton.__init__( self, {
			'name'     : name,
			'size'     : (48,48),
			'draggable': vp.TRUE,
			'editmode' : vp.FALSE
			})

	# ------------------------------------------------------------------
	# Notification methods:

	def LabelChangedCB( self, value ):
		if self.IsRealized():
			if self.editmode == vp.TRUE:
				self._width.UpdateInternal()  # internal width
				if self.width.value != None and self.height.value != None:
					self.natural_size = (self.width.value, self.height.value)
				self.ExecuteCallback( 'size_changed', self )
			else:
				self.SetSize( self.natural_size[0], self.natural_size[1] )



# -------------------------------------------------------------------------
# CLASS:         CircleIcon
#
# INHERITS FROM: (Circle,Icon) : Primitive : Graphic : \
#                (TreeNode, DatumDict) : Object
#
# DESCRIPTION: 
#

class CircleIcon( Circle, Icon ):

	def __init__( self, master_object, name ):
		Icon.__init__( self, master_object )
		Circle.__init__( self, {
			  'name'     : name,
			  'size'     : (48,48),
			  'fill'     : vp.TRUE,
			  'outline'  : vp.TRUE,
			  'show_name': vp.TRUE,
			  'draggable': vp.TRUE
			  })
		self.SetXResources({ 'foreground': 'turquoise3' })
			  

# -------------------------------------------------------------------------
# CLASS:         NestableRectangle
#
# INHERITS FROM: (Composite, Primitive) : Graphic : (TreeNode, DatumDict) :\
#                Object
#
# DESCRIPTION: 
#

class NestableRectangle( Nestable, Composite, Rectangle ):

	icon_class = RectangleIcon

	# ------------------------------------------------------------------
	# Disambiguate methods from superclasses:

	_Subscribe        = Primitive._Subscribe
	_UnSubscribe      = Primitive._UnSubscribe
	RedrawOn          = Primitive.RedrawOn
	RedrawOff         = Primitive.RedrawOff
	ProcessArgs       = Primitive.ProcessArgs

	CreatePrimitive   = Rectangle.CreatePrimitive
	_Redraw           = Rectangle._Redraw

	# ------------------------------------------------------------------
	# Init methods

	def __init__( self, argdict = {} ):
		argdict = self.MergeDefaults(argdict, {
			'pos'      : (0,0),
			'size'     : (100,100),
			'fill'     : vp.TRUE,
			'outline'  : vp.TRUE,
			'show_name': vp.TRUE
			})
		Nestable.__init__( self, argdict )


# -------------------------------------------------------------------------
# CLASS:         NestableCircle
#
# INHERITS FROM: (Composite, Primitive) : Graphic : (TreeNode, DatumDict) :\
#                Object
#
# DESCRIPTION: 
#

class NestableCircle( Nestable, Composite, Circle ):

	icon_class = CircleIcon

	# ------------------------------------------------------------------
	# Disambiguate methods from superclasses:

	_Subscribe        = Primitive._Subscribe
	_UnSubscribe      = Primitive._UnSubscribe
	RedrawOn          = Primitive.RedrawOn
	RedrawOff         = Primitive.RedrawOff
	ProcessArgs       = Primitive.ProcessArgs

	CreatePrimitive   = Circle.CreatePrimitive
	_Redraw           = Circle._Redraw

	# ------------------------------------------------------------------
	# Init methods

	def __init__( self, argdict = {} ):
		argdict = self.MergeDefaults(argdict, {
			'pos'      : (0,0),
			'size'     : (100,100),
			'fill'     : vp.TRUE,
			'outline'  : vp.TRUE,
			'show_name': vp.FALSE
			})
		Nestable.__init__( self, argdict )


# -------------------------------------------------------------------------
# CLASS:         NestableCanvas
#
# INHERITS FROM: (Nestable, Canvas) : (Composite, WidgetObject) : Graphic : \
#                (TreeNode, DatumDict) : Object
#
# DESCRIPTION: 
#

class NestableCanvas( Nestable, Canvas ):

	icon_class = ButtonIcon
	
	def __init__( self, argdict = {} ):
		argdict = self.MergeDefaults(argdict, {
			'pos'      : (0,0),
			'size'     : (100,100),
			'titlebar' : vp.TRUE,
			'xres'     : {'background': 'grey80', 'borderWidth': 2}
			})
		Nestable.__init__( self, argdict )

		if self.titlebar:
			from MiscGraphic import Label

			self.title = Label({
				'name': self.GetName(),
				'xres': {'background': 'white', 'borderWidth': 1}
				})
			self.AddChild( self.title )
			

	def InitAddedNode( self, child ):
		if hasattr( self, 'title' ) and child == self.title:
			Canvas.InitAddedNode( self, child )
			
			if self.draggable:
				self.title.Subscribe( vp.MOUSE_DOWN, self.MouseDownEH, None )
				self.title.Subscribe( vp.MOUSE2_DRAG, self.Mouse2DragEH, None )
				self.title.Subscribe( vp.MOUSE_UP, self.MouseUpEH, None )
		else:
			Nestable.InitAddedNode( self, child )

