# -------------------------------------------------------------------------
# MODULE:      MiscDialogs
#
# DESCRIPTION: 
#     Contains the InfoDialog, QuestionDialog, FileSelectionDialog, and
#     TextEntryDialog classes.
#
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, per@cwi.nl


import Xm, Xmd, X
import vp

from Dialog import Dialog

debug = vp.FALSE


# -------------------------------------------------------------------------
# CLASS:         InfoDialog
#
# INHERITS FROM: Dialog : WidgetObject : (TreeNode,Datum) : Object
#
# DESCRIPTION: 
#     Provides an easy way for displaying information messages such as:
#     "Disk almost full", "Compilation completed", etc. 
#
#     One instance of this class will be created by the Application class. 
#     This instance is accessible to the whole application via a global 
#     variable in the "vp" module: > vp.theInfoDialog.
#
# USAGE:
#     Messages can be displayed by calling the "vp.theInfoDialog.Post" method,
#     ex:
#     ---------------------------------------------------------------------
#     import vp
#     ...
#         ...
#         vp.theInfoDialog.Post( message, OkCB, CancelCB, None )
#     ---------------------------------------------------------------------
#

class InfoDialog( Dialog ):
	
	def SetDefaultXRes( self ):
		self.xresources['title']        = self.GetName()
		self.xresources['resizePolicy'] = Xmd.RESIZE_ANY


	def CreateWidget( self ):
		self.w = vp.theApplication.GetWidget().CreateInformationDialog( 
			                                 'InfoDialog', self.xresources)
		self.w.AddCallback( 'okCallback', self.wOkCB, None )
		self.w.AddCallback( 'cancelCallback', self.wCancelCB, None )
		self.w.AddCallback( 'helpCallback', self.wHelpCB, None )
		self.InitWidget()
		self.w.ManageChild()


		
# -------------------------------------------------------------------------
# CLASS:         QuestionDialog
#
# INHERITS FROM: Dialog : WidgetObject : (TreeNode,Datum) : Object
#
# DESCRIPTION: 
#     Provides an easy, application-wide, method for asking questions. The
#     application will be blocked until the question has been answered.
#
# USAGE:
#     Same as the InfoDialog class.
#

class QuestionDialog( Dialog ):

	def SetDefaultXRes( self ):
		self.xresources['title']        = self.GetName()
		self.xresources['resizePolicy'] = Xmd.RESIZE_ANY
		self.xresources['dialogStyle']  = Xmd.DIALOG_FULL_APPLICATION_MODAL
		

	def CreateWidget( self ):
		self.w = vp.theApplication.GetWidget().CreateQuestionDialog( 
			                              'QuestionDialog',self.xresources)
		self.w.AddCallback( 'okCallback', self.wOkCB, None )
		self.w.AddCallback( 'cancelCallback', self.wCancelCB, None )
		self.w.AddCallback( 'helpCallback', self.wHelpCB, None )
		self.InitWidget()
		self.w.ManageChild()
		
	
# -------------------------------------------------------------------------
# CLASS:         FileSelectionDialog
#
# INHERITS FROM: Dialog : WidgetObject : (TreeNode,DatumDict) : Object
#
# DESCRIPTION: 
#     This instance is accessible to the whole application via a global 
#     variable in the "vp" module: > vp.theFileSelectionDialog.
#

class FileSelectionDialog( Dialog ):
	
	# ------------------------------------------------------------------
	# Initialization and finalization methods

	def SetDefaultXRes( self ):
		self.xresources['title']        = self.GetName()
		self.xresources['resizePolicy'] = Xmd.RESIZE_NONE
		

	def CreateWidget( self ): 
		problem_resources = {}
		for key in self.xresources.keys():
			if key in ['directory', 'pattern']:
				problem_resources[key] = self.xresources[key]
				del self.xresources[key]

		self.w = vp.theApplication.GetWidget().CreateFileSelectionDialog( 
 		                                       'FileSelectionDialog',
			                                   self.xresources )
		self.w.ManageChild()
		self.w.AddCallback( 'okCallback', self.wOkCB, None )
		self.w.AddCallback( 'cancelCallback', self.wCancelCB, None )
		self.w.AddCallback( 'helpCallback', self.wHelpCB, None )
		self.InitWidget()
		self.w.SetValues( problem_resources )


	# ------------------------------------------------------------------
	# Access methods

	def GetFilename( self ):
		#
		# Return the filename in the selection field
		#
		filename = None

		if self.IsRealized():
			tw = self.w.FileSelectionBoxGetChild( Xmd.DIALOG_TEXT )
			try:
				filename = tw.TextFieldGetString( )
			except TypeError:
				filename = tw.TextGetString( )

		return filename


	def GetFilenameFilter( self ):
		#
		# Return the filter used to select the files in the file list.
		#
		ftext = None

		if self.IsRealized():
			tw = self.w.FileSelectionBoxGetChild( Xmd.DIALOG_FILTER_TEXT )
			try:
				ftext = tw.TextFieldGetString( )
			except TypeError:
				ftext = tw.TextGetString( )

		return ftext


	# ------------------------------------------------------------------
	# Modify methods

	def SetDirectory( self, dir ): self.SetXResources({ 'directory': dir })

	def SetPattern( self, pat ): self.SetXResources({ 'pattern': pat })


# -------------------------------------------------------------------------
# CLASS:         PromptDialog
#
# INHERITS FROM: Dialog : WidgetObject : (TreeNode,DatumDict) : Object
#
# DESCRIPTION: 
#

class PromptDialog( Dialog ):
	
	def SetDefaultXRes( self ):
		self.xresources['title']        = self.GetName()
		self.xresources['resizePolicy'] = Xmd.RESIZE_NONE
		

	def CreateWidget( self ): 
		self.w = vp.theApplication.GetWidget().CreatePromptDialog( 
 		                                'PromptDialog', 
										self.xresources )
		self.w.ManageChild()
		self.w.AddCallback( 'okCallback', self.wOkCB, None )
		self.w.AddCallback( 'cancelCallback', self.wCancelCB, None )
		self.w.AddCallback( 'helpCallback', self.wHelpCB, None )
		self.InitWidget()


	def SetMessage( self, message ): self.w.selectionLabelString = message


	def GetText( self ):
		text = None

		if self.w != None:
			text = self.w.SelectionBoxGetChild( Xmd.DIALOG_TEXT )
			try:
				text = text.TextFieldGetString( )
			except TypeError:
				text = text.TextGetString( )

		return text


# -------------------------------------------------------------------------
# CLASS:         WorkingDialog
#
# INHERITS FROM: Dialog : WidgetObject : (TreeNode,DatumDict) : Object
#
# DESCRIPTION: 
#

class WorkingDialog( Dialog ):
	
	def SetDefaultXRes( self ):
		self.xresources['title']        = self.GetName()
		self.xresources['resizePolicy'] = Xmd.RESIZE_NONE
		

	def CreateWidget( self ): 
		self.w = vp.theApplication.GetWidget().CreateWorkingDialog( 
 		                                 'WorkingDialog', 
										 self.xresources )
		self.w.ManageChild()
		self.w.AddCallback( 'okCallback', self.wOkCB, None )
		self.w.AddCallback( 'cancelCallback', self.wCancelCB, None )
		self.w.AddCallback( 'helpCallback', self.wHelpCB, None )
		self.InitWidget()

