# -------------------------------------------------------------------------
# MODULE:      Lines
#
# DESCRIPTION: 
#     Contains line primitives.
#
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, per@cwi.nl

from math    import sqrt, pow

import vp, point

from Primitive import Primitive
from Datum     import Datum, Tuple, BBoxPos, BBoxSize

debug       = vp.FALSE


# -------------------------------------------------------------------------
# CLASS:         Line
#
# INHERITS FROM: Primitive : Graphic : (TreeNode,DatumDict) : Object
#
# DESCRIPTION:
#

class Line( Primitive ):

	# ------------------------------------------------------------------
	# Init methods

	def __init__( self, argdict = {} ):
		argdict = self.MergeDefaults(argdict, { 
			  'begin': (0, 0),
			  'end'  : (10, 10),
			  })
		Primitive.__init__( self, argdict )

		if not 'line_width' in self.xresources.keys():
			self.SetXResources({ 'line_width': 1 })

		if hasattr( self, 'wc_mode' ): 
			#
			# 'wc_mode' means that the 'begin' and 'end' points are given 
			# in window (widget) coordinates instead of relative to the
			# (primitive) parent. This is done so that a line can cross 
			# 'primitive' borders.
			#
			self._Redraw  = self._WcRedraw
			self.IsPicked = self._WcIsPicked

		
	def ProcessArgs( self, argdict ):
		#
		# Overridden from Graphic in order to process 'begin' and 'end' 
		# arguments correctly.
		#
		if debug: print 'Line.ProcessArgs: argdict =', argdict

		if argdict.has_key( 'begin' ) and argdict.has_key( 'end' ):
			if argdict.has_key( 'pos' ): del argdict['pos']
			if argdict.has_key( 'size' ): del argdict['size']

		Primitive.ProcessArgs( self, argdict )


	# ------------------------------------------------------------------
	# Datum methods:

	def AddDatums( self ):
		#
		# A line is defined by its begin and end points 
		#
		self.AddDatum('wc_x', Datum(0))
		self.AddDatum('wc_y', Datum(0))
		self.AddDatum('bx', Datum(0))
		self.AddDatum('by', Datum(0))
		self.AddDatum('ex', Datum(0))
		self.AddDatum('ey', Datum(0))

		bx = self.bx
		by = self.by
		ex = self.ex
		ey = self.ey

		self.AddDatum('begin', Tuple(bx, by))
		self.AddDatum('end',   Tuple(ex, ey))

		# In addition a Line, just as all other Graphic objects, has a bbox
		# defined by the datums: 'x', 'y', 'width', and 'height'

		self.AddDatum('x',      BBoxPos( bx, ex ))
		self.AddDatum('y',      BBoxPos( by, ey ))
		self.AddDatum('width',  BBoxSize( bx, ex ))
		self.AddDatum('height', BBoxSize( by, ey ))

		# add redraw-callback-datum

		self.AddFunctionDatum('_redraw', self.RedrawCB)
		self.x.LinkTo( self._redraw )
		self.y.LinkTo( self._redraw )
		self.width.LinkTo( self._redraw )
		self.height.LinkTo( self._redraw )


	# ------------------------------------------------------------------
	# Drawing methods:

	def _Redraw( self ):
		#
		# Begin and end points are given in coordinates relative to the parent
		#
		begin = self.begin.value
		end   = self.end.value
		ori   = (self.parent.wc_x.value, self.parent.wc_y.value)
		self.GetGC().DrawLine( ori[0]+begin[0], ori[1]+begin[1], 
			                   ori[0]+end[0],   ori[1]+end[1] )


	def _WcRedraw( self ):
		#
		# Begin and end points are given in window coordinates
		#
		begin = self.begin.value
		end   = self.end.value
		self.GetGC().DrawLine( begin[0], begin[1], end[0],   end[1] )


	# ------------------------------------------------------------------
	# Query methods:

	def IsPicked( self, x, y ):
		#
		# 'x' and 'y' are in widget coordinates, while 'begin' and 'end' are 
		# relative to parent.
		#
		if self.x.value != self.wc_x.value: x = x - self.wc_x.value
		if self.y.value != self.wc_y.value: y = y - self.wc_y.value

		if self.show_cntrl_pnts == vp.TRUE:
			bb = self._GetWcBBox()
			return (self._IsControlPointPicked( x, y, bb ) or \
				  point.is_point_on_line( (x,y), 
				                           (self.begin.value, self.end.value),
										   3 ))
		else:
			return point.is_point_on_line( (x,y), 
				                           (self.begin.value, self.end.value),
										   3 )
		
	def _WcIsPicked( self, x, y ):
		#
		# In this case the bbox, begin, and end points are in widget 
		# coordinates
		#
		if self.show_cntrl_pnts == vp.TRUE:
			bb = self.GetBBox()
			return (self._IsControlPointPicked( x, y, bb ) or \
				  point.is_point_on_line( (x,y), 
				                           (self.begin.value, self.end.value),
										   3 ))
		else:
			return point.is_point_on_line( (x,y), 
				                           (self.begin.value, self.end.value),
										   3 )
		
		
		
# -------------------------------------------------------------------------
# CLASS:         Arrow
#
# INHERITS FROM: Line : Primitive : Graphic : (TreeNode,DatumDict) : Object
#
# DESCRIPTION:
#

class Arrow( Line ):

	def __init__( self, argdict = {} ):
		self.head_length = 20
		self.head_width  = 10
		Line.__init__( self, argdict )


	def _Redraw( self ):
		#
		# Begin and end points are given in coordinates relative to the parent
		#
		begin = self.begin.value
		end   = self.end.value
		ori   = (self.parent.wc_x.value, self.parent.wc_y.value)
		self.GetGC().DrawLine( ori[0]+begin[0], ori[1]+begin[1], 
			                   ori[0]+end[0],   ori[1]+end[1] )

		dx  = end[0] - begin[0]
		dy  = end[1] - begin[1]
		len = sqrt( dx*dx + dy*dy )
		if len:
			sv  = (dx/len, dy/len)              # slope vector
			nv  = (-dy/len, dx/len)             # normal vector
		else:
			sv = nv = (0.0, 0.0)
		end = (ori[0]+end[0],ori[1]+end[1]) # tranform end to wc

		p1 = (int(end[0] - sv[0]*self.head_length + nv[0]*self.head_width/2),
			  int(end[1] - sv[1]*self.head_length + nv[1]*self.head_width/2))

		p2 = (int(end[0] - sv[0]*self.head_length - nv[0]*self.head_width/2),
			  int(end[1] - sv[1]*self.head_length - nv[1]*self.head_width/2))

		self.GetGC().FillPolygon( [(end[0],end[1]), p1, p2], 0, 0 )


	def _WcRedraw( self ):
		#
		# Begin and end points are given in window coordinates
		#
		begin = self.begin.value
		end   = self.end.value
		self.GetGC().DrawLine( begin[0], begin[1], end[0],   end[1] )

		dx  = end[0] - begin[0]
		dy  = end[1] - begin[1]
		len = sqrt( dx*dx + dy*dy )
		if len:
			sv  = (dx/len, dy/len)             # slope vector
			nv  = (-dy/len, dx/len)            # normal vector
		else:
			sv = nv = (0.0, 0.0)

		p1 = (int(end[0] - sv[0]*self.head_length + nv[0]*self.head_width/2),
			  int(end[1] - sv[1]*self.head_length + nv[1]*self.head_width/2))

		p2 = (int(end[0] - sv[0]*self.head_length - nv[0]*self.head_width/2),
			  int(end[1] - sv[1]*self.head_length - nv[1]*self.head_width/2))

		self.GetGC().FillPolygon( [(end[0],end[1]), p1, p2], 0, 0 )



